function M = hippo_main_axis_rotation(fn_img, fn_out, p1, p2, theta, interp)
% 
% FUNCTION hippo_main_axis_rotation
% PURPOSE  Rotate an image so that the principal axis (specified by
%          a pair of points) is along the z-axis
% USAGE:
%       hippo_main_axis_rotation(fn_img, fn_out, p1, p2, theta, interp)
% PARAMETERS:
%       fn_img              Image file name
%       fn_out              Output file name
%       p1, p2              Points defining the main axis (spacial coords)
%       theta               Rotation around the main axis
%       interp              Interpolation parameter
% OUTPUT:
%       The new matrix from image space to patient space

% Compute the point p3
v3 = cross(p2-p1, cross(p2-p1, [0 0 1]'));
v3 = v3 / norm(v3);
p3 = 0.5 * (p1 + p2) + v3;      

% a is half of the distance from p1 to p2
a = 0.5 * norm(p2 - p1);

% b is the distance from p1 to the point on (p1, p2) that's closest to p3
b = - dot(p2 - p1, p1 - p3) / dot(p2 - p1, p2 - p1)

% c is the distance from p3 to interval (p1, p2)
c = norm(p3 - (p1 + b * (p2 - p1)))

% Use procrustes to map the points. Remember, x is sagittal, y is coronal
% and z is axial
Q(1,:) = [0 -a 0];
Q(2,:) = [0 a 0];
Q(3,:) = [c * sin(theta), 2*a*b-a, c * cos(theta)];

[proc z tran] = procrustes(Q, [p1'; p2'; p3'])
Mp=eye(4);
Mp(1:3,1:3)=tran.T';
Mp(1:3,4)=tran.c(1,:)';
Mp

% Construct the final matrix   
[p1'; p2'; p3'] * tran.T + tran.c

g1 = tran.T' * p1 + tran.c(1,:)'

a1 = Mp*[p1;1]; a2 = Mp*[p2;1]; a3 = Mp*[p3;1];

fprintf('Point (%f,%f,%f) maps to (%f,%f,%f,%f)\n', p1, a1);
fprintf('Point (%f,%f,%f) maps to (%f,%f,%f,%f)\n', p2, a2);
fprintf('Point (%f,%f,%f) maps to (%f,%f,%f,%f)\n', p3, a3);
fprintf('Distance p1, p2 = %f / %f\n',norm(p1-p2), norm(a1-a2));
fprintf('Distance p1, p3 = %f / %f\n',norm(p1-p3), norm(a1-a3));
fprintf('Distance p2, p3 = %f / %f\n',norm(p2-p3), norm(a2-a3));

vol = spm_vol(fn_img);
M = Mp * vol.mat;

% Create an array of corners
K0(1,:) = [1 1 1];
K0(8,:) = vol.dim(1:3);
K0(2,:) = [K0(1,1) K0(1,2) K0(8,3)];
K0(3,:) = [K0(1,1) K0(8,2) K0(1,3)];
K0(4,:) = [K0(8,1) K0(1,2) K0(1,3)];
K0(5,:) = [K0(1,1) K0(8,2) K0(8,3)];
K0(6,:) = [K0(8,1) K0(1,2) K0(8,3)];
K0(7,:) = [K0(8,1) K0(8,2) K0(1,3)];
K0(:,4) = 1;

% Transform the corners to new space
K1 = (Mp * vol.mat * K0')';

% Look at the extents of the image in space coordinates
c1 = min(K1(:,1:3));
c2 = max(K1(:,1:3));
fprintf('Image extent : %f %f %f\n', c1, c2);
fprintf('Image size: %d %d %d\n', round((c2-c1) / 0.4));

% Create a copy of the image 
vcpy = vol; vcpy.fname = [tempname() '.img'];
spm_create_vol(vcpy);
spm_write_vol(vcpy, spm_read_vols(vol));

% Create a 'reference' image
vox = max(sqrt(sum(vcpy.mat(1:3,1:3).^2)));

vref.fname = [tempname() '.img'];
vref.dim = round([(c2-c1) / vox, vcpy.dim(4)]);
M2 = diag([1/vox 1/vox 1/vox 1]);
M2(1:3,4) = 0.5 * [1 1 1] .* vref.dim(1:3);
vref.mat = inv(M2);
vref.pinfo = vcpy.pinfo
vref = spm_create_vol(vref, 'noopen')
vref.mat

fprintf('p1 goes to image coord %f %f %f %f\n', inv(vref.mat) * Mp * [p1; 1]);
fprintf('p2 goes to image coord %f %f %f %f\n', inv(vref.mat) * Mp * [p2; 1]);
fprintf('p3 goes to image coord %f %f %f %f\n', inv(vref.mat) * Mp * [p3; 1]);

% Apply the transformation to the input data
flg_res.which=1; flg_res.interp=interp; flg_res.mean=0;
vcpy.mat = spm_get_space(vcpy.fname, Mp * vol.mat);
spm_reslice([vref, vcpy], flg_res);

% Copy the images
[pa1 fn1 ex1 ve1] = fileparts(vcpy.fname);
[pa2 fn2 ex2 ve2] = fileparts(fn_out);
copyfile(fullfile(pa1, [fn1 '.img']), fullfile(pa2, [fn2 '.img']), 'f');
copyfile(fullfile(pa1, [fn1 '.hdr']), fullfile(pa2, [fn2 '.hdr']), 'f');
copyfile(fullfile(pa1, [fn1 '.mat']), fullfile(pa2, [fn2 '.mat']), 'f');
copyfile(fullfile(pa1, ['r' fn1 '.img']), fullfile(pa2, ['r' fn2 '.img']), 'f');
copyfile(fullfile(pa1, ['r' fn1 '.hdr']), fullfile(pa2, ['r' fn2 '.hdr']), 'f');
copyfile(fullfile(pa1, ['r' fn1 '.mat']), fullfile(pa2, ['r' fn2 '.mat']), 'f');

hippo_delete_image(fullfile(pa1, ['r' fn1 '.img']));
hippo_delete_image(vref.fname);
hippo_delete_image(vcpy.fname);

M = inv(vref.mat) * (Mp * vol.mat);