function recon_sems(path, fn_analyze)
% Reconstructs SEMS file
% Usage:
%       recon_sems(path, fn_analyze)
% Parameters:
%       path        Pathname of the directory containing the 'fid' file
%       fn_analyze  Analyze filename where to save the output
%

% Get the dimensions of the image
np = read_procpar(path,'np') / 2;
nv = read_procpar(path,'nv');
ns = read_procpar(path,'ns');

% Get the FOV specification
lro = read_procpar(path,'lro');
lpe = read_procpar(path,'lpe');

% Get the slice positions
pss = read_procpar(path,'pss');

% Check if pss is sorted and if not sort it
[pss ipss] = sort(pss);

% Compute the voxel size in mm
vox(1) = 10 * lro / np;
vox(2) = 10 * lpe / nv;
vox(3) = 10 * (pss(end) - pss(1)) / (ns - 1);
vox(4) = 1;
fprintf('Voxel Size: (%d, %d, %d)\n',vox(1:3));
fprintf('Dimensions: %d %d %d\n', np/2, nv, ns);

% Read the FID file
[RE,IM,NP,NB,NT,HDR]=read_fid(path);
NP=np; NB=nv; NT=ns;
fprintf('Read FID: NP = %d, NB = %d, NT = %d\n',NP,NB,NT);

% Create the output image
Vi.fname = fn_analyze;
Vi.dim = [NP NB NT 4];
Vi.mat = diag(vox);
Vi.pinfo = [1;0;0];
Vo = spm_create_vol(Vi);

% Allocate data for output
Y = zeros(NP,NB,NT);
ZR = zeros(NP,NB,NT);
ZI = zeros(NP,NB,NT);

% Loop over the slices
for i = 1:NT
    
    % Get the slice of the K-space
    j = ipss(i);
    kslice = complex(RE(:,j:NT:NT*NB),IM(:,j:NT:NT*NB));
    
    % Apply baseline correction to the slice (why?)
    kslice = baseline_correction(kslice, 8);
    
    % Compute the FFT
    islice = fftshift(fft2(fftshift(kslice)));
    
    % Store the slice in Y
    Y(:,:,i) = abs(islice);
    ZR(:,:,i) = real(islice);
    ZI(:,:,i) = imag(islice);
    
    % Display the slice
    % imagesc(abs(islice));
    % colormap('gray');
    % axis image;
end

% Create a volume for the complex data
Vci = Vi;
Vci.fname = hippo_filename(Vi.fname,'p','re_');
Vco = spm_create_vol(Vci);
spm_write_vol(Vco, ZR);
Vci.fname = hippo_filename(Vi.fname,'p','im_');
Vco = spm_create_vol(Vci);
spm_write_vol(Vco, ZI);

% Save the output image
spm_write_vol(Vo, Y);
