/*!
 * \file   SbiaOdvbaAlgorithm.h
 * \brief  Implements Optimally-Discriminative Voxel-Based Analysis (ODVBA).
 * \author Tianhao Zhang, revised by Andreas Schuh
 * \date   12/9/2010
 *
 * $Revision: 143 $
 * $Id: SbiaOdvbaAlgorithm.h 143 2011-02-08 18:17:30Z schuha@UPHS.PENNHEALTH.PRV $
 *
 * <b>Last changed:</b>
 * $Author: schuha@UPHS.PENNHEALTH.PRV $
 * $Date: 2011-02-08 13:17:30 -0500 (Tue, 08 Feb 2011) $
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * For copyright information see Copyright section of project
 * ReadMe.html in the project's root directory.
 *
 * Contact: sbia-software@uphs.upenn.edu
 */

#pragma once
#ifndef SbiaOdvbaAlgorithm_h
#define SbiaOdvbaAlgorithm_h


#include <cv.h>
#include <boost/any.hpp>

#include <boost/numeric/ublas/matrix.hpp> 
#include <boost/numeric/ublas/matrix_proxy.hpp>
#include <boost/numeric/ublas/vector.hpp> 
#include <boost/numeric/ublas/vector_proxy.hpp> 
#include <boost/numeric/ublas/operation.hpp> 
#include <boost/numeric/ublas/operation_sparse.hpp> 
#include <boost/numeric/ublas/matrix_sparse.hpp> 
#include <boost/numeric/ublas/vector_sparse.hpp> 
#include <boost/numeric/ublas/symmetric.hpp>

#include <boost/numeric/ublas/io.hpp> 

#include <boost/numeric/bindings/atlas/cblas3.hpp>
#include <boost/numeric/bindings/atlas/cblas2.hpp>
#include <boost/numeric/bindings/traits/ublas_matrix.hpp>

#include "SbiaOdvbaConfig.h" // Keep as last include statement
                             // (document reason(s) for exceptions)!


SBIA_ODVBA_NAMESPACE_BEGIN


//////////////////////////////////////////////////////////////////////////////
// Data types
//////////////////////////////////////////////////////////////////////////////

/*!
 * \struct Database
 * \brief  Brief comment about data structures purpose.
 *
 * \The Database includes the necessary inputs, e.g., the input data(m x n), the neihghborhoods(NE x nn), the index(non-zeros voxels), and n1(number of group1);
 */
struct Database
{
	int    n1;    ///< Number of subjects belonging to group 1.
	CvMat *X;     ///< Voxel data per subject where the first n1 subjects belong
	              ///< to group 1 and the remaining n2 = n - n1 subjects belong to group 2.
	CvMat *NI;    ///< the matrix to store the neighorhoods (nn x NE or NE x nn check!)
	CvMat *index; ///< the vector to store the index of template (1 x m)

	/*!
	 * \brief Default constructor.
	 */
	Database ();

	/*!
	 * \brief Destrutor.
	 *
	 * \note The destructor does NOT release any data. Call method
	 *       release () instead manually if the data should be released.
	 *
	 * \see release ()	 
	 */
	~Database ();

	/*!
	 * \brief Release assigned data.
	 */	 	
	void release ();

private:

	/*!
	 * \brief Copy constructor.
	 *
	 * \note Intentionally not implemented!
	 *
	 * \param [in] other Other instance.
	 */
	Database (const Database &other);

	/*!
	 * \brief Assignment operator.
	 *
	 * \note Intentionally not implemented!
	 *
	 * \param [in] rhs Right-hand side.
	 */
	Database &operator= (const Database &rhs);

}; // struct Database

/*!
 * \struct Options
 * \brief  Parameters of ODVBA method.
 */
struct Options
{
	double gamma; ///< Gamma in Eq. (4) of the MICCAI paper.
	double phi;   ///< Phi in Eq. (11), the discrimination degree, of the MICCAI paper.
	              ///< It is a tuning parameter which aims at reducing the number of outliers.
	int    nPerm; ///< Number of permutations tested during permutation test.
	CvMat *perms; ///< Matrix of permutations. Each row represents one permutation of the
	              ///< numbers [1..n], where n is the number of subjects. Only used for
	              ///< regression testing as here deterministic results are required
	              ///< which can be compared to the expected results.
	              ///< If nPerm is zero, the specified pre-computed permutations are
	              ///< are used for the group analysis. Otherwise, nPerm random permutations
	              ///< are generated during the group analysis and stored in this matrix.
	              ///< Hence, in the latter case, the matrix must have nPerm rows.

	/*!
	 * \brief Default constructor which sets the default values of the parameters.
	 */
	Options ()
	{
		gamma = 1.0e-6;
		phi   = 1;
		nPerm = 100;
		perms = NULL;
	}

	/*!
	 * \brief Destructor.
	 */
	~Options ()
	{
		if (perms)
		{
			cvReleaseMat (&perms);
			perms = NULL;
		}
	}

}; // struct Options

/*!
 * \struct NDPData
 * \brief  Data used by function NDP.
 *
 * By defining this data structure and instantiating it only once, the running
 * time of the ODVBA method is improved due to the avoidance of unnecessary
 * (de-)allocations of temporary memory.
 *
 * \see NDP ()
 */
struct NDPData
{
	boost::numeric::ublas::matrix<double> M;          ///< Temporary matrix with columns of data belonging to group 1/2 only.
	boost::numeric::ublas::matrix<double> MT;         ///< Transpose of M.
	boost::numeric::ublas::vector<double> e;          ///< Used for mean computation using matrix multiplication, i.e.,
		                                              ///< M * [1/n 1/n ...]' = [meanOfRow1 meanOfRow2 ...]'
	boost::numeric::ublas::vector<double> intraMean1; ///< Mean of voxel values of group 1.
	boost::numeric::ublas::matrix<double> intraCov1;  ///< Covariance of voxel values of group 1.
	boost::numeric::ublas::vector<double> intraMean2; ///< Mean of voxel values of group 2.
	boost::numeric::ublas::matrix<double> intraCov2;  ///< Covariance of voxel values of group 2.
	CvMat                                *A_rep;      ///< gamma * S_W + S_B, see MICCAI paper.
	CvMat                                *Eigvector;  ///< Eigenvectors of A_rep.
	CvMat                                *Eigvalue;   ///< Eigenvalues of A_rep.
	CvMat                                *A;          ///< Positive definite matrix A defined in MICCAI paper.
	boost::numeric::ublas::matrix<double> Ap;         ///< A^+, positive elements of A.
	boost::numeric::ublas::matrix<double> An;         ///< A^-, negative negative elements of A.
	boost::numeric::ublas::vector<double> a;
	boost::numeric::ublas::vector<double> c;
	boost::numeric::ublas::vector<double> w;          ///< Result of NDP.

	/*!
	 * \brief Contructor which initializes the data structure.
	 *
	 * \param [in] m Number of voxels per subject. If zero is passed as argument,
	 *               the data structure remains uninitialized and has to be
	 *               initialized manually (done by the function NDP).
	 */
	NDPData (const int m = 0);

	/*!
	 * \brief Destructor.
	 */
	~NDPData ();

private:

	/*!
	 * \brief Copy constructor.
	 *
	 * \note Intentionally not implemented!
	 *
	 * \param [in] other Other instance.
	 */
	NDPData (const NDPData &other);

	/*!
	 * \brief Assignment operator.
	 *
	 * \note Intentionally not implemented!
	 *
	 * \param [in] rhs Right-hand side.
	 */
	NDPData &operator= (const NDPData &rhs);

}; // struct NDPData

/*!
 * \struct CompMapData
 * \brief  Data used by function compMap.
 *
 * By defining this data structure and instantiating it only once, the running
 * time of the ODVBA method is improved due to the avoidance of unnecessary
 * (de-)allocations of temporary memory.
 *
 * \see compMap ()
 */
struct CompMapData
{
	boost::numeric::ublas::matrix<double> theta;  ///< Learning set constructed by the subvolume vectors from all subjects.
	boost::numeric::ublas::matrix<double> thetaT; ///< Transpose of theta.
	boost::numeric::ublas::vector<double> weight; ///< Projected learning set using projection direction obtained via NDP.
	CvMat                                *delta;  ///< Discriminative degree for each voxel and each neighborhood.
	NDPData                               ndp;    ///< Temporary data used by NDP function.
	CvMat                                *W;      ///< Stores the coffients for all learning sets.

	/*!
	 * \brief Constructor which initializes the data structures.
	 *
	 * If any passed argument is zero, the data structure remains
	 * uninitialized and has to be initialized manually
	 * (done by the function compMap).
	 *
	 * \param [in] m Number of voxels per subject (i.e., voxels in template).
	 * \param [in] n Number of subjects.
	 * \param [in] N Number of neighborhoods.
	 * \param [in] k Number of voxels per neighborhood.
	 */
	CompMapData (const int m = 0,
	             const int n = 0,
	             const int N = 0,
	             const int k = 0);

	/*!
	 * \brief Destructor.
	 */
	~CompMapData ();

private:

	/*!
	 * \brief Copy constructor.
	 *
	 * \note Intentionally not implemented!
	 *
	 * \param [in] other Other instance.
	 */
	CompMapData (const CompMapData &other);

	/*!
	 * \brief Assignment operator.
	 *
	 * \note Intentionally not implemented!
	 *
	 * \param [in] rhs Right-hand side.
	 */
	CompMapData &operator= (const CompMapData &rhs);

}; // struct CompMapData

//////////////////////////////////////////////////////////////////////////////
// Non-negative Discriminative Projection (NDP)
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Non-negative Discriminative Projection (NDP).
 *
 * Call this function with each learning set to get the matrix W.
 *
 * \see Section 2.1 of the MICCAI paper.
 *
 * \see compMap ()
 *
 * \param [in] theta An input learning set constructed from the neighborhoods
 *                   centered at a particular voxel. Each column of the matrix
 *                   corresponds to a subject, where the first n1 subjects belong
 *                   to group 1 and the remaining n2 = n - n1 subjects to group 2.
 *                   The columns of the learning set further correspond to the
 *                   subvolume vectors which were constructed by randomly sampling
 *                   the voxels neighborhood within the subject's original image.
 * \param [in] n1    The number of subjects belonging to group 1.
 * \param [in] gamma Gamma in the discrimination degree defined in the Eq. (4)
 *                   of the MICCAI paper.
 * \param [out] tmp  Temporary data/Previously allocated memory which is reused
 *                   to improve the performance of the ODVBA method.
 *
 * \return The vector w which stores the coffients for the input learning set
 *         (note that the result is also stored in tmp.w).
 */
boost::numeric::ublas::vector<double> NDP (const boost::numeric::ublas::matrix<double> &theta,
                                           const int                    n1,
                                           const double                 gamma,
                                           NDPData                     &tmp);

//////////////////////////////////////////////////////////////////////////////
// Determination of each voxel's statistic
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Determine statistics of each voxel using NDP.
 *
 * \see Section 2.2 of the MICCAI paper.
 *
 * \see NDP ()
 *
 * \param [in]  data     Input database including original image data,
 *                       template mask (index) and the matrix specifying the
 *                       pre-computed neighborhoods.
 * \param [in]  opt      Parameters of the ODVBA method.
 * \param [out] map      Map which stores the determined statistics of each voxel.
 *                       Has to be allocated by caller using cvCreateMat (m, 1, CV_32FC1),
 *                       where m is the number of voxels per subject, i.e., the number
 *                       of non-zero voxels in the template.
 * \param [out] tmp      Temporary memory re-used for different executions of compMap.
 *                       This improves the performance of the ODVBA method by
 *                       avoiding unnecessary (de-)allocations of temporary memory.
 */
void compMap (const Database &data, const Options &opt, CvMat *map, CompMapData &tmp);

//////////////////////////////////////////////////////////////////////////////
// Permutation tests
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Perform ODVBA to get statistical value using permutation tests.
 *
 * \see Section 2.3 of the MICCAI paper.
 *
 * \see compMap ()
 *
 * \param [in] data      The original data: number of voxels per subject x number of subjects.
 * \param [in] opt       Parameters of the ODVBA method.
 * \param [in] initial   Whether to compute initial results first using
 *                       non-permuted input data.
 * \param [in] verbosity Verbosity of messages. If less or equal to zero,
 *                       no messages are printed to stdout. Otherwise, the
 *                       higher the verbosity level the more verbose messages
 *                       are generated. Errors and warnings are printed to
 *                       stderr in any case.
 * \param [in] msgPrefix String which is prefixed each printed message.
 *                       No whitespace is printed between the prefix and the
 *                       actual message. Include whitespace in prefix if desired.
 *
 * \return map of statistic values
 *         Caller is responsible for releasing the memory using cvReleaseMat ().
 */
CvMat *performAnalysis (const Database &data,
                        const Options  &opt,
                        bool            initial   = true,
                        int             verbosity = 0,
                        const char     *msgPrefix = "");

//////////////////////////////////////////////////////////////////////////////
// p-value / p-image
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Get final map of p-values from results of permutation tests.
 *
 * \see permutationTests ()
 *
 * \param [in] maps Result of permutation tests.
 *
 * \return Map of p-values or NULL on error.
 *         Caller is responsible for releasing the memory using cvReleaseMat ().
 */
CvMat *getPValue (CvMat *maps);

/*!
 * \brief Creates the image of p-values.
 *
 * \see permutationTests ()
 *
 * \param [in] maps     Result of permutation tests.
 * \param [in] index    Indices of non-zero template voxels.
 * \param [in] nVoxels  Number of voxels in original input images. 
 * \param [in] pValueIn Calculated p-values (optional).
 *                      If NULL, the p-values are implicitly calculated using getPValue ().
 *
 * \return Image of p-values of size nVoxels or NULL on error.
 *         Caller is responsible for releasing the memory using cvReleaseMat ().
 */
CvMat *getPImage (CvMat *maps, CvMat *index, int nVoxels, CvMat *pValueIn = NULL);


SBIA_ODVBA_NAMESPACE_END


#endif // SbiaOdvbaAlgorithm_h
