#! /bin/bash

# \file   OdvbaMPIOnSGE.sh
# \brief  Bash script used to submit SGE job for parallel execution of OdvbaMPI.
# \author Andreas Schuh
# \date   01/31/11
#
# $Revision: 138 $
# $Id: OdvbaMPIOnSGE.sh 138 2011-02-03 04:29:05Z schuha@UPHS.PENNHEALTH.PRV $
#
# <b>Last changed:</b>
# $Author: schuha@UPHS.PENNHEALTH.PRV $
# $Date: 2011-02-02 23:29:05 -0500 (Wed, 02 Feb 2011) $
#
# The variable NSLOTS used by this script is set by SGE to match the number
# of slots requested using the -pe option. A specification of the hosts
# is not required. For testing purposes, however, set the variable NSLOTS to
# the number of slots and the variable MYMACHINES to the complete option needed
# to specify the machines for mpiexec manually prior to executing this script,
# as in (BASH syntax)
#
#  MYMACHINES="-machinefile ~/mymachines" NSLOTS=42 SbiaExecOdvbaMPI.sh ...
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# For copyright information see Copyright section of project
# ReadMe.html in the project's root directory.
#
# Contact: sbia-software@uphs.upenn.edu

# ////////////////////////////////////////////////////////////////////////////
# SGE
# ////////////////////////////////////////////////////////////////////////////

# SGE will read options that are treated by the shell as comments. The SGE
# parameters must begin with the characters "#$", followed by the option.
# An option given as "##$" is hence disabled.
#
# Environment:
#$ -S /bin/bash # shell used under SGE
#$ -cwd         # run command in current working directory
#
# Save the output:
##$ -o OdvbaMPI.$JOB_ID.log
##$ -e OdvbaMPI.$JOB_ID.log
#
# Email notifications:
##$ -M please_specify_on_command_line # Email address
#$ -m b # Send mail at the beginning of the job
#$ -m e # Send mail at the end of the job
#$ -m a # Send mail in case the job is aborted
#
# Parallel environment:
#$ -pe openmpi 6 # Default number of CPUs requested. Give this option with a
#                # different number on the command line in order to overwrite
#                # the default.
#$ -l openmpi    # Request access to the "openmpi" resource.

##############################################################################
# variables
##############################################################################

progName=${0##*/} # name of this script

version_major=1   # major version number
version_minor=0   # minor version number
version_patch=0   # version patch number

# version string
version="$version_major.$version_minor.$version_patch"

verbosity=0       # verbosity of output messages
odvba=OdvbaMPI    # name of ODVBA exectuable
args=""           # arguments for ODVBA executable

# path to ODVBA exectuable
bin="@PROJECT_EXECUTABLE_DIR@"

##############################################################################
# functions
##############################################################################

# ////////////////////////////////////////////////////////////////////////////
# version / usage / help
# ////////////////////////////////////////////////////////////////////////////

# \brief Prints version information
version ()
{
	echo "$progName $version [using `$bin/$odvba --version`]"
}

# \brief Prints usage information
usage ()
{
	version
	echo
	echo "Description:"
	echo "  Script used to run $odvba as parallel job on SGE."
	echo "  For a description of $odvba, see below."
	echo
	echo "Usage:"
	echo "  qsub [qsub options] $progName [options of $odvba]"
	echo
	echo "Common qsub options:"
	echo "  [-pe openmpi <int>]    : Specify number of CPUs to be requested."
	echo "  [-M me@uphs.upenn.edu] : Email address for email notifications."
	echo "                           Note that the script enables notifications"
	echo "                           on begin, end, and abortion already."
	echo
	echo "Example:"
	echo "  qsub -pe openmpi 10 $progName [options of $odvba]"
	echo
	echo
	$bin/$odvba -u
}

# \brief Prints help.
help ()
{
	usage
}

# ////////////////////////////////////////////////////////////////////////////
# options
# ////////////////////////////////////////////////////////////////////////////

options ()
{
	for arg in $@
	do
		case "$arg" in
			-u|--usage)
				usage
				exit 0
				;;
			-h|--help)
				help
				exit 0
				;;
			-V|--version)
				version
				exit 0
				;;
		esac
	done
}

##############################################################################
# main
##############################################################################

# ----------------------------------------------------------------------------
# load required modules
# ----------------------------------------------------------------------------

. /usr/share/Modules/init/bash

if [ $? != 0 ]
then
	echo "Failure to load /usr/share/Modules/init/bash"
	exit 1
fi

module load openmpi/1.3.3

if [ $? != 0 ]
then
	echo "Failed to load module openmpi/1.3.3"
	exit 1
fi

# ----------------------------------------------------------------------------
# parse options
# ----------------------------------------------------------------------------

# parse options to overwrite usage, help, and version information
# \note This has to be done after the Open MPI module was loaded!
options $*

# ----------------------------------------------------------------------------
# default output to enable identification of job in output files
# ----------------------------------------------------------------------------

echo "Command: $progName"
echo "Arguments: $*"
echo -e "Executing in: \c"; pwd
echo -e "Executing on: \c"; hostname
echo -e "Executed at:  \c"; date
echo "----- STDOUT from $progName below this line -----"

echo "Command: $progName" 1>&2
echo "Arguments: $*" 1>&2
(echo -e "Executing in: \c"; pwd) 1>&2
(echo -e "Executing on: \c"; hostname) 1>&2
(echo -e "Executed at:  \c"; date) 1>&2
echo "----- STDERR from $progName below this line -----" 1>&2

# ----------------------------------------------------------------------------
# run ODVBA
# ----------------------------------------------------------------------------

# \note $OPENMPI is set by the "openmpi/1.3.3" module
# \note $NSLOTS is set by SGE to match the number of slots requested
# \note $MYMACHINES is only used for script testing purposes
# \note The option "--mca orte_forward_job_control 1" enables jobs to be
#       suspended and restarted by SGE depending on the load on the server.
echo "Executing $bin/$odvba"
mpiexec --prefix $OPENMPI -x PATH="$PATH" -x LD_LIBRARY_PATH="$LD_LIBRARY_PATH" -np $NSLOTS $MYMACHINES --mca orte_forward_job_control 1 $bin/$odvba $*
