/*!
 * \file  CreateIndex.cpp
 * \brief Command line tool which creates the index file (template).
 *
 * For copyright information please see Copyright.txt in the root
 * directory of the project.
 *
 * Contact: SBIA Group <sbia-software@uphs.upenn.edu>
 */

#include <getopt.h>
#include <stdio.h>

#include "SbiaOdvbaUtilities.h"

#include "MainAux.h"


using namespace sbia::odvba;


//////////////////////////////////////////////////////////////////////////////
// usage / help
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Prints usage information / help.
 *
 * \param progName Name of program.
 */
void usage (const char *progName)
{
	version (progName);
	printf ("\n");
	printf ("Description:\n");
	printf ("  This program determines the indices of voxels in the set of input images\n");
	printf ("  whose average voxel value exceeds a certain threshold. Hence, it determines\n");
	printf ("  the indices of voxels relevant to the group analysis and outputs them to a\n");
	printf ("  text file which is used as input to the main program which performs the\n");
	printf ("  group analysis.\n");
	printf ("  \n");
	printf ("  T. Zhang and C. Davatzikos; ODVBA: Optimally-Dicriminative ");
	printf ("  Voxel-Based Analysis\n");
	printf ("\n");
	printf ("Usage:\n");
	printf ("  %s [options] <subjects.txt>\n", progName);
	printf ("\n");
	printf ("Required options:\n");
	printf ("  <subjects.txt> : The subjets list which specifies the input data given as NIfTI-1 images.\n");
	printf ("\n");
	printf ("Options:\n");
	printf ("  [-o --out <filename>] : Filename of the output index file.\n");
	printf ("                          The default value is 'index.txt'.\n");
	printf ("  [-h --help]           : Print help and exit.\n");
	printf ("  [-u --usage]          : Print usage information and exit.\n");
	printf ("  [-V --version]        : Print version information and exit.\n");
	printf ("  [-v --verbose]        : Enable verbose messages. Can be specified multiple times\n");
	printf ("                          to increase the verbosity.\n");
	printf ("\n");
	printf ("Example:\n");
	printf ("  %s subjects.txt -o index.txt\n", progName);
}

//////////////////////////////////////////////////////////////////////////////
// main
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Main function of program.
 *
 * \param [in] argc Number of command line arguments.
 * \param [in] argv Command line arguments.
 *
 * \return Exit status of program.
 *
 * \retval EXIT_SUCCESS on success.
 * \retval EXIT_FAILURE on failure.
 */
int main (int argc, char *argv[])
{
	const char *progName = getProgName (argv);
	bool        ok       = true;

	static struct option long_options [] =
	{
		{"out",     required_argument, NULL, 'o'},
		{"usage",   no_argument,       NULL, 'u'},
		{"help",    no_argument,       NULL, 'h'},
		{"version", no_argument,       NULL, 'V'},
		{"Version", no_argument,       NULL, 'V'},
		{"verbose", no_argument,       NULL, 'v'},
        {0, 0, 0, 0}
	}; // struct long_options

	int c      = -1;
	int optidx = 0;

	// default options
	int         verbosity   = 0;           // verbosity of messages
	const char *sublistFile = NULL;        // subjects list file
	const char *indexFile   = "index.txt"; // index file

	while ((c = getopt_long (argc, argv, "o:uhVv", long_options, &optidx)) != -1)
	{
		switch (c)
		{
		case 'o':
			indexFile = optarg;
			break;

		case 'u':
			// fall-through intended
		case 'h':
			usage (progName);
			exit (EXIT_SUCCESS);

		case 'V':
			version (progName);
			exit (EXIT_SUCCESS);

		case 'v':
			++ verbosity;
			break;

		case '?':
			// getopt_long already printed an error message
			usage (progName);
			exit (EXIT_FAILURE);
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1)
	{
		usage (progName);
		exit (EXIT_FAILURE);
	}

	sublistFile = argv [0];

	// read input data
	CvMat *data = NULL;

	printf ("Parsing subject list and reading image data\n");
	fflush (stdout);

	ok = ((data = readData  (sublistFile)) != NULL);

	// create index
	CvMat *index = NULL;

	if (ok)
	{
		printf ("Generating index\n");
		fflush (stdout);

		ok = (index = createIndex (data)) != NULL;
	}

	// write index
	if (ok)
	{
		printf ("Writing index to file '%s'\n", indexFile);
		fflush (stdout);

		ok = writeCvMat (indexFile, index, "%.0f");
	}

	// clean up
	if (index)
	{
		cvReleaseMat (&index);
		index = NULL;
	}

	if (data)
	{
		cvReleaseMat (&data);
		data = NULL;
	}

	exit (ok ? EXIT_SUCCESS : EXIT_FAILURE);
}
