/*!
 * \file  SbiaOdvbaUtilities.cpp
 * \brief Common utility functions used by the ODVBA implementations.
 *
 * For copyright information please see Copyright.txt in the root
 * directory of the project.
 *
 * Contact: SBIA Group <sbia-software@uphs.upenn.edu>
 */

#include "SbiaOdvbaUtilities.h"

#include <fstream>
#include <algorithm>
#include <string>


namespace sbia
{

namespace odvba
{


//////////////////////////////////////////////////////////////////////////////
// system
//////////////////////////////////////////////////////////////////////////////

/*!
 * \brief Extract directory path from filename.
 *
 * \param [in] filename Path of file.
 *
 * \return Directory part of filename.   
 */
std::string getFileDirectory (const char *filename)
{
	std::string            dir (filename);
	std::string::size_type pos = dir.find_last_of ("/\\");

	return pos == std::string::npos ? "." : dir.substr (0, pos);
}

/*!
 * \brief Get absolute path.
 *
 * \param [in] root     Root directory (used for relative paths).
 * \param [in] filename Relative or absolute path of file or directory.
 *
 * \return Absolute path of file or directory.
 */ 
std::string getAbsPath (const char *root, const char *filename)
{
	if (filename [0] == '/') return filename;
	else return std::string (root) + '/' + filename; 
}

//////////////////////////////////////////////////////////////////////////////
// IO
//////////////////////////////////////////////////////////////////////////////

// ---------------------------------------------------------------------------
// read input datasets
// ---------------------------------------------------------------------------

/*!
 * \brief Validate NIfTI image attributes.
 *
 * \see readData ()
 *
 * \param [in] nim Input NIfTI image.
 * \param [in] ref Reference NIfTI image, e.g., first input image.
 *
 * \return Whether or not the input image nim is valid.
 */
bool checkImage (const nifti_image * const nim, const nifti_image * const ref)
{
	if (!nim) return false;

	// check image attributes independent of reference image
	//
	// \note Example images generated by resampleImage.py at SBIA had
	//       ndim == 3 even though they should have nim->ndim == 4?
	//       Anyways, the check of nx, ny, nz, and nt should be sufficient.
	if (/*nim->ndim != 4 ||*/nim->nx <= 1 || nim->ny <= 1 || nim->nz <= 1 || nim->nt != 1) return false;

	if (nim->datatype != DT_FLOAT && nim->datatype != DT_SIGNED_SHORT) return false;

	// no reference image given, so image is ok
	if (!ref) return true;

	// compare to reference image
	if (nim->nx != ref->nx || nim->ny != ref->ny || nim->nz != ref->nz) return false;

	// \todo Add more checks, e.g., orientation, pixel dimensions...

	return true;
}

/****************************************************************************/
CvMat *readData (const char *filename, int *n1, nifti_1_header *hdr1)
{
	char str [1024]; // temporary memory to read strings from file

	// open subject list file
	FILE *fpList = fopen (filename, "r");

	if (!fpList)
	{
		fprintf (stderr, "Failed to open subject list '%s'\n", filename);
		return NULL;
	}

	// parse header
	int         nSubjects1 = 0; // number of subjects in group 1
	int         nSubjects2 = 0; // number of subjects in group 2
	std::string dataDir;        // default data root directory

	if (fscanf (fpList, "%d %d", &nSubjects1, &nSubjects2) != 2 || nSubjects1 < 0 || nSubjects2 < 0)
	{
		fprintf (stderr, "Failed to parse subject list '%s': number of subjects\n", filename);
		return NULL;
	}

	if (fscanf (fpList, "%s", &str) != 1)
	{
		fprintf (stderr, "Failed to parse subject list '%s': data root\n", filename);
		return NULL;
	}

	dataDir = getAbsPath (getFileDirectory (filename).c_str (), str);

	// process input images
	std::string  dataFile;  // path to data file (NIfTI-1 header file)
	nifti_image *nim1;      // image attributes of first input image
	                        // used to check whether other input images
	                        // have identical attributes

	// read header information of first input image
	if (fscanf (fpList, "%s", &str) != 1)
	{
		fprintf (stderr, "Failed to parse subject list '%s': image filename\n", filename);
		return NULL;
	}

	dataFile = getAbsPath (dataDir.c_str (), str);
	nim1     = nifti_image_read (dataFile.c_str (), 0);

	if (!nim1)
	{
		fprintf (stderr, "Failed to read NIfTI-1 header of file '%s'\n", dataFile.c_str ());
		return NULL;
	}

	if (!checkImage (nim1, NULL))
	{
		fprintf (stderr, "Invalid or unsupported NIfTI-1 image file '%s'. Only DT_SIGNED_SHORT and DT_FLOAT supported.\n", dataFile.c_str ());
		nifti_image_free (nim1);
		return NULL;
	}

	// allocate matrix for output data
	CvMat *data = cvCreateMat (nim1->nx * nim1->ny * nim1->nz, nSubjects1 + nSubjects2, CV_32FC1);

	if (!data)
	{
		fprintf (stderr, "Failed to allocate memory\n");
		nifti_image_free (nim1);
		return NULL;
	}

	// read data of each subject
	bool         ok  = true; // whether everything is fine yet
	nifti_image *nim = NULL; // current input NIfTI image

	for (int i = 0; ok && i < data->cols; ++ i)
	{
		// header of first input image was already read
		if (i == 0)
		{
			nim = nim1;
		}
		// read image header
		else
		{
			if (fscanf (fpList, "%s", &str) != 1)
			{
				fprintf (stderr, "Failed to parse subject list '%s': image filename\n", filename);
				ok = false;
				break;
			}

			dataFile = dataDir + '/' + str;

			// read NIfTI-1 image header
			nim = nifti_image_read (dataFile.c_str (), 0);
	
			if (!nim)
			{
				fprintf (stderr, "Failed to open image file '%s'\n", dataFile.c_str ());
				ok = false;
				continue;
			}

			// verify image attributes
			if (!checkImage (nim, nim1))
			{
				fprintf (stderr, "Attributes of image '%s' differ from attributes of first image\n", dataFile.c_str ());
				ok = false;
				continue;
			}
		}

		// read image data (optionally compressed)
		znzFile fpData = nifti_image_open (dataFile.c_str (), "rb", &nim);

		if (znz_isnull (fpData))
		{
			fprintf (stderr, "Failed to open data file of image '%s'\n", dataFile.c_str ());
			ok = false;
		}

		// get image offset: a negative offset means to figure from end of file
		int ioff = 0;

		if (ok)
		{
			if (nim->iname_offset < 0)
			{
				if (nifti_is_gzfile (nim->iname))
				{
					fprintf (stderr, "Invalid offset for compressed data file of image '%s'\n", dataFile.c_str ());
					ok = false;
				}
				else
				{
					int ii = nifti_get_filesize (nim->iname);

					if (ii <= 0)
					{
						fprintf (stderr, "Data file of image '%s' is empty\n", dataFile.c_str ());
						ok = false;
					}
					else
					{
						int ntot = static_cast <int> (nifti_get_volsize (nim));
						ioff = ((ii > ntot) ? (ii - ntot) : 0);
					}
				}
			}
			else
			{
				ioff = nim->iname_offset;
			}
		}

		// seek to the appropriate read position
		if (ok && znzseek (fpData, static_cast <long> (ioff), SEEK_SET) < 0)
		{
			fprintf (stderr, "Failed to seek to offset %u of data file of image '%s'\n", static_cast <unsigned int> (ioff), dataFile.c_str ());
			ok = false;
		}

		// read image data
		if (nim->datatype == DT_SIGNED_SHORT)
		{
			short tmp;

			for (int k = 0; ok && k < data->rows; ++ k)
			{
				if (nifti_read_buffer (fpData, &tmp, sizeof (short), nim) == sizeof (short))
				{
					cvSetReal2D (data, k, i, static_cast <double> (tmp));
				}
				else
				{
					fprintf (stderr, "Failed to read data of image '%s'\n", dataFile.c_str ());
					ok = false;
				}
			}
		}
		else if (nim->datatype == DT_FLOAT)
		{
			float tmp;

			for (int k = 0; ok && k < data->rows; ++ k)
			{
				if (nifti_read_buffer (fpData, &tmp, sizeof (float), nim) == sizeof (float))
				{
					cvSetReal2D (data, k, i, static_cast <double> (tmp));
				}
				else
				{
					fprintf (stderr, "Failed to read data of image '%s'\n", dataFile.c_str ());
					ok = false;
				}
			}
		}
		else
		{
			fprintf (stderr, "Image '%s' has unsupported voxel type. Only DT_SIGNED_SHORT and DT_FLOAT are supported.\n", dataFile.c_str ());
			ok = false;
		}

		// clean up
		if (!znz_isnull (fpData))
		{
			znzclose (fpData);
		}

		if (nim && nim != nim1)
		{
			nifti_image_free (nim);
			nim = NULL;
		}
	}

	fclose (fpList);

	// normalize data
	if (ok) normData (data, 100);

	// other return values
	if (n1)   *n1   = nSubjects1;
	if (hdr1) *hdr1 = nifti_convert_nim2nhdr (nim1);

	// clean up
	nifti_image_free (nim1);

	if (!ok)
	{
		cvReleaseMat (&data);
		data = NULL;
	}

    return data;
}

/****************************************************************************/
void normData (CvMat *data, const int meanValue)
{
	// calculate normalization factor
	double norm = meanValue / cvAvg (data).val [0];

	// normalize data in-place
	for (int i = 0; i < data->rows ; ++ i)
	{
		for (int j = 0; j < data->cols; ++ j)
		{
			cvSetReal2D (data, i, j, norm * cvGetReal2D (data, i, j));
		}
	}
}

// ---------------------------------------------------------------------------
// read/write OpenCV matrix
// ---------------------------------------------------------------------------

/****************************************************************************/
CvMat *readCvMat (const char *filename, bool hdr)
{
	bool ok   = true;
	int  rows = 0;
	int  cols = 0;

	// open file
	FILE *fp = fopen (filename, "r");

	if (fp == NULL)
	{
		fprintf (stderr, "Failed to open file '%s'\n", filename);
		return NULL;
	}

	// parse header to extract matrix dimensions (if applicable)
	if (hdr)
	{
		if (fscanf (fp, "%d %d", &rows, &cols) != 2 || rows < 0 || cols < 0)
		{
			fprintf (stderr, "Failed to parse header of file '%s': matrix dimensions\n", filename);
			ok = false;
		}
	}
	// otherwise scan file first to determine matrix dimensions
	else
	{
		int  c;
		int  cnt  = 0;
		bool empty = true; // flag used to detect empty lines 
		bool skip  = true; // flag used to skip whitespaces (init with true to allow whitespaces before first entry)

		while (ok && (c = fgetc (fp)) != EOF)
		{
			if (c == '\n')
			{
				if (empty)
				{
					// skip empty line
				}
				else
				{
					++ rows;
					++ cnt;

					if (rows > 1)
					{
						if (cnt != cols)
						{
							fprintf (stderr, "Number of columns in row %d differs from first row\n", rows);
							ok = false;
						}
					}
					else cols = cnt;

					cnt   = 0;
					empty = true;
					skip  = true;
				}
			}
			else if (c == '\r')
			{
				// ignore it, '\n' will be next...
			}
			else if (c == ' ' || c == '\t')
			{
				if (!skip)
				{
					++ cnt;
					skip = true;
				}
			}
			else
			{
				empty = false;
				skip  = false;
			}
		}

		if (ferror (fp) || rows == 0 || cols == 0)
		{
			fprintf (stderr, "Invalid input file '%s'\n", filename);
			ok = false;
		}

		fclose (fp);
		fp = NULL;
	}

	// allocate matrix
	CvMat *matrix = NULL;
	
	if (ok)
	{
		if ((matrix = cvCreateMat (rows, cols, CV_32FC1)) == NULL)
		{
			fprintf (stderr, "Failed to allocate memory\n");
			ok = false;
		}
	}

	// re-open file (if necessary)
	if (ok && fp == NULL)
	{
		ok = ((fp = fopen (filename, "r")) != NULL);
	}

	// read matrix elements
	float tmp;

	for (int i = 0; ok && i < rows; ++ i)
	{
		for (int j = 0; ok && j < cols; ++ j)
		{
			if (fscanf (fp, "%f", &tmp) == 1)
			{
				cvSetReal2D (matrix, i, j, static_cast <double> (tmp));
			}
			else ok = false;
		}
	}

	if (!ok)
	{
		fprintf (stderr, "Failed to read data from file '%s'\n", filename);
		cvReleaseMat (&matrix);
		matrix = NULL;
	}

	// close file
	if (fp)
	{
		fclose (fp);
		fp = NULL;
	}

	return matrix;
}

/****************************************************************************/
bool writeCvMat (const char *filename, const CvMat *matrix, const char *fmt, bool hdr)
{
	// open file
	FILE *fp = fopen (filename, "w");

	if (fp == NULL)
	{
		fprintf (stderr, "Failed to open file '%s' for writing\n", filename);
		return false;
	}

	// write header
	if (hdr) fprintf (fp, "%d %d\n", matrix->rows, matrix->cols);

	// write matrix elements
	for (int i = 0; i < matrix->rows; ++ i)
	{
		for (int j = 0; j < matrix->cols; ++ j)
		{
			if (j > 0) fprintf (fp, " ");
			fprintf (fp, fmt, cvGetReal2D (matrix, i, j));
		}
		fprintf (fp, "\n");
	}

	// close file
	fclose (fp);

	return true;
}

// ---------------------------------------------------------------------------
// write OpenCV matrix as image
// ---------------------------------------------------------------------------

/****************************************************************************/
bool writeRawImage (const char *filename, const CvMat *image)
{
	// open file
	FILE *fp = fopen (filename, "wb");

	if (fp == NULL)
	{
		fprintf (stderr, "Failed to open file '%s'\n", filename);
		return false;
	}

	// write matrix elements as raw image data
	float tmp;

	if (image->cols > 0)
	{
		for (int j = 0; j < image->cols; ++ j)
		{
			for (int i = 0; i < image->rows; ++ i)
			{
				tmp = static_cast <float> (cvGetReal2D (image, i, j));
				fwrite (&tmp, sizeof (float), 1, fp);
			}
		}
	}
	else
	{
		for (int i = 0; i < image->rows; ++ i)
		{
			tmp = static_cast <float> (cvGetReal1D (image, i));
			fwrite (&tmp, sizeof (float), 1, fp);
		}
	}

	fclose (fp);
	return true;
}

/****************************************************************************/
bool writeNiftiImage (const char *filename, const nifti_1_header &hdrIn, const CvMat *image)
{
	const int dim[8] = {4, hdrIn.dim[1], hdrIn.dim[2], hdrIn.dim[3], 1, 0, 0, 0};

	// check consistency of matrix dimensions and image dimensions
	if (dim[1] * dim[2] * dim[3] < image->rows * image->cols)
	{
		fprintf (stderr, "Invalid output image dimensions\n");
		return false;
	}

	// create new image header
	nifti_1_header *hdr = nifti_make_new_header (dim, NIFTI_TYPE_FLOAT32);

	if (!hdr)
	{
		fprintf (stderr, "Failed to initialize image header\n");
		return false;
	}

	// default header values
	// (write hdr/img files by default; may be overwritten according to prefix filename extension)
	strcpy (hdr->magic, "ni1");

	// copy selected header elements
	hdr->dim_info    = hdrIn.dim_info;
	hdr->intent_p1   = hdrIn.intent_p1;
	hdr->intent_p2   = hdrIn.intent_p2;
	hdr->intent_p3   = hdrIn.intent_p3;
	hdr->intent_code = hdrIn.intent_code;
	hdr->pixdim[1]   = hdrIn.pixdim[1];
	hdr->pixdim[2]   = hdrIn.pixdim[2];
	hdr->pixdim[3]   = hdrIn.pixdim[3];
	hdr->xyzt_units  = hdrIn.xyzt_units;
	hdr->cal_min     = 0;
	hdr->cal_max     = 1;
	memcpy (hdr->descrip, hdrIn.descrip, 80);
	hdr->qform_code  = hdrIn.qform_code;
	hdr->sform_code  = hdrIn.sform_code;
	hdr->quatern_b   = hdrIn.quatern_b;
	hdr->quatern_c   = hdrIn.quatern_c;
	hdr->quatern_d   = hdrIn.quatern_d;
	hdr->qoffset_x   = hdrIn.qoffset_x;
	hdr->qoffset_y   = hdrIn.qoffset_y;
	hdr->qoffset_z   = hdrIn.qoffset_z;
	memcpy (hdr->srow_x, hdrIn.srow_x, 4);
	memcpy (hdr->srow_y, hdrIn.srow_y, 4);
	memcpy (hdr->srow_z, hdrIn.srow_z, 4);
	memcpy (hdr->intent_name, hdrIn.intent_name, 16);

	// convert NIfTI-1 header to nifti_image structure
	nifti_image *nim = nifti_convert_nhdr2nim (*hdr, filename);

	free (hdr);
	hdr = NULL;

	if (!nim)
	{
		fprintf (stderr, "Failed to create NIfTI-1 image structure\n");
		return false;
	}

	// write image header
	znzFile fpData = nifti_image_write_hdr_img (nim, 2, "wb");

	// write image data
	if (znz_isnull (fpData))
	{
		fprintf (stderr, "Failed to write image data\n");
		nifti_image_free (nim);
		return false;
	}

	float tmp; // temporary voxel data

	if (image->cols > 0)
	{
		for (int j = 0; j < image->cols; ++ j)
		{
			for (int i = 0; i < image->rows; ++ i)
			{
				tmp = static_cast <float> (cvGetReal2D (image, i, j));
				nifti_write_buffer (fpData, &tmp, sizeof (float));
			}
		}
	}
	else
	{
		for (int i = 0; i < image->rows; ++ i)
		{
			tmp = static_cast <float> (cvGetReal1D (image, i));
			nifti_write_buffer (fpData, &tmp, sizeof (float));
		}
	}

	nifti_image_free (nim);
	znzclose (fpData);

	return true;
}

//////////////////////////////////////////////////////////////////////////////
// initialization
//////////////////////////////////////////////////////////////////////////////

/****************************************************************************/
CvMat *createIndex (const CvMat *data)
{
	CvMat *index1 = cvCreateMat (data->rows, 1, CV_32FC1);

	if (!index1) return NULL;

	cvZero (index1);

	int count = 0;

	for (int i = 0; i < data->rows; ++ i)
	{
		double me = 0;

		for (int j = 0; j < data->cols; ++ j)
		{
			me = me + cvGetReal2D (data, i, j);
		}

		me = me / data->cols;

		if (me > 40)
		{
			cvSetReal1D (index1, i, static_cast <double> (i + 1));
			++ count;
		}
	}

	if (count == 0)
	{
		fprintf (stderr, "No \"non-zero\" voxels found\n");
		return NULL;
	}

	int idx = 0;

	CvMat *index2 = cvCreateMat (count, 1, CV_32FC1);

	if (!index2)
	{
		cvReleaseMat (&index1);
		return NULL;
	}

	for (int i = 0; i < data->rows; ++ i)
	{
		double ele = cvGetReal1D (index1, i);

		if (ele > 0)
		{
			cvSetReal1D (index2, idx, ele);
			++ idx;
		}
	}

	cvReleaseMat (&index1);
	index1 = NULL;

	return index2;
}

/****************************************************************************/
CvMat *createNI (const CvMat *index,
                 const int    dimx,   const int dimy,   const int dimz,
                 const int    sizNIx, const int sizNIy, const int sizNIz,
                 const int    numNI,  const int numVox)
{
	const int m = index->rows; // number of non-zero voxels

	// check arguments
	if (sizNIx < 1 || sizNIy < 1 || sizNIz < 1)
	{
		fprintf (stderr, "createNI (): invalid argument for parameter 'sizNI*'\n");
		return NULL;
	}

	if (numNI < 1 || numNI > m)
	{
		fprintf (stderr, "createNI (): invalid argument for parameter 'numNI'\n");
		return NULL;
	}

	if (numVox < 1)
	{
		fprintf (stderr, "createNI (): invalid argument for parameter 'numVox'\n");
		return NULL;
	}

	// allocate memory
	CvMat *NI       = cvCreateMat (numVox, numNI, CV_32FC1); // neighborhood(s)
	CvMat *samples  = cvCreateMat (m,      1,     CV_32FC1); // indices of centering voxels (only first numNI used)
	CvMat *x_vector = cvCreateMat (m,      1,     CV_32FC1); // maps voxel index to x coordinate of voxel
	CvMat *y_vector = cvCreateMat (m,      1,     CV_32FC1); // maps voxel index to y coordinate of voxel
	CvMat *z_vector = cvCreateMat (m,      1,     CV_32FC1); // maps voxel index to z coordinate of voxel

	if (!NI || !samples || !x_vector || !y_vector || !z_vector)
	{
		fprintf (stderr, "Failed to allocated required memory\n");

		if (x_vector) cvReleaseMat (&x_vector);
		if (y_vector) cvReleaseMat (&y_vector);
		if (z_vector) cvReleaseMat (&z_vector);
		if (samples)  cvReleaseMat (&samples);
		if (NI)       cvReleaseMat (&NI);

		return NULL;
	}

	// create mapping from voxel indices to (x, y, z) coordinates
	// index [i] => (x_vector [i], y_vector [i], z_vector [i])
	for (int i = 0; i < m; ++ i)
	{
		double idx = cvGetReal1D(index, i);

		double z_temp = ceil ((idx / dimx) / dimy);
		double y_temp = ceil ((idx - (z_temp - 1) * dimx * dimy) / dimx);
		double x_temp = idx - (z_temp - 1) * dimx * dimy - (y_temp - 1) * dimx;

		cvSetReal1D (x_vector, i, x_temp);
		cvSetReal1D (y_vector, i, y_temp);	
		cvSetReal1D (z_vector, i, z_temp);
	}

	// randomly select numNI voxels out of the m non-zero voxels
	{
		int *ra = randperm (m);

		for (int i = 0; i < numNI; ++ i)
		{
			cvSetReal1D (samples, i, static_cast <double> (ra [i] - 1));
		}

		delete [] ra;
		ra = NULL;
	}

	// for each neighborhood, randomly select numVox voxels
	int ni1 [(2 * sizNIx + 1) * (2 * sizNIy + 1) * (2 * sizNIz + 1)];
	int ni2 [numVox];

	for (int i = 0; i < numNI; ++ i)
	{
		// map index of neighborhood center to (x, y, z)
		int r = static_cast <int> (cvGetReal1D (samples,  i));

		int xi = static_cast <int> (cvGetReal1D (x_vector, r));
		int yi = static_cast <int> (cvGetReal1D (y_vector, r));
		int zi = static_cast <int> (cvGetReal1D (z_vector, r));

		// determine boundaries of neighborhood
		int minx = xi - sizNIx;
		if (minx < 1) minx = 1;

		int maxx = xi + sizNIx;
		if (maxx > dimx) maxx = dimx;

		int miny = yi - sizNIy;
		if (miny < 1) miny = 1;

		int maxy = yi + sizNIy;
		if (maxy > dimy) maxy = dimy;

		int minz = zi - sizNIz;
		if (minz < 1) minz = 1;

		int maxz = zi + sizNIz;
		if (maxz > dimz) maxz = dimz;

		// determine indices of all neighboring non-zero voxels
		int num_ni = 0;

		for (int j = 0; j < m; ++ j)
		{
			int xj = static_cast <int> (cvGetReal1D (x_vector, j));
			int yj = static_cast <int> (cvGetReal1D (y_vector, j));
			int zj = static_cast <int> (cvGetReal1D (z_vector, j));

			if ((xj < maxx + 1) && (xj > minx - 1) &&
			    (yj < maxy + 1) && (yj > miny - 1) &&
			    (zj < maxz + 1) && (zj > minz - 1))
			{
				ni1 [num_ni] = j + 1;
				++ num_ni;
			}
		}

		while (num_ni < numVox)
		{
			ni1 [num_ni] = 0; // note: zero indicates an invalid index
			++ num_ni;
		}

		// randomly select numVox voxels of these neighboring voxels
		{
			int *ra = randperm (num_ni);

			for (int ss = 0; ss < numVox; ++ ss)
			{
				ni2 [ss] = ni1 [ra [ss] - 1];
				
			}

			delete [] ra;
			ra = NULL;
		}

		// set first entry to center of neighborhood
		int flag = 0;

		for (int tt = 0; tt < numVox; ++ tt)
		{
			if (ni2 [tt] == r + 1)
			{
				int tmp  = ni2 [0];
				ni2 [0]  = ni2 [tt];
				ni2 [tt] = tmp;

				flag = 1;
				break;
			}
		}

		if (flag == 0) ni2 [0] = r + 1;

		// copy neighborhood to column i of NI
		for (int ss = 0; ss < numVox; ++ ss)
		{
			cvSetReal2D (NI, ss, i, ni2 [ss]);
		}
	}

	// clean up
	cvReleaseMat (&x_vector);
	x_vector = NULL;

	cvReleaseMat (&y_vector);
	y_vector = NULL;

	cvReleaseMat (&z_vector);
	z_vector = NULL;

	cvReleaseMat (&samples);
	samples = NULL;

	return NI;
}

/****************************************************************************/
CvMat *createX (CvMat *data, CvMat *index)
{
	const unsigned int n = data->cols;  // number of subjects
	const unsigned int m = index->rows; // number of voxels per subject (template)

	CvMat *X = NULL;

	try
	{
		X = cvCreateMat (m, n, CV_32FC1);
	}
	catch (...)
	{
		fprintf (stderr, "Failed to allocate memory for matrix X\n");
	}

	if (!X) return NULL;

	bool ok = true;
	int  idx;

	for (int i = 0; ok && i < index->rows; ++ i)
	{
		for (int j = 0; ok && j < data->cols; ++ j)
		{
			idx = static_cast <int> (cvGetReal1D (index, i)) - 1;

			if (0 <= idx && idx < data->rows)
			{
				cvSetReal2D (X, i, j, cvGetReal2D (data, idx, j));
			}
			else
			{
				ok = false;
			}	
		}
	}

	if (!ok)
	{
		fprintf (stderr, "Invalid index '%d'\n", idx);
		cvReleaseMat (&X);
		X = NULL;
	}

	return X;
}

//////////////////////////////////////////////////////////////////////////////
// random numbers / generation of permutations
//////////////////////////////////////////////////////////////////////////////

/****************************************************************************/
int *random (int n, int seed)
{
	int *d = new int [n];

	srand (static_cast <unsigned int>(seed < 0 ? time (NULL) : seed));

	for (int i = 0; i < n; ++ i)
	{
	   d [i] = rand ();
	}

	return d;
}

/****************************************************************************/
int *randperm (int n, int seed)
{
	int *d     = random (n, seed);
 	int *rep_d = new int [n];

	for (int i = 0; i < n; ++ i)
	{
		rep_d [i] = d [i];
	}

	std::sort (d, d + n);

	int *ind_sort = new int [n];

	for (int i = 0; i < n; ++ i)
	{
		for (int j = 0; j < n; ++ j)
		{
			if (rep_d [i] == d [j])
			{
				ind_sort [j] = i + 1;
			}
		}
	}

	delete [] d;
	delete [] rep_d;

	return ind_sort;
}


} // namespace odvba

} // namespace sbia
