/**
 * @file  odvba.h
 * @brief Implements Optimally-Discriminative Voxel-Based Analysis (ODVBA).
 *
 * Copyright (c) 2010-2012 University of Pennsylvania. All rights reserved.
 * See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.
 *
 * Contact: SBIA Group <sbia-software at uphs.upenn.edu>
 *
 * @ingroup PublicAPI
 */

#pragma once
#ifndef _SBIA_ODVBA_ODVBA_H
#define _SBIA_ODVBA_ODVBA_H


#include <cv.h>
#include <boost/any.hpp>

#include <boost/numeric/ublas/matrix.hpp> 
#include <boost/numeric/ublas/matrix_proxy.hpp>
#include <boost/numeric/ublas/vector.hpp> 
#include <boost/numeric/ublas/vector_proxy.hpp> 
#include <boost/numeric/ublas/operation.hpp> 
#include <boost/numeric/ublas/operation_sparse.hpp> 
#include <boost/numeric/ublas/matrix_sparse.hpp> 
#include <boost/numeric/ublas/vector_sparse.hpp> 
#include <boost/numeric/ublas/symmetric.hpp>

#include <boost/numeric/ublas/io.hpp> 

#include <boost/numeric/bindings/atlas/cblas3.hpp>
#include <boost/numeric/bindings/atlas/cblas2.hpp>
#include <boost/numeric/bindings/traits/ublas_matrix.hpp>

#include <sbia/odvba/config.h>


namespace sbia
{

namespace odvba
{


// ===========================================================================
// data types
// ===========================================================================

/**
 * @struct Database
 * @brief  Input data to perform_analysis().
 *
 * The Database includes the necessary inputs, e.g., the input data (m x n),
 * the neihghborhoods (NE x nn), the index (non-zeros voxels), and n1
 * (number of subjects in group1).
 *
 * @ingroup PublicAPI
 */
struct Database
{
    /// Number of subjects belonging to group 1.
    int n1;
    /// @brief Voxel data per subject.
    ///
    /// The first n1 subjects belong to group 1 and the
    /// remaining n2 = n - n1 subjects belong to group 2.
    CvMat* X;
    /// The vector to store the index of template.
    CvMat* index;
    /// The matrix to store the neighorhoods.
    CvMat* NI;

    /**
     * @brief Default constructor.
     */
    Database();

    /**
     * @brief Destrutor.
     *
     * @note The destructor does NOT release any data. Call method
     *       release () instead manually if the data should be released.
     *
     * @sa release ()   
     */
    ~Database();

    /**
     * @brief Release assigned data.
     */     
    void release();

private:

    /**
     * @brief Copy constructor.
     *
     * @note Intentionally not implemented!
     *
     * @param [in] other Other instance.
     */
    Database(const Database& other);

    /**
     * @brief Assignment operator.
     *
     * @note Intentionally not implemented!
     *
     * @param [in] rhs Right-hand side.
     */
    Database& operator=(const Database& rhs);

}; // struct Database

/**
 * @struct Options
 * @brief  Parameters of ODVBA method.
 *
 * @ingroup PublicAPI
 */
struct Options
{
    /// Gamma in Eq. (4) of the MICCAI paper.
    double gamma;
    /// @brief Phi in Eq. (11), the discrimination degree, of the MICCAI paper.
    ///
    /// It is a tuning parameter which aims at reducing the number of outliers.
    double phi;
    /// Number of permutations tested during permutation test.
    int nPerm;
    /// @brief Matrix of permutations.
    ///
    /// Each row represents one permutation of the
    /// numbers [1..n], where n is the number of subjects. Only used for
    /// regression testing as here deterministic results are required
    /// which can be compared to the expected results.
    /// If nPerm is zero, the specified pre-computed permutations are
    /// are used for the group analysis. Otherwise, nPerm random permutations
    /// are generated during the group analysis and stored in this matrix.
    /// Hence, in the latter case, the matrix must have nPerm rows.
    CvMat* perms;
    /// Seed used for initialization of random number generator.
    unsigned int seed;

    /**
     * @brief Default constructor which sets the default values of the parameters.
     */
    Options()
    {
        gamma = 1.0e-6;
        phi   = 1;
        nPerm = 100;
        perms = NULL;
        seed  = 1;
    }

    /**
     * @brief Destructor.
     */
    ~Options()
    {
        if (perms) {
            cvReleaseMat(&perms);
            perms = NULL;
        }
    }

}; // struct Options

/**
 * @struct NDPData
 * @brief  Data used by function NDP.
 *
 * By defining this data structure and instantiating it only once, the running
 * time of the ODVBA method is improved due to the avoidance of unnecessary
 * (de-)allocations of temporary memory.
 *
 * @sa NDP()
 */
struct NDPData
{
    /// Temporary matrix with columns of data belonging to group 1/2 only.
    boost::numeric::ublas::matrix<double> M;
    /// Transpose of M.
    boost::numeric::ublas::matrix<double> MT;
    /// Used for mean computation using matrix multiplication, i.e.,
    /// M * [1/n 1/n ...]' = [meanOfRow1 meanOfRow2 ...]'.
    boost::numeric::ublas::vector<double> e;
    /// Mean of voxel values of group 1.
    boost::numeric::ublas::vector<double> intraMean1;
    /// Covariance of voxel values of group 1.
    boost::numeric::ublas::matrix<double> intraCov1;
    /// Mean of voxel values of group 2.
    boost::numeric::ublas::vector<double> intraMean2;
    /// Covariance of voxel values of group 2.
    boost::numeric::ublas::matrix<double> intraCov2;  
    /// gamma * S_W + S_B, see MICCAI paper.
    CvMat* A_rep;      
    /// Eigenvectors of A_rep.
    CvMat* Eigvector;  
    /// Eigenvalues of A_rep.
    CvMat* Eigvalue;
    /// Positive definite matrix A defined in MICCAI paper.
    CvMat* A;          
    /// A^+, positive elements of A.
    boost::numeric::ublas::matrix<double> Ap;
    /// A^-, negative negative elements of A.
    boost::numeric::ublas::matrix<double> An;
    boost::numeric::ublas::vector<double> a;
    boost::numeric::ublas::vector<double> c;
    /// Result of NDP.
    boost::numeric::ublas::vector<double> w;

    /**
     * @brief Contructor which initializes the data structure.
     *
     * @param [in] m Number of voxels per subject. If zero is passed as argument,
     *               the data structure remains uninitialized and has to be
     *               initialized manually (done by the function NDP).
     */
    NDPData(const int m = 0);

    /**
     * @brief Destructor.
     */
    ~NDPData();

private:

    /**
     * @brief Copy constructor.
     *
     * @note Intentionally not implemented!
     *
     * @param [in] other Other instance.
     */
    NDPData(const NDPData& other);

    /**
     * @brief Assignment operator.
     *
     * @note Intentionally not implemented!
     *
     * @param [in] rhs Right-hand side.
     */
    NDPData& operator=(const NDPData& rhs);

}; // struct NDPData

/**
 * @struct CompMapData
 * @brief  Data used by function compMap.
 *
 * By defining this data structure and instantiating it only once, the running
 * time of the ODVBA method is improved due to the avoidance of unnecessary
 * (de-)allocations of temporary memory.
 *
 * @sa compMap()
 */
struct CompMapData
{
    /// Learning set constructed by the subvolume vectors from all subjects.
    boost::numeric::ublas::matrix<double> theta;
    /// Transpose of theta.
    boost::numeric::ublas::matrix<double> thetaT;
    /// Projected learning set using projection direction obtained via NDP.
    boost::numeric::ublas::vector<double> weight;
    /// Discriminative degree for each voxel and each neighborhood.
    CvMat* delta;
    /// Temporary data used by NDP function.
    NDPData ndp;
    /// Stores the coffients for all learning sets.
    CvMat* W;

    /**
     * @brief Constructor which initializes the data structures.
     *
     * If any passed argument is zero, the data structure remains
     * uninitialized and has to be initialized manually
     * (done by the function compMap).
     *
     * @param [in] m Number of voxels per subject (i.e., voxels in template).
     * @param [in] n Number of subjects.
     * @param [in] N Number of neighborhoods.
     * @param [in] k Number of voxels per neighborhood.
     */
    CompMapData(const int m = 0,
                const int n = 0,
                const int N = 0,
                const int k = 0);

    /**
     * @brief Destructor.
     */
    ~CompMapData();

private:

    /**
     * @brief Copy constructor.
     *
     * @note Intentionally not implemented!
     *
     * @param [in] other Other instance.
     */
    CompMapData(const CompMapData& other);

    /**
     * @brief Assignment operator.
     *
     * @note Intentionally not implemented!
     *
     * @param [in] rhs Right-hand side.
     */
    CompMapData& operator=(const CompMapData& rhs);

}; // struct CompMapData

// ===========================================================================
// initialization
// ===========================================================================

/**
 * @brief Create index, i.e., the union of indices of non-zero voxels of the
 *        data of all subjects.
 *
 * Only voxels whose index is listed in the index data structure are considered
 * during the voxel-based analysis. The index can be considered as template
 * mask of the input volumes.
 *
 * @param [in] data Volume data of all subjects, where the each column
 *                  corresponds to one subject.
 *
 * @returns Index data structure or NULL on error (i.e., memory allocation failed).
 *          The returned matrix has to be released by caller using cvReleaseMat().
 *
 * @ingroup PublicAPI
 */
CvMat* create_index(const CvMat* data);

/**
 * @brief Create neighborhoods.
 *
 * This function first selects randomly numNI voxels of all non-zero voxels.
 * Then, numVox voxels are randomly drawn from the voxels which lie within a
 * neighborhood of size (2 * sizNIx + 1) * (2 * sizNIy + 1) * (2 * sizNIz + 1)
 * centered at each of the voxels selected. If the random selection contained
 * invalid voxels, i.e., ones with voxel value zero, the corresponding entry
 * in the output matrix is set to the invalid index zero.
 *
 * All the neighborhoods corresponding to all the voxels can be used to get the 
 * discriminative coefficients, but it is really computationally expensive. By 
 * selecting random neighborhoods, the number of the involved learning sets can 
 * be reduced and therefore improve the performance of the group analysis.
 *
 * @param [in] index  The index of non-zero voxels.
 * @param [in] dimx   The x dimension of the image.
 * @param [in] dimy   The y dimension of the image.
 * @param [in] dimz   The z dimension of the image.
 * @param [in] sizNIx Radius of each neighborhood along x dimension (excluding center voxel).
 * @param [in] sizNIy Radius of each neighborhood along y dimension (excluding center voxel).
 * @param [in] sizNIz Radius of each neighborhood along z dimension (excluding center voxel).
 * @param [in] numNI  Number of neighborhoods.
 * @param [in] numVox Number of voxels randomly selected from each neighborhood.
 *
 * @returns Neighborhood data structure or NULL on error (e.g., memory allocation failed).
 *          The returned matrix has to be released by caller using cvReleaseMat().
 *
 * @ingroup PublicAPI
 */
CvMat* create_ni(const CvMat* index,
                 const int dimx,   const int dimy,   const int dimz,
                 const int sizNIx, const int sizNIy, const int sizNIz,
                 const int numNI,  const int numVox);

/**
 * @brief Create data matrix X from original input and index.
 *
 * @param [in] data  Input data of all subjects.
 * @param [in] index Index, see createIndex().
 *
 * @returns Created data matrix X or NULL on failure.
 *          The returned memory has to be released using cvReleaseMat().
 *
 * @see read_data()
 * @see create_index()
 * @see read_index()
 *
 * @ingroup PublicAPI
 */
CvMat* create_x(CvMat* data, CvMat* index);

// ===========================================================================
// non-negative discriminative projection (NDP)
// ===========================================================================

/**
 * @brief Non-negative Discriminative Projection (NDP).
 *
 * Call this function with each learning set to get the matrix W.
 * @sa Section 2.1 of the MICCAI paper.
 *
 * @param [in]  theta An input learning set constructed from the neighborhoods
 *                    centered at a particular voxel. Each column of the matrix
 *                    corresponds to a subject, where the first n1 subjects belong
 *                    to group 1 and the remaining n2 = n - n1 subjects to group 2.
 *                    The columns of the learning set further correspond to the
 *                    subvolume vectors which were constructed by randomly sampling
 *                    the voxels neighborhood within the subject's original image.
 * @param [in]  n1    The number of subjects belonging to group 1.
 * @param [in]  gamma Gamma in the discrimination degree defined in the Eq. (4)
 *                    of the MICCAI paper.
 * @param [out] tmp   Temporary data/Previously allocated memory which is reused
 *                    to improve the performance of the ODVBA method.
 *
 * @returns The vector w which stores the coffients for the input learning set
 *          (note that the result is also stored in tmp.w).
 *
 * @sa comp_map()
 *
 * @ingroup PublicAPI
 */
boost::numeric::ublas::vector<double> ndp(const boost::numeric::ublas::matrix<double>& theta,
                                          const int                    n1,
                                          const double                 gamma,
                                          NDPData&                     tmp);

// ===========================================================================
// determination of each voxel's statistic
// ===========================================================================

/**
 * @brief Determine statistics of each voxel using NDP.
 *
 * @sa Section 2.2 of the MICCAI paper.
 *
 * @sa ndp()
 *
 * @param [in]  data Input database including original image data,
 *                   template mask (index) and the matrix specifying the
 *                   pre-computed neighborhoods.
 * @param [in]  opt  Parameters of the ODVBA method.
 * @param [out] map  Map which stores the determined statistics of each voxel.
 *                   Has to be allocated by caller using cvCreateMat (m, 1, CV_32FC1),
 *                   where m is the number of voxels per subject, i.e., the number
 *                   of non-zero voxels in the template.
 * @param [out] tmp  Temporary memory re-used for different executions of compMap.
 *                   This improves the performance of the ODVBA method by
 *                   avoiding unnecessary (de-)allocations of temporary memory.
 *
 * @ingroup PublicAPI
 */
void comp_map(const Database& data, const Options& opt, CvMat* map, CompMapData& tmp);

// ===========================================================================
// permutation tests
// ===========================================================================

/**
 * @brief Perform ODVBA to get statistical value using permutation tests.
 *
 * @sa Section 2.3 of the MICCAI paper.
 *
 * @param [in] data      The original data: number of voxels per subject x
 *                       number of subjects.
 * @param [in] opt       Parameters of the ODVBA method.
 * @param [in] initial   Whether to compute initial results first using
 *                       non-permuted input data.
 * @param [in] verbose   Verbosity of messages. If less or equal to zero,
 *                       no messages are printed to stdout. Otherwise, the
 *                       higher the verbose level the more verbose messages
 *                       are generated. Errors and warnings are printed to
 *                       stderr in any case.
 * @param [in] msgPrefix String which is prefixed each printed message.
 *                       No whitespace is printed between the prefix and the
 *                       actual message. Include whitespace in prefix if desired.
 *
 * @returns map of statistic values
 *          Caller is responsible for releasing the memory using cvReleaseMat().
 *
 * @sa comp_map()
 *
 * @ingroup PublicAPI
 */
CvMat* perform_analysis(const Database& data,
                        const Options&  opt,
                        bool            initial   = true,
                        int             verbose   = 0,
                        const char*     msgPrefix = "");

// ===========================================================================
// p-value / p-image
// ===========================================================================

/**
 * @brief Get final map of p-values from results of permutation tests.
 *
 * @param [in] maps Result of permutation tests.
 *
 * @returns Map of p-values or NULL on error.
 *          Caller is responsible for releasing the memory using cvReleaseMat().
 *
 * @sa perform_analysis()
 *
 * @ingroup PublicAPI
 */
CvMat* get_p_value(CvMat* maps);

/**
 * @brief Create the image of p-values.
 *
 * @param [in] maps     Result of permutation tests.
 * @param [in] index    Indices of non-zero template voxels.
 * @param [in] nVoxels  Number of voxels in original input images. 
 * @param [in] pValueIn Calculated p-values (optional).
 *                      If NULL, the p-values are implicitly calculated using
 *                      the function get_p_value().
 *
 * @returns Image of p-values of size nVoxels or NULL on error.
 *          Caller is responsible for releasing the memory using cvReleaseMat().
 *
 * @sa get_p_value()
 * @sa perform_analysis()
 *
 * @ingroup PublicAPI
 */
CvMat* get_p_image(CvMat* maps, CvMat* index, int nVoxels, CvMat* pValueIn = NULL);


} // namespace odvba

} // namespace sbia


#endif // _SBIA_ODVBA_ODVBA_H
