/**
 * @file  ODVBA.cpp
 * @brief Implements Optimally-Discriminative Voxel-Based Analysis (ODVBA).
 *
 * Copyright (c) 2015 University of Pennsylvania. All rights reserved.
 * See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.
 *
 * Contact: SBIA Group <sbia-software at uphs.upenn.edu>
 */
 
#include "CAlgorithm.h"
#include "cbicaCmdParser.h"
#include <omp.h>

int main(int argc, char **argv)
{
	ExDatabase datapara;
	Options opt;
	char* sublist;
  std::string outputDirectory;
	
  cbica::CmdParser parser = cbica::CmdParser(argc, argv, "ODVBA");
  parser.addRequiredParameter("d", "dataList", "string", "none", "Input text file containing the full paths of the images");
  parser.addRequiredParameter("o", "outputDir", "string", "none", "Output directory to place written files", "Upon successful completion, expect to see the following files:", "p.nii, {index, map, permResult, perm222}.txt");
  parser.addRequiredParameter("n", "numNi", "integer", "5000-20000; default: 10000", "Number of neighborhoods to consider", "At most, the neighborhood around each non-zero voxel can be considered");
  parser.addOptionalParameter("s", "sizeNi", "integer", "9-17; default: 15", "Size of neighborhood");
  parser.addOptionalParameter("c", "count", "integer", "200-500; default: 400", "Count of voxels in each neighborhood");
  parser.addOptionalParameter("p", "permsNum", "integer", ">100; default: 200", "The number of permutations to test");
  parser.addOptionalParameter("e", "exponent", "double", "0-10; default: 1", "Exponent phi of discrimination degree", "Refer Eq.(11) of MICCAI paper for details");
  parser.exampleUsage("-d subjects.txt -s 15 -n 10000 -c 400 -p 200 -e 1 ~/ODVBA_results/");
  
  if (argc <= 1)
	{
		parser.echoUsage();
		return EXIT_FAILURE;
	}
  
  int tmpPosition;
	if (parser.compareParameter("u", tmpPosition))
	{
		parser.echoUsage();
		return EXIT_SUCCESS;
	}
	
	if (parser.compareParameter("h", tmpPosition))
	{
		parser.echoHelp();
		return EXIT_SUCCESS;
	}
	if (parser.compareParameter("v", tmpPosition))
	{
		parser.echoVersion();
		return EXIT_SUCCESS;
	}
	if (parser.compareParameter("d", tmpPosition))
	{
		std::cout << "'dataList' read.\n";
		sublist = argv[tmpPosition+1];
	}     
	if (parser.compareParameter("o", tmpPosition))
	{
		std::cout << "'outputDir' read.\n";
		outputDirectory = argv[tmpPosition+1];
    outputDirectory += "/";
	}   
  if (parser.compareParameter("s", tmpPosition))
	{
		std::cout << "'sizeNi' read.\n";
		datapara.s = atol(argv[tmpPosition+1]);
	}
	if (parser.compareParameter("n", tmpPosition))
	{
		std::cout << "'numNi' read.\n";
		datapara.nn = atol(argv[tmpPosition + 1]);
	}
	if (parser.compareParameter("c", tmpPosition))
	{
		std::cout << "'count' read.\n";
		datapara.NE = atoi(argv[tmpPosition + 1]);
	}
	if (parser.compareParameter("p", tmpPosition))
	{
		std::cout << "'permsNum' read.\n";
		opt.nPerm = atoi(argv[tmpPosition + 1]);
	}
	if (parser.compareParameter("e", tmpPosition))
	{
		std::cout << "'exponent' read.\n";
		opt.aa = atoi(argv[tmpPosition + 1]);
	}
  printf("\n The input options are below:"); 
  printf("\n ===============================================\n");
  printf("\n sublist:      %s",sublist);
  printf("\n s:      %d",datapara.s);  
  printf("\n n:      %ld",datapara.nn);
  printf("\n c:      %d",datapara.NE);
  printf("\n p:      %d",opt.nPerm);
  printf("\n e:      %f",opt.aa);
  printf("\n ==============================================\n");
  /*****************************************************/
  transferData *transData = new transferData;  
  int subject_id      = 2;
  transData->SetInput (sublist, subject_id);
  transData->GetInput();
  transData->Read_nifti_Data();
  datapara.n = (transData->_num_group1+transData->_num_group2);	    
  datapara.n1=transData->_num_group1;
  datapara.n2=transData->_num_group2;
  int dimx=transData->_xdim;
  int dimy=transData->_ydim;
  int dimz=transData->_zdim;
  opt.nn=datapara.nn;
  int ni_size_x = static_cast<int>(round(datapara.s/ 2 - 0.5));
  int ni_size_y = static_cast<int>(round(datapara.s / 2 - 0.5));
  int ni_size_z = static_cast<int>(round(datapara.s / 2 - 0.5));
  
  /*****************************************************/
  ublas::matrix<double> rand_matrix(datapara.n,opt.nPerm);
  srand(time( NULL )); 
  for (int ii = 0; ii < opt.nPerm; ii++ )  
  {  
    int* ra=randperm(datapara.n); 
	  for (int jj= 0; jj < datapara.n; jj++ ) 
	  {
	    rand_matrix(jj,ii)= *(ra+jj)-1;
	  }
  }
  /*****************************************************/
  ublas::matrix<double> index = create_index(transData->origin_data);
  datapara.m=index.size1();
  writeData(index, std::string(outputDirectory + "index.txt").c_str());
  ublas::matrix<double> NI =create_ni(index,dimx, dimy, dimz,ni_size_x, ni_size_y, ni_size_z,datapara.nn, datapara.NE);
  datapara.NI=NI;
  ublas::matrix<double> X(datapara.m,transData->_num_group1+transData->_num_group2);
  createX(X, transData->origin_data,index); 
  /*****************************************************/
  opt.nNeighbor=datapara.NE;
  opt.nSample=datapara.n;
	MapParas mParas;
  ublas::matrix<double> fea(opt.nNeighbor,opt.nSample);
	mapResults mapW;
	mapW.map1= ublas::matrix<double>(datapara.m,1);
	mapW.W1= ublas::matrix<double>(opt.nNeighbor,datapara.nn);
  Var_matrix mat;
	mat= init_mat(mat,opt.nNeighbor,transData->_num_group1, transData->_num_group2, opt.nSample,datapara.nn,opt.nSample);
  /*****************************************************/
	mapW.map1=ublas::zero_matrix<double>(mapW.map1.size1(),mapW.map1.size2());
	ublas::matrix<double> map1=ublas::zero_matrix<double>(mapW.map1.size1(),mapW.map1.size2());
	map1=compMap(mapW,X,opt,mat,datapara,fea,mParas);
  /*****************************************************/
  ublas::matrix<double> permResult(opt.nPerm+1, datapara.m);
	permResult=ublas::zero_matrix<double>(permResult.size1(),permResult.size2());
	for ( int i = 0; i < datapara.m; i++ )
  {
    permResult(0,i)=map1(i,0);
  }
  //  writeData(map1, std::string(outputDirectory + "map.txt").c_str());
  /****************************************************/
  //char str[5];
  string name1="./";	

  /*****************************************************/
  int i;
  const int threads = omp_get_max_threads(); // obtain maximum number of threads available on machine  
  #pragma omp parallel for num_threads(threads) 
  for (i = 1; i < opt.nPerm+1; i++ )  
  {  
	  mapResults mapW2;
    mapW2.map1= ublas::matrix<double>(datapara.m,1);
    mapW2.W1= ublas::matrix<double>(opt.nNeighbor,datapara.nn);
    ublas::matrix<double> fea2(opt.nNeighbor,opt.nSample);
    ublas::matrix<double> map2=ublas::zero_matrix<double>(mapW2.map1.size1(),mapW2.map1.size2());
    mapW2.map1=ublas::zero_matrix<double>(mapW.map1.size1(),mapW.map1.size2());
	  mapW2.W1=ublas::zero_matrix<double>(mapW.W1.size1(),mapW.W1.size2());  
	  ublas::matrix<double> X2(datapara.m,transData->_num_group1+transData->_num_group2);
	  X2=ublas::zero_matrix<double>(datapara.m,transData->_num_group1+transData->_num_group2);
	  for ( int k = 0 ; k < datapara.n ; k++ )  
	  {  
      for ( int j = 0 ; j < datapara.m; j++ )	
	    {  
	      X2(j,rand_matrix(k,i))=X(j,k);	
	    }       
	  }  
    map2=compMap(mapW2,X2,opt,mat,datapara,fea2,mParas);
    for ( int j = 0; j < datapara.m; j++ )
    {
      permResult(i,j)=map2(j,0);
    }
  }
      
  /*****************************************************/  
  //string savename_txt = outputDirectory + "perm.txt";  
  // writeData(permResult, savename_txt.c_str());  
  /*****************************************************/  
  ublas::matrix<double> pvalue(datapara.m,1);
  ublas::matrix<double> pimage(transData->_xdim*transData->_ydim*transData->_zdim,1);
  pvalue=ublas::zero_matrix<double>(pvalue.size1(),pvalue.size2());
  for ( int j = 1; j < opt.nPerm+1; j++ )       
  {     
    for ( int i = 0; i < datapara.m; i++ )   
    {
      if ((permResult(j,i)>permResult(0,i))||(permResult(j,i)==permResult(0,i)))   
        pvalue(i,0)=pvalue(i,0)+1;   
    }   
  }   
  for ( int i = 0; i < transData->_xdim*transData->_ydim*transData->_zdim; i++ )   
  {      
    pimage(i,0)=1;   
  }   
  for( int i = 0; i < datapara.m; i++ )   
  {     
    pimage(int(index(i,0)),0)=pvalue(i,0)/opt.nPerm;   
  }   
  /*****************************************************/  
  string savename = outputDirectory + "p.nii";  
  transData->write_nifti_img(pimage, savename.c_str());  
  /*****************************************************/  
  std::cout << "\n ODVBA successfully completed.\n";
  return 0;
}
