#! /usr/bin/env python

##############################################################################
# @file  gondola-crossval.py
# @brief Perform steps of cross-validation experiments.
#
# Copyright (c) 2012 University of Pennsylvania. All rights reserved.
# See http://www.rad.upenn.edu/sbia/software/license.html or COPYING file.
#
# Contact: SBIA Group <sbia-software at uphs.upenn.edu>
##############################################################################

import os
import sys
import re
import time
import shlex
import numpy
import scipy

from ConfigParser import SafeConfigParser as ConfigParser
from getopt       import getopt, GetoptError

from sbiautilities import pyxel
from gondola       import basis


# ============================================================================
# constants
# ============================================================================

# names of configuration sections
COMMANDS = 'commands'
SETTINGS = 'settings'
TRAINING = 'training'
TESTING  = 'testing'

## @brief Implemented experiment steps.
STEPS = ['learn', 'show', 'extract', 'search', 'classify', 'summarize']

## @brief Maps meta-classifier name to name of underlying core classifier.
META_TO_CORE_CLASSIFIER = {
    'Bagging Logistic':        'Logistic',
    'Bagging SMO':             'SMO',
    'Boosted Simple Logistic': 'Simple Logistic'
}

# ============================================================================
# help
# ============================================================================

# ----------------------------------------------------------------------------
## @brief Print usage information.
def print_help():
    print """\
Usage:
  %(EXENAME)s <step> [<step>...] [options]
  %(EXENAME)s [options] <step> [<step>...]

Description:
  This command executes individual steps of an experiment for each cross-validation
  fold. The input subsets of training and testing data and corresponding
  configuration files are conveniently generated by the gondola-config utility.

  The following experiment steps are implemented and have to be executed in
  the given order.

  1. learn        Learn the best basis vectors from the training data.
  2. show         Optionally save basis vectors to images for visualization.
  3. extract      Extract features from both training and testing sets using
                  the learned basis vectors. The extracted features are saved
                  in Weka format (.arff file).
  4. search       Find the best parameters for each classifier named in the
                  configuration file. The found parameters are stored in CSV format.
  5. classify     Classify both training and testing data using the found best
                  parameters. The classification results are stored in CSV format.
  6. summarize    Generate performance report for each classifier.

Required arguments:
  <step>                    Step to perform. Note that this argument can be given multiple
                            times and that the named steps are executed in the order they
                            appear in the argument list. If 'all' is given, every step is
                            executed in the order specified above.
 
Optional arguments:
  -c, --configfile <file>   Configuration file for this program.
                            (default: crossval.cfg)
  --foldids <id>[,<id>...]  Comma separated list of fold IDs. The IDs of all folds are
                            usually specified in the [%(SETTINGS)s] section of the
                            configuration file. This option, if specified, overwrites
                            this configuration option. It can be used in particular to
                            only run the experiment for a subset of all folds.
  --continue                Continue learning after previous run was interrupted.
  -m, --macroavg            Instead of collecting all data into a single CSV table
                            and compute the average accuracy from this table, compute
                            the average for each cross-validation fold first and
                            finally compute the average of these averaged results.
                            The standard deviation is reported if this option is given.
  -v, --verbose             Increase verbosity of output messages.
  -s, --simulate            Simulate execution, i.e., print commands that would be
                            executed only without actually executing them.
  -h, --help                Show help and exit.

Configuration:
  The configuration file for this command must contain the sections:
  [%(COMMANDS)s], [%(SETTINGS)s], [%(TRAINING)s], and [%(TESTING)s].
  
  The [%(COMMANDS)s] section specifies the commands to execute as subprocesses for
  each experiment step. Note that default arguments for these commands can be
  specified as well. This program will simply add additional arguments to the
  command-line. The configuration options for each of these sections are:

  [%(COMMANDS)s]
  learn:       Command to learn the basis vectors.
  show:        Command to save basis vectors to images.
  extract:     Command to extract the features.
  search:      Command to find the best parameters for each classifier.
  classify:    Command to train classifiers and classify data samples.

  the commands that can be used and are included with the GONDOLA package are
  "gondola learn", "gondola-sbia learn", and "gondola-sbia-sge learn" for the
  learning step. Similarly, you can use "gondola show" or "gondola extract"
  for the conversion of the basis vectors to images or the feature extraction
  step, respectively. Also in this case, "gondola-sbia" or "gondola-sbia-sge"
  are valid substitudes for "gondola". Note that these small wrapper scripts
  are intended for use at SBIA only and will not work for other environments.
  For the parameter search, the command "wekaParamSearchForClassifier" is
  provided and the "wekaClassifier" is used by default for the training and
  classification step. This classifier implements several different classifiers.
  See the help of each command for more information (these auxiliary commands
  can be found in the lib/ directory of the GONDOLA installation).

Example configuration:
  See below for an example configuration of a typical cross-validation
  experiment that can be used with this program. Note that this configuration
  file, saved in the file <expdir>/crossval.cfg, can be generated by
  gondola-config and may then be edited to make the desired changes.
  For example, the "gondola learn" command used for the learning of the basis
  vectors has been replaced below by a command that can be used within the
  computing environment at SBIA to perform the learning using separate batch jobs,
  one for each cross-validation experiment, resulting in a parallel processing of
  these cross-validations.

  [%(COMMANDS)s]
  learn:            qsub gondola-sbia learn
  show:             gondola show
  extract:          gondola
  search:           wekaParamSearchForClassifier -i -w
  classify:         wekaClassifier
  jobstat:          qstat -j %%(jobid)s

  [%(SETTINGS)s]
  foldids:          1, 2, 3, 4, 5
  classifiers:      Logistic, SMO, Simple Logistic, Bayesian, Random Forest
  summaryfile:      Summary for %%(classifer)s Classifier.txt

  [%(TRAINING)s]
  configfile:       gondola.cfg
  imagelistfile:    %%(foldid)s/traindata.lst
  idlistfile:       %%(foldid)s/trainids.lst
  datafile:         %%(foldid)s/gondola.mat
  basisimagedir:    %%(foldid)s/basisimages
  featuresfile:     %%(foldid)s/trainfeatures.arff
  bestparamsfile:   %%(foldid)s/bestparams.csv
  resultfile:       %%(foldid)s/%%(classifier)sClassifierResults.csv
  resultcolumn:     ClassLabelCV%%(foldid)s

  [%(TESTING)s]
  imagelistfile:    %%(foldid)s/testdata.lst
  idlistfile:       %%(foldid)s/testids.lst
  featuresfile:     %%(foldid)s/testfeatures.arff
  resultfile:       %%(foldid)s/%%(classifier)sClassifierResults.csv
  resultcolumn:     ClassLabel
 
Examples:
  %(EXENAME)s learn

    Performs the first step of the experiment, i.e., the learning of the basis
    vectors. The configuration file of the cross-validation experiment has to
    be located in the current working directory with the name crossval.cfg, the
    default of the -c (--configfile) option. The consecutive steps, e.g.,
    the extraction of the features once the basis vectors are learned, are
    similarly executed by simply providing the name of the step as argument
    instead of "learn".

  %(EXENAME)s show extract

    This command saves the learned basis vectors to images and further
    extracts the features for both training and learning sets.

  %(EXENAME)s all

    This command performs all steps of the cross-validation experiment at once.

    Note that this command will not detect if a step failed if this step was
    performed as parallel job using a batch queuing system, e.g., when the
    command gondola-sbia is used at SBIA in place of gondola. Therefore, check
    the log files to verify that each step was successful or simply run each
    step separately instead.
""" % \
    {
        'EXENAME':  basis.exename(),
        'COMMANDS': COMMANDS,
        'SETTINGS': SETTINGS,
        'TRAINING': TRAINING,
        'TESTING':  TESTING
    }
    basis.print_contact()

# ============================================================================
# auxiliary functions
# ============================================================================

# ----------------------------------------------------------------------------
## @brief Get the best classifier parameters.
#
# This function reads in the CSV file written by the bestparam search and
# finds the extracts the found best parameters for the specified classifier.
#
# @param [in] bestparamsfile CSV file generated by bestparam search.
# @param [in] classifier     Name of classifier.
#
# @returns Tuple (params, extraparams), where @c params are the best
#          parameters for the specified classifier, and @c extraparams are the
#          additional parameters if the classifier is a meta-classifier and an
#          empty string otherwise.
def bestparams(bestparamsfile, classifier):
    # load result of best parameter search
    csv = pyxel.Pyxel()
    csv.load(bestparamsfile)
    # get parameters for best classifier
    param = csv.grab(classifier, 'Parameters')
    # get extra parameters of meta-classifier
    extraparam = None
    if 'Bag' in classifier or 'Boost' in classifier:
        extraparam = csv.grab(META_TO_CORE_CLASSIFIER[classifier], 'Parameters')
    if not extraparam: extraparam = ''
    return param, extraparam

# ----------------------------------------------------------------------------
def accuracy(p, colname='ClassLabel'):
    # find out how many classes are in the experiment
    numSamples = {}      # this is a list containing list of labels
    labels = p.column(col=2)[1]   # get the column of the actual labels
    for l in labels:
        if not(l in numSamples.keys()):
            numSamples.update({l:0})
    numLabels = len(numSamples.keys()) 
    # count number of samples per class
    labels = p.column(col=2)[1]
    for l in labels:
        numSamples[l] = numSamples[l] + 1  
    numLabels = len(numSamples.keys())
    confusionMatrix = scipy.mat( numpy.zeros( (numLabels,numLabels) ) ) 
    hdr_actual = '%s-actual' % colname
    hdr_predic = '%s-prediction' % colname
    mapLabels = {}
    cnt = 0
    for l in numSamples.keys():
       mapLabels.update({l:cnt})
       cnt = cnt + 1
    confusionMatrix = scipy.mat( numpy.zeros( (numLabels,numLabels) ) )  
    hdr_actual = '%s-actual' % colname
    hdr_predic = '%s-prediction' % colname
    actualLabels = p.column(hdr_actual)
    predictLabels = p.column(hdr_predic)
    for cnt in range(0,len(actualLabels[0])):
        pLabel = mapLabels[predictLabels[1][cnt]]     # predicted label
        aLabel = mapLabels[actualLabels[1][cnt]]      # actual Label
        if ( not(  predictLabels[0][cnt] ==   actualLabels[0][cnt]   ) ):     # it is just a sanity check, this event should happen ever. Just to make sure that it is comparing the same subject
              assert False, "This event should NOT happend ever !!! Are you sure you are using the correct Pyxel version??? I am comparing labels of two different subjects !! " 
        confusionMatrix[aLabel,pLabel] = confusionMatrix[aLabel,pLabel] + 1.0
    return confusionMatrix
 
# ----------------------------------------------------------------------------
# this function summarize the results into text and csv files
def write_summary(classifier, csvTrainFile_List, csvTestFile_List, resFile, aveMode, colname):
    if not aveMode in [1, 2]:
        assert False, "Invalid value for aveMode parameter of write_summary() function: %s" % repr(aveMode)
    # find out home many classes are in the experiment
    numSamples = {}      # this is a list containing list of labels
    for fn in csvTestFile_List:
        p = pyxel.Pyxel(fn)
        if len(p) > 1:                  # first row are the column headers
            labels = p.column(col=2)[1] # get the column of the actual labels
            for l in labels:
                if not(l in numSamples.keys()):
                    numSamples.update({l:0})
        else:
            raise Exception("File %s\ndoes either not exist or contains no data rows !!!" % fn)
    numLabels = len(numSamples.keys()) 
    # read all test csv files and keep them into a single file
    totalNumSamplesPerClass = []
    res_test = pyxel.Pyxel()
    for fn in csvTestFile_List:
        p = pyxel.Pyxel(fn)
        totalNumSamplesPerClass.append(len(p.getIds()))
        res_test = res_test + p
        res_test.contract()
    # count number of samples per class
    labels = res_test.column(col=2)[1]
    for l in labels:
        numSamples[l] = numSamples[l] + 1.0  
    # Do one of the following: aveMode=1) collect accuracy of all experiments and avearge, or aveMode=2) collect all csv file into one csv file and compute average
    if aveMode == 1:
        confusionMatrix = accuracy(res_test, colname)
    else:
        confusionMatrix = [] 
        for fn in csvTestFile_List:
            p = pyxel.Pyxel(fn)
            tmpConfMatrix = accuracy(p, colname)
            confusionMatrix.append(tmpConfMatrix)
    # write a report to the text file
    text_file = open(resFile, "a+")
    sys.stdout.write('\n'); text_file.write('\n')
    if aveMode == 1:
        line = "Confusion matrix of %s classifier" % classifier
    else:
        line = "Confusion matrices of %s classifier" % classifier
    line = ' '.join([line, "(numbers of classified samples NOT percentages):\n"])
    print line; text_file.write(line + '\n')
    if aveMode == 1:
        for l in confusionMatrix:
            line = str(l)
            print line; text_file.write(line + '\n')
        sys.stdout.write('\n'); text_file.write('\n')
        line = "Normalized trace of confusion matrix: " + str(confusionMatrix.trace()/numpy.sum(totalNumSamplesPerClass))
        print line; text_file.write(line + '\n')
    else:
        cnt = 0
        accList = []
        for cfMat in confusionMatrix:
            if cnt > 0:
                sys.stdout.write('\n'); text_file.write('\n')
            line = "Confusion matrix %d =" % (cnt + 1)
            print line; text_file.write(line + '\n')
            for l in cfMat:
                line = str(l) 
                print line; text_file.write(line + '\n')
            line = "\n"
            print line; text_file.write(line + '\n')
            line = "Normalized trace of confusion matrix: " + str(cfMat.trace()/totalNumSamplesPerClass[cnt])
            print line; text_file.write(line + '\n')
            accList.append(cfMat.trace()/totalNumSamplesPerClass[cnt])
            cnt = cnt + 1
        line = "Average accuracy [standard deviation]: %f [%f] " % (numpy.mean(accList), numpy.std(accList))
        print line; text_file.write(line + '\n')
    sys.stdout.write('\n'); text_file.write('\n')
    text_file.close()

# ----------------------------------------------------------------------------
## @brief Get list of fold IDs of cross-validation experiments.
#
# @param [in] cfg ConfigParser instance.
def getfoldids(cfg):
    if cfg.has_option(SETTINGS, 'foldids'):
        return [s.strip() for s in cfg.get(SETTINGS, 'foldids').split(',')]
    elif cfg.has_option(SETTINGS, 'numfolds'):
        return [str(i) for i in range(1, cfg.getint(SETTINGS, 'numfolds') + 1)]
    else: raise Exception("Missing 'foldids' or 'numfolds' option in [%s]!" % SETTINGS)

# ----------------------------------------------------------------------------
## @brief Execute subprocess and return job IDs if batch job was submitted.
#
# This function executes a subprocess and parses the output to extract the
# job ID if the command submitted a batch job to a batch queuing system.
# At the moment, only SGE is supported. If no job ID is matched in the
# command output, it is assumed that the subprocess exits after the command
# has been executed.
#
# @returns List of job IDs or an empty list.
def execute(cmd, verbose=0, simulate=False):
    if not type(cmd) is list: cmd = shlex.split(cmd)
    for i in range(len(cmd)):
        path = basis.exepath(cmd[i])
        if path: cmd[i] = path
    (status, stdout) = basis.execute(cmd, verbose=verbose, simulate=simulate, stdout=True)
    m = re.match(r"^.*Your job (?P<jobid>[0-9]+) \(\".*\"\) has been submitted.*$", stdout)
    if not m is None: return m.group('jobid')
    else:             return None

# ============================================================================
# steps
# ============================================================================

# ----------------------------------------------------------------------------
## @brief Learn basis vectors from each training set.
#
# @param [in] cfg ConfigParser instance.
def learn(cfg):
    # get configuration values
    foldids    = getfoldids(cfg)
    configdir  = cfg.get       (SETTINGS, 'configdir')
    continue_  = cfg.getboolean(SETTINGS, 'continue')
    verbose    = cfg.getint    (SETTINGS, 'verbose')
    simulate   = cfg.getboolean(SETTINGS, 'simulate')
    # run step for each subset
    jobids = []
    for foldid in foldids:
        cfgvars = {'foldid': foldid}
        cfgvars['configfile']    = os.path.join(configdir, cfg.get(TRAINING, 'configfile',    0, cfgvars))
        cfgvars['imagelistfile'] = os.path.join(configdir, cfg.get(TRAINING, 'imagelistfile', 0, cfgvars))
        cfgvars['datafile']      = os.path.join(configdir, cfg.get(TRAINING, 'datafile',      0, cfgvars))
        cmd = shlex.split(cfg.get(COMMANDS, 'learn', 0, cfgvars))
        if continue_: cmd.append('--continue')
        jobid = execute(cmd, verbose=verbose, simulate=simulate)
        if jobid: jobids.append(jobid)
    return jobids

# ----------------------------------------------------------------------------
## @brief Convert basis vectors to images.
def show(cfg):
    # get configuration values
    foldids   = getfoldids(cfg)
    configdir = cfg.get       (SETTINGS, 'configdir')
    verbose   = cfg.getint    (SETTINGS, 'verbose')
    simulate  = cfg.getboolean(SETTINGS, 'simulate')
    # run step for each subset
    jobids = []
    for foldid in foldids:
        cfgvars  = {'foldid': foldid}
        cfgvars['imagelistfile'] = os.path.join(configdir, cfg.get(TRAINING, 'imagelistfile', 0, cfgvars))
        cfgvars['datafile']      = os.path.join(configdir, cfg.get(TRAINING, 'datafile',      0, cfgvars))
        cfgvars['basisimagedir'] = os.path.join(configdir, cfg.get(TRAINING, 'basisimagedir', 0, cfgvars))
        cmd   = shlex.split(cfg.get(COMMANDS, 'show', 0, cfgvars))
        jobid = execute(cmd, verbose=verbose, simulate=simulate)
        if jobid: jobids.append(jobid)
    return jobids

# ----------------------------------------------------------------------------
## @brief Extract features given trained basis vectors.
#
# @param [in] cfg ConfigParser object.
def extract(cfg):
    # get configuration values
    foldids    = getfoldids(cfg)
    configdir  = cfg.get       (SETTINGS, 'configdir')
    verbose    = cfg.getint    (SETTINGS, 'verbose')
    simulate   = cfg.getboolean(SETTINGS, 'simulate')
    # run step for each subset
    jobids = []
    for foldid in foldids:
        for s in (TRAINING, TESTING):
            cfgvars = {'foldid': foldid}
            cfgvars['datafile'] = os.path.join(configdir, cfg.get(TRAINING, 'datafile', 0, cfgvars))
            cfgvars['imagelistfile'] = os.path.join(configdir, cfg.get(s, 'imagelistfile', 0, cfgvars))
            cfgvars['idlistfile']    = os.path.join(configdir, cfg.get(s, 'idlistfile',    0, cfgvars))
            cfgvars['featuresfile']  = os.path.join(configdir, cfg.get(s, 'featuresfile',  0, cfgvars))
            cmd   = shlex.split(cfg.get(COMMANDS, 'extract', 0, cfgvars))
            jobid = execute(cmd, verbose=verbose, simulate=simulate)
            if jobid: jobids.append(jobid)
    return jobids

# ----------------------------------------------------------------------------
## @brief Find the best parameters for each classifier.
#
# @param [in] cfg ConfigParser object.
def search(cfg):
    # get configuration values
    foldids     = getfoldids(cfg)
    configdir   = cfg.get       (SETTINGS, 'configdir')
    verbose     = cfg.getint    (SETTINGS, 'verbose')
    simulate    = cfg.getboolean(SETTINGS, 'simulate')
    classifiers = cfg.get       (SETTINGS, 'classifiers')
    # run step for each subset
    jobids = []
    for foldid in foldids:
        cfgvars = {'foldid': foldid, 'classifiers': classifiers}
        cfgvars['featuresfile']   = os.path.join(configdir, cfg.get(TRAINING, 'featuresfile',   0, cfgvars))
        cfgvars['bestparamsfile'] = os.path.join(configdir, cfg.get(TRAINING, 'bestparamsfile', 0, cfgvars))
        cmd   = shlex.split(cfg.get(COMMANDS, 'search', 0, cfgvars))
        jobid = execute(cmd, verbose=verbose, simulate=simulate)
        if jobid: jobids.append(jobid)
    return jobids

# ----------------------------------------------------------------------------
## @brief Train classifiers using best parameters and classify samples.
#
# @param [in] cfg ConfigParser object.
def classify(cfg):
    # get configuration values
    foldids       = getfoldids(cfg)
    configdir     = cfg.get       (SETTINGS, 'configdir')
    verbose       = cfg.getint    (SETTINGS, 'verbose')
    simulate      = cfg.getboolean(SETTINGS, 'simulate')
    classifiers   = [c.strip() for c in cfg.get(SETTINGS, 'classifiers').split(',')]
    # run step for each subset
    jobids = []
    for foldid in foldids:
        for cls in classifiers:
            cfgvars = {'foldid': foldid, 'classifier': cls.replace(' ', '')}
            cfgvars['bestparamsfile']        = os.path.join(configdir, cfg.get(TRAINING, 'bestparamsfile', 0, cfgvars))
            cfgvars['training.featuresfile'] = os.path.join(configdir, cfg.get(TRAINING, 'featuresfile',   0, cfgvars))
            cfgvars['training.resultfile']   = os.path.join(configdir, cfg.get(TRAINING, 'resultfile',     0, cfgvars))
            cfgvars['training.resultheader'] =                         cfg.get(TRAINING, 'resultcolumn',   0, cfgvars)
            cfgvars['testing.featuresfile']  = os.path.join(configdir, cfg.get(TESTING,  'featuresfile',   0, cfgvars))
            cfgvars['testing.resultfile']    = os.path.join(configdir, cfg.get(TESTING,  'resultfile',     0, cfgvars))
            cfgvars['testing.resultheader']  =                         cfg.get(TESTING,  'resultcolumn',   0, cfgvars)
            try:
                (cfgvars['bestparams'], cfgvars['extraparams']) = bestparams(cfgvars['bestparamsfile'], cls)
            except IOError, e:
                sys.stderr.write("File with best parameters for each classifier could not be opened.\n")
                sys.stderr.write("Have you run the parameter \"search\" step already ?\n")
                raise e
            if not cfgvars['bestparams']:
                raise Exception('Failed to get best parameters of %s classifier!\n' \
                                ' Check the file %s with the best parameters.\n'\
                                ' Was the best parameter search successful and complete?' \
                                       % (cls, cfgvars['bestparamsfile']))
            cmd   = shlex.split(cfg.get(COMMANDS, 'classify', 0, cfgvars))
            jobid = execute(cmd, verbose=verbose, simulate=simulate)
            if jobid: jobids.append(jobid)
    return jobids

# ----------------------------------------------------------------------------
## @brief Summarize results and generate report.
#
# @param [in] cfg ConfigParser object.
def summarize(cfg, ):
    # get configuration values
    foldids     = getfoldids(cfg)
    classifiers = [c.strip() for c in cfg.get(SETTINGS, 'classifiers').split(',')]
    configdir   = cfg.get   (SETTINGS, 'configdir')
    avgmode     = cfg.getint(SETTINGS, 'avgmode')
    colname     = cfg.get   (TESTING,  'resultcolumn', 0, {'foldid': 0, 'classifier': ''})
    # remove previous summary reports
    for cls in classifiers:
        cfgvars = {'classifier': cls.replace(' ', '')}
        summaryfile = os.path.join(configdir, cfg.get(SETTINGS, 'summaryfile', 0, cfgvars))
        if os.path.isfile(summaryfile):
            os.remove(summaryfile)
    # create summary report for each classifier
    for cls in classifiers:
        cfgvars = {'classifier': cls.replace(' ', '')}
        trainresultfiles = []
        testresultfiles  = []
        summaryfile = os.path.join(configdir, cfg.get(SETTINGS, 'summaryfile', 0, cfgvars))
        for foldid in foldids:
            cfgvars['foldid'] = foldid
            trainresultfiles.append(os.path.join(configdir, cfg.get(TRAINING, 'resultfile', 0, cfgvars)))
            testresultfiles .append(os.path.join(configdir, cfg.get(TESTING,  'resultfile', 0, cfgvars)))
        write_summary(cls, trainresultfiles, testresultfiles, summaryfile, avgmode, colname)
    return []

# ============================================================================
# main
# ============================================================================

# ----------------------------------------------------------------------------
## @brief Perform specified steps for all cross-validation sets.
def main(steps, cfg):
    verbose = cfg.getint(SETTINGS, 'verbose')
    for i in range(len(steps)):
        steps[i] = steps[i].lower()
        if not steps[i] in STEPS:
            raise Exception("Invalid processing step: %s! Valid steps are: %s"
                                % (steps[i], ', '.join(STEPS)))
    for step in steps:
        if step == 'all':
            steps = STEPS
            break
    for step in steps:
        # run next step
        if   step == 'learn':     jobids = learn    (cfg)
        elif step == 'show':      jobids = show     (cfg)
        elif step == 'extract':   jobids = extract  (cfg)
        elif step == 'search':    jobids = search   (cfg)
        elif step == 'classify':  jobids = classify (cfg)
        elif step == 'summarize': jobids = summarize(cfg)
        else: assert False, 'unhandled step'
        # wait for submitted batch jobs
        if cfg.getboolean(SETTINGS, 'sync') and cfg.has_option(COMMANDS, 'jobstat'):
            while jobids:
                cmd = cfg.get(COMMANDS, 'jobstat', 0, {'jobid': jobids[-1]})
                if cmd: status = basis.execute(cmd, allow_fail=True, quiet=True)
                else:   break
                if status != 0:
                    if verbose > 0: print "Job %s finished." % jobid
                    jobids.pop()
                else:
                    if verbose > 0: print "Job %s running. Waiting for job to finish." % jobid
                    time.sleep(60)

# ----------------------------------------------------------------------------
## @brief Parse command-line arguments and call main() function.
if __name__ == '__main__':
    if len(sys.argv) == 1:
        print_help()
        sys.exit(1)

    steps      = []
    sync       = True
    configfile = 'crossval.cfg'
    foldids    = None
    avgmode    = 1
    continue_  = False
    simulate   = False
    verbose    = 0

    # getopt ignores all options (such as --foldids) following a positional
    # argument (such as 'learn'). we would like to allow for both of the
    # following uses, however, where the second version is actually preferred:
    # $ gondola-crossval --configfile crossval.cfg --foldids 1,2,3 learn
    # $ gondola-crossval learn --configfile crossval.cfg --foldids 1,2,3
    i = 1
    while i < len(sys.argv) and sys.argv[i].lower() in STEPS:
        steps.append(sys.argv[i])
        i += 1
    if i < len(sys.argv):
        try:
            opts, args = getopt(sys.argv[i:], "c:msvh",
                                ["configfile=", "foldids=", "macroavg",
                                 "continue", "sync", "nosync", "simulate", "verbose", "help"])
            steps.extend(args)
        except GetoptError, e:
            sys.stderr.write("%s\n" % str(e))
            sys.exit(1)

        for o, a in opts:
            if o in ('-h', '--help'):
                print_help()
                sys.exit(0)
            elif o in ('-v', '--verbose'):
                verbose += 1
            elif o in ('-s', '--simulate'):
                simulate = True
            elif o == '--foldids':
                foldids = a
            elif o == '--sync':
                sync = True
            elif o == '--nosync':
                sync = False
            elif o in ('-m', '--macroavg'):
                avgmode = 2
            elif o in ('-c', '--configfile'):
                configfile = a
            else:
                assert False, "unhandled option: %s" % o

    if not steps:
        sys.stderr.write("No experiment steps specified! Implemented steps are: %s\n" % ', '.join(STEPS))
        sys.exit(1)
    for i in range(len(steps)):
        steps[i] = steps[i].lower()
        if not steps[i] in STEPS:
            sys.stderr.write("Invalid processing step: %s! Valid steps are: %s\n"
                                % (steps[i], ', '.join(STEPS)))
            sys.exit(1)
    if not os.path.isfile(configfile):
        if configfile == 'crossval.cfg':
            sys.stderr.write("Missing configuration file 'crossval.cfg'!\n")
        else:
            sys.stderr.write("Configuration file '%s' does not exist!\n")
        sys.exit(1)

    try:
        # read configuration from file
        configfile = os.path.realpath(configfile)
        cfg = ConfigParser()
        cfg.readfp(open(configfile), configfile)
        # add configuration values given by command-line
        if foldids: cfg.set(SETTINGS, 'foldids', foldids)
        cfg.set(SETTINGS, 'sync',      str(sync))
        cfg.set(SETTINGS, 'continue',  str(continue_))
        cfg.set(SETTINGS, 'avgmode',   str(avgmode))
        cfg.set(SETTINGS, 'verbose',   str(verbose))
        cfg.set(SETTINGS, 'simulate',  str(simulate))
        cfg.set(SETTINGS, 'configdir', os.path.dirname(configfile))
        # run main function which performs the specified steps
        main(steps, cfg)
    except Exception, e:
        sys.stderr.write("%s\n" % str(e))
        sys.exit(1)
    sys.exit(0)
