///////////////////////////////////////////////////////////////////////////////////////
// DTIProcessingManager.cpp
// Developed by Saima Rathore
// Copyright (c) 2014 University of Pennsylvania. All rights reserved.
// See http://www.cbica.upenn.edu/sbia/software/license.html or COYPING file.
// Contact: SBIA Group <sbia-software at uphs.upenn.edu>
///////////////////////////////////////////////////////////////////////////////////////


#include "cbicaDTIProcessingManager.h"

#define Malloc(type, n) (type *)malloc((n)*sizeof(type))

namespace cbica
{
  DTIProcessingManager::DTIProcessingManager()
  {
    max_line_len = 5000;
    line = NULL;
  }

  DTIProcessingManager::~DTIProcessingManager()
  {
  }
  char* DTIProcessingManager::readline(FILE *input)
  {
    int len;

    if (fgets(line, max_line_len, input) == NULL)
      return NULL;

    while (strrchr(line, '\n') == NULL)
    {
      max_line_len *= 2;
      line = (char *)realloc(line, max_line_len);
      len = (int)strlen(line);
      if (fgets(line + len, max_line_len - len, input) == NULL)
        break;
    }
    return line;
  }

  void DTIProcessingManager::FlipYOrientationInBVecFile(std::string filename)
  {
    std::vector<long double> XOrientation;
    std::vector<long double> YOrientation;
    std::vector<long double> ZOrientation;
    //std::string::size_type sz;

    int LineNumber = 1;
    max_line_len = 5000;
    FILE *fp = fopen(filename.c_str(), "r");
    if (fp == NULL)
      exit(1);
    line = Malloc(char, max_line_len);
    while (readline(fp) != NULL)
    {
      char *p = strtok(line, "\t"); // label
      char * first = strtok(line, " ");
      std::string firstElement(first);
      //char* pEnd;
      if (LineNumber == 1)
        XOrientation.push_back(std::atoll(firstElement.c_str()));
      else if (LineNumber == 2)
        YOrientation.push_back(std::atoll(firstElement.c_str()));
      else if (LineNumber == 3)
        ZOrientation.push_back(std::atoll(firstElement.c_str()));
      while (1)
      {
        p = strtok(NULL, " ");
        if (p == NULL || *p == '\n') // check '\n' as ' ' may be after the last feature
          break;
        std::string element(p);
        if (LineNumber == 1)
          XOrientation.push_back(std::atoll(element.c_str()));
        else if (LineNumber == 2)
          YOrientation.push_back(std::atoll(element.c_str()));
        else if (LineNumber == 3)
          ZOrientation.push_back(std::atoll(element.c_str()));
      }
      LineNumber++;
    }
    fclose(fp);
    for (unsigned int i = 1; i < YOrientation.size(); i++)
      YOrientation[i] = -1 * YOrientation[i];

    fp = fopen(filename.c_str(), "w");
    for (unsigned int index = 0; index < XOrientation.size(); index++)
    {
      if (index<XOrientation.size() - 1)
        fprintf(fp, "%.15Lf ", XOrientation[index]);
      else
        fprintf(fp, "%.15Lf\n", XOrientation[index]);
    }
    for (unsigned int index = 0; index < YOrientation.size(); index++)
    {
      if (index<YOrientation.size() - 1)
        fprintf(fp, "%.15Lf ", YOrientation[index]);
      else
        fprintf(fp, "%.15Lf\n", YOrientation[index]);
    }
    for (unsigned int index = 0; index < ZOrientation.size(); index++)
    {
      if (index<ZOrientation.size() - 1)
        fprintf(fp, "%.15Lf ", ZOrientation[index]);
      else
        fprintf(fp, "%.15Lf", ZOrientation[index]);
    }
    fclose(fp);
  }
  void GetImageInfo(std::string fName, itk::ImageIOBase::IOPixelType *pixelType, itk::ImageIOBase::IOComponentType *componentType)
  {
    itk::ImageIOBase::Pointer imageIO;
    //~ try
    //~ {
    imageIO = itk::ImageIOFactory::CreateImageIO(fName.c_str(), itk::ImageIOFactory::ReadMode);
    if (imageIO)
    {
      imageIO->SetFileName(fName);
      imageIO->ReadImageInformation();
      *pixelType = imageIO->GetPixelType();
      *componentType = imageIO->GetComponentType();
    }
    else
    {
      std::cout << "Could not read the input image information from " <<
        fName << std::endl;
      //TODO should throw exception
      exit(EXIT_FAILURE);
    }

  }

  std::vector<itk::Image<float, 3>::Pointer> DTIProcessingManager::dtiMainFunction(std::string dwiFile, std::string maskFile, std::string bvalFile, std::string gradFile, std::string outputDir)
  {
    typedef itk::Image<float, 3> ScalarImageType;
    std::vector<ScalarImageType::Pointer> vectorOfDTIScalars;

    std::string prefix = "tensor";
    bool prefix_flag = 1;
    bool outputDir_flag = 1;

    std::string outputBasename = "";
    if (outputDir_flag)
      outputBasename += outputDir;

    if (prefix_flag)
      outputBasename += prefix;
    else
      outputBasename += "";

    itk::ImageIOBase::IOPixelType       maskPixelType;
    itk::ImageIOBase::IOComponentType   maskCompType;
    GetImageInfo(maskFile, &maskPixelType, &maskCompType);

    //Die if it isn't scalar.
    if (maskPixelType != itk::ImageIOBase::SCALAR)
    {
      std::cerr << "Mask is expected to be a Scalar image" << std::endl;
    }

    switch (maskCompType)
    {
    case itk::ImageIOBase::UCHAR:
    {
      vectorOfDTIScalars = runOverMaskType<unsigned char>(dwiFile, gradFile, bvalFile, outputBasename, maskFile, true);
      break;
    }
    case itk::ImageIOBase::SHORT:
    {
      vectorOfDTIScalars = runOverMaskType<short>(dwiFile, gradFile, bvalFile, outputBasename, maskFile, true);
      break;
    }
    default:
    {
      std::cerr << "Mask is expected to be either a unsigned char or a signed short Image" << std::endl;
      std::cerr << "Please invectigate the image type of your supplied mask" << std::endl;
      break;
    }
    }
    return vectorOfDTIScalars;
  }
  void DTIProcessingManager::GetImageInfo(std::string fName, itk::ImageIOBase::IOPixelType *pixelType, itk::ImageIOBase::IOComponentType *componentType)
  {
    itk::ImageIOBase::Pointer imageIO;
    imageIO = itk::ImageIOFactory::CreateImageIO(fName.c_str(), itk::ImageIOFactory::ReadMode);
    if (imageIO)
    {
      imageIO->SetFileName(fName);
      imageIO->ReadImageInformation();
      *pixelType = imageIO->GetPixelType();
      *componentType = imageIO->GetComponentType();
    }
    else
    {
      std::cout << "Could not read the input image information from " <<
        fName << std::endl;
      //TODO should throw exception
      exit(EXIT_FAILURE);
    }

  }

  void DTIProcessingManager::FlipAndShiftNiftiFile(std::string inputfilename, std::string outputfilename)
  {

    typedef short                      InputPixelType;
    // typedef short                       MiddlePixelType;
    typedef short                       OutputPixelType;
    typedef itk::Image< InputPixelType, 4 >    InputImageType;
    // typedef itk::Image< MiddlePixelType, 4 >    MiddleImageType;
    typedef itk::Image< OutputPixelType, 4 >    OutputImageType;

    typedef itk::FlipImageFilter< InputImageType> FlipType;
    FlipType::Pointer flip = FlipType::New();

    typedef itk::ImageFileReader< InputImageType  >  ReaderType;
    typedef itk::ImageFileWriter< OutputImageType >  WriterType;

    ReaderType::Pointer reader = ReaderType::New();
    WriterType::Pointer writer = WriterType::New();

    reader->SetFileName(inputfilename);
    writer->SetFileName(outputfilename);


    FlipType::FlipAxesArrayType flipAxesSet;

    flipAxesSet[0] = 0;
    flipAxesSet[1] = -1;
    flipAxesSet[2] = 0;
    flipAxesSet[3] = 0;

    flip->SetFlipAxes(flipAxesSet);
    flip->FlipAboutOriginOff();
    flip->SetInput(reader->GetOutput());
    flip->Update();



    //itk::NiftiImageIO::Pointer nifti_io = itk::NiftiImageIO::New();
    OutputImageType::Pointer image = OutputImageType::New();
    image = flip->GetOutput();
    image->SetOrigin(reader->GetOutput()->GetOrigin());
    writer->SetInput(image);
    //writer->SetImageIO(nifti_io);
    try
    {
      writer->Update();
    }
    catch (itk::ExceptionObject & err)
    {
      std::cerr << "ExceptionObject caught !" << std::endl;
      std::cerr << err << std::endl;
    }

  }
}