///////////////////////////////////////////////////////////////////////////////////////
// fMainWindow.cpp
// Developed by Saima Rathore
// Copyright (c) 2014 University of Pennsylvania. All rights reserved.
// See http://www.cbica.upenn.edu/sbia/software/license.html or COYPING file.
// Contact: SBIA Group <sbia-software at uphs.upenn.edu>
///////////////////////////////////////////////////////////////////////////////////////


#include "fMainWindow.h"
#include "fRecurrenceDialog.h"
#include "fPreprocessingDialog.h"
#include "fRegistrationDialog.h"
#include "itkCSVArray2DFileReader.h"
#include "OutputInteractorStyleNavigator.h"

#include "qobject.h"
#include "fHelpDialog.h"
//vtkStandardNewMacro(myVtkInteractorStyleImage);

int GetNumberOfDimensions(vtkImageData* input)
{
	int dim = 0;
#if VTK_MAJOR_VERSION <= 5
	int* extent = input->GetWholeExtent();
#else
	int* extent = input->GetExtent();
#endif
	if (extent[4] != extent[5])
	{
		dim = 3;
	}
	else if (extent[3] != extent[4])
	{
		dim = 2;
	}
	else if (extent[0] != extent[1])
	{
		dim = 1;
	}
	return dim;
}


fMainWindow::fMainWindow()
{
	setupUi(this);
	mFeatureExtractionObj		= new FeatureExtractionClass();
	mPreprocessingObj			= new PreprocessingPipelineClass();
	mClassificationObj			= new SVMClassificationClass();
	mFeatureReductionObj		= new FeatureReductionClass();
	mFeatureScalingObj			= new FeatureScalingClass();
	mNifiDataManager			= new NiftiDataManager();
	mOutputManager				= new OutputWritingManager();
	mEGFRPredictor				= new EGFRStatusPredictor();
	mRecurrenceEstimator		= new RecurrenceEstimator(mNifiDataManager, mOutputManager, mFeatureReductionObj, mFeatureScalingObj, mFeatureExtractionObj, mClassificationObj);
	mGeodesicSegmentation		= new GeodesicSegmentation(); 
	recurrencePanel				= new fRecurrenceDialog();
	preprocessingPanel			= new fPreprocessingDialog();
	registrationPanel		    = new fRegistrationDialog();
  
  imagetype = "";

	t1cePath = "";

	mFont.setFamily(QString::fromUtf8("Calibri"));
	this->setFont(mFont);

	QString msg = "PHI Estimator";
#ifdef SW_VER
	msg += " - v" + tr(SW_VER);
#endif
	this->setWindowTitle(msg);

	mInputPathName = "";
	mMainWidget = this;
	mCurrentSelectedImageId = "";
	mCurrentPickedImageId = "";
	mCurrentPickedImageIndex = 0;
	mCurrentNearPoints = 0;
	mCurrentFarPoints = 0;
	mCurrentInitPoints = 0;



	mLandmarks = new Landmarks(LANDMARK_TYPE_LANDMARKS);
	mSeedPoints = new Landmarks(LANDMARK_TYPE_SEED_POINTS);
	mTissuePoints = new Landmarks(LANDMARK_TYPE_TISSUE_POINTS);


	mMask = vtkSmartPointer<vtkImageData>::New();

	mSlicerManagers.resize(0);
	imagesTable->setColumnCount(3);
	imagesTable->setSelectionBehavior(QAbstractItemView::SelectRows);
	imagesTable->setSelectionMode(QAbstractItemView::SingleSelection);

	nonViewingImagesTable->setColumnCount(3);
	nonViewingImagesTable->setSelectionBehavior(QAbstractItemView::SelectRows);
	nonViewingImagesTable->setSelectionMode(QAbstractItemView::SingleSelection);

	//masksTable->setSelectionBehavior(QAbstractItemView::SelectRows);
	//masksTable->setSelectionMode(QAbstractItemView::SingleSelection);

	overlayTable->setColumnCount(3);
	overlayTable->setSelectionBehavior(QAbstractItemView::SelectRows);
	overlayTable->setSelectionMode(QAbstractItemView::SingleSelection);

	overlayLabel->setEnabled(false);
	overlayUse->setChecked(false);
	overlayUse->setEnabled(false);
	overlayTable->setEnabled(false);
	overlaySlider->setEnabled(false);
	overlaySlider->setRange(0, 10);
	overlaySlider->setValue(5);
	connect(overlayUse, SIGNAL(stateChanged(int)), this, SLOT(overlayUseStateChanged(int)));
	connect(overlaySlider, SIGNAL(valueChanged(int)), this, SLOT(overlaySliderChanged()));

	connect(overlayTable, SIGNAL(itemSelectionChanged()), this, SLOT(overlayChanged()));
	connect(overlayTable, SIGNAL(itemClicked(QTableWidgetItem*)), this, SLOT(overlayChanged(QTableWidgetItem*)));


	SetNonViewingImagesTable();
	//SetMasksTable();
	SetPresetComboBox();

	// init the sliders
	verticalSliders.push_back(AxialViewSlider);
	verticalSliders.push_back(CoronalViewSlider);
	verticalSliders.push_back(SaggitalViewSlider);
	for (int i = 0; i < 3; i++)
	{
		verticalSliders[i]->hide();
	}
	connect(AxialViewSlider, SIGNAL(valueChanged(int)), this, SLOT(AxialViewSliderChanged()));
	connect(CoronalViewSlider, SIGNAL(valueChanged(int)), this, SLOT(CoronalViewSliderChanged()));
	connect(SaggitalViewSlider, SIGNAL(valueChanged(int)), this, SLOT(SaggitalViewSliderChanged()));

	//connect(actionLoad_Images, SIGNAL(triggered()), this, SLOT(OpenImages()));
	connect(actionLoad_T1_Images, SIGNAL(triggered()), this, SLOT(OpenT1Images()));
	connect(actionLoad_T2_Images, SIGNAL(triggered()), this, SLOT(OpenT2Images()));
	connect(actionLoad_T1CE_Images, SIGNAL(triggered()), this, SLOT(OpenT1CEImages()));
	connect(actionLoad_T2_FLAIR_Images, SIGNAL(triggered()), this, SLOT(OpenT2FlairImages()));
	connect(actionLoad_Perfusion_Images, SIGNAL(triggered()), this, SLOT(OpenPerfusionImages()));
	connect(actionLoad_DTI_Images, SIGNAL(triggered()), this, SLOT(OpenDTIImages()));
	connect(actionLoad_Recurrence_Images, SIGNAL(triggered()), this, SLOT(OpenRecurrenceImages()));
	
	connect(actionLoad_Dicom_Images, SIGNAL(triggered()), this, SLOT(OpenDicomImages()));
	connect(actionLoad_Nifti_Images, SIGNAL(triggered()), this, SLOT(OpenNiftiImages()));

  
	connect(actionExit, SIGNAL(triggered()), this, SLOT(close()));
	connect(actionAbout, SIGNAL(triggered()), this, SLOT(about()));
  connect(actionHelp, SIGNAL(triggered()), this, SLOT(help()));
	connect(actionShortcuts, SIGNAL(triggered()), this, SLOT(shortcuts()));


	connect(actionAppEGFR, SIGNAL(triggered()), this, SLOT(ApplicationEGFR()));
	connect(actionAppRecurrence, SIGNAL(triggered()), this, SLOT(ApplicationRecurrence()));
	connect(actionAppGeodesic, SIGNAL(triggered()), this, SLOT(ApplicationGeodesic()));
	connect(actionPreprocessingDenoise, SIGNAL(triggered()), this, SLOT(ImageDenoising()));
	connect(actionPreprocessingBias, SIGNAL(triggered()), this, SLOT(ImageBiasCorrection()));
	connect(actionPreprocessingRegister, SIGNAL(triggered()), this, SLOT(ImageRegistration()));
	connect(actionPreprocessingCustom, SIGNAL(triggered()), this, SLOT(CustomPreprocessing()));

	connect(imagesTable, SIGNAL(itemSelectionChanged()), this, SLOT(DisplayChanged()));
	connect(imagesTable, SIGNAL(itemClicked(QTableWidgetItem*)), this, SLOT(DisplayChanged(QTableWidgetItem*)));

	connect(windowSpinBox, SIGNAL(editingFinished()), this, SLOT(WindowLevelEdited()));
	connect(levelSpinBox, SIGNAL(editingFinished()), this, SLOT(WindowLevelEdited()));
	connect(presetComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(UpdateWindowLevel()));
	connect(thresholdSlider, SIGNAL(valueChanged(int)), this, SLOT(thresholdSliderChanged()));
	//
	thresholdSlider->setRange(0, 255);
	thresholdSlider->setValue(20);
	if (presetComboBox->currentIndex() == PRESET_THRESHOLD) 
	{
		thresholdLabel->setEnabled(true);
		thresholdSlider->setEnabled(true);
	}
	else 
	{
		thresholdLabel->setEnabled(false);
		thresholdSlider->setEnabled(false);
	}

	connect(tumorPanel, SIGNAL(UpdateRenderWindows()), this, SLOT(UpdateRenderWindows()));
	connect(tumorPanel, SIGNAL(SetActiveLandmarksType(int, int, int)), this, SLOT(SetActiveLandmarksType(int, int, int)));
	connect(tumorPanel, SIGNAL(MoveSlicerCursor(double, double, double)), this, SLOT(MoveSlicerCursor(double, double, double)));

	connect(drawingPanel, SIGNAL(DrawNearButtonClicked()), this, SLOT(EnableNearRegionDrawing()));
	connect(drawingPanel, SIGNAL(DrawFarButtonClicked()), this, SLOT(EnableFarRegionDrawing()));
	//connect(drawingPanel, SIGNAL(DrawInitButtonClicked()), this, SLOT(EnableInitDrawing()));
	connect(drawingPanel, SIGNAL(EscapeButtonClicked()), this, SLOT(EscapeMode()));
	//connect(drawingPanel, SIGNAL(EraseAllButtonClicked()), this, SLOT(EraseAllTheDrawings()));
	connect(drawingPanel, SIGNAL(EraseSelectedButtonClicked()), this, SLOT(SetDrawingModeToEraserMode()));
	connect(drawingPanel, SIGNAL(EraseNearButtonClicked()), this, SLOT(EraseNearDrawing()));
	connect(drawingPanel, SIGNAL(EraseFarButtonClicked()), this, SLOT(EraseFarDrawing()));
	//connect(drawingPanel, SIGNAL(EraseInitButtonClicked()), this, SLOT(EraseInitDrawing()));
  
  connect(actionSave_ROI_Images, SIGNAL(triggered()),this, SLOT(SaveDrawing()));
  connect(actionSave_Nifti_Images, SIGNAL(triggered()), this, SLOT(SaveImage()));
  connect(actionLoad_ROI_Images, SIGNAL(triggered()), this, SLOT(LoadDrawing()));


	//connect(drawingPanel, SIGNAL(SaveSeedDrawingButtonClicked()), this, SLOT(SaveSeedDrawing()));

	//connect(drawingPanel, SIGNAL(LoadSeedDrawingButtonClicked()), this, SLOT(LoadSeedDrawing()));
	connect(drawingPanel, SIGNAL(CurrentBrushSizeChanged(int)), this, SLOT(ChangeBrushSize(int)));
	

	connect(recurrencePanel, SIGNAL(SubjectBasedRecurrenceEstimate(std::string, bool, bool, bool, bool, bool, bool,bool)), this, SLOT(StartRecurrenceEstimate(std::string, bool, bool, bool, bool, bool, bool,bool)));
	connect(recurrencePanel, SIGNAL(ExistingModelBasedRecurrenceEstimate(std::string, std::string, std::string, bool, bool, bool, bool, bool, bool, bool)), this, SLOT(RecurrenceEstimateOnExistingModel(std::string, std::string, std::string, bool, bool, bool, bool, bool, bool, bool)));
	connect(recurrencePanel, SIGNAL(TrainNewModel(std::string, std::string, bool, bool, bool, bool, bool, bool,bool)), this, SLOT(TrainNewModelOnGivenData(std::string, std::string, bool, bool, bool, bool, bool, bool,bool)));
	connect(registrationPanel, SIGNAL(Registration(std::string, std::vector<std::string>, std::vector<std::string>)), this, SLOT(Registration(std::string,std::vector<std::string>, std::vector<std::string>)));

	connect(this, SIGNAL(SeedPointsFocused(bool)), tumorPanel, SLOT(sTableFocused(bool)));
	connect(this, SIGNAL(TissuePointsFocused(bool)), tumorPanel, SLOT(tTableFocused(bool)));
	connect(tabWidget, SIGNAL(currentChanged(int)), this, SLOT(SetActiveLandmarkTypeTab(int)));
	connect(infoPanel, SIGNAL(MoveSlicerCursor(double, double, double, int)), this, SLOT(MoveSlicerCursor(double, double, double, int)));

	AxialViewWidget->hide();
	CoronalViewWidget->hide();
	SaggitalViewWidget->hide();
	infoPanel->show();

	windowLabel->setEnabled(false);
	windowSpinBox->setEnabled(false);
	levelLabel->setEnabled(false);
	levelSpinBox->setEnabled(false);
	thresholdLabel->setEnabled(false);
	thresholdSlider->setEnabled(false);
	presetLabel->setEnabled(false);
	presetComboBox->setEnabled(false);

	setAcceptDrops(true);

	mDrawMode = DRAW_MODE_NORMAL;
	mDrawSize = 1;
	statusBar()->showMessage(QString::fromUtf8("Normal Mode"));
	//TestEGFRFunction();

	//------------------------work to do for classification testing-----------------------
	//typedef itk::CSVArray2DFileReader<double> ReaderType;
	//ReaderType::Pointer reader = ReaderType::New();
	//reader->SetFileName("C:/Users/rathores/Downloads/libsvm-3.21/libsvm-3.21/modifiedData.csv");
	//reader->SetFieldDelimiterCharacter(',');
	//reader->HasColumnHeadersOff();
	//reader->HasRowHeadersOff();
	//reader->Parse();
	//typedef vnl_matrix<double> MatrixType;
	//MatrixType data = reader->GetArray2DDataObject()->GetMatrix();
	//
	//FILE *t;
	//t = fopen("TrainingData.txt", "w");

	//for (int i = 0; i < data.rows(); i++)
	//{
	//	fprintf(t, "%f ", data[i][data.cols() - 1]);
	//	for (int j = 0; j < data.cols() - 1; j++)
	//		fprintf(t, "%d:%lf ", j + 1, data[i][j]);
	//	fprintf(t, "\n");
	//}
	//fclose(t);
	//VariableSizeMatrixType dataMatrix;
	//dataMatrix.SetSize(data.rows(), data.cols());
	//for (int i = 0; i < data.rows(); i++)
	//	for (int j = 0; j < data.cols(); j++)
	//		dataMatrix[i][j] = data(i, j);

	//this->mClassificationObj->Training(dataMatrix);
}

fMainWindow::~fMainWindow()
{
	for (int i = 0; i < (int)mSlicerManagers.size(); i++) 
	{
		if (mSlicerManagers[i] != NULL) {
			delete mSlicerManagers[i];
		}
	}

	if (mLandmarks) {
		delete mLandmarks;
	}
	if (mSeedPoints) {
		delete mSeedPoints;
	}
	if (mTissuePoints) {
		delete mTissuePoints;
	}
}

void fMainWindow::help()
{
  fHelpDialog mHelpDlg;
  mHelpDlg.exec();
  //mHelpDlg.setModal(false);
  //mHelpDlg.show();
}


void fMainWindow::about()
{
	QString msg = "Peritumoral Heterogeneity Index (PHI) Estimator\n";
	msg += "  Version " + tr(PROJECT_VERSION) + "\n\n";
	//
  msg += "Algorithm Developer:\n";
  msg += "\tSpyridon Bakas, Ph.D.\n\n";
  //
  msg += "Software Engineers:\n";
  msg += "\tSaima Rathore, Ph.D.\n";
  msg += "\tSarthak Pati, M.Sc.\n\n";
	//
	msg += "Copyright (c) 2015 University of Pennsylvania. All rights reserved.\n";
	msg += "Contact: SBIA Group <sbia-software at uphs.upenn.edu>";
  //
  msg += "\n\nNotes:\n\n- Qt is a registered trademark of The Qt Company Ltd. and/or its subsidiaries. All other trademarks are the property of their respective owners. \n\n";
	//
  msg += "- This package also uses ITK and VTK libraries, which are publised by Kitware, Inc. All rights to these components are subject to the terms and licenses of their respective owners. All trademarks are the property of their respective owners.\n\n";
  //
	QMessageBox box(this);
	box.setIcon(QMessageBox::Information);
	box.addButton(QMessageBox::Ok);
	box.setText(msg);
	box.setWindowTitle(tr("About"));
	box.setFont(mFont);
	box.exec();
}

void fMainWindow::shortcuts()
{
	QString msg = "Keyboard:\n\n";
	msg += "<1-4>:    showing images 1-4\n";
	msg += "<h>:    hide cross hair and landmarks\n";
	msg += "<r>:    reset view geometry\n";
	msg += "<a>:    reset intensity scale (window and level)\n";
	msg += "<space>:    add a landmark\n";
	msg += "<shift> + <space>:    set landmark radius for tumor seed points,\n";
	msg += "        set additional points for tissue points\n\n";
	//
	msg += "<f>:    toggle foreground drawing mode\n";
	msg += "<b>:    toggle background drawing mode\n";
	msg += "<e>:    toggle erase mode in drawing mode\n";
	msg += "<1-9>:    change brush size\n";
	msg += "<esc>:    quit drawing mode\n";
	//
	msg += "\nMouse:\n\n";
	msg += "<left button> + <move>:    move cross hair\n";
	msg += "<wheel up/down>:    move slice\n";
	msg += "<right button> + <move>:    change intensity scale\n";
	msg += "<ctrl> + <wheel up/down>:    change image scale (zoom in/out)\n";
	msg += "<ctrl> + <left button> + <move>:    move image center (pan)\n";
	//
	QMessageBox box(this);
	box.setIcon(QMessageBox::Information);
	box.addButton(QMessageBox::Ok);
	box.setText(msg);
	box.setWindowTitle(tr("Shortcuts"));
	box.setFont(mFont);
	box.exec();
}
void fMainWindow::resizeEvent(QResizeEvent *event)
{
	UpdateRenderWindows();
}

void fMainWindow::OpenImages(int imagetype,std::vector<std::string> directoryname)
{
	if (imagetype == IMAGE_DICOM)
	{
    std::vector<std::string> subDirectories = cbica::subdirectoriesInDirectory(directoryname[0]);
		//check whether it is perfusion only
		/*if (subDirectories.size() == 1)
		{
			if (subDirectories[0] == "PERFUSION")
			{
				LoadNonViewingImages(directoryname[0] + "/" + subDirectories[0], IMAGE_DICOM,  IMAGE_TYPE_PERFUSION, RUNNING_MODE_NORMAL);
				std::vector<std::string> localDirectoryName;
				localDirectoryName.push_back(directoryname[0] + "/" + subDirectories[0]);

				LoadSlicerImages(localDirectoryName, IMAGE_DICOM, IMAGE_TYPE_T1CE,IMAGE_TYPE_PERFUSION, RUNNING_MODE_NORMAL);
				LoadSlicerImages(localDirectoryName, IMAGE_DICOM,IMAGE_TYPE_T2FLAIR, IMAGE_TYPE_PERFUSION, RUNNING_MODE_NORMAL);
			}
			else
				return;
		}*/
		
		//check for other options
		for (unsigned int i = 0; i < subDirectories.size(); i++)
		{
			if (subDirectories[i] == "T1CE")
			{
				std::vector<std::string> localDirectoryName;
				localDirectoryName.push_back(directoryname[0] + "/" + subDirectories[i]);
				t1cePath = directoryname[0] + "/" + subDirectories[i];
				LoadSlicerImages(localDirectoryName, IMAGE_DICOM, IMAGE_TYPE_T1CE, IMAGE_TYPE_UNDEFINED, RUNNING_MODE_NORMAL);
			}
		}

		for (unsigned int i = 0; i < subDirectories.size(); i++)
		{
			if (subDirectories[i] == "T1")
				LoadNonViewingImages(directoryname[0] + "/" + subDirectories[i], IMAGE_DICOM, IMAGE_TYPE_T1, RUNNING_MODE_NORMAL);
			else if (subDirectories[i] == "T2")
				LoadNonViewingImages(directoryname[0] + "/" + subDirectories[i], IMAGE_DICOM, IMAGE_TYPE_T2, RUNNING_MODE_NORMAL);
			else if (subDirectories[i] == "DTI")
				LoadNonViewingImages(directoryname[0] + "/" + subDirectories[i], IMAGE_DICOM, IMAGE_TYPE_DTI, RUNNING_MODE_NORMAL);
			else if (subDirectories[i] == "PERFUSION")
				LoadNonViewingImages(directoryname[0] + "/" + subDirectories[i], IMAGE_DICOM, IMAGE_TYPE_PERFUSION, RUNNING_MODE_NORMAL);

			else if (subDirectories[i] == "FLAIR")
			{
				std::vector<std::string> localDirectoryName;
				localDirectoryName.push_back(directoryname[0] + "/" + subDirectories[i]);
				LoadSlicerImages(localDirectoryName, IMAGE_DICOM,IMAGE_TYPE_T2FLAIR, IMAGE_TYPE_UNDEFINED, RUNNING_MODE_NORMAL);
			}
		}
	}
	else
	{
		LoadSlicerImages(directoryname, IMAGE_TYPE_RECURRENCE_OUTPUT, IMAGE_TYPE_UNDEFINED, IMAGE_TYPE_UNDEFINED, RUNNING_MODE_NORMAL);
	}

}

void fMainWindow::SaveImage()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index < 0 || index >= (int)mSlicerManagers.size()) {
		return;
	}
	//
	QString extensions = IMAGES_EXTENSIONS;
	extensions += ";;All Files (*)";
	QString file;
	QFileDialog dialog(this, tr("Save Image"), mInputPathName, extensions);
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setAcceptMode(QFileDialog::AcceptSave);
	dialog.selectFile(QString(mSlicerManagers[index]->mFileName.c_str()));
	int ret = dialog.exec();
	if (ret == QDialog::Accepted) 
  {
		file = dialog.selectedFiles()[0];
		if (mSlicerManagers[index]->GetPreset() == PRESET_THRESHOLD)
		{
			typedef itk::Image<float, 3> ImageType;
			ImageType::Pointer img = convertVtkToItkFloat(mSlicerManagers[index]->mImage);
			ImageType::RegionType region = img->GetLargestPossibleRegion();
			double threshold = mSlicerManagers[index]->mThreshold;
			//
			typedef itk::ImageDuplicator< ImageType > DuplicatorType;
			DuplicatorType::Pointer duplicator = DuplicatorType::New();
			duplicator->SetInputImage(img);
			duplicator->Update();
			ImageType::Pointer seg = duplicator->GetOutput();
			//
			int vd_x, vd_y, vd_z;
			int i, j, k;
			//
			vd_x = region.GetSize()[0];
			vd_y = region.GetSize()[1];
			vd_z = region.GetSize()[2];
			//	
			for (k = 0; k < vd_z; k++) 
			{
				for (j = 0; j < vd_y; j++) 
				{
					for (i = 0; i < vd_x; i++) 
					{
						ImageType::IndexType pixelIndex;
						pixelIndex[0] = i;
						pixelIndex[1] = j;
						pixelIndex[2] = k;

						if (img->GetPixel(pixelIndex) <= threshold) 
						{
							seg->SetPixel(pixelIndex, 1);
						}
						else 
						{
							seg->SetPixel(pixelIndex, 0);
						}
					}
				}
			}
			//
			typedef itk::ImageFileWriter<ImageType > WriterType;
			WriterType::Pointer writer = WriterType::New();
			writer->SetFileName(file.toStdString());
			writer->SetInput(seg);
			writer->Update();
		}
		else {
			std::string InputPixelType = mSlicerManagers[index]->mImage->GetScalarTypeAsString();
			if (InputPixelType == "short") {
				typedef itk::Image<short, 3> ImageType;
				ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[index]->mImage);
				typedef itk::ImageFileWriter<ImageType > WriterType;
				WriterType::Pointer writer = WriterType::New();
				writer->SetFileName(file.toStdString());
				writer->SetInput(img);
				writer->Update();
			}
			else if (InputPixelType == "unsigned short") {
				typedef itk::Image<unsigned short, 3> ImageType;
				ImageType::Pointer img = convertVtkToItk<unsigned short, 3>(mSlicerManagers[index]->mImage);
				typedef itk::ImageFileWriter<ImageType > WriterType;
				WriterType::Pointer writer = WriterType::New();
				writer->SetFileName(file.toStdString());
				writer->SetInput(img);
				writer->Update();
			}
			else if (InputPixelType == "char") {
				typedef itk::Image<char, 3> ImageType;
				ImageType::Pointer img = convertVtkToItk<char, 3>(mSlicerManagers[index]->mImage);
				typedef itk::ImageFileWriter<ImageType > WriterType;
				WriterType::Pointer writer = WriterType::New();
				writer->SetFileName(file.toStdString());
				writer->SetInput(img);
				writer->Update();
			}
			else if (InputPixelType == "unsigned char") {
				typedef itk::Image<unsigned char, 3> ImageType;
				ImageType::Pointer img = convertVtkToItk<unsigned char, 3>(mSlicerManagers[index]->mImage);
				typedef itk::ImageFileWriter<ImageType > WriterType;
				WriterType::Pointer writer = WriterType::New();
				writer->SetFileName(file.toStdString());
				writer->SetInput(img);
				writer->Update();
			}
			else if (InputPixelType == "int") {
				typedef itk::Image<int, 3> ImageType;
				ImageType::Pointer img = convertVtkToItk<int, 3>(mSlicerManagers[index]->mImage);
				typedef itk::ImageFileWriter<ImageType > WriterType;
				WriterType::Pointer writer = WriterType::New();
				writer->SetFileName(file.toStdString());
				writer->SetInput(img);
				writer->Update();
			}
			else if (InputPixelType == "unsigned int") {
				typedef itk::Image<unsigned int, 3> ImageType;
				ImageType::Pointer img = convertVtkToItk<unsigned int, 3>(mSlicerManagers[index]->mImage);
				typedef itk::ImageFileWriter<ImageType > WriterType;
				WriterType::Pointer writer = WriterType::New();
				writer->SetFileName(file.toStdString());
				writer->SetInput(img);
				writer->Update();
			}
			else if (InputPixelType == "double") {
				typedef itk::Image<double, 3> ImageType;
				ImageType::Pointer img = convertVtkToItk<double, 3>(mSlicerManagers[index]->mImage);
				typedef itk::ImageFileWriter<ImageType > WriterType;
				WriterType::Pointer writer = WriterType::New();
				writer->SetFileName(file.toStdString());
				writer->SetInput(img);
				writer->Update();
			}
			else if (InputPixelType == "float") {
				typedef itk::Image<float, 3> ImageType;
				ImageType::Pointer img = convertVtkToItk<float, 3>(mSlicerManagers[index]->mImage);
				typedef itk::ImageFileWriter<ImageType > WriterType;
				WriterType::Pointer writer = WriterType::New();
				writer->SetFileName(file.toStdString());
				writer->SetInput(img);
				writer->Update();
			}
			else {
				std::cerr << "Error, input pixel type : " << InputPixelType << " unknown !" << std::endl;
			}
		}
	}
}

void fMainWindow::InitMask(vtkImageData* image)
{
	int extent[6];
	double spacing[3];
	double origin[3];
	int i, j, k;

	image->GetExtent(extent);
	image->GetSpacing(spacing);
	image->GetOrigin(origin);

	mMask->Initialize();
	mMask->SetExtent(extent);
	mMask->SetSpacing(spacing);
	mMask->SetOrigin(origin);
#if VTK_MAJOR_VERSION <= 5
	mMask->SetScalarTypeToFloat();
	mMask->SetNumberOfScalarComponents(1);
	mMask->AllocateScalars();
#else
	mMask->AllocateScalars(VTK_FLOAT, 1);
#endif
	{
		int vd_x, vd_y, vd_z;
		vd_x = mMask->GetDimensions()[0];
		vd_y = mMask->GetDimensions()[1];
		vd_z = mMask->GetDimensions()[2];
		float* pData = (float*)mMask->GetScalarPointer();
		for (k = 0; k < vd_z; k++) 
		{
			for (j = 0; j < vd_y; j++) 
			{
				for (i = 0; i < vd_x; i++) 
				{
					*pData = 0;
					pData++;
				}
			}
		}
	}
}
void fMainWindow::SetDrawMode(int mode, int size) 
{
	mDrawMode = mode;
	mDrawSize = size;

	int color[4] = { 0, 0, 0, 0 };

	if (mDrawMode == DRAW_MODE_REC) 
	{
		statusBar()->showMessage(QString::fromUtf8("ReDrawing Mode"));
		//
		color[0] = 255;
		color[1] = 0;
		color[2] = 0;
		color[3] = 255;
	}
	else if (mDrawMode == DRAW_MODE_NREC) 
	{
		statusBar()->showMessage(QString::fromUtf8("Drawing Mode: Far Region"));
		color[0] = 0;
		color[1] = 255;
		color[2] = 0;
		color[3] = 255;
	}
	else if (mDrawMode == DRAW_MODE_SEED)
	{
		statusBar()->showMessage(QString::fromUtf8("Seeding Mode"));
		color[0] = 0;
		color[1] = 0;
		color[2] = 255;
		color[3] = 255;
	}
	else if (mDrawMode == DRAW_MODE_ERASE) 
	{
		statusBar()->showMessage(QString::fromUtf8("Erase Mode"));
		color[0] = 255;
		color[1] = 255;
		color[2] = 255;
		color[3] = 255;
	}
	else if (mDrawMode == DRAW_MODE_NORMAL) 
		statusBar()->showMessage(QString::fromUtf8("Normal Mode"));

	if (mode == DRAW_MODE_REC || mode == DRAW_MODE_NREC || mode == DRAW_MODE_SEED || mode == DRAW_MODE_ERASE)
	{
		QImage img(33, 33, QImage::Format_ARGB32);
		for (int j = -16; j <= 16; j++) 
		{
			for (int i = -16; i <= 16; i++) 
			{
				if (abs(i) <= mDrawSize && abs(j) <= mDrawSize) 
				{
					img.setPixel(i + 16, j + 16, qRgba(color[0], color[1], color[2], color[3]));
				}
				else 
				{
					img.setPixel(i + 16, j + 16, qRgba(0, 0, 0, 0));
				}
				//*/
			}
		}

		AxialViewWidget->setCursor(QCursor(QPixmap::fromImage(img)));
		CoronalViewWidget->setCursor(QCursor(QPixmap::fromImage(img)));
		SaggitalViewWidget->setCursor(QCursor(QPixmap::fromImage(img)));
	}
	else if (mode == DRAW_MODE_NORMAL)
	{
		AxialViewWidget->unsetCursor();
		CoronalViewWidget->unsetCursor();
		SaggitalViewWidget->unsetCursor();
	}
}

void fMainWindow::LoadNonViewingImages(std::string directoryname, int imagetype, int subimagetype,int runningmode)
{
	for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
	{
		if (mNonViewingImageManager[index]->GetPathFileName() == directoryname)
		{
			return;
		}
	}
	SimpleImageManager* nonViewingImage = new SimpleImageManager();
	nonViewingImage->SetPathFileName(directoryname);
	nonViewingImage->SetFileName(directoryname);
	if (imagetype == IMAGE_NIFTI)
		nonViewingImage->ReadGivenNonViewingNiftiImage(directoryname, subimagetype);
	else
		nonViewingImage->ReadGivenNonViewingDicomImage(directoryname, subimagetype);

	if (nonViewingImage->GetLastEncounteredError() != "")
		delete nonViewingImage;
	else
		mNonViewingImageManager.push_back(nonViewingImage);


	//Add image Information in NonViewing images table
	//----------------------------------------------
	if (runningmode == RUNNING_MODE_CLUSTER)
		return;

	nonViewingImage->mImageType = imagetype;
	nonViewingImage->mImageSubType = subimagetype;

	std::string strImageType;
	if (nonViewingImage->mImageSubType == IMAGE_TYPE_PERFUSION)
		strImageType = "PERFUS";
	else if (nonViewingImage->mImageSubType == IMAGE_TYPE_T1)
		strImageType = "T1";
	else if (nonViewingImage->mImageSubType == IMAGE_TYPE_T2)
		strImageType = "T2";
	else if (nonViewingImage->mImageSubType == IMAGE_TYPE_DTI)
		strImageType = "DTI";

	int rowindex = nonViewingImagesTable->rowCount();
	nonViewingImagesTable->setRowCount(rowindex + 1);

	QFileInfo fileinfo(nonViewingImage->GetFileName().c_str());
	QString id = directoryname.c_str() + QString::number(mNonViewingImageManager.size() - 1);
	{
		QTableWidgetItem *item = new QTableWidgetItem(fileinfo.fileName());
		item->setData(Qt::UserRole, directoryname.c_str());

		QTablePushButton* cButton = new QTablePushButton;
		cButton->setItem(item);
		cButton->setText(tr("X"));
		cButton->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);
		cButton->setMaximumWidth(20);

		QString in = QString::fromStdString(strImageType);
		QTableWidgetItem *item2 = new QTableWidgetItem(in);
		item2->setData(Qt::UserRole, in.toStdString().c_str());


		if (nonViewingImage->mImageSubType == IMAGE_TYPE_T1)
			connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseNonViewingT1Image(QTableWidgetItem*)));
		else if (nonViewingImage->mImageSubType == IMAGE_TYPE_T2)
			connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseNonViewingT2Image(QTableWidgetItem*)));
		else if (nonViewingImage->mImageSubType == IMAGE_TYPE_PERFUSION)
			connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseNonViewingPerfusionImage(QTableWidgetItem*)));
		else if (nonViewingImage->mImageSubType == IMAGE_TYPE_DTI)
			connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseNonViewingDTIImage(QTableWidgetItem*)));


		nonViewingImagesTable->setCellWidget(rowindex, TAB_IMAGES_COLUMN_CLOSE, cButton);
		nonViewingImagesTable->setItem(rowindex, TAB_IMAGES_COLUMN_NAME, item);
		nonViewingImagesTable->setItem(rowindex, TAB_IMAGES_COLUMN_TYPE, item2);

		nonViewingImagesTable->resizeRowsToContents();
	}
}
void fMainWindow::LoadSlicerImages(std::vector<std::string> directories, int imagetype, int subimagetype, int loadingtype, bool bSkipDup)
{
	bool bFirstLoad = false;
	if (mSlicerManagers.size() == 0)
	{
		bFirstLoad = true;
	}

	int NumberOfDirectories = (int)directories.size();

	QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

#ifdef USE_PROCESSDIALOG
	fProgressDialog progress("Loading " + directories[0], 1);
	progress.SetProgress(0, NumberOfDirectories);
	qApp->processEvents();
#endif

	int numberofsuccesulreads = 0;
	for (int i = 0; i < NumberOfDirectories; i++)
	{
#ifdef USE_PROCESSDIALOG
		progress.SetText("Loading " + directories[i]);
		progress.SetProgress(i, NumberOfDirectories);
		qApp->processEvents();
#endif
		if (bSkipDup && loadingtype == IMAGE_TYPE_UNDEFINED)
		{
			bool bDup = false;
			for (int j = 0; j < (int)mSlicerManagers.size(); j++)
			{
				if (directories[i] == mSlicerManagers[j]->GetPathFileName())
				{
					bDup = true;
				}
			}
			if (bDup)
			{
				continue;
			}
		}
		SlicerManager* imageManager = new SlicerManager(3, mLandmarks, mSeedPoints, mTissuePoints);
		imageManager->mImageSubType = subimagetype;
		imageManager->mLoadingType = loadingtype; 
		bool SetImageSucceed = false;

		itk::Image<float, 3>::Pointer t1ceImage;
		if (imagetype == IMAGE_NIFTI)
			SetImageSucceed = imageManager->SetImage(directories[i], t1ceImage, IMAGE_NIFTI); 
		else
		{
			if (subimagetype == IMAGE_TYPE_T2FLAIR)
				t1ceImage = mSlicerManagers[0]->mITKImage;
			SetImageSucceed = imageManager->SetImage(directories[i], t1ceImage,IMAGE_DICOM);
		}

		if (!SetImageSucceed)
		{
			QApplication::restoreOverrideCursor();
			QString error = "Cannot open file: \n";
			error += imageManager->GetLastError().c_str();
			QMessageBox::information(this, tr("Reading problem"), error);
			delete imageManager;
		}
		else
		{
			if (mSlicerManagers.size() >= 1)
			{
				char dimension_mismatch_messgae[1024];
				char spacing_mismatch_messgae[1024];
				char origin_mismatch_messgae[1024];


				bool bDiff = false;
				if (imageManager->GetImage()->GetDimensions()[0] != mSlicerManagers[0]->GetImage()->GetDimensions()[0] ||
					imageManager->GetImage()->GetDimensions()[1] != mSlicerManagers[0]->GetImage()->GetDimensions()[1] ||
					imageManager->GetImage()->GetDimensions()[2] != mSlicerManagers[0]->GetImage()->GetDimensions()[2])
				{
					sprintf(dimension_mismatch_messgae, "Dimensions: %d %d %d != %d %d %d",
						imageManager->GetImage()->GetDimensions()[0], imageManager->GetImage()->GetDimensions()[1], imageManager->GetImage()->GetDimensions()[2],
						mSlicerManagers[0]->GetImage()->GetDimensions()[0], mSlicerManagers[0]->GetImage()->GetDimensions()[1], mSlicerManagers[0]->GetImage()->GetDimensions()[2]);
					bDiff = true;
				}
				if (imageManager->GetImage()->GetSpacing()[0] != mSlicerManagers[0]->GetImage()->GetSpacing()[0] ||
					imageManager->GetImage()->GetSpacing()[1] != mSlicerManagers[0]->GetImage()->GetSpacing()[1] ||
					imageManager->GetImage()->GetSpacing()[2] != mSlicerManagers[0]->GetImage()->GetSpacing()[2])
				{
					sprintf(spacing_mismatch_messgae, "Spacing: %.3f %.3f %.3f != %.3f %.3f %.3f",
						imageManager->GetImage()->GetSpacing()[0], imageManager->GetImage()->GetSpacing()[1], imageManager->GetImage()->GetSpacing()[2],
						mSlicerManagers[0]->GetImage()->GetSpacing()[0], mSlicerManagers[0]->GetImage()->GetSpacing()[1], mSlicerManagers[0]->GetImage()->GetSpacing()[2]);
					bDiff = true;
				}
				if (imageManager->GetImage()->GetOrigin()[0] != mSlicerManagers[0]->GetImage()->GetOrigin()[0] ||
					imageManager->GetImage()->GetOrigin()[1] != mSlicerManagers[0]->GetImage()->GetOrigin()[1] ||
					imageManager->GetImage()->GetOrigin()[2] != mSlicerManagers[0]->GetImage()->GetOrigin()[2])
				{
					sprintf(origin_mismatch_messgae, "Origin: %.3f %.3f %.3f != %.3f %.3f %.3f",
						imageManager->GetImage()->GetOrigin()[0], imageManager->GetImage()->GetOrigin()[1], imageManager->GetImage()->GetOrigin()[2],
						mSlicerManagers[0]->GetImage()->GetOrigin()[0], mSlicerManagers[0]->GetImage()->GetOrigin()[1], mSlicerManagers[0]->GetImage()->GetOrigin()[2]);
					bDiff = true;
				}
				if (bDiff)
				{
					/*QApplication::restoreOverrideCursor();
					QString error = origin_mismatch_messgae;
					error += "\n\n";
					error += spacing_mismatch_messgae;
					error += "\n\n";
					error += dimension_mismatch_messgae;

					error += "\n\nNot //consistent with previous images: \n";

					QMessageBox::information(this, tr("Reading problem"), error);
					delete imageManager;
					continue;*/
				}
			}

			imageManager->SetMask(mMask);
			int rowIndex = (int)mSlicerManagers.size();
			imagesTable->setRowCount(rowIndex + 1);
			overlayTable->setRowCount(rowIndex + 1);

			mSlicerManagers.push_back(imageManager);


			QFileInfo fileinfo(imageManager->GetFileName().c_str());
			QString id = directories[i].c_str() + QString::number(mSlicerManagers.size() - 1);
			//
			{
				QTableWidgetItem *item = new QTableWidgetItem(fileinfo.fileName());
				item->setData(Qt::UserRole, id.toStdString().c_str());

				QTablePushButton* cButton = new QTablePushButton;
				cButton->setItem(item);
				cButton->setText(tr("X"));
				cButton->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);
				cButton->setMaximumWidth(20);
				connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseImage(QTableWidgetItem*)));



				std::string strImageType;
				if (subimagetype == IMAGE_TYPE_T1CE)
					strImageType = " T1CE ";
				else if (subimagetype == IMAGE_TYPE_T2FLAIR)
					strImageType = " FLAIR ";
				else if (subimagetype == IMAGE_TYPE_RECURRENCE_OUTPUT)
					strImageType = " R-MAP ";
			
				QLabel * label = new QLabel;
				label->setText(QString::fromStdString(strImageType));
				imagesTable->setCellWidget(rowIndex, TAB_IMAGES_COLUMN_CLOSE, cButton);
				imagesTable->setCellWidget(rowIndex, TAB_IMAGES_COLUMN_TYPE, label);
				imagesTable->setItem(rowIndex, TAB_IMAGES_COLUMN_NAME, item);
				imagesTable->resizeRowsToContents();
				imagesTable->resizeColumnsToContents();
			}
						{
							QTableWidgetItem *item = new QTableWidgetItem(fileinfo.fileName());
							item->setData(Qt::UserRole, id.toStdString().c_str());

							QTablePushButton* cButton = new QTablePushButton;
							cButton->setItem(item);
							cButton->setText(tr("X"));
							cButton->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);
							cButton->setMaximumWidth(20);
							connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseImage(QTableWidgetItem*)));



							std::string strImageType;
							if (subimagetype == IMAGE_TYPE_T1CE)
								strImageType = " T1CE ";
							else if (subimagetype == IMAGE_TYPE_T2FLAIR)
								strImageType = " FLAIR ";
							else if (subimagetype == IMAGE_TYPE_RECURRENCE_OUTPUT)
								strImageType = " R-MAP ";

							QLabel * label = new QLabel;
							label->setText(QString::fromStdString(strImageType));
							overlayTable->setCellWidget(rowIndex, TAB_IMAGES_COLUMN_CLOSE, cButton);
							overlayTable->setCellWidget(rowIndex, TAB_IMAGES_COLUMN_TYPE, label);
							overlayTable->setItem(rowIndex, TAB_IMAGES_COLUMN_NAME, item);
							overlayTable->resizeRowsToContents();
						}

			mSlicerManagers.back()->SetId(id.toStdString());

			connect(mSlicerManagers.back(), SIGNAL(currentImageChanged(std::string)), this, SLOT(CurrentImageChanged(std::string)));
			connect(mSlicerManagers.back(), SIGNAL(currentPickedImageChanged(std::string)), this, SLOT(CurrentPickedImageChanged(std::string)));
			connect(mSlicerManagers.back(), SIGNAL(UpdatePosition(int, double, double, double, double, double, double, double)), this, SLOT(MousePositionChanged(int, double, double, double, double, double, double, double)));
			connect(mSlicerManagers.back(), SIGNAL(WindowLevelChanged()), this, SLOT(WindowLevelChanged()));	
			connect(mSlicerManagers.back(), SIGNAL(UpdateSlice(int, int)), this, SLOT(UpdateSlice(int, int)));
			connect(mSlicerManagers.back(), SIGNAL(UpdateSliceRange(int, int, int)),this, SLOT(UpdateSliceRange(int, int, int)));
			connect(mSlicerManagers.back(), SIGNAL(UpdateLinkManager(std::string, int, double, double, double)),this, SLOT(UpdateLinkManager(std::string, int, double, double, double)));
			connect(mSlicerManagers.back(), SIGNAL(UpdateLinkedNavigation(std::string, SlicerManager*, Slicer*)),this, SLOT(UpdateLinkedNavigation(std::string, SlicerManager*, Slicer*)));
			connect(mSlicerManagers.back(), SIGNAL(ChangeImageWithOrder(SlicerManager*, int)),this, SLOT(ChangeImageWithOrder(SlicerManager*, int)));
			connect(mSlicerManagers.back(), SIGNAL(UpdateBorderWidgetInMain(double, double,double,double)), this, SLOT(UpdateBorderWidget(double, double,double,double)));
			connect(mSlicerManagers.back(), SIGNAL(UpdateBorderWidgetInMain(double, double)), this, SLOT(UpdateBorderWidget(double, double)));

			connect(mSlicerManagers.back(), SIGNAL(SeedPointsAdded()),tumorPanel, SLOT(sAddPoint()));
			connect(mSlicerManagers.back(), SIGNAL(SeedPointsAdded(int, bool)),tumorPanel, SLOT(sAddPoint(int, bool)));
			connect(mSlicerManagers.back(), SIGNAL(TissuePointsAdded(int)),tumorPanel, SLOT(tAddPoint(int)));

			InitSlicers();

			if (bFirstLoad)
			{
				InitMask(mSlicerManagers.back()->mImage);
			}
			{
				for (int i = 0; i < (int)mSlicerManagers.back()->mSlicers.size(); i++)
				{
					mSlicerManagers.back()->mSlicers[i]->SetMask(mSlicerManagers.back()->GetMask());
				}
			}

			if (directories[i].find("scan_label_map") != std::string::npos) {
				mSlicerManagers.back()->SetPreset(PRESET_LABEL);
			}
			if (directories[i].find("gt") != std::string::npos) {
				mSlicerManagers.back()->SetPreset(PRESET_LABEL2);
			}

			numberofsuccesulreads++;
		}
	}
	if (numberofsuccesulreads) 
	{
		AxialViewWidget->show();
		CoronalViewWidget->show();
		SaggitalViewWidget->show();
		infoPanel->show();

		overlayLabel->setEnabled(true);
		overlayUse->setEnabled(true); 
		
		windowLabel->setEnabled(true);
		windowSpinBox->setEnabled(true);
		levelLabel->setEnabled(true);
		levelSpinBox->setEnabled(true);
		presetLabel->setEnabled(true);
		presetComboBox->setEnabled(true);

		if (bFirstLoad)
		{
			for (int i = 0; i < 3; i++)
			{
				mSlicerManagers.back()->GetSlicer(i)->SetInitPosition();
			}
			QTableWidgetItem* item = GetItemFromSlicerManager(mSlicerManagers.back());
			item->setSelected(true);
			DisplayChanged(item);
		}
		else
		{
			QTableWidgetItem* item = NULL;
			for (int i = 0; i < (int)mSlicerManagers.size(); i++)
			{
				item = GetItemFromSlicerManager(mSlicerManagers[i]);
				if (item->isSelected())
				{
					break;
				}
			}
			DisplayChanged(item);
		}

		if (mSlicerManagers.size() > 1)
		{
			for (int i = 0; i < (int)mSlicerManagers.size(); i++)
			{
				for (int j = i + 1; j < (int)mSlicerManagers.size(); j++)
				{
					AddLink(QString::fromStdString(mSlicerManagers[i]->GetId()), QString::fromStdString(mSlicerManagers[j]->GetId()));
				}
			}
		}

		if (bFirstLoad) {
			QTableWidgetItem* item = GetItemFromSlicerManager(mSlicerManagers.back());
			item->setSelected(true);
			DisplayChanged(item);
		}
		else 
    {
			QTableWidgetItem* item = NULL;
			for (int i = 0; i < (int)mSlicerManagers.size(); i++) 
      {
				item = GetItemFromSlicerManager(mSlicerManagers[i]);
				if (item->isSelected()) 
        {
					break;
				}
			}
			DisplayChanged(item);
		}

		InitDisplay();
	}

	QApplication::restoreOverrideCursor();
}

void fMainWindow::CurrentImageChanged(std::string id)
{
	if (id == mCurrentSelectedImageId) {
		return;
	}
	int selected = 0;
	for (int i = 0; i < imagesTable->rowCount(); i++) {
		if (imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString().toStdString() == id) {
			selected = i;
		}
		else {
			imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->setSelected(false);
		}
	}
	imagesTable->item(selected, TAB_IMAGES_COLUMN_NAME)->setSelected(true);
	mCurrentSelectedImageId = id;
	emit SelectedImageHasChanged(mSlicerManagers[selected]);
}

void fMainWindow::CurrentPickedImageChanged(std::string id)
{
	if (id == mCurrentPickedImageId) {
		return;
	}
	int selected = 0;
	for (int i = 0; i < imagesTable->rowCount(); i++) {
		if (imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString().toStdString() == id) {
			selected = i;
		}
		else {
			imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->setSelected(false);
		}
	}
	imagesTable->item(selected, TAB_IMAGES_COLUMN_NAME)->setSelected(true);
	mCurrentPickedImageId = id;
	mCurrentPickedImageIndex = selected;
}

void fMainWindow::ImageInfoChanged()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index < 0 || index >= (int)mSlicerManagers.size()) {
		return;
	}

	vtkSmartPointer<vtkMatrix4x4> transformation;
	QString image = imagesTable->selectedItems()[0]->data(Qt::DisplayRole).toString();

	vtkSmartPointer<vtkImageData> imageSelected;
	vtkSmartPointer<vtkTransform> transformSelected;
	imageSelected = mSlicerManagers[index]->GetSlicer(0)->GetImage();
	transformSelected = mSlicerManagers[index]->GetSlicer(0)->GetTransform();

	//int dimension = GetNumberOfDimensions(imageSelected);
	std::string vtktype = mSlicerManagers[index]->GetImage()->GetScalarTypeAsString();
	QString pixelType = vtktype.c_str();

	infoPanel->setFileName(image);
	infoPanel->setSizePixel(imageSelected->GetDimensions()[0], imageSelected->GetDimensions()[1], imageSelected->GetDimensions()[2]);
	infoPanel->setOrigin(imageSelected->GetOrigin()[0], imageSelected->GetOrigin()[1], imageSelected->GetOrigin()[2]);
	infoPanel->setSpacing(imageSelected->GetSpacing()[0], imageSelected->GetSpacing()[1], imageSelected->GetSpacing()[2]);
	transformation = transformSelected->GetMatrix();
	tumorPanel->SetCurrentSPoints(mSlicerManagers[index]->mSeedPoints);
	tumorPanel->SetCurrentTPoints(mSlicerManagers[index]->mTissuePoints);
	tumorPanel->SetCurrentPath(mInputPathName.toStdString());

	for (int i = 0; i < 3; i++)
	{
		mSlicerManagers[index]->UpdateInfoOnCursorPosition(i);
	}

	WindowLevelChanged();

	// reset SliceManager order
	for (int j = 0; j < (int)mSlicerManagers.size(); j++) {
		mSlicerManagers[j]->SetOrder(j);
	}
	for (int i = 0; i < imagesTable->rowCount(); i++) {
		QString id_table = imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString();
		for (int j = 0; j < (int)mSlicerManagers.size(); j++) {
			QString id_sm = mSlicerManagers[j]->GetId().c_str();
			if (id_table == id_sm) {
				mSlicerManagers[j]->SetOrder(i);
				break;
			}
		}
	}
}

void fMainWindow::DisplayChanged()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) 
	{
		return;
	}
	DisplayChanged(items[0]);
}
void fMainWindow::DisplayChanged(QTableWidgetItem *clickedItem)
{
	int slicerManagerIndex = GetSlicerIndexFromItem(clickedItem);
	if (slicerManagerIndex < 0 || slicerManagerIndex >= (int)mSlicerManagers.size())
	{
		return;
	}

	QTableWidgetItem* clickedParentItem = imagesTable->item(slicerManagerIndex, TAB_IMAGES_COLUMN_NAME);
	for (int i = 0; i < (int)mSlicerManagers.size(); i++)
	{
		QTableWidgetItem* currentParentItem = imagesTable->item(i, TAB_IMAGES_COLUMN_NAME);
		if (currentParentItem != clickedParentItem)
		{
			for (int j = 0; j < 3; j++)
			{
				mSlicerManagers[i]->UpdateSlicer(j, false);
			}
		}
		else
		{
			int VisibleInWindow = 0;
			for (int j = 0; j < 3; j++)
			{
				mSlicerManagers[slicerManagerIndex]->UpdateSlicer(j, true);
				mSlicerManagers[slicerManagerIndex]->UpdateInfoOnCursorPosition(j);
				DisplaySliders(slicerManagerIndex, j);
				//
				if (mSlicerManagers[slicerManagerIndex]->GetSlicer(j)->GetActive())
				{
					VisibleInWindow = j;
				}
			}

			mSlicerManagers[slicerManagerIndex]->Picked();
			mSlicerManagers[slicerManagerIndex]->UpdateViews(VisibleInWindow);
			mSlicerManagers[slicerManagerIndex]->UpdateLinked(VisibleInWindow);
			mSlicerManagers[slicerManagerIndex]->UpdateInfoOnCursorPosition(VisibleInWindow);
		}
		
		for (int j = 0; j < 3; j++) 
		{
			mSlicerManagers[i]->mSlicers[j]->RemoveOverlay();
		}
	}
	overlaySlider->setValue(5);
	QList<QTableWidgetSelectionRange> ranges = overlayTable->selectedRanges();
	if (!ranges.empty())
		overlayTable->setRangeSelected(ranges[0], false);
	UpdateRenderWindows();

	ImageInfoChanged();
}



QString fMainWindow::Get4x4MatrixDoubleAsString(vtkSmartPointer<vtkMatrix4x4> matrix, const int precision)
{
	std::ostringstream strmatrix;

	unsigned width[4];
	for (unsigned int j = 0; j < 4; j++) {
		double absmax = 0.;
		for (unsigned int i = 0; i < 4; i++) {
			absmax = std::max(absmax, vnl_math_abs(matrix->GetElement(i, j)));
		}
		unsigned ndigits = (unsigned)std::max(0., std::log10(absmax)) + 1;
		width[j] = precision + ndigits + 3;
	}

	for (unsigned int i = 0; i < 4; i++) {
		for (unsigned int j = 0; j < 4; j++) {
			strmatrix.setf(ios::fixed, ios::floatfield);
			strmatrix.precision(precision);
			strmatrix.fill(' ');
			strmatrix.width(width[j]);
			strmatrix << std::right << matrix->GetElement(i, j);
		}
		strmatrix << std::endl;
	}
	QString result = strmatrix.str().c_str();
	return result;
}

int fMainWindow::GetSlicerIndexFromItem(QTableWidgetItem* item)
{
	if (item != NULL) {
		QString id = item->data(Qt::UserRole).toString();
		std::string id_string = id.toStdString();
		for (int i = 0; i < imagesTable->rowCount(); i++)
		{
			QString id_table_string = imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString();
			std::string table_string = id_table_string.toStdString();
			if (imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString() == id)
			{
				return i;
			}
		}
	}
	return -1;
}

QTableWidgetItem* fMainWindow::GetItemFromSlicerManager(SlicerManager* sm)
{
	QString id = sm->GetId().c_str();
	for (int i = 0; i < imagesTable->rowCount(); i++)
	{
		if (imagesTable->item(i, TAB_IMAGES_COLUMN_NAME)->data(Qt::UserRole).toString() == id)
		{
			return imagesTable->item(i, TAB_IMAGES_COLUMN_NAME);
		}
	}
	return NULL;
}

void fMainWindow::InitSlicers()
{
	if (mSlicerManagers.size()) {
		mSlicerManagers.back()->GenerateDefaultLookupTable();

		mSlicerManagers.back()->SetSlicerWindow(0, AxialViewWidget->GetRenderWindow());
		mSlicerManagers.back()->SetSlicerWindow(1, CoronalViewWidget->GetRenderWindow());
		mSlicerManagers.back()->SetSlicerWindow(2, SaggitalViewWidget->GetRenderWindow());
	}
}

void fMainWindow::InitDisplay()
{
	if (mSlicerManagers.size()) 
	{
		for (int j = 0; j < 3; j++) 
		{
			InteractorStyleNavigator* style = InteractorStyleNavigator::New();
			style->SetAutoAdjustCameraClippingRange(1);
			for (int i = 0; i < imagesTable->rowCount(); i++) 
			{
				mSlicerManagers[i]->SetInteractorStyleNavigator(j, style);
				//
				mSlicerManagers[i]->UpdateLinkedNavigation(mSlicerManagers[i]->GetSlicer(0), true);
			}
			style->Delete();
		}
	}
}

void fMainWindow::DisplaySliders(int slicer, int window)
{
	int range[2];
	mSlicerManagers[slicer]->GetSlicer(window)->GetSliceRange(range);
	int position = mSlicerManagers[slicer]->GetSlicer(window)->GetSlice();

	bool showVertical = false;
	if (GetNumberOfDimensions(mSlicerManagers[slicer]->GetSlicer(window)->GetImage()) >= 3) {
		showVertical = true;
	}
	if (showVertical) {
		verticalSliders[window]->show();
	}
	else {
		verticalSliders[window]->hide();
	}
	verticalSliders[window]->setRange(range[0], range[1]);
	verticalSliders[window]->setValue(position);
}

void fMainWindow::CloseImage(QTableWidgetItem* item)
{
	int index = GetSlicerIndexFromItem(item);
	if (index < 0 || index >= (int)mSlicerManagers.size())
	{
		return;
	}
	if (mSlicerManagers.size() > 1) 
	{
		for (int k = 0; k < (int)mSlicerManagers.size() - 1; k++)
		{
			if (k != index) 
			{
				RemoveLink(QString::fromStdString(mSlicerManagers[k]->GetId()), QString::fromStdString(mSlicerManagers[index]->GetId()));
			}
		}
	}
	std::vector<SlicerManager*>::iterator Manageriter = mSlicerManagers.begin();
	for (int i = 0; i < index; i++)
	{
		Manageriter++;
	}

	mSlicerManagers[index]->RemoveActors();
	delete mSlicerManagers[index];
	mSlicerManagers.erase(Manageriter);

	overlayTable->removeRow(index); 
	imagesTable->removeRow(index);
	
	if (mSlicerManagers.size() >= 1)
	{
		QTableWidgetItem* item_tmp = GetItemFromSlicerManager(mSlicerManagers.back());
		item_tmp->setSelected(true);
		DisplayChanged(item_tmp);
	}
	else
	{
		this->ResetNumberOfPoints();
		AxialViewWidget->hide();
		CoronalViewWidget->hide();
		SaggitalViewWidget->hide();

		for (int i = 0; i < 3; i++)
		{
			verticalSliders[i]->hide();
		}
		//
		windowLabel->setEnabled(false);
		windowSpinBox->setEnabled(false);
		levelLabel->setEnabled(false);
		levelSpinBox->setEnabled(false);
		thresholdLabel->setEnabled(false);
		thresholdSlider->setEnabled(false);
		presetLabel->setEnabled(false);
		presetComboBox->setEnabled(false);

		mLandmarks->Clear();
		mSeedPoints->Clear();
		mTissuePoints->Clear();
	}

	InitDisplay();
}

void fMainWindow::MousePositionChanged(int visibility, double x, double y, double z, double X, double Y, double Z, double value)
{
	infoPanel->setCurrentInfo(visibility, x, y, z, X, Y, Z, value);
}

void fMainWindow::WindowLevelChanged()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index < 0 || index >= (int)mSlicerManagers.size())
	{
		return;
	}
	windowSpinBox->setValue(mSlicerManagers[index]->GetColorWindow());
	levelSpinBox->setValue(mSlicerManagers[index]->GetColorLevel());
	thresholdSlider->setValue(mSlicerManagers[index]->GetThresholdIndex());
	presetComboBox->setCurrentIndex(mSlicerManagers[index]->GetPreset());
	if (presetComboBox->currentIndex() == PRESET_THRESHOLD)
	{
		thresholdLabel->setEnabled(true);
		thresholdSlider->setEnabled(true);
	}
	else
	{
		thresholdLabel->setEnabled(false);
		thresholdSlider->setEnabled(false);
	}
}

void fMainWindow::WindowLevelEdited()
{
	presetComboBox->setCurrentIndex(PRESET_USER);
	UpdateWindowLevel();
}

void fMainWindow::SetWindowLevel(double w, double l)
{
	windowSpinBox->setValue(w);
	levelSpinBox->setValue(l);
	presetComboBox->setCurrentIndex(PRESET_USER);
	UpdateWindowLevel();
}

void fMainWindow::UpdateWindowLevel()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index >= 0 && index < (int)mSlicerManagers.size()) {
		mSlicerManagers[index]->SetColorWindow(windowSpinBox->value());
		mSlicerManagers[index]->SetColorLevel(levelSpinBox->value());
		mSlicerManagers[index]->SetPreset(presetComboBox->currentIndex());
		mSlicerManagers[index]->Render();
		//
		if (presetComboBox->currentIndex() == PRESET_THRESHOLD) {
			thresholdLabel->setEnabled(true);
			thresholdSlider->setEnabled(true);
		}
		else {
			thresholdLabel->setEnabled(false);
			thresholdSlider->setEnabled(false);
		}
		//
		WindowLevelChanged();
	}
}

void fMainWindow::thresholdSliderChanged()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index >= 0 && index < (int)mSlicerManagers.size()) {
		mSlicerManagers[index]->SetThresholdIndex(thresholdSlider->value());
		mSlicerManagers[index]->SetPreset(mSlicerManagers[index]->GetPreset());
		mSlicerManagers[index]->Render();
		WindowLevelChanged();
	}
}

void fMainWindow::UpdateLinkManager(std::string id, int slicer, double x, double y, double z)
{
	for (int i = 0; i < (int)mSlicerManagers.size(); i++) 
	{
		if (mSlicerManagers[i]->GetId() == id) 
		{
			mSlicerManagers[i]->GetSlicer(slicer)->SetCurrentPosition(x, y, z);
			mSlicerManagers[i]->UpdateViews(slicer);
			break;
		}
	}
}

void fMainWindow::UpdateLinkedNavigation(std::string id, SlicerManager * sm, Slicer* refSlicer)
{
	for (int i = 0; i < (int)mSlicerManagers.size(); i++) 
	{
		if (id == mSlicerManagers[i]->GetId()) 
		{
			mSlicerManagers[i]->UpdateLinkedNavigation(refSlicer);
		}
	}
}

void fMainWindow::CloseImage()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	CloseImage(items[0]);
}

void fMainWindow::ResetTransformationToIdentity()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index >= 0 && index < (int)mSlicerManagers.size()) {
		mSlicerManagers[index]->ResetTransformationToIdentity();
		ImageInfoChanged();
	}
}

void fMainWindow::AddLink(QString image1, QString image2)
{
	unsigned int sm1 = 0;
	unsigned int sm2 = 0;

	for (int i = 0; i < (int)mSlicerManagers.size(); i++) 
	{
		if (image1.toStdString() == mSlicerManagers[i]->GetId()) 
		{
			mSlicerManagers[i]->AddLink(image2.toStdString());
			sm1 = i;
		}
		if (image2.toStdString() == mSlicerManagers[i]->GetId()) 
		{
			mSlicerManagers[i]->AddLink(image1.toStdString());
			sm2 = i;
		}
	}

	emit UpdateLinkedNavigation(mSlicerManagers[sm1]->GetId(), mSlicerManagers[mCurrentPickedImageIndex], mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0));
	emit UpdateLinkedNavigation(mSlicerManagers[sm2]->GetId(), mSlicerManagers[mCurrentPickedImageIndex], mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0));
}

void fMainWindow::RemoveLink(QString image1, QString image2)
{
	for (int i = 0; i < (int)mSlicerManagers.size(); i++) 
	{
		if (image1.toStdString() == mSlicerManagers[i]->GetId()) 
		{
			mSlicerManagers[i]->RemoveLink(image2.toStdString());
		}
		if (image2.toStdString() == mSlicerManagers[i]->GetId()) {
			mSlicerManagers[i]->RemoveLink(image1.toStdString());
		}
	}
}

void fMainWindow::ChangeImageWithOrder(SlicerManager *sm, int order)
{
	if (mSlicerManagers.size() <= 1) {
		return;
	}
	if (order >= (int)mSlicerManagers.size()) {
		return;
	}
	//if (sm != mSlicerManagers[order]) {
	//	return;
	//}

	QTableWidgetItem* item;
	item = GetItemFromSlicerManager(mSlicerManagers[order]);
	item->setSelected(true);
	DisplayChanged(item);
}

void fMainWindow::AxialViewSliderChanged()
{
	static int value = -1;
	if (value == AxialViewSlider->value()) {
		return;
	}
	else {
		value = AxialViewSlider->value();
	}
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index >= 0 && index < (int)mSlicerManagers.size()) {
		if (mSlicerManagers[index]->GetSlicer(0)->GetSlice() != value) {
			mSlicerManagers[index]->GetSlicer(0)->SetSlice(value);
			mSlicerManagers[index]->VerticalSliderHasChanged(0, value);
			mSlicerManagers[index]->UpdateSlice(0);
		}
	}
}

void fMainWindow::SaggitalViewSliderChanged()
{
	static int value = -1;
	if (value == SaggitalViewSlider->value())
	{
		return;
	}
	else {
		value = SaggitalViewSlider->value();
	}
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index >= 0 && index < (int)mSlicerManagers.size()) {
		if (mSlicerManagers[index]->GetSlicer(2)->GetSlice() != value) {
			mSlicerManagers[index]->GetSlicer(2)->SetSlice(value);
			mSlicerManagers[index]->VerticalSliderHasChanged(2, value);
			mSlicerManagers[index]->UpdateSlice(2);
		}
	}
}

void fMainWindow::CoronalViewSliderChanged()
{
	static int value = -1;
	if (value == CoronalViewSlider->value())
	{
		return;
	}
	else
	{
		value = CoronalViewSlider->value();
	}
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index >= 0 && index < (int)mSlicerManagers.size()) {
		if (mSlicerManagers[index]->GetSlicer(1)->GetSlice() != value) {
			mSlicerManagers[index]->GetSlicer(1)->SetSlice(value);
			mSlicerManagers[index]->VerticalSliderHasChanged(1, value);
			mSlicerManagers[index]->UpdateSlice(1);
		}
	}
}

void fMainWindow::UpdateSlice(int slicer, int slice)
{
	if (slicer == 0) {
		AxialViewSlider->setValue(slice);
	}
	else if (slicer == 1) {
		CoronalViewSlider->setValue(slice);
	}
	else if (slicer == 2) {
		SaggitalViewSlider->setValue(slice);
	}
}

void fMainWindow::UpdateSliceRange(int slicer, int min, int max)
{
	int position = int((min + max) / 2);
	if (slicer == 0) {
		AxialViewSlider->setValue(position);
		AxialViewSlider->setRange(min, max);
	}
	else if (slicer == 1) {
		CoronalViewSlider->setValue(position);
		CoronalViewSlider->setRange(min, max);
	}
	else if (slicer == 2) {
		SaggitalViewSlider->setValue(position);
		SaggitalViewSlider->setRange(min, max);
	}
}

void fMainWindow::UpdateRenderWindows()
{
	if (imagesTable->rowCount() <= 0) 
	{
		return;
	}
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	QTableWidgetItem* item = items[0];
	if (item == NULL) {
		return;
	}
	int index = GetSlicerIndexFromItem(item);
	if (index >= 0 && index < (int)mSlicerManagers.size()) 
	{
		mSlicerManagers[index]->Render();
	}
	//*/
}

void fMainWindow::SetActiveLandmarksType(int type, int row, int col)
{
	if (type == LANDMARK_TYPE_LANDMARKS)
	{
		emit LandmarksFocused(true);
		emit SeedPointsFocused(false);
		emit TissuePointsFocused(false);
	}
	else if (type == LANDMARK_TYPE_SEED_POINTS)
	{
		emit LandmarksFocused(false);
		emit SeedPointsFocused(true);
		emit TissuePointsFocused(false);
	}
	else if (type == LANDMARK_TYPE_TISSUE_POINTS)
	{
		emit LandmarksFocused(false);
		emit SeedPointsFocused(false);
		emit TissuePointsFocused(true);
	}
	else
	{
		emit LandmarksFocused(false);
		emit SeedPointsFocused(false);
		emit TissuePointsFocused(false);
	}

	for (int i = 0; i < (int)mSlicerManagers.size(); i++) 
		mSlicerManagers[i]->SetCurrentLandmarksType(type, row, col);

	UpdateRenderWindows();
}
void fMainWindow::SetActiveLandmarkTypeTab(int current)
{
	if (current == TAB_IMAGES)
	{
		SetActiveLandmarksType(LANDMARK_TYPE_NONE, 0, 0);
	}
	else if (current == TAB_TUMOR)
	{
		SetActiveLandmarksType(LANDMARK_TYPE_NONE, 0, 0);
		tumorPanel->SetCurrentSelectedTissueType();
	}
	else if (current == TAB_LANDMARKS)
	{
		SetActiveLandmarksType(LANDMARK_TYPE_LANDMARKS, 0, 0);
	}
}

void fMainWindow::MoveSlicerCursor(double x, double y, double z, int mode)
{
	if (mCurrentPickedImageIndex < 0 || mCurrentPickedImageIndex >= (int)mSlicerManagers.size()) {
		return;
	}
	//
	if (mode == 0) {
		// x, y, z are LPS
		mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->SetCurrentPosition(x, y, z);
		//
		mSlicerManagers[mCurrentPickedImageIndex]->Picked();
		mSlicerManagers[mCurrentPickedImageIndex]->UpdateViews(0);
		mSlicerManagers[mCurrentPickedImageIndex]->UpdateLinked(0);
		mSlicerManagers[mCurrentPickedImageIndex]->UpdateInfoOnCursorPosition(0);
	}
	else if (mode == 1) {
		// x, y, z are pixel
		x = x * mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetSpacing()[0] + mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetOrigin()[0];
		y = y * mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetSpacing()[1] + mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetOrigin()[1];
		z = z * mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetSpacing()[2] + mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->GetInput()->GetOrigin()[2];
		//
		mSlicerManagers[mCurrentPickedImageIndex]->GetSlicer(0)->SetCurrentPosition(x, y, z);
		//
		mSlicerManagers[mCurrentPickedImageIndex]->Picked();
		mSlicerManagers[mCurrentPickedImageIndex]->UpdateViews(0);
		mSlicerManagers[mCurrentPickedImageIndex]->UpdateLinked(0);
		mSlicerManagers[mCurrentPickedImageIndex]->UpdateInfoOnCursorPosition(0);
	}
}

void fMainWindow::NewClassificationRadioChecked()
{
	svmModelButton->setEnabled(false);
	svmModelFileName->setEnabled(false);
}
void fMainWindow::ExistingClassificationRadioChecked()
{
	svmModelFileName->setEnabled(true);
	svmModelButton->setEnabled(true);
}
void fMainWindow::CreateClassificationModelRadioChecked()
{
	existingMaskDirectoryName->setEnabled(true);
	existingMasksButton->setEnabled(true);
}
void fMainWindow::EnableNearRegionDrawing()
{
	if (mDrawMode == DRAW_MODE_REC)
		SetDrawMode(DRAW_MODE_NORMAL, mDrawSize);
	else
		SetDrawMode(DRAW_MODE_REC, mDrawSize);
}
void fMainWindow::EnableFarRegionDrawing()
{
	if (mDrawMode == DRAW_MODE_NREC) 
		SetDrawMode(DRAW_MODE_NORMAL, mDrawSize);
	else
		SetDrawMode(DRAW_MODE_NREC, mDrawSize);
}
void fMainWindow::EscapeMode()
{
	SetDrawMode(DRAW_MODE_NORMAL, mDrawSize);
}
void fMainWindow::EnableInitDrawing()
{
	if (mDrawMode == DRAW_MODE_SEED)
		SetDrawMode(DRAW_MODE_NORMAL,mDrawSize);
	else
		SetDrawMode(DRAW_MODE_SEED, mDrawSize);
	return;
}
VectorVectorDouble fMainWindow::FormulateDrawingPointsForEdemaSegmentation()
{
	VectorVectorDouble Indices;
	if (mSlicerManagers.size() <= 0)
		return Indices;

	typedef itk::Image<short, 3> ImageType;
	ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	IteratorType maskIt(img, img->GetLargestPossibleRegion());
	maskIt.GoToBegin();
	while (!maskIt.IsAtEnd())
	{
		if (maskIt.Get() == 1 || maskIt.Get() == 2)
		{
			std::vector<double> localIndex;
			localIndex.push_back(maskIt.GetIndex()[0]);
			localIndex.push_back(maskIt.GetIndex()[1]);
			localIndex.push_back(maskIt.GetIndex()[2]);

			Indices.push_back(localIndex);
		}
		++maskIt;
	}
	return Indices;
}
VectorVectorDouble fMainWindow::GetNearRegionIndices()
{
	VectorVectorDouble Indices;
	if (mSlicerManagers.size() <= 0)
		return Indices;

	typedef itk::Image<short, 3> ImageType;
	ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	IteratorType maskIt(img, img->GetLargestPossibleRegion());
	maskIt.GoToBegin();
	while (!maskIt.IsAtEnd())
	{
		if (maskIt.Get() == 1)
		{
			std::vector<double> localIndex;
			localIndex.push_back(maskIt.GetIndex()[0]);
			localIndex.push_back(maskIt.GetIndex()[1]);
			localIndex.push_back(maskIt.GetIndex()[2]);

			Indices.push_back(localIndex);
		}
		++maskIt;
	}
	return Indices;
}
VectorVectorDouble fMainWindow::GetFarRegionIndices()
{
	VectorVectorDouble Indices;
	if (mSlicerManagers.size() <= 0)
		return Indices;

	typedef itk::Image<short, 3> ImageType;
	ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	IteratorType maskIt(img, img->GetLargestPossibleRegion());
	maskIt.GoToBegin();
	while (!maskIt.IsAtEnd())
	{
		if (maskIt.Get() == 2)
		{
			std::vector<double> localIndex;
			localIndex.push_back(maskIt.GetIndex()[0]);
			localIndex.push_back(maskIt.GetIndex()[1]);
			localIndex.push_back(maskIt.GetIndex()[2]);

			Indices.push_back(localIndex);
		}
		++maskIt;
	}
	return Indices;
}
VectorVectorDouble fMainWindow::FormulateDrawingPointsForTumorSegmentation()
{
  VectorVectorDouble Indices;
	if (mSlicerManagers.size() <= 0)
		return Indices;

	typedef itk::Image<short, 3> ImageType;
	ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	IteratorType maskIt(img, img->GetLargestPossibleRegion());
	maskIt.GoToBegin();
	while (!maskIt.IsAtEnd())
	{
		if (maskIt.Get() == 3)
		{
			std::vector<double> localIndex;
			localIndex.push_back(maskIt.GetIndex()[0]);
			localIndex.push_back(maskIt.GetIndex()[1]);
			localIndex.push_back(maskIt.GetIndex()[2]);

			Indices.push_back(localIndex);
		}
		++maskIt;
	}
	return Indices;
}

void fMainWindow::SaveDrawing()
{

	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty())
		return;
	int index = GetSlicerIndexFromItem(items[0]);

	QString extensions = IMAGES_EXTENSIONS;
	extensions += ";;All Files (*)";
	QString file;
	QFileDialog dialog(this, tr("Save Image"), mInputPathName, extensions);
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setAcceptMode(QFileDialog::AcceptSave);
	dialog.selectFile(QString(mSlicerManagers[index]->mFileName.c_str()));
	int ret = dialog.exec();
	if (ret == QDialog::Accepted)
	{
		file = dialog.selectedFiles()[0];
		std::string filename = file.toStdString();

		typedef itk::Image<short, 3> ImageType;
		ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);

    std::vector<ImageType::IndexType> finalNearIndices;
    std::vector<ImageType::IndexType> finalFarIndices;



		std::vector<ImageType::IndexType> nearIndices;
		std::vector<ImageType::IndexType> farIndices;
		typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
		IteratorType maskIt(img, img->GetLargestPossibleRegion());
		maskIt.GoToBegin();
		while (!maskIt.IsAtEnd())
		{
			if (maskIt.Get() == 1)
				nearIndices.push_back(maskIt.GetIndex());
			else if (maskIt.Get() == 2)
				farIndices.push_back(maskIt.GetIndex());
			++maskIt;
		}
		std::string errormsg = "";
		if (nearIndices.size() == 0 || farIndices.size() == 0)
			errormsg = "Draw both the near and far region before saving. Currently either one of both of them are missing.";
		QMessageBox box(this);
		box.setIcon(QMessageBox::Information);
		box.addButton(QMessageBox::Ok);

		if (errormsg.length() > 0)
		{
			box.setText(QString::fromStdString(errormsg));
			box.setWindowTitle(tr("Error message"));
			box.setFont(mFont);
			box.exec();
			return;
		}

    std::vector< std::vector < ImageType::IndexType > >  stats = StartEGFREstimateDrawing<ImageType>();
    if (stats[0].size() != nearIndices.size() || stats[1].size() != farIndices.size())
    {
      QMessageBox::StandardButton reply;
      reply = QMessageBox::question(this, "Warning", "Save Near/Far masks using only voxels with corresponding perfusion (DSC) data? (If \"No\" is pressed all originally drawn voxels will be saved.)", QMessageBox::Yes | QMessageBox::No);
      if (reply == QMessageBox::Yes) 
      {
        finalNearIndices = stats[0];
        finalFarIndices = stats[1];
      }
      else 
      {
        finalNearIndices = nearIndices;
        finalFarIndices = farIndices;
      }
    }
    else
    {
      finalNearIndices = nearIndices;
      finalFarIndices = farIndices;
    }
    //Save Near/Far masks using only voxels with perfusion (DSC) data? (If "No" is pressed all drawn voxels will be saved.)

		//save actual near and far region according to the current image type
		std::string InputPixelType = mSlicerManagers[0]->mImage->GetScalarTypeAsString();
		//std::string subjectname = imagesTable->selectedItems()[0]->data(Qt::DisplayRole).toString().toStdString();
		//subjectname = subjectname.substr(0, subjectname.find("_"));
		if (InputPixelType == "short")
		{
			typedef itk::Image<short, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mImage);
      mOutputManager->WriteNearFarMasks<ImageType>(img, filename, finalNearIndices, finalFarIndices);
		}
		else if (InputPixelType == "unsigned short")
		{
			typedef itk::Image<unsigned short, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<unsigned short, 3>(mSlicerManagers[0]->mImage);
      mOutputManager->WriteNearFarMasks<ImageType>(img, filename, finalNearIndices, finalFarIndices);
		}
		else if (InputPixelType == "char")
		{
			typedef itk::Image<char, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<char, 3>(mSlicerManagers[0]->mImage);
      mOutputManager->WriteNearFarMasks<ImageType>(img, filename, finalNearIndices, finalFarIndices);
		}
		else if (InputPixelType == "unsigned char")
		{
			typedef itk::Image<unsigned char, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<unsigned char, 3>(mSlicerManagers[0]->mImage);
      mOutputManager->WriteNearFarMasks<ImageType>(img, filename, finalNearIndices, finalFarIndices);
		}
		else if (InputPixelType == "int")
		{
			typedef itk::Image<int, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<int, 3>(mSlicerManagers[0]->mImage);
      mOutputManager->WriteNearFarMasks<ImageType>(img, filename, finalNearIndices, finalFarIndices);
		}
		else if (InputPixelType == "unsigned int")
		{
			typedef itk::Image<unsigned int, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<unsigned int, 3>(mSlicerManagers[0]->mImage);
      mOutputManager->WriteNearFarMasks<ImageType>(img, filename, finalNearIndices, finalFarIndices);
		}
		else if (InputPixelType == "double")
		{
			typedef itk::Image<double, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<double, 3>(mSlicerManagers[0]->mImage);
      mOutputManager->WriteNearFarMasks<ImageType>(img, filename, finalNearIndices, finalFarIndices);
		}
		else if (InputPixelType == "float")
		{
			typedef itk::Image<float, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<float, 3>(mSlicerManagers[0]->mImage);
      mOutputManager->WriteNearFarMasks<ImageType>(img, filename, finalNearIndices, finalFarIndices);
		}
		else
		{
			std::cerr << "Error, input pixel type : " << InputPixelType << " unknown !" << std::endl;
		}

		std::string msg = "Mask saved in: " + filename;
		box.setText(QString::fromStdString(msg));
		box.setWindowTitle(tr("Information"));
		box.setFont(mFont);
		box.exec();
	}
}

void fMainWindow::SaveSeedDrawing()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty())
		return;
	int index = GetSlicerIndexFromItem(items[0]);

	QString extensions = IMAGES_EXTENSIONS;
	extensions += ";;All Files (*)";
	QString file;
	QFileDialog dialog(this, tr("Save Image"), mInputPathName, extensions);
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setAcceptMode(QFileDialog::AcceptSave);
	dialog.selectFile(QString(mSlicerManagers[index]->mFileName.c_str()));
	int ret = dialog.exec();
	if (ret == QDialog::Accepted)
	{
		file = dialog.selectedFiles()[0];
		std::string filename = file.toStdString();



		typedef itk::Image<short, 3> ImageType;
		ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
		std::vector<ImageType::IndexType> seedIndices;
		typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
		IteratorType maskIt(img, img->GetLargestPossibleRegion());
		maskIt.GoToBegin();
		while (!maskIt.IsAtEnd())
		{
			if (maskIt.Get() == INIT_POINT_LABEL)
				seedIndices.push_back(maskIt.GetIndex());
			++maskIt;
		}
		std::string errormsg = "";
		if (seedIndices.size() == 0)
			errormsg = "Draw seed points before saving.";
		QMessageBox box(this);
		box.setIcon(QMessageBox::Information);
		box.addButton(QMessageBox::Ok);

		if (errormsg.length() > 0)
		{
			box.setText(QString::fromStdString(errormsg));
			box.setWindowTitle(tr("Error message"));
			box.setFont(mFont);
			box.exec();
			return;
		}

		//save actual near and far region according to the current image type
		std::string InputPixelType = mSlicerManagers[0]->mImage->GetScalarTypeAsString();
		std::string subjectname = imagesTable->selectedItems()[0]->data(Qt::DisplayRole).toString().toStdString();
		subjectname = subjectname.substr(0, subjectname.find("_"));
		if (InputPixelType == "short")
		{
			typedef itk::Image<short, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mImage);
			mOutputManager->WriteSeedMasks<ImageType>(img, filename, seedIndices);
		}
		else if (InputPixelType == "unsigned short")
		{
			typedef itk::Image<unsigned short, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<unsigned short, 3>(mSlicerManagers[0]->mImage);
			mOutputManager->WriteSeedMasks<ImageType>(img, filename, seedIndices);
		}
		else if (InputPixelType == "char")
		{
			typedef itk::Image<char, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<char, 3>(mSlicerManagers[0]->mImage);
			mOutputManager->WriteSeedMasks<ImageType>(img, filename, seedIndices);
		}
		else if (InputPixelType == "unsigned char")
		{
			typedef itk::Image<unsigned char, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<unsigned char, 3>(mSlicerManagers[0]->mImage);
			mOutputManager->WriteSeedMasks<ImageType>(img, filename, seedIndices);
		}
		else if (InputPixelType == "int")
		{
			typedef itk::Image<int, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<int, 3>(mSlicerManagers[0]->mImage);
			mOutputManager->WriteSeedMasks<ImageType>(img, filename, seedIndices);
		}
		else if (InputPixelType == "unsigned int")
		{
			typedef itk::Image<unsigned int, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<unsigned int, 3>(mSlicerManagers[0]->mImage);
			mOutputManager->WriteSeedMasks<ImageType>(img, filename, seedIndices);
		}
		else if (InputPixelType == "double")
		{
			typedef itk::Image<double, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<double, 3>(mSlicerManagers[0]->mImage);
			mOutputManager->WriteSeedMasks<ImageType>(img, subjectname, seedIndices);
		}
		else if (InputPixelType == "float")
		{
			typedef itk::Image<float, 3> ImageType;
			ImageType::Pointer img = convertVtkToItk<float, 3>(mSlicerManagers[0]->mImage);
			mOutputManager->WriteSeedMasks<ImageType>(img, filename, seedIndices);
		}
		else
		{
			std::cerr << "Error, input pixel type : " << InputPixelType << " unknown !" << std::endl;
		}

		std::string msg = "Mask saved: " + filename;
		box.setText(QString::fromStdString(msg));
		box.setWindowTitle(tr("Information"));
		box.setFont(mFont);
		box.exec();
	}
}

void fMainWindow::EraseAllTheDrawings()
{
	if (mSlicerManagers.size() <= 0)
		return;

	typedef itk::Image<short, 3> ImageType;
	ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
	std::vector<ImageType::IndexType> Indices;
	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	IteratorType maskIt(img, img->GetLargestPossibleRegion());
	maskIt.GoToBegin();
	while (!maskIt.IsAtEnd())
	{
		if (maskIt.Get() == 1 || maskIt.Get() == 2 || maskIt.Get() == 3)
			Indices.push_back(maskIt.GetIndex());
		++maskIt;
	}

	for (unsigned int i = 0; i < Indices.size(); i++)
	{
		float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)Indices[i][0], (int)Indices[i][1], (int)Indices[i][2]);
		*pData = 0;
	}
	this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
	this->mSlicerManagers[0]->Render();
	//this->ResetNumberOfPoints();
}
void fMainWindow::EraseFarDrawing()
{
	if (mSlicerManagers.size() <= 0)
		return;

	typedef itk::Image<short, 3> ImageType;
	ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
	std::vector<ImageType::IndexType> farIndices;
	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	IteratorType maskIt(img, img->GetLargestPossibleRegion());
	maskIt.GoToBegin();
	while (!maskIt.IsAtEnd())
	{
		if (maskIt.Get() == 2)
			farIndices.push_back(maskIt.GetIndex());
		++maskIt;
	}

	for (unsigned int i = 0; i < farIndices.size(); i++)
	{
		float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)farIndices[i][0], (int)farIndices[i][1], (int)farIndices[i][2]);
		*pData = 0;
	}
	this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
	this->mSlicerManagers[0]->Render();
	UpdateNumberOfPointsInTable();
}
void fMainWindow::EraseInitDrawing()
{
	if (mSlicerManagers.size() <= 0)
		return;

	typedef itk::Image<short, 3> ImageType;
	ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
	std::vector<ImageType::IndexType> farIndices;
	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	IteratorType maskIt(img, img->GetLargestPossibleRegion());
	maskIt.GoToBegin();
	while (!maskIt.IsAtEnd())
	{
		if (maskIt.Get() == 3)
			farIndices.push_back(maskIt.GetIndex());
		++maskIt;
	}

	for (unsigned int i = 0; i < farIndices.size(); i++)
	{
		float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)farIndices[i][0], (int)farIndices[i][1], (int)farIndices[i][2]);
		*pData = 0;
	}
	this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
	this->mSlicerManagers[0]->Render();
	UpdateNumberOfPointsInTable();
}
void fMainWindow::EraseNearDrawing()
{
	if (mSlicerManagers.size() <= 0)
		return;

	typedef itk::Image<short, 3> ImageType;
	ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
	std::vector<ImageType::IndexType> nearIndices;
	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	IteratorType maskIt(img, img->GetLargestPossibleRegion());
	maskIt.GoToBegin();
	while (!maskIt.IsAtEnd())
	{
		if (maskIt.Get() == 1)
			nearIndices.push_back(maskIt.GetIndex());
		++maskIt;
	}

	for (unsigned int i = 0; i < nearIndices.size(); i++)
	{
		float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)nearIndices[i][0], (int)nearIndices[i][1], (int)nearIndices[i][2]);
		*pData = 0;
	}
	this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
	this->mSlicerManagers[0]->Render();
	UpdateNumberOfPointsInTable();
}
void fMainWindow::SetDrawingModeToEraserMode()
{
	if (mDrawMode == DRAW_MODE_ERASE)
		SetDrawMode(DRAW_MODE_NORMAL, mDrawSize);
	else
		SetDrawMode(DRAW_MODE_ERASE, mDrawSize);
}
void fMainWindow::OpenRecurrenceImages()
{
	std::vector<std::string> vector;
	std::string directoryname;
	QString directory = QFileDialog::getExistingDirectory(this, tr("Open Directory"), "/home", QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks | QFileDialog::DontUseNativeDialog);
	if (directory.isNull())
		return;
	else
	{
		mInputPathName = itksys::SystemTools::GetFilenamePath(directory.toStdString()).c_str();
		directoryname = directory.toStdString();
		vector.push_back(directoryname);
	}
	OpenImages(IMAGE_TYPE_RECURRENCE_OUTPUT, vector);
}

void fMainWindow::OpenT1Images()
{

}

void fMainWindow::OpenT2Images()
{

}

void fMainWindow::OpenT1CEImages()
{

}

void fMainWindow::OpenT2FlairImages()
{

}

void fMainWindow::OpenPerfusionImages()
{

}

void fMainWindow::OpenDTIImages()
{

}

void fMainWindow::BulkTestingNiftiData()
{

}
template <class ImageType>
void fMainWindow::FormulateNearFarPoints( std::vector<typename ImageType::IndexType> & nearIndices,  std::vector<typename ImageType::IndexType> & farIndices)
{
	if (mSlicerManagers.size() <= 0)
		return;

	// typedef itk::Image<short, 3> ImageType;
	typename ImageType::Pointer img = convertVtkToItk<typename ImageType::PixelType, 3>(mSlicerManagers[0]->mMask);
	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	IteratorType maskIt(img, img->GetLargestPossibleRegion());
	maskIt.GoToBegin();
	while (!maskIt.IsAtEnd())
	{
		if (maskIt.Get() == 1)
		{
			nearIndices.push_back(maskIt.GetIndex());
		}
		else if (maskIt.Get() == 2)
			farIndices.push_back(maskIt.GetIndex());
		++maskIt;
	}
}
void fMainWindow::TestEGFRFunction()
{
	return;

	//std::vector<std::string> subjectNames;
	//std::vector<std::string> nPaths;
	//std::vector<std::string> fPaths;

	//subjectNames.push_back("AAAA");
	//subjectNames.push_back("AAAB");
	//subjectNames.push_back("AAAC");
	//subjectNames.push_back("AAAE");
	//subjectNames.push_back("AAAF");
	//subjectNames.push_back("AAAG");
	//subjectNames.push_back("AAAK");
	//subjectNames.push_back("AAAL");
	//subjectNames.push_back("AAAM");
	//subjectNames.push_back("AAAN");
	//subjectNames.push_back("AAAP");
	//subjectNames.push_back("AAAR");
	//subjectNames.push_back("AABA");
	//subjectNames.push_back("AABE");
	//subjectNames.push_back("AABH");
	//subjectNames.push_back("AABM");
	//subjectNames.push_back("AABN");
	//subjectNames.push_back("AABO");
	//subjectNames.push_back("AABP");
	//subjectNames.push_back("AABY");
	//subjectNames.push_back("AACV");
	//subjectNames.push_back("AAKY");
	//subjectNames.push_back("AALT");
	//subjectNames.push_back("AALV");
	//subjectNames.push_back("AALX");
	//subjectNames.push_back("AAMD");
	//subjectNames.push_back("AAME");
	//subjectNames.push_back("AAMH");
	//subjectNames.push_back("AANB");
	//subjectNames.push_back("AANG");
	//subjectNames.push_back("AANI");
	//subjectNames.push_back("AANK");
	//subjectNames.push_back("AANP"); subjectNames.push_back("AANR"); subjectNames.push_back("AANX"); subjectNames.push_back("AANZ"); 
	//subjectNames.push_back("AAOB"); subjectNames.push_back("AAOD"); subjectNames.push_back("AAOH"); subjectNames.push_back("AAOK"); 
	//subjectNames.push_back("AAOR"); subjectNames.push_back("AAPB"); subjectNames.push_back("AAPS"); 
	//	subjectNames.push_back("AAPV"); subjectNames.push_back("AAPW"); subjectNames.push_back("AAPY"); subjectNames.push_back("AAPZ"); 
	//	subjectNames.push_back("AAQA"); subjectNames.push_back("AAQB"); subjectNames.push_back("AAQC"); subjectNames.push_back("AAQD"); 
	//	subjectNames.push_back("AAQE"); subjectNames.push_back("AAQF"); subjectNames.push_back("AAQG"); subjectNames.push_back("AAQI"); 
	//	subjectNames.push_back("AAQJ"); subjectNames.push_back("AAQK"); subjectNames.push_back("AAQL"); 
	//	subjectNames.push_back("AAQM"); subjectNames.push_back("AAQN"); subjectNames.push_back("AAQO"); subjectNames.push_back("AAQP"); 
	//	subjectNames.push_back("AAQQ"); subjectNames.push_back("AAQR"); subjectNames.push_back("AAQS"); subjectNames.push_back("AAQT"); 
	//	subjectNames.push_back("AAQU"); subjectNames.push_back("AAQV"); subjectNames.push_back("AAQX"); subjectNames.push_back("AAQY"); 
	//	subjectNames.push_back("AAQZ"); subjectNames.push_back("AARB"); subjectNames.push_back("AARD"); 
	//	subjectNames.push_back("AARF"); subjectNames.push_back("AARH"); subjectNames.push_back("AARO"); subjectNames.push_back("AARW"); 
	//	subjectNames.push_back("AARZ"); subjectNames.push_back("AASA"); subjectNames.push_back("AASE"); subjectNames.push_back("AASF"); 
	//	subjectNames.push_back("AASG"); subjectNames.push_back("AASH"); subjectNames.push_back("AASK"); subjectNames.push_back("AASN"); 
	//	subjectNames.push_back("AASO"); subjectNames.push_back("AASR"); subjectNames.push_back("AASS"); 
	//	subjectNames.push_back("AASU"); subjectNames.push_back("AASV"); subjectNames.push_back("AASW"); subjectNames.push_back("AASX"); 
	//	subjectNames.push_back("AASY"); subjectNames.push_back("AASZ"); subjectNames.push_back("AATA"); subjectNames.push_back("AATB"); 
	//	subjectNames.push_back("AATC"); subjectNames.push_back("AATD"); subjectNames.push_back("AATE"); subjectNames.push_back("AATF"); 
	//	subjectNames.push_back("AATP"); subjectNames.push_back("AATV"); subjectNames.push_back("AATW"); 
	//	subjectNames.push_back("AATX"); subjectNames.push_back("AAUC"); subjectNames.push_back("AAUE"); subjectNames.push_back("AAUF"); 
	//	subjectNames.push_back("AAUM"); subjectNames.push_back("AAUN"); subjectNames.push_back("AAUQ"); subjectNames.push_back("AAUR"); 
	//	subjectNames.push_back("AAVS"); subjectNames.push_back("AAVV"); subjectNames.push_back("AAWA"); subjectNames.push_back("AAWB"); 
	//	subjectNames.push_back("AAWG"); subjectNames.push_back("AAWH"); subjectNames.push_back("AAWI"); 
	//	subjectNames.push_back("AAWJ"); subjectNames.push_back("AAWK"); subjectNames.push_back("AAXI"); subjectNames.push_back("AAXJ"); 
	//	subjectNames.push_back("AAXL"); subjectNames.push_back("AAXM"); subjectNames.push_back("AAXN"); subjectNames.push_back("AAXO"); 
	//	subjectNames.push_back("AAXQ"); subjectNames.push_back("AAXV"); subjectNames.push_back("AAXW"); subjectNames.push_back("AAYI"); subjectNames.push_back("AAYV"); 
	//	subjectNames.push_back("AAYW"); 
	//	subjectNames.push_back("AAZA"); 
	//	subjectNames.push_back("AAZD"); 
	//	subjectNames.push_back("AAZF"); 
	//	subjectNames.push_back("AAZH"); 
	//	subjectNames.push_back("AAZJ");


	//	nPaths.push_back("AAAA_Near_LPS.nii.gz");
	//	nPaths.push_back("AAAB_Near_LPS.nii.gz");
	//	nPaths.push_back("AAAC_Near_LPS.nii.gz");
	//	nPaths.push_back("AAAE_Near_LPS.nii.gz");
	//	nPaths.push_back("AAAF_Near_LPS.nii.gz");
	//	nPaths.push_back("AAAG_Near_LPS.nii.gz");
	//	nPaths.push_back("AAAK_Near_LPS.nii.gz");
	//	nPaths.push_back("AAAL_Near.nii.gz");
	//	nPaths.push_back("AAAM_Near.nii.gz");
	//	nPaths.push_back("AAAN_Near_LPS.nii.gz");
	//	nPaths.push_back("AAAP_Near.nii.gz");
	//	nPaths.push_back("AAAR_Near_LPS.nii.gz");
	//	nPaths.push_back("AABA_Near_LPS.nii.gz");
	//	nPaths.push_back("AABE_Near_LPS.nii.gz");
	//	nPaths.push_back("AABH_Near_LPS.nii.gz");
	//	nPaths.push_back("AABM_Near_LPS.nii.gz");
	//	nPaths.push_back("AABN_Near.nii.gz");
	//	nPaths.push_back("AABO_Near.nii.gz");
	//	nPaths.push_back("AABP_Near_LPS.nii.gz");
	//	nPaths.push_back("AABY_Near_LPS.nii.gz");
	//	nPaths.push_back("AACV_Near_LPS.nii.gz");
	//	nPaths.push_back("AAKY_Near_LPS.nii.gz");
	//	nPaths.push_back("AALT_Near_LPS.nii.gz");
	//	nPaths.push_back("AALV_Near.nii.gz");
	//	nPaths.push_back("AALX_Near.nii.gz");
	//	nPaths.push_back("AAMD_Near_LPS.nii.gz");
	//	nPaths.push_back("AAME_Near_LPS.nii.gz");
	//	nPaths.push_back("AAMH_Near_LPS.nii.gz");
	//	nPaths.push_back("AANB_Near.nii.gz");
	//	nPaths.push_back("AANG_Near_LPS.nii.gz");
	//	nPaths.push_back("AANI_Near.nii.gz");
	//	nPaths.push_back("AANK_Near_LPS.nii.gz");
	//	nPaths.push_back("AANP_Near_LPS.nii.gz");
	//	nPaths.push_back("AANR_Near.nii.gz");
	//	nPaths.push_back("AANX_Near_LPS.nii.gz");
	//	nPaths.push_back("AANZ_Near_LPS.nii.gz");
	//	nPaths.push_back("AAOB_Near_LPS.nii.gz");
	//	nPaths.push_back("AAOD_Near_LPS.nii.gz");
	//	nPaths.push_back("AAOH_Near_LPS.nii.gz");
	//	nPaths.push_back("AAOK0_Near.nii.gz");
	//	nPaths.push_back("AAOR_Near_LPS.nii.gz");
	//	nPaths.push_back("AAPB0_Near.nii.gz");
	//	nPaths.push_back("AAPS0_Near.nii.gz");
	//	nPaths.push_back("AAPV0_Near.nii.gz");
	//	nPaths.push_back("AAPW0_Near.nii.gz");
	//	nPaths.push_back("AAPY0_Near.nii.gz");
	//	nPaths.push_back("AAPZ0_Near.nii.gz");
	//	nPaths.push_back("AAQA_Near_LPS.nii.gz");
	//	nPaths.push_back("AAQB0_Near.nii.gz");
	//	nPaths.push_back("AAQC0_Near.nii.gz");
	//	nPaths.push_back("AAQD0_Near.nii.gz");
	//	nPaths.push_back("AAQE_Near.nii.gz");
	//	nPaths.push_back("AAQF_Near.nii.gz");
	//	nPaths.push_back("AAQG_Near_LPS.nii.gz");
	//	nPaths.push_back("AAQI_Near.nii.gz");
	//	nPaths.push_back("AAQJ_Near.nii.gz");
	//	nPaths.push_back("AAQK_Near_LPS.nii.gz");
	//	nPaths.push_back("AAQL_Near.nii.gz");
	//	nPaths.push_back("AAQM0_Near.nii.gz");
	//	nPaths.push_back("AAQN_Near_LPS.nii.gz");
	//	nPaths.push_back("AAQO0_Near.nii.gz");
	//	nPaths.push_back("AAQP0_Near.nii.gz");
	//	nPaths.push_back("AAQQ0_Near.nii.gz");
	//	nPaths.push_back("AAQR0_Near.nii.gz");
	//	nPaths.push_back("AAQS0_Near.nii.gz");
	//	nPaths.push_back("AAQT0_Near.nii.gz");
	//	nPaths.push_back("AAQU0_Near.nii.gz");
	//	nPaths.push_back("AAQV0_Near.nii.gz");
	//	nPaths.push_back("AAQX_Near_LPS.nii.gz");
	//	nPaths.push_back("AAQY0_Near.nii.gz");
	//	nPaths.push_back("AAQZ_Near_LPS.nii.gz");
	//	nPaths.push_back("AARB0_Near.nii.gz");
	//	nPaths.push_back("AARD0_Near.nii.gz");
	//	nPaths.push_back("AARF0_Near.nii.gz");
	//	nPaths.push_back("AARH0_Near.nii.gz");
	//	nPaths.push_back("AARO0_Near.nii.gz");
	//	nPaths.push_back("AARW0_Near.nii.gz");
	//	nPaths.push_back("AARZ_Near_LPS.nii.gz");
	//	nPaths.push_back("AASA0_Near.nii.gz");
	//	nPaths.push_back("AASE0_Near.nii.gz");
	//	nPaths.push_back("AASF_Near_LPS.nii.gz");
	//	nPaths.push_back("AASG0_Near.nii.gz");
	//	nPaths.push_back("AASH0_Near.nii.gz");
	//	nPaths.push_back("AASK0_Near.nii.gz");
	//	nPaths.push_back("AASN0_Near.nii.gz");
	//	nPaths.push_back("AASO0_Near.nii.gz");
	//	nPaths.push_back("AASR0_Near.nii.gz");
	//	nPaths.push_back("AASS0_Near1.nii.gz");
	//	nPaths.push_back("AASU0_Near.nii.gz");
	//	nPaths.push_back("AASV_Near_LPS.nii.gz");
	//	nPaths.push_back("AASW0_Near.nii.gz");
	//	nPaths.push_back("AASX0_Near.nii.gz");
	//	nPaths.push_back("AASY0_Near.nii.gz");
	//	nPaths.push_back("AASZ0_Near.nii.gz");
	//	nPaths.push_back("AATA0_Near.nii.gz");
	//	nPaths.push_back("AATB_Near_LPS.nii.gz");
	//	nPaths.push_back("AATC0_Near.nii.gz");
	//	nPaths.push_back("AATD0_Near.nii.gz");
	//	nPaths.push_back("AATE0_Near.nii.gz");
	//	nPaths.push_back("AATF0_Near.nii.gz");
	//	nPaths.push_back("AATP_Near_LPS.nii.gz");
	//	nPaths.push_back("AATV_Near_LPS.nii.gz");
	//	nPaths.push_back("AATW_Near_LPS.nii.gz");
	//	nPaths.push_back("AATX0_Near.nii.gz");
	//	nPaths.push_back("AAUC_Near_LPS.nii.gz");
	//	nPaths.push_back("AAUE0_Near.nii.gz");
	//	nPaths.push_back("AAUF_Near_LPS.nii.gz");
	//	nPaths.push_back("AAUM_Near_LPS.nii.gz");
	//	nPaths.push_back("AAUN0_Near.nii.gz");
	//	nPaths.push_back("AAUQ0_Near.nii.gz");
	//	nPaths.push_back("AAUR0_Near.nii.gz");
	//	nPaths.push_back("AAVS_Near_LPS.nii.gz");
	//	nPaths.push_back("AAVV0_Near.nii.gz");
	//	nPaths.push_back("AAWA_Near_LPS.nii.gz");
	//	nPaths.push_back("AAWB0_Near.nii.gz");
	//	nPaths.push_back("AAWG_Near_LPS.nii.gz");
	//	nPaths.push_back("AAWH0_Near.nii.gz");
	//	nPaths.push_back("AAWI_Near_LPS.nii.gz");
	//	nPaths.push_back("AAWJ_Near_LPS.nii.gz");
	//	nPaths.push_back("AAWK_Near_LPS.nii.gz");
	//	nPaths.push_back("AAXI0_Near.nii.gz");
	//	nPaths.push_back("AAXJ0_Near.nii.gz");
	//	nPaths.push_back("AAXL0_Near.nii.gz");
	//	nPaths.push_back("AAXM0_Near.nii.gz");
	//	nPaths.push_back("AAXN0_Near.nii.gz");
	//	nPaths.push_back("AAXO0_Near.nii.gz");
	//	nPaths.push_back("AAXQ0_Near.nii.gz");
	//	nPaths.push_back("AAXV0_Near.nii.gz");
	//	nPaths.push_back("AAXW0_Near.nii.gz");
	//	nPaths.push_back("AAYI0_Near.nii.gz");
	//	nPaths.push_back("AAYV_Near.nii.gz");
	//	nPaths.push_back("AAYW_Near.nii.gz");
	//	nPaths.push_back("AAZA_Near.nii.gz");
	//	nPaths.push_back("AAZD_Near.nii.gz");
	//	nPaths.push_back("AAZF_Near.nii.gz");
	//	nPaths.push_back("AAZH_Near.nii.gz");
	//	nPaths.push_back("AAZJ_Near.nii.gz");


	//	fPaths.push_back("AAAA_Far_LPS.nii.gz");
	//	fPaths.push_back("AAAB_Far_LPS.nii.gz");
	//	fPaths.push_back("AAAC_Far_LPS.nii.gz");
	//	fPaths.push_back("AAAE_Far_LPS.nii.gz");
	//	fPaths.push_back("AAAF_Far_LPS.nii.gz");
	//	fPaths.push_back("AAAG_Far_LPS.nii.gz");
	//	fPaths.push_back("AAAK_Far_LPS.nii.gz");
	//	fPaths.push_back("AAAL_Far.nii.gz");
	//	fPaths.push_back("AAAM_Far.nii.gz");
	//	fPaths.push_back("AAAN_Far_LPS.nii.gz");
	//	fPaths.push_back("AAAP_Far.nii.gz");
	//	fPaths.push_back("AAAR_Far_LPS.nii.gz");
	//	fPaths.push_back("AABA_Far_LPS.nii.gz");
	//	fPaths.push_back("AABE_Far_LPS.nii.gz");
	//	fPaths.push_back("AABH_Far_LPS.nii.gz");
	//	fPaths.push_back("AABM_Far_LPS.nii.gz");
	//	fPaths.push_back("AABN_Far.nii.gz");
	//	fPaths.push_back("AABO_Far.nii.gz");
	//	fPaths.push_back("AABP_Far_LPS.nii.gz");
	//	fPaths.push_back("AABY_Far_LPS.nii.gz");
	//	fPaths.push_back("AACV_Far_LPS.nii.gz");
	//	fPaths.push_back("AAKY_Far_LPS.nii.gz");
	//	fPaths.push_back("AALT_Far_LPS.nii.gz");
	//	fPaths.push_back("AALV_Far.nii.gz");
	//	fPaths.push_back("AALX_Far.nii.gz");
	//	fPaths.push_back("AAMD_Far_LPS.nii.gz");
	//	fPaths.push_back("AAME_Far_LPS.nii.gz");
	//	fPaths.push_back("AAMH_Far_LPS.nii.gz");
	//	fPaths.push_back("AANB_Far.nii.gz");
	//	fPaths.push_back("AANG_Far_LPS.nii.gz");
	//	fPaths.push_back("AANI_Far.nii.gz");
	//	fPaths.push_back("AANK_Far_LPS.nii.gz");
	//	fPaths.push_back("AANP_Far_LPS.nii.gz");
	//	fPaths.push_back("AANR_Far.nii.gz");
	//	fPaths.push_back("AANX_Far_LPS.nii.gz");
	//	fPaths.push_back("AANZ_Far_LPS.nii.gz");
	//	fPaths.push_back("AAOB_Far_LPS.nii.gz");
	//	fPaths.push_back("AAOD_Far_LPS.nii.gz");
	//	fPaths.push_back("AAOH_Far_LPS.nii.gz");
	//	fPaths.push_back("AAOK_Far.nii.gz");
	//	fPaths.push_back("AAOR_Far_LPS.nii.gz");
	//	fPaths.push_back("AAPB0_Far.nii.gz");
	//	fPaths.push_back("AAPS0_Far.nii.gz");
	//	fPaths.push_back("AAPV0_Far.nii.gz");
	//	fPaths.push_back("AAPW0_Far.nii.gz");
	//	fPaths.push_back("AAPY0_Far.nii.gz");
	//	fPaths.push_back("AAPZ0_Far.nii.gz");
	//	fPaths.push_back("AAQA_Far_LPS.nii.gz");
	//	fPaths.push_back("AAQB0_Far.nii.gz");
	//	fPaths.push_back("AAQC0_Far.nii.gz");
	//	fPaths.push_back("AAQD0_Far.nii.gz");
	//	fPaths.push_back("AAQE_Far.nii.gz");
	//	fPaths.push_back("AAQF_Far.nii.gz");
	//	fPaths.push_back("AAQG_Far_LPS.nii.gz");
	//	fPaths.push_back("AAQI_Far.nii.gz");
	//	fPaths.push_back("AAQJ_Far.nii.gz");
	//	fPaths.push_back("AAQK_Far_LPS.nii.gz");
	//	fPaths.push_back("AAQL_Far.nii.gz");
	//	fPaths.push_back("AAQM0_Far.nii.gz");
	//	fPaths.push_back("AAQN_Far_LPS.nii.gz");
	//	fPaths.push_back("AAQO0_Far.nii.gz");
	//	fPaths.push_back("AAQP0_Far.nii.gz");
	//	fPaths.push_back("AAQQ0_Far.nii.gz");
	//	fPaths.push_back("AAQR0_Far.nii.gz");
	//	fPaths.push_back("AAQS0_Far.nii.gz");
	//	fPaths.push_back("AAQT0_Far.nii.gz");
	//	fPaths.push_back("AAQU0_Far.nii.gz");
	//	fPaths.push_back("AAQV0_Far.nii.gz");
	//	fPaths.push_back("AAQX_Far_LPS.nii.gz");
	//	fPaths.push_back("AAQY0_Far.nii.gz");
	//	fPaths.push_back("AAQZ0_Far.nii.gz");
	//	fPaths.push_back("AARB0_Far.nii.gz");
	//	fPaths.push_back("AARD0_Far.nii.gz");
	//	fPaths.push_back("AARF0_Far.nii.gz");
	//	fPaths.push_back("AARH0_Far.nii.gz");
	//	fPaths.push_back("AARO0_Far.nii.gz");
	//	fPaths.push_back("AARW0_Far.nii.gz");
	//	fPaths.push_back("AARZ0_Far.nii.gz");
	//	fPaths.push_back("AASA0_Far.nii.gz");
	//	fPaths.push_back("AASE0_Far.nii.gz");
	//	fPaths.push_back("AASF_Far_LPS.nii.gz");
	//	fPaths.push_back("AASG0_Far.nii.gz");
	//	fPaths.push_back("AASH0_Far.nii.gz");
	//	fPaths.push_back("AASK0_Far.nii.gz");
	//	fPaths.push_back("AASN0_Far.nii.gz");
	//	fPaths.push_back("AASO0_Far.nii.gz");
	//	fPaths.push_back("AASR0_Far.nii.gz");
	//	fPaths.push_back("AASS0_Far1.nii.gz");
	//	fPaths.push_back("AASU0_Far.nii.gz");
	//	fPaths.push_back("AASV_Far_LPS.nii.gz");
	//	fPaths.push_back("AASW0_Far.nii.gz");
	//	fPaths.push_back("AASX0_Far.nii.gz");
	//	fPaths.push_back("AASY0_Far.nii.gz");
	//	fPaths.push_back("AASZ0_Far.nii.gz");
	//	fPaths.push_back("AATA0_Far.nii.gz");
	//	fPaths.push_back("AATB_Far_LPS.nii.gz");
	//	fPaths.push_back("AATC0_Far.nii.gz");
	//	fPaths.push_back("AATD0_Far.nii.gz");
	//	fPaths.push_back("AATE0_Far.nii.gz");
	//	fPaths.push_back("AATF0_Far.nii.gz");
	//	fPaths.push_back("AATP_Far_LPS.nii.gz");
	//	fPaths.push_back("AATV0_Far.nii.gz");
	//	fPaths.push_back("AATW_Far_LPS.nii.gz");
	//	fPaths.push_back("AATX0_Far.nii.gz");
	//	fPaths.push_back("AAUC_Far_LPS.nii.gz");
	//	fPaths.push_back("AAUE0_Far.nii.gz");
	//	fPaths.push_back("AAUF_Far_LPS.nii.gz");
	//	fPaths.push_back("AAUM_Far_LPS.nii.gz");
	//	fPaths.push_back("AAUN0_Far.nii.gz");
	//	fPaths.push_back("AAUQ0_Far.nii.gz");
	//	fPaths.push_back("AAUR0_Far.nii.gz");
	//	fPaths.push_back("AAVS_Far_LPS.nii.gz");
	//	fPaths.push_back("AAVV0_Far.nii.gz");
	//	fPaths.push_back("AAWA_Far_LPS.nii.gz");
	//	fPaths.push_back("AAWB0_Far.nii.gz");
	//	fPaths.push_back("AAWG_Far_LPS.nii.gz");
	//	fPaths.push_back("AAWH_Far_LPS.nii.gz");
	//	fPaths.push_back("AAWI0_Far.nii.gz");
	//	fPaths.push_back("AAWJ_Far_LPS.nii.gz");
	//	fPaths.push_back("AAWK_Far_LPS.nii.gz");
	//	fPaths.push_back("AAXI0_Far.nii.gz");
	//	fPaths.push_back("AAXJ0_Far.nii.gz");
	//	fPaths.push_back("AAXL0_Far.nii.gz");
	//	fPaths.push_back("AAXM0_Far.nii.gz");
	//	fPaths.push_back("AAXN0_Far.nii.gz");
	//	fPaths.push_back("AAXO0_Far.nii.gz");
	//	fPaths.push_back("AAXQ0_Far.nii.gz");
	//	fPaths.push_back("AAXV0_Far.nii.gz");
	//	fPaths.push_back("AAXW0_Far.nii.gz");
	//	fPaths.push_back("AAYI0_Far.nii.gz");
	//	fPaths.push_back("AAYV_Far.nii.gz");
	//	fPaths.push_back("AAYW_Far.nii.gz");
	//	fPaths.push_back("AAZA_Far.nii.gz");
	//	fPaths.push_back("AAZD_Far.nii.gz");
	//	fPaths.push_back("AAZF_Far.nii.gz");
	//	fPaths.push_back("AAZH_Far.nii.gz");
	//	fPaths.push_back("AAZJ_Far.nii.gz");



	////	int a = subjectNames.size();
	////	int b = nPaths.size();
	//std::vector<double> distances;

	//for (int sid = 136; sid <=150; sid++) // why hard coded?
	//{
	//	std::string subjectname = subjectNames[sid];
	//	typedef itk::Image<float, 3> ImageType;
	//	typedef itk::Image<float, 4> PerfusionImageType;
	//	std::vector<ImageType::Pointer> Perfusion_Registered;
	//	Perfusion_Registered.resize(5);

	//	ImageType::Pointer nImagePointer;
	//	ImageType::Pointer fImagePointer;
	//	PerfusionImageType::Pointer pImagePointer;
	//	std::string pPath = "Z:/brain_tumor/Brain_Tumor_2015/Protocols/PreProcessing/" + subjectNames[sid] + "/" + subjectNames[sid] + "0/" + subjectNames[sid] + "0_perf/" + subjectNames[sid] + "0_perf_pp.nii.gz";
	//	std::string nPath = "Z:/EGFR/forSaima/" + subjectNames[sid] + "/" + nPaths[sid];
	//	std::string fPath = "Z:/EGFR/forSaima/" + subjectNames[sid] + "/" + fPaths[sid];

	//	pImagePointer = mNifiDataManager->Read4DNiftiImage(pPath);
	//	nImagePointer = mNifiDataManager->ReadNiftiImage(nPath);
	//	fImagePointer = mNifiDataManager->ReadNiftiImage(fPath);

	//	std::vector<ImageType::IndexType > nearIndices;
	//	std::vector<ImageType::IndexType > farIndices;

	//	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	//	IteratorType nIt(nImagePointer, nImagePointer->GetLargestPossibleRegion());
	//	IteratorType fIt(fImagePointer, nImagePointer->GetLargestPossibleRegion());
	//	nIt.GoToBegin();
	//	while (!nIt.IsAtEnd())
	//	{
	//		if (nIt.Get() != 0)
	//			nearIndices.push_back(nIt.GetIndex());
	//		if (fIt.Get() != 0)
	//			farIndices.push_back(fIt.GetIndex());
	//		++nIt; ++fIt;
	//	}
	//	double distance = mEGFRPredictor->PredictEGFRStatus<ImageType, PerfusionImageType>(pImagePointer, Perfusion_Registered, nearIndices, farIndices, IMAGE_NIFTI);
	//	distances.push_back(distance);
	//}
	//typedef vnl_matrix<double> MatrixType;
	//MatrixType data;
	//data.set_size(1, 138);
	//for (int j = 0; j <138; j++)
	//		data(0, j) = distances[j];
	//typedef itk::CSVNumericObjectFileWriter<double, 1, 138> WriterType;
	//WriterType::Pointer writer = WriterType::New();
	//writer->SetFileName("BC.csv");
	//writer->SetInput(&data);
	//writer->Write();
}

void fMainWindow::StartEGFREstimate()
{
	bool t1ceDataPresent = false;
	bool t2flairDataPresent = false;
	bool perfusionDataPresent = false;
	
	fProgressDialog progress("Pre-Processing", 1);
	progress.SetProgress(0, 100);
	qApp->processEvents();


	UpdateNumberOfPointsInTable();
	
	if (CheckCompletenessOfInputDataForEGFR(t1ceDataPresent, t2flairDataPresent, perfusionDataPresent)==false)
		return;
	
	progress.SetProgress(5, 100);
	qApp->processEvents();

		typedef itk::Image<float, 3> ImageType;
		typedef itk::Image<float, 4> PerfusionImageType;
		typedef ImageType::Pointer ImagePointerType;
		ImagePointerType T1CEImagePointer;
		ImagePointerType T2FlairImagePointer;
		std::vector<ImagePointerType>	PerfusionImagePointer;
		typedef itk::Image<float, 3> InternalImageType;
		itk::MultiResolutionImageRegistrationMethod<ImageType, ImageType>::Pointer Registrar;
		std::vector<ImagePointerType> Perfusion_Registered;
		PerfusionImageType::Pointer perfusionImage = NULL;

		std::vector<ImageType::IndexType> nearIndices;
		std::vector<ImageType::IndexType> farIndices;
		FormulateNearFarPoints<ImageType>(nearIndices, farIndices);
		std::string imagetype = "";

		for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
		{
			if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1CE)
				T1CEImagePointer = mSlicerManagers[index]->mITKImage;
			else
				T2FlairImagePointer = mSlicerManagers[index]->mITKImage;
		}
		for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
		{
			if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_PERFUSION && mNonViewingImageManager[index]->mImageType == IMAGE_DICOM)
			{
				for (unsigned int seriesindex = 0; seriesindex < mNonViewingImageManager[index]->mNVImageSeriesReaderPointer.size(); seriesindex++)
					PerfusionImagePointer.push_back(mNonViewingImageManager[index]->mNVImageSeriesReaderPointer[seriesindex]);
				imagetype = "dicom";
			}
			else if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_PERFUSION && mNonViewingImageManager[index]->mImageType == IMAGE_NIFTI)
			{
				perfusionImage = mNonViewingImageManager[index]->mPerfusionImagePointer;
				imagetype = "nifti";
			}
		}
		Perfusion_Registered.resize(PerfusionImagePointer.size());
		if (imagetype == "dicom")
		{
			Registrar = mPreprocessingObj->Registration<ImageType, InternalImageType>(T1CEImagePointer, PerfusionImagePointer[0]);
			progress.SetProgress(10, 100);
			qApp->processEvents();

			for (unsigned int index = 0; index < PerfusionImagePointer.size(); index++)
			{
				Perfusion_Registered[index] = mPreprocessingObj->ResampleTransform<ImageType>(Registrar, T1CEImagePointer, PerfusionImagePointer[index]);
				progress.SetProgress((index+1)*2+10, 100);
				qApp->processEvents();
			}
		}
    std::vector<double> EGFRStatusParams;
		if (imagetype == "dicom")
      EGFRStatusParams = mEGFRPredictor->PredictEGFRStatus<ImageType, PerfusionImageType>(perfusionImage, Perfusion_Registered, nearIndices, farIndices, IMAGE_DICOM);
		else
      EGFRStatusParams = mEGFRPredictor->PredictEGFRStatus<ImageType, PerfusionImageType>(perfusionImage, Perfusion_Registered, nearIndices, farIndices, IMAGE_NIFTI);

      QString msg;
      msg = "PHI index      = " + QString::number(EGFRStatusParams[0]) + "\n\nNear/Far perfusion drop ratio = " + QString::number(EGFRStatusParams[1] / EGFRStatusParams[2]) + "\n\nNear ROI voxels used = " + QString::number(EGFRStatusParams[3]) + "/" + QString::number(nearIndices.size()) +
        "\nFar ROI voxels used =   " + QString::number(EGFRStatusParams[4]) + "/" + QString::number(farIndices.size());

      QMessageBox box(this);
      box.setIcon(QMessageBox::Information);
      box.addButton(QMessageBox::Ok);
      box.setText(msg);
      box.setWindowTitle(tr("Analysis outcome"));
      box.setFont(mFont);
      box.exec();

}




void fMainWindow::StartRecurrenceEstimate(std::string outputdirectory, bool t1DataPresent, bool t1ceDataPresent, bool t2DataPresent, bool t2flairDataPresent, bool dtiDataPresent, bool perfusionDataPresent, bool distanceDataPresent)
{
	int imagetype;
	bool useOtherModalities = false;

	mOutputManager->SetOutputDirectoryPath(outputdirectory);
	
	UpdateNumberOfPointsInTable();
	if (!this->CheckCompletenessOfInputData(t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, perfusionDataPresent, dtiDataPresent))
		return;

	if (t1DataPresent || t2DataPresent || t1ceDataPresent || t2flairDataPresent || dtiDataPresent)
		useOtherModalities = true;

	typedef itk::Image<float, 3> ImageType;
	typedef ImageType::Pointer ImagePointerType;
	ImagePointerType T1CEImagePointer;
	ImagePointerType T2FlairImagePointer;
	ImagePointerType T1ImagePointer;
	ImagePointerType T2ImagePointer;
	std::vector<ImagePointerType>	PerfusionImagePointer;
	std::vector<ImagePointerType>	DTIImagePointer;

	ImagePointerType FinalT1CEImagePointer;
	ImagePointerType FinalT2FlairImagePointer;
	ImagePointerType FinalT1ImagePointer;
	ImagePointerType FinalT2ImagePointer;
	std::vector<ImagePointerType>	FinalPerfusionImagePointer;
	std::vector<ImagePointerType>	FinalDTIImagePointer;

	std::vector<std::string> filenames;
	std::string t1cebasefilename;
	filenames.resize(6);

	for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
	{
		if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1CE)
		{
			imagetype = mSlicerManagers[index]->mImageType;
			T1CEImagePointer = mSlicerManagers[index]->mITKImage;
			filenames[0] = mSlicerManagers[index]->mFileName;
			t1cebasefilename = mSlicerManagers[index]->mPathFileName;
		}
		else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T2FLAIR)
		{
			T2FlairImagePointer = mSlicerManagers[index]->mITKImage;
			filenames[1] = mSlicerManagers[index]->mFileName;
		}
		else
			std::cout << "unknown message type" << std::endl;
	}
	for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
	{
		if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_T1)
		{
			filenames[2] = mNonViewingImageManager[index]->mFileName;
			for (unsigned int seriesindex = 0; seriesindex < mNonViewingImageManager[index]->mNVImageSeriesReaderPointer.size(); seriesindex++)
				T1ImagePointer = RescaleImageIntensity(mNonViewingImageManager[index]->mNVImageSeriesReaderPointer[seriesindex]);
		}
		else if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_T2)
		{
			filenames[3] = mNonViewingImageManager[index]->mFileName;
			for (unsigned int seriesindex = 0; seriesindex < mNonViewingImageManager[index]->mNVImageSeriesReaderPointer.size(); seriesindex++)
				T2ImagePointer = RescaleImageIntensity(mNonViewingImageManager[index]->mNVImageSeriesReaderPointer[seriesindex]);
		}
		else if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_PERFUSION)
		{
			filenames[4] = mNonViewingImageManager[index]->mFileName;
			for (unsigned int seriesindex = 0; seriesindex < mNonViewingImageManager[index]->mNVImageSeriesReaderPointer.size(); seriesindex++)
				PerfusionImagePointer.push_back(mNonViewingImageManager[index]->mNVImageSeriesReaderPointer[seriesindex]);
		}
		else if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_DTI)
		{
			filenames[5] = mNonViewingImageManager[index]->mFileName;
			for (unsigned int seriesindex = 0; seriesindex < mNonViewingImageManager[index]->mNVImageSeriesReaderPointer.size(); seriesindex++)
				DTIImagePointer.push_back(RescaleImageIntensity(mNonViewingImageManager[index]->mNVImageSeriesReaderPointer[seriesindex]));
		}
	}
	//------------------------------------------noise reduction----------------------------------------------
	if (imagetype == IMAGE_DICOM)
	{
		if (t2flairDataPresent)
			mPreprocessingObj->DenoisingMethodCall<ImageType>(T2FlairImagePointer);
		if (t1ceDataPresent)
			mPreprocessingObj->DenoisingMethodCall<ImageType>(T1CEImagePointer);
		if (t1DataPresent)
			mPreprocessingObj->DenoisingMethodCall<ImageType>(T1ImagePointer);
		if (t2DataPresent)
			mPreprocessingObj->DenoisingMethodCall<ImageType>(T2ImagePointer);
		if (perfusionDataPresent)
			for (unsigned int index = 0; index < PerfusionImagePointer.size(); index++)
				mPreprocessingObj->DenoisingMethodCall<ImageType>(PerfusionImagePointer[index]);
		if (dtiDataPresent)
			for (unsigned int index = 0; index < DTIImagePointer.size(); index++)
				mPreprocessingObj->DenoisingMethodCall<ImageType>(DTIImagePointer[index]);

		mOutputManager->WriteOutput<ImageType>(T1CEImagePointer, T2FlairImagePointer, T1ImagePointer, T2ImagePointer, PerfusionImagePointer, DTIImagePointer, filenames, WRITING_TYPE_SUSAN,
			t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, perfusionDataPresent, dtiDataPresent);

		//------------------------------------------bias correction----------------------------------------------
		ImagePointerType T1_N3Corrected_ImagePointer;
		ImagePointerType T2_N3Corrected_ImagePointer;
		ImagePointerType T1CE_N3Corrected_ImagePointer;
		ImagePointerType T2Flair_N3Corrected_ImagePointer;
		std::vector<ImagePointerType> Perfusion_N3Corrected_ImagePointer;
		std::vector<ImagePointerType> DTI_N3Corrected_ImagePointer;

		if (t2flairDataPresent)
			T2Flair_N3Corrected_ImagePointer = mPreprocessingObj->N3BiasCorrectionMethodCall<ImageType>(T2FlairImagePointer);
		if (t1ceDataPresent)
			T1CE_N3Corrected_ImagePointer = mPreprocessingObj->N3BiasCorrectionMethodCall<ImageType>(T1CEImagePointer);
		if (t1DataPresent)
			T1_N3Corrected_ImagePointer = mPreprocessingObj->N3BiasCorrectionMethodCall<ImageType>(T1ImagePointer);
		if (t2DataPresent)
			T2_N3Corrected_ImagePointer = mPreprocessingObj->N3BiasCorrectionMethodCall<ImageType>(T2ImagePointer);

		if (perfusionDataPresent)
			for (unsigned int index = 0; index < PerfusionImagePointer.size(); index++)
				Perfusion_N3Corrected_ImagePointer[index] = mPreprocessingObj->N3BiasCorrectionMethodCall<ImageType>(PerfusionImagePointer[index]);
		if (dtiDataPresent)
			for (unsigned int index = 0; index < DTIImagePointer.size(); index++)
				DTI_N3Corrected_ImagePointer[index] = mPreprocessingObj->N3BiasCorrectionMethodCall<ImageType>(DTIImagePointer[index]);

		mOutputManager->WriteOutput<ImageType>(T1CE_N3Corrected_ImagePointer, T2Flair_N3Corrected_ImagePointer, T1_N3Corrected_ImagePointer, T2_N3Corrected_ImagePointer, Perfusion_N3Corrected_ImagePointer, DTI_N3Corrected_ImagePointer, filenames, WRITING_TYPE_BIASCORRECTION,
			t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, perfusionDataPresent, dtiDataPresent);

		//-------------------------------------------------------registration------------------------------------------
		typedef itk::Image<float, 3> InternalImageType;
		itk::MultiResolutionImageRegistrationMethod<ImageType, ImageType>::Pointer Registrar;
		ImagePointerType T2Flair_Registered = T2Flair_N3Corrected_ImagePointer;
		ImagePointerType T1_Registered;
		ImagePointerType T2_Registered;
		std::vector<ImagePointerType> Perfusion_Registered;
		Perfusion_Registered.resize(Perfusion_N3Corrected_ImagePointer.size());
		std::vector<ImagePointerType> DTI_Registered;
		DTI_Registered.resize(DTI_N3Corrected_ImagePointer.size());

		if (t2DataPresent)
		{
			Registrar = mPreprocessingObj->Registration<ImageType, InternalImageType>(T1CE_N3Corrected_ImagePointer, T2_N3Corrected_ImagePointer);
			T2_Registered = mPreprocessingObj->ResampleTransform<ImageType>(Registrar, T1CE_N3Corrected_ImagePointer, T2_N3Corrected_ImagePointer);
		}
		if (t1DataPresent)
		{
			Registrar = mPreprocessingObj->Registration<ImageType, InternalImageType>(T1CE_N3Corrected_ImagePointer, T1_N3Corrected_ImagePointer);
			T1_Registered = mPreprocessingObj->ResampleTransform<ImageType>(Registrar, T1CE_N3Corrected_ImagePointer, T1_N3Corrected_ImagePointer);
		}
		if (perfusionDataPresent)
		{
			Registrar = mPreprocessingObj->Registration<ImageType, InternalImageType>(T1CE_N3Corrected_ImagePointer, Perfusion_N3Corrected_ImagePointer[0]);
			for (unsigned int index = 0; index < Perfusion_N3Corrected_ImagePointer.size(); index++)
				Perfusion_Registered[index] = mPreprocessingObj->ResampleTransform<ImageType>(Registrar, T1CE_N3Corrected_ImagePointer, Perfusion_N3Corrected_ImagePointer[index]);
		}
		if (dtiDataPresent)
		{
			Registrar = mPreprocessingObj->Registration<ImageType, InternalImageType>(T1CE_N3Corrected_ImagePointer, DTI_N3Corrected_ImagePointer[0]);
			for (unsigned int index = 0; index < DTI_N3Corrected_ImagePointer.size(); index++)
				DTI_Registered[index] = mPreprocessingObj->ResampleTransform<ImageType>(Registrar, T1CE_N3Corrected_ImagePointer, DTI_N3Corrected_ImagePointer[index]);
		}
		mOutputManager->WriteOutput<ImageType>(T1CE_N3Corrected_ImagePointer, T2Flair_Registered, T1_Registered, T2_Registered, Perfusion_Registered, DTI_Registered, filenames, WRITING_TYPE_REGISTRATION, t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, perfusionDataPresent, dtiDataPresent);

		FinalT1ImagePointer = T1_Registered;
		FinalT2ImagePointer = T2_Registered;
		FinalT1CEImagePointer = T1CE_N3Corrected_ImagePointer;
		FinalT2FlairImagePointer = T2Flair_N3Corrected_ImagePointer;
		FinalPerfusionImagePointer = Perfusion_Registered;
		FinalDTIImagePointer = DTI_Registered;
	}
	else
	{
		FinalT1ImagePointer		= T1ImagePointer;
		FinalT2ImagePointer		= T2ImagePointer;
		FinalT1CEImagePointer	= T1CEImagePointer;
		FinalT2FlairImagePointer = T2FlairImagePointer;
		FinalPerfusionImagePointer = PerfusionImagePointer;
		FinalDTIImagePointer = DTIImagePointer;
	}

	//--------------------------------------------load edema and tumor mask--------------------------------------------------
	VectorVectorDouble allDrawingPoints = FormulateDrawingPointsForEdemaSegmentation();
	ImagePointerType edemaMask = mPreprocessingObj->Edema3DSegmentationInGivenImage<ImageType>(FinalT2FlairImagePointer, allDrawingPoints);

	typedef itk::Image<short, 3> ImageTypeGeodesic;
	ImageTypeGeodesic::Pointer Inp = ImageTypeGeodesic::New();
	ImageTypeGeodesic::Pointer tumorMask = ImageTypeGeodesic::New();
	ImageType::Pointer tumorMaskFinal = ImageType::New();

	typedef itk::CastImageFilter<ImageType, ImageTypeGeodesic> CastFilterType;
	CastFilterType::Pointer castFilter = CastFilterType::New();
	castFilter->SetInput(FinalT1CEImagePointer);
	Inp = castFilter->GetOutput();
	Inp->Update();

	if (distanceDataPresent)
	{
		VectorVectorDouble tumorPoints = FormulateDrawingPointsForTumorSegmentation();
		//tumorMask = mGeodesicSegmentation->GeodesicSegmentation3D<ImageTypeGeodesic>(Inp, tumorPoints);

		typedef itk::CastImageFilter<ImageTypeGeodesic, ImageType> CastFilterBackType;
		CastFilterBackType::Pointer castFilter2 = CastFilterBackType::New();
		castFilter2->SetInput(tumorMask);
		tumorMaskFinal = castFilter2->GetOutput();
		tumorMaskFinal->Update();
	}
	mRecurrenceEstimator->RecurrenceEstimateOnGivenSubject<ImageType>(edemaMask, tumorMaskFinal, FinalT1CEImagePointer, FinalT2FlairImagePointer, FinalT1ImagePointer, FinalT2ImagePointer, FinalPerfusionImagePointer, FinalDTIImagePointer, IMAGE_NIFTI, t1DataPresent, t2DataPresent, t1ceDataPresent, t2flairDataPresent, dtiDataPresent, perfusionDataPresent, distanceDataPresent, useOtherModalities, t1cebasefilename, this->GetNearRegionIndices(),this->GetFarRegionIndices());
	ShowMessage("Recurrence estimate for the given subject has been saved at the specified location.");
}

template<class ImageType>
void fMainWindow::DisplayRecurrenceEstimationOutput(typename ImageType::Pointer RecurrenceProbabilityMap/*, typename ImageType::Pointer NonRecurrenceProbabilityMap, typename ImageType::Pointer LabelMap*/)
{
	typedef itk::ImageToVTKImageFilter<ImageType> ConnectorType;
	typename ConnectorType::Pointer connector = ConnectorType::New();
	connector->SetInput(RecurrenceProbabilityMap);

	vtkSmartPointer<vtkImageViewer2> imageViewer = vtkSmartPointer<vtkImageViewer2>::New();
	imageViewer->SetInput(connector->GetOutput());

	vtkSmartPointer<vtkTextProperty> sliceTextProp = vtkSmartPointer<vtkTextProperty>::New();
	sliceTextProp->SetFontFamilyToCourier();
	sliceTextProp->SetFontSize(20);
	sliceTextProp->SetVerticalJustificationToBottom();
	sliceTextProp->SetJustificationToLeft();

	vtkSmartPointer<vtkTextMapper> sliceTextMapper = vtkSmartPointer<vtkTextMapper>::New();
	std::string msg = StatusMessage::Format(imageViewer->GetSliceMin(), imageViewer->GetSliceMax());
	sliceTextMapper->SetInput(msg.c_str());
	sliceTextMapper->SetTextProperty(sliceTextProp);

	vtkSmartPointer<vtkActor2D> sliceTextActor = vtkSmartPointer<vtkActor2D>::New();
	sliceTextActor->SetMapper(sliceTextMapper);
	sliceTextActor->SetPosition(15, 10);

	vtkSmartPointer<vtkTextProperty> usageTextProp = vtkSmartPointer<vtkTextProperty>::New();
	usageTextProp->SetFontFamilyToCourier();
	usageTextProp->SetFontSize(14);
	usageTextProp->SetVerticalJustificationToTop();
	usageTextProp->SetJustificationToLeft();

	vtkSmartPointer<vtkTextMapper> usageTextMapper = vtkSmartPointer<vtkTextMapper>::New();
	usageTextMapper->SetInput("- Slice with mouse wheel\n  or Up/Down-Key\n- Zoom with pressed right\n  mouse button while dragging");
	usageTextMapper->SetTextProperty(usageTextProp);

	vtkSmartPointer<vtkActor2D> usageTextActor = vtkSmartPointer<vtkActor2D>::New();
	usageTextActor->SetMapper(usageTextMapper);
	usageTextActor->GetPositionCoordinate()->SetCoordinateSystemToNormalizedDisplay();
	usageTextActor->GetPositionCoordinate()->SetValue(0.05, 0.95);

	vtkSmartPointer<vtkRenderWindowInteractor> renderWindowInteractor = vtkSmartPointer<vtkRenderWindowInteractor>::New();

	vtkSmartPointer<myVtkInteractorStyleImage> myInteractorStyle = vtkSmartPointer<myVtkInteractorStyleImage>::New();

	myInteractorStyle->SetImageViewer(imageViewer);
	myInteractorStyle->SetStatusMapper(sliceTextMapper);

	imageViewer->SetupInteractor(renderWindowInteractor);
	renderWindowInteractor->SetInteractorStyle(myInteractorStyle);
	imageViewer->GetRenderer()->AddActor2D(sliceTextActor);
	imageViewer->GetRenderer()->AddActor2D(usageTextActor);

	imageViewer->Render();
	imageViewer->GetRenderer()->ResetCamera();
	imageViewer->Render();
	renderWindowInteractor->Start();
}

void fMainWindow::SetMasksTable()
{
	//	masksTable->horizontalHeaderItem(0)->setText(QApplication::translate("fMainWindow", "Mask type", 0, QApplication::UnicodeUTF8));
	//masksTable->horizontalHeaderItem(0)->setText(QApplication::translate("fMainWindow", " # ", 0, QApplication::UnicodeUTF8));


	//masksTable->verticalHeaderItem(MASK_RECURRENCE_ROW)->setText(QApplication::translate("fMainWindow", "Recurrence", 0, QApplication::UnicodeUTF8));
	//masksTable->verticalHeaderItem(MASK_NON_RECURRENCE_ROW)->setText(QApplication::translate("fMainWindow", "NonRecurrence e", 0, QApplication::UnicodeUTF8));
	//masksTable->verticalHeaderItem(MASK_TOTAL_ROW)->setText(QApplication::translate("fMainWindow", "Total", 0, QApplication::UnicodeUTF8));

	//masksTable->resizeRowsToContents();
	////masksTable->resizeColumnsToContents();
	//masksTable->setColumnWidth(0, 130);
}

void fMainWindow::SetNonViewingImagesTable()
{
	nonViewingImagesTable->horizontalHeaderItem(TAB_IMAGES_COLUMN_CLOSE)->setText(QApplication::translate("fMainWindow", "X", 0, QApplication::UnicodeUTF8));
	nonViewingImagesTable->horizontalHeaderItem(TAB_IMAGES_COLUMN_TYPE)->setText(QApplication::translate("fMainWindow", "Type", 0, QApplication::UnicodeUTF8));
  nonViewingImagesTable->horizontalHeaderItem(TAB_IMAGES_COLUMN_NAME)->setText(QApplication::translate("fMainWindow", "      Name     ", 0, QApplication::UnicodeUTF8));

	imagesTable->horizontalHeaderItem(TAB_IMAGES_COLUMN_CLOSE)->setText(QApplication::translate("fMainWindow", "X", 0, QApplication::UnicodeUTF8));
	imagesTable->horizontalHeaderItem(TAB_IMAGES_COLUMN_TYPE)->setText(QApplication::translate("fMainWindow", "Type", 0, QApplication::UnicodeUTF8));
	imagesTable->horizontalHeaderItem(TAB_IMAGES_COLUMN_NAME)->setText(QApplication::translate("fMainWindow", "      Name     ", 0, QApplication::UnicodeUTF8));

	overlayTable->horizontalHeaderItem(TAB_IMAGES_COLUMN_CLOSE)->setText(QApplication::translate("fMainWindow", "X", 0, QApplication::UnicodeUTF8));
	overlayTable->horizontalHeaderItem(TAB_IMAGES_COLUMN_TYPE)->setText(QApplication::translate("fMainWindow", "Type", 0, QApplication::UnicodeUTF8));
	overlayTable->horizontalHeaderItem(TAB_IMAGES_COLUMN_NAME)->setText(QApplication::translate("fMainWindow", "      Name     ", 0, QApplication::UnicodeUTF8));


	imagesTable->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
	//masksTable->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
	nonViewingImagesTable->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
	overlayTable->horizontalHeader()->setResizeMode(QHeaderView::Stretch);

	imagesTable->setColumnWidth(TAB_IMAGES_COLUMN_CLOSE, 40);
	imagesTable->setColumnWidth(TAB_IMAGES_COLUMN_TYPE, 150);
	imagesTable->setColumnWidth(TAB_IMAGES_COLUMN_NAME, 125);

	nonViewingImagesTable->setColumnWidth(TAB_IMAGES_COLUMN_CLOSE, 40);
	nonViewingImagesTable->setColumnWidth(TAB_IMAGES_COLUMN_TYPE, 150);
	nonViewingImagesTable->setColumnWidth(TAB_IMAGES_COLUMN_NAME, 125);


}

void fMainWindow::CloseNonViewingT1Image(QTableWidgetItem* item)
{
	int itemIndexToBeDeleted = 0;
	nonViewingImagesTable->removeRow(item->row());

	for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
	{
		if (mNonViewingImageManager[index]->mImageType == IMAGE_TYPE_T1)
		{
			itemIndexToBeDeleted = index;
			delete mNonViewingImageManager[index];
			break;
		}
	}
	std::vector<SimpleImageManager*>::iterator simpleImageIterator = mNonViewingImageManager.begin();
	for (int i = 0; i < itemIndexToBeDeleted; i++)
		simpleImageIterator++;
	mNonViewingImageManager.erase(simpleImageIterator);
}
void fMainWindow::CloseNonViewingT2Image(QTableWidgetItem* item)
{
	int itemIndexToBeDeleted = 0;
	nonViewingImagesTable->removeRow(item->row());

	for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
	{
		if (mNonViewingImageManager[index]->mImageType == IMAGE_TYPE_T2)
		{
			itemIndexToBeDeleted = index;
			delete mNonViewingImageManager[index];
			break;
		}
	}
	std::vector<SimpleImageManager*>::iterator simpleImageIterator = mNonViewingImageManager.begin();
	for (int i = 0; i < itemIndexToBeDeleted; i++)
		simpleImageIterator++;
	mNonViewingImageManager.erase(simpleImageIterator);
}

void fMainWindow::CloseNonViewingPerfusionImage(QTableWidgetItem* item)
{
	int itemIndexToBeDeleted = 0;
	nonViewingImagesTable->removeRow(item->row());

	for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
	{
		if (mNonViewingImageManager[index]->mImageType == IMAGE_TYPE_PERFUSION)
		{
			itemIndexToBeDeleted = index;
			delete mNonViewingImageManager[index];
			break;
		}
	}
	std::vector<SimpleImageManager*>::iterator simpleImageIterator = mNonViewingImageManager.begin();
	for (int i = 0; i < itemIndexToBeDeleted; i++)
		simpleImageIterator++;
	mNonViewingImageManager.erase(simpleImageIterator);
}
void fMainWindow::CloseNonViewingDTIImage(QTableWidgetItem* item)
{
	int itemIndexToBeDeleted = 0;
	nonViewingImagesTable->removeRow(item->row());

	for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
	{
		if (mNonViewingImageManager[index]->mImageType == IMAGE_TYPE_DTI)
		{
			itemIndexToBeDeleted = index;
			delete mNonViewingImageManager[index];
			break;
		}
	}
	std::vector<SimpleImageManager*>::iterator simpleImageIterator = mNonViewingImageManager.begin();
	for (int i = 0; i < itemIndexToBeDeleted; i++)
		simpleImageIterator++;
	mNonViewingImageManager.erase(simpleImageIterator);
}

void fMainWindow::UpdateNumberOfPointsInTable()
{
	if (mSlicerManagers.size() <= 0)
		return;

	typedef itk::Image<short, 3> ImageType;
	ImageType::Pointer img = convertVtkToItk<short, 3>(mSlicerManagers[0]->mMask);
	int nearCounter=0;
	int  farCounter=0;
	int  initCounter = 0;
	typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
	IteratorType maskIt(img, img->GetLargestPossibleRegion());
	maskIt.GoToBegin();
	while (!maskIt.IsAtEnd())
	{
		if (maskIt.Get() == 1)
			nearCounter++;
		else if (maskIt.Get() == 2)
			farCounter++;
		else if (maskIt.Get() == 3)
			initCounter++;
		++maskIt;
	}
	mCurrentNearPoints = nearCounter;
	mCurrentFarPoints = farCounter;
	mCurrentInitPoints = initCounter;
}

void fMainWindow::SetPresetComboBox()
{
	presetComboBox->addItem("Auto Scale");
	presetComboBox->addItem("User Scale");
	presetComboBox->addItem("Label Map");
	presetComboBox->addItem("Label map 2");
	presetComboBox->addItem("Threshold");
	presetComboBox->addItem("Probability");
}
void fMainWindow::ResetNumberOfPoints()
{
	UpdateNumberOfPointsInTable();
}
void fMainWindow::GetOutputDirectory()
{
	QString directory = QFileDialog::getExistingDirectory(this, tr("Open Directory"), "/home", QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks | QFileDialog::DontUseNativeDialog);
	if (!directory.isNull())
	{
		mOutputManager->SetOutputDirectoryPath(directory.toStdString());
		outputDirectoryName->setText(directory);
		if (!mOutputManager->SetupOutputFolders())
			return;
	}
}
bool fMainWindow::CheckCompletenessOfInputData(bool & t1DataPresent, bool & t2DataPresent, bool & t1ceDataPresent, bool & t2flairDataPresent, bool & perfusionDataPresent, bool & dtiDataPresent)
{
	std::string msg = "Missing elements:\n";


	for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
	{
		if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_T1)
			t1DataPresent = true;
		else if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_T2)
			t2DataPresent = true;
		else if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_PERFUSION)
			perfusionDataPresent = true;
		else if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_DTI)
			dtiDataPresent = true;
	}

	for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
	{
		if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1CE)
			t1ceDataPresent = true;
		else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T2FLAIR)
			t2flairDataPresent = true;
	}
	if (mOutputManager->GetOutputDirectoryPath() == "")
		msg = "Output directory";
	if (t1ceDataPresent == false && t2flairDataPresent == false)
		msg = msg + "\n" + "T1CE Data.";
	if (mCurrentNearPoints == 0)
		msg = msg + "\n" + "Recurrence mask.";
	if (mCurrentFarPoints == 0)
		msg = msg + "\n" + "Non-Recurrence mask.";

	if (msg != "Missing elements:\n")
	{
		QMessageBox box(this);
		box.setIcon(QMessageBox::Information);
		box.addButton(QMessageBox::Ok);
		box.setText(QString::fromStdString(msg));
		box.setWindowTitle(tr("Missing Data"));
		box.setFont(mFont);
		box.exec();
		return false;
	}
	else
		return true;
}
bool fMainWindow::CheckCompletenessOfInputDataForEGFR(bool & t1ceDataPresent, bool & t2flairDataPresent, bool & perfusionDataPresent)
{
	std::string msg = "Missing elements:\n";
	for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
	{
		if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_PERFUSION)
			perfusionDataPresent = true;
	}
	for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
	{
		if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1CE)
			t1ceDataPresent = true;
		else if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T2FLAIR)
			t2flairDataPresent = true;
	}
	if (t1ceDataPresent == false && t2flairDataPresent == false)
		msg = msg + "\n" + "T1CE image.";
	if (perfusionDataPresent == false)
		msg = msg + "\n" + "Perfusion image.";
	if (mCurrentNearPoints == 0)
		msg = msg + "\n" + "Near region.";
	if (mCurrentFarPoints == 0)
		msg = msg + "\n" + "Far region.";
	//if (mCurrentInitPoints == 0)
	//	msg = msg + "\n" + "Initial seed mask.";

	if (msg != "Missing elements:\n")
	{
		QMessageBox box(this);
		box.setIcon(QMessageBox::Information);
		box.addButton(QMessageBox::Ok);
		box.setText(QString::fromStdString(msg));
		box.setWindowTitle(tr("Missing elements"));
		box.setFont(mFont);
		box.exec();
		return false;
	}
	else
		return true;
}

void fMainWindow::RecurrenceEstimateOnExistingModel(std::string modeldirectory, std::string inputdirectory,std::string outputdirectory, bool useT1Data, bool useT1CEData, bool useT2Data, bool useT2FlairData, bool useDTIData, bool usePerfData,bool useDistData)
{
	mRecurrenceEstimator->RecurrenceEstimateOnExistingModel(modeldirectory, inputdirectory, outputdirectory,useT1Data,useT1CEData,useT2Data,useT2FlairData, useDTIData,usePerfData,useDistData);
	if (mRecurrenceEstimator->mLastEncounteredError != "")
		ShowMessage(mRecurrenceEstimator->mLastEncounteredError);
	else
		ShowMessage("Recurrence maps have been generated for the given subjects.");
}

itk::Image<float, 3>::Pointer fMainWindow::FlipGivenImage(itk::Image<float, 3>::Pointer image)
{
	typedef itk::Image<float, 3> ImageType;
	typedef itk::FlipImageFilter< ImageType> FlipType;
	FlipType::Pointer flip = FlipType::New();
	FlipType::FlipAxesArrayType flipAxesSet;
	flipAxesSet[0] = 0;
	flipAxesSet[1] = -1;
	flipAxesSet[2] = 0;
	flip->SetFlipAxes(flipAxesSet);
	flip->FlipAboutOriginOff();
	flip->SetInput(image);
	flip->Update();
	ImageType::Pointer outputimage = ImageType::New();
	outputimage = flip->GetOutput();
	outputimage->SetOrigin(image->GetOrigin());
	return outputimage;
}

itk::Image<float, 3>::Pointer fMainWindow::RescaleImageIntensity(itk::Image<float, 3>::Pointer image)
{
	typedef itk::Image<float, 3> ImageType;
	typedef itk::RescaleIntensityImageFilter< ImageType, ImageType > RescaleFilterType;
	RescaleFilterType::Pointer rescaleFilter = RescaleFilterType::New();
	rescaleFilter->SetInput(image);
	rescaleFilter->SetOutputMinimum(0);
	rescaleFilter->SetOutputMaximum(255);
	rescaleFilter->Update();

	ImageType::Pointer outputimage = rescaleFilter->GetOutput();

	return outputimage;

}
itk::Image<float,4>::Pointer fMainWindow::RescalePerfusionImage(itk::Image<float, 4>::Pointer image)
{
	typedef itk::Image<float, 4> ImageType;
	typedef itk::RescaleIntensityImageFilter< ImageType, ImageType > RescaleFilterType;
	RescaleFilterType::Pointer rescaleFilter = RescaleFilterType::New();
	rescaleFilter->SetInput(image);
	rescaleFilter->SetOutputMinimum(0);
	rescaleFilter->SetOutputMaximum(255);
	rescaleFilter->Update();
	ImageType::Pointer outputimage = rescaleFilter->GetOutput();
	return outputimage;
}

int fMainWindow::GetFeatureVectorSize(bool useT1Data, bool useT1CEData, bool useT2FlairData, bool useT2Data, bool useDTIData, bool usePerfData, bool useDistData)
{
	int size=0;
	if (useT1Data)
		size++;
	if (useT2Data)
		size++;
	if (useT2FlairData)
		size++;
	if (useT1CEData)
		size++;
	if (useDTIData)
		size=size+4;
	if (usePerfData)
		size=size+5;
	
	if (useDistData)
		size=size+1;
	return size;
}

void fMainWindow::TrainNewModelOnGivenData(std::string directoryname, std::string outputdirectory, bool useT1Data, bool useT1CEData, bool useT2Data, bool useT2FlairData, bool useDTIData, bool usePerfData,bool useDistData)
{
	mRecurrenceEstimator->TrainNewModelOnGivenData(directoryname, outputdirectory, useT1Data, useT1CEData, useT2Data, useT2FlairData, useDTIData, usePerfData, useDistData);
	//mOutputManager->SetOutputDirectoryPath(outputdirectory);
	//bool useOtherModalities = false;
	//if (useT1CEData || useDTIData || useT1Data || useT2Data || useT2FlairData)
	//	useOtherModalities = true;

	//std::vector<std::string> t1ceFileNames;
	//std::vector<std::string> t1FileNames;
	//std::vector<std::string> t2FileNames;
	//std::vector<std::string> t2FlairFileNames;
	//std::vector<std::string> axFileNames;
	//std::vector<std::string> faFileNames;
	//std::vector<std::string> radFileNames;
	//std::vector<std::string> trFileNames;
	//std::vector<std::string> perfFileNames;
	//std::vector<std::string> labelNames;
	//std::vector<std::string> nearFileNames;
	//std::vector<std::string> farFileNames;
	//std::vector<std::string> rejectedSubjectNames;
	//std::vector<std::string> qualifiedSubjectNames;
	//LoadQualifiedSubjectsFromGivenDirectory("training",directoryname, useT1Data, useT1CEData, useT2Data, useT2FlairData, useDTIData, usePerfData, useDistData,
	//	 qualifiedSubjectNames, t1FileNames, t1ceFileNames, t2FileNames, t2FlairFileNames, axFileNames, faFileNames, radFileNames, trFileNames, perfFileNames, labelNames, nearFileNames, farFileNames, rejectedSubjectNames);

	//std::vector<std::string> vector;
	//std::vector<std::vector<double>> mNearIntensities;
	//std::vector<std::vector<double>> mFarIntensities;
	//std::vector<double> tNearIntensities;
	//std::vector<double> tFarIntensities;
	//std::vector<std::vector<double>> pNearIntensities;
	//std::vector<std::vector<double>> pFarIntensities;
	//std::vector<std::vector<double>> perfusionIntensities;
	//std::vector<std::vector<double>> otherIntensities;
	//std::vector<double> distanceIntensities;


	//	//std::string subjectPath = directoryname + "/" + subjectNames[sid];
	//	//std::string dataFolder = "Z:/brain_tumor/human_3T/Recurrence_study/postprocessing/";
	//	//std::string t2flairPath = dataFolder + subjectNames[sid].substr(0, subjectNames[sid].length() - 1) + "/" + subjectNames[sid] + "_flair_sus_n3_r_strip.img";
	//	//std::string t1cePath = dataFolder + subjectNames[sid].substr(0, subjectNames[sid].length() - 1) + "/" + subjectNames[sid] + "_t1ce_sus_n3_strip.img";
	//	//std::string t1Path = dataFolder + subjectNames[sid].substr(0, subjectNames[sid].length() - 1) + "/" + subjectNames[sid] + "_t1_sus_n3_r_strip.img";
	//	//std::string t2Path = dataFolder + subjectNames[sid].substr(0, subjectNames[sid].length() - 1) + "/" + subjectNames[sid] + "_t2_sus_n3_r_strip.img";
	//	//std::string perfPath = "Z:/brain_tumor/human_3T/Recurrence_study/perfusion/perfusion_21/T0/4D/" + subjectNames[sid] + "_perf_4D_r_strip.nii.gz";

	//	//std::string axPath = "Z:/brain_tumor/human_3T/Recurrence_study/DTI_RCBV_Orig/" + subjectNames[sid] + "/" + subjectNames[sid] + "_AX_to_t1ce_str.nii.gz";
	//	//std::string faPath = "Z:/brain_tumor/human_3T/Recurrence_study/DTI_RCBV_Orig/" + subjectNames[sid] + "/" + subjectNames[sid] + "_FA_to_t1ce_str.nii.gz";
	//	//std::string radPath = "Z:/brain_tumor/human_3T/Recurrence_study/DTI_RCBV_Orig/" + subjectNames[sid] + "/" + subjectNames[sid] + "_RAD_to_t1ce_str.nii.gz";
	//	//std::string trPath = "Z:/brain_tumor/human_3T/Recurrence_study/DTI_RCBV_Orig/" + subjectNames[sid] + "/" + subjectNames[sid] + "_TR_to_t1ce_str.nii.gz";
	//	//
	//	//std::string glistrPath;
	//	//if (subjectNames[sid] == "AAAB0" || subjectNames[sid] == "AAAF0" || subjectNames[sid] == "AALV0")
	//	//	glistrPath = "Z:/RecurrenceData/gData/" + subjectNames[sid] + "_Label_VP.nii.gz"; 
	//	//else
	//	//	glistrPath = "Z:/brain_tumor/human_3T/Recurrence_study/GLISTR_Results/Label_map/" + subjectNames[sid] + "/" + subjectNames[sid] + "_labels.nii.gz";
	//	//	

	//	//std::vector<std::string> filePaths;
	//	//std::string recurPath = subjectPath + "/" + subjectNames[sid] +"_Infiltrated.img";
	//	//std::string nonrecurPath = subjectPath + "/" + subjectNames[sid] + "_Pure.img";

	//for (int i = 0; i < t1ceFileNames.size();i++)
	//{
	//	typedef itk::Image<float, 3> ImageType;
	//	typedef itk::Image<float, 4> PerfusionImageType;
	//	ImageType::Pointer recurImagePointer;
	//	ImageType::Pointer nonrecurImagePointer;
	//	ImageType::Pointer T1CEImagePointer;
	//	ImageType::Pointer T2FlairImagePointer;
	//	ImageType::Pointer T1ImagePointer;
	//	ImageType::Pointer T2ImagePointer;
	//	ImageType::Pointer AXImagePointer;
	//	ImageType::Pointer RADImagePointer;
	//	ImageType::Pointer FAImagePointer;
	//	ImageType::Pointer TRImagePointer;
	//	ImageType::Pointer LabelImagePointer;
	//	PerfusionImageType::Pointer perfImagePointer;
	//	std::vector<ImageType::Pointer> perfusionDicomVector;

	//	LabelImagePointer		= mNifiDataManager->ReadNiftiImage(labelNames[i]); 
	//	recurImagePointer		= mNifiDataManager->ReadNiftiImage(nearFileNames[i]);
	//	nonrecurImagePointer	= mNifiDataManager->ReadNiftiImage(farFileNames[i]);

	//	if (usePerfData)
	//		perfImagePointer		= mNifiDataManager->Read4DNiftiImage(perfFileNames[i]);
	//	if (useT1CEData)
	//		T1CEImagePointer		= RescaleImageIntensity(mNifiDataManager->ReadNiftiImage(t1ceFileNames[i]));
	//	if (useT2FlairData)
	//		T2FlairImagePointer		= RescaleImageIntensity(mNifiDataManager->ReadNiftiImage(t2FlairFileNames[i]));
	//	if (useT1Data)
	//		T1ImagePointer			= RescaleImageIntensity(mNifiDataManager->ReadNiftiImage(t1FileNames[i]));
	//	if (useT2Data)
	//		T2ImagePointer			= RescaleImageIntensity(mNifiDataManager->ReadNiftiImage(t2FileNames[i]));
	//	if (useDTIData)
	//	{
	//		AXImagePointer = RescaleImageIntensity(mNifiDataManager->ReadNiftiImage(axFileNames[i]));
	//		RADImagePointer = RescaleImageIntensity(mNifiDataManager->ReadNiftiImage(radFileNames[i]));
	//		FAImagePointer = RescaleImageIntensity(mNifiDataManager->ReadNiftiImage(faFileNames[i]));
	//		TRImagePointer = RescaleImageIntensity(mNifiDataManager->ReadNiftiImage(trFileNames[i]));
	//	}

	//	std::vector<std::vector<double>> mNearIntensitiesPerSub;
	//	std::vector<std::vector<double>> mFarIntensitiesPerSub;
	//	std::vector<std::vector<double>> pNearIntensitiesPerSub;
	//	std::vector<std::vector<double>> pFarIntensitiesPerSub;
	//	std::vector<double> tNearIntensitiesPerSub;
	//	std::vector<double> tFarIntensitiesPerSub;

	//	int imagetype = IMAGE_NIFTI;
	//	mNifiDataManager->LoadTrainingData(LabelImagePointer, recurImagePointer, nonrecurImagePointer, T1CEImagePointer, T2FlairImagePointer, T1ImagePointer, T2ImagePointer, perfImagePointer, perfusionDicomVector, AXImagePointer, FAImagePointer, RADImagePointer, TRImagePointer, pNearIntensitiesPerSub, pFarIntensitiesPerSub, mNearIntensitiesPerSub, mFarIntensitiesPerSub, tNearIntensitiesPerSub, tFarIntensitiesPerSub, imagetype,
	//		useT1Data,useT2Data,useT1CEData,useT2FlairData,useDTIData,usePerfData,useDistData);

	//	for (int i = 0; i < mNearIntensitiesPerSub.size(); i++)
	//	{
	//		if (useOtherModalities)
	//			mNearIntensities.push_back(mNearIntensitiesPerSub[i]);
	//		if (usePerfData)
	//			pNearIntensities.push_back(pNearIntensitiesPerSub[i]);
	//		if (useDistData)
	//			tNearIntensities.push_back(tNearIntensitiesPerSub[i]);
	//	}
	//	for (int i = 0; i < mFarIntensitiesPerSub.size(); i++)
	//	{
	//		if (useOtherModalities)
	//			mFarIntensities.push_back(mFarIntensitiesPerSub[i]);
	//		if (usePerfData)
	//			pFarIntensities.push_back(pFarIntensitiesPerSub[i]);
	//		if (useDistData)
	//			tFarIntensities.push_back(tFarIntensitiesPerSub[i]);
	//	}
	//}
	//int nearSamples = 0;
	//if (mNearIntensities.size()>0)
	//	nearSamples = mNearIntensities.size();
	//else if (pNearIntensities.size()>0)
	//	nearSamples = pNearIntensities.size();
	//else if (tNearIntensities.size()>0)
	//	nearSamples = tNearIntensities.size();

	//int farSamples = 0;
	//if (mFarIntensities.size()>0)
	//	farSamples = mFarIntensities.size();
	//else if (pFarIntensities.size()>0)
	//	farSamples = pFarIntensities.size();
	//else if (tFarIntensities.size()>0)
	//	farSamples = tFarIntensities.size();


	//int totalSamples = nearSamples + farSamples;

	//for (int i = 0; i < nearSamples; i++)
	//{
	//	if (usePerfData)
	//		perfusionIntensities.push_back(pNearIntensities[i]);
	//	if (useOtherModalities)
	//		otherIntensities.push_back(mNearIntensities[i]);
	//	if (useDistData)
	//		distanceIntensities.push_back(tNearIntensities[i]);
	//}
	//for (int i = 0; i < farSamples; i++)
	//{
	//	if (usePerfData)
	//		perfusionIntensities.push_back(pFarIntensities[i]);
	//	if (useOtherModalities)
	//		otherIntensities.push_back(mFarIntensities[i]);
	//	if (useDistData)
	//		distanceIntensities.push_back(tFarIntensities[i]);
	//}
	//VariableLengthVectorType perfMeanVector;
	//vtkSmartPointer<vtkTable> reducedPerfusionFeatures;
	////------------------------------------------reduce perfusion intensities------------------------------------------
	//if (usePerfData)
	//{
	//	perfMeanVector = mFeatureReductionObj->ComputeMeanOfGivenFeatureVectors(perfusionIntensities);
	//	reducedPerfusionFeatures = mFeatureReductionObj->GetDiscerningPerfusionTimePoints(perfusionIntensities);
	//}
	////-----------------------------------develope final near and far vectors------------------------------------------
	//std::vector<std::vector<double>> fNearIntensities;
	//std::vector<std::vector<double>> fFarIntensities;
	//for (int i = 0; i < nearSamples; i++)
	//{
	//	std::vector<double> cIntensityVectorPerSub; 
	//	if (usePerfData)
	//		for (int j = 0; j < NO_OF_PCS; j++)
	//			cIntensityVectorPerSub.push_back(reducedPerfusionFeatures->GetValue(i, j).ToDouble());

	//	if (useOtherModalities)
	//		for(int j = 0; j < otherIntensities[i].size(); j++)
	//			cIntensityVectorPerSub.push_back(otherIntensities[i][j]);

	//	if(useDistData)
	//		cIntensityVectorPerSub.push_back(distanceIntensities[i]);

	//	fNearIntensities.push_back(cIntensityVectorPerSub);
	//}

	//for (int i = nearSamples; i < totalSamples; i++)
	//{
	//	std::vector<double> cIntensityVectorPerSub;
	//	if (usePerfData)
	//		for (int j = 0; j < NO_OF_PCS; j++)
	//			cIntensityVectorPerSub.push_back(reducedPerfusionFeatures->GetValue(i, j).ToDouble());

	//	if (useOtherModalities)
	//		for (int j = 0; j < otherIntensities[i].size(); j++)
	//			cIntensityVectorPerSub.push_back(otherIntensities[i][j]);
	//	
	//	if (useDistData)
	//		cIntensityVectorPerSub.push_back(distanceIntensities[i]);
	//	fFarIntensities.push_back(cIntensityVectorPerSub);
	//}
	////---------------------training data formulation-----------------------------------
	//mFeatureExtractionObj->FormulateTrainingData(fNearIntensities, fFarIntensities);
	//VariableSizeMatrixType TrainingData = mFeatureExtractionObj->GetTrainingData();
	//

	////typedef vnl_matrix<double> MatrixType;
	////MatrixType data;

	////data.set_size(1404,6);
	////for (int i = 0; i < TrainingData.Rows(); i++)
	////	for (int j = 0; j < TrainingData.Cols(); j++)
	////		data(i, j) = TrainingData[i][j];
	////typedef itk::CSVNumericObjectFileWriter<double, 1404,6> WriterType;
	////WriterType::Pointer writer = WriterType::New();
	////writer->SetFileName("tData.csv");
	////writer->SetInput(&data);
	////writer->Write();



	//VariableSizeMatrixType ScaledTrainingData = mFeatureScalingObj->ScaleGivenTrainingFeatures(TrainingData);
	//
	////MatrixType sdata;
	////sdata.set_size(1404,6);
	////for (int i = 0; i < ScaledTrainingData.Rows(); i++)
	////	for (int j = 0; j < ScaledTrainingData.Cols(); j++)
	////		sdata(i, j) = ScaledTrainingData[i][j];
	////typedef itk::CSVNumericObjectFileWriter<double, 1404,6> WriterTypeS;
	////WriterTypeS::Pointer writerS = WriterTypeS::New();
	////writerS->SetFileName("sData.csv");
	////writerS->SetInput(&sdata);
	////writerS->Write();



	//VariableSizeMatrixType ResampledTrainingData = mFeatureExtractionObj->ResampleTrainingData(ScaledTrainingData, nearSamples, farSamples);
	////------------------------------------------saving model related features---------------------------------------------------

	////MatrixType rdata;
	////rdata.set_size(20000,6);
	////for (int i = 0; i < 20000; i++)
	////	for (int j = 0; j < ResampledTrainingData.Cols(); j++)
	////		rdata(i, j) = 0;
	////if (ResampledTrainingData.Rows() < 20000)
	////{
	////	for (int i = 0; i < ResampledTrainingData.Rows(); i++)
	////		for (int j = 0; j < ResampledTrainingData.Cols(); j++)
	////			rdata(i, j) = ResampledTrainingData[i][j];
	////}
	////else
	////{
	////	for (int i = 0; i < 20000; i++)
	////		for (int j = 0; j < ResampledTrainingData.Cols(); j++)
	////			rdata(i, j) = ResampledTrainingData[i][j];
	////}
	////typedef itk::CSVNumericObjectFileWriter<double, 20000,6> WriterTypeR;
	////WriterTypeR::Pointer writerR = WriterTypeR::New();
	////writerR->SetFileName("rData.csv");
	////writerR->SetInput(&rdata);
	////writerR->Write();

	//FILE *t;
	//t = fopen("TrainingData.txt","w");
	//
	//for (int i = 0; i < ResampledTrainingData.Rows(); i++)
	//{
	//	fprintf(t, "%f ", ResampledTrainingData[i][ResampledTrainingData.Cols() - 1]);
	//	for (int j = 0; j < ResampledTrainingData.Cols() - 1; j++)
	//		fprintf(t, "%d:%lf ", j + 1, ResampledTrainingData[i][j]);
	//	fprintf(t, "\n");
	//}
	//fclose(t);

	//int size = GetFeatureVectorSize(useT1Data, useT2Data, useT1CEData, useT2FlairData, useDTIData, usePerfData, useDistData);
	//mOutputManager->SaveModelResults(ScaledTrainingData, mFeatureScalingObj->GetMeanVector(), mFeatureScalingObj->GetStdVector(), perfMeanVector, mFeatureReductionObj->GetPCATransformationMatrix(),
	//	useT1Data, useT2Data, useT1CEData, useT2FlairData, useDTIData, usePerfData, useDistData, size);

	////------------------------------------------training process---------------------------------------------------

	//QMessageBox box(this);
	//box.setIcon(QMessageBox::Information);
	//box.addButton(QMessageBox::Ok); 

	//if (mClassificationObj->Training(ResampledTrainingData) == 0)
	//	box.setText(QString::fromStdString(mClassificationObj->GetLastEncounteredError()));
	//else
	//	box.setText(QString::fromStdString("Training complete."));

	//box.setWindowTitle(tr("Message"));
	//box.setFont(mFont);
	//box.exec();
	//return;

	////-----------------clear all the training parameters-------------------------
	//this->mFeatureReductionObj->ResetParameters();
	//this->mFeatureScalingObj->ResetParameters();
}


void fMainWindow::RunInClusterMode()
{
	//LoadNonViewingImages("", IMAGE_TYPE_T1CE, RUNNING_MODE_CLUSTER);
	//LoadNonViewingImages("", IMAGE_TYPE_T2FLAIR, RUNNING_MODE_CLUSTER);
	//LoadNonViewingImages("", IMAGE_TYPE_T1, RUNNING_MODE_CLUSTER);
	//LoadNonViewingImages("", IMAGE_TYPE_T2, RUNNING_MODE_CLUSTER);
	//LoadNonViewingImages("", IMAGE_TYPE_PERFUSION, RUNNING_MODE_CLUSTER);

	//typedef itk::Image< float,3>         ImageType;
	//typedef itk::ImageFileReader<ImageType> ReaderType;
	//ReaderType::Pointer reader = ReaderType::New();
	//reader->SetFileName("");
	//ImageType::RegionType region = reader->GetOutput()->GetLargestPossibleRegion();

	//for (int x_index = 0; x_index < region.GetSize()[0]; x_index++)
	//	for (int y_index = 0; y_index <region.GetSize()[1]; y_index++)
	//		for (int z_index = 0; z_index <region.GetSize()[2]; z_index++)
	//		{
	//			ImageType::IndexType pointIndex;
	//			pointIndex[0] = x_index;
	//			pointIndex[1] = y_index;
	//			pointIndex[2] = z_index;

	//			if (reader->GetOutput()->GetPixel(pointIndex) == 255)
	//			{
	//				singlePoint point;
	//				point.image_coordinates[0] = x_index;
	//				point.image_coordinates[1] = y_index;
	//				point.image_coordinates[2] = z_index;
	//				mNearPoints->mDrawingPointsVector.push_back(point);
	//			}

	//mOutputManager->SetOutputDirectoryPath("");
	//mOutputManager->SetupOutputFolders();
	//StartRecurrenceEstimate(RUNNING_MODE_CLUSTER);
}


itk::Image<float, 3>::Pointer fMainWindow::GetFarRegionMask()
{
	int index = -1;

	for (unsigned int i = 0; i < mSlicerManagers.size(); i++)
	{
		if (mSlicerManagers[i]->mImageType == IMAGE_TYPE_T2FLAIR)
			index = i;
	}
	typedef itk::Image<float, 3> ImageType;
	ImageType::Pointer image = convertVtkToItk<float, 3>(mSlicerManagers[index]->mImage);

	itk::ImageRegionIterator<ImageType> imageIterator(image, image->GetLargestPossibleRegion());
	while (!imageIterator.IsAtEnd())
	{
		imageIterator.Set(0);
		++imageIterator;
	}

	for (unsigned int i = 0; i < mTissuePoints->mLandmarks.size(); i++)
	{
		ImageType::IndexType pointIndex;
		pointIndex[0] = mTissuePoints->mLandmarks[i].image_coordinates[0];
		pointIndex[1] = mTissuePoints->mLandmarks[i].image_coordinates[1];
		pointIndex[2] = mTissuePoints->mLandmarks[i].image_coordinates[2];
		image->SetPixel(pointIndex, 255);
	}
	return image;
}

template<class ImageType>
void fMainWindow::SetTransformedFarRegionPoints(typename ImageType::Pointer FarRegionMask)
{
	// itk::NiftiImageIO::Pointer nifti_io = itk::NiftiImageIO::New();
	typedef itk::ImageFileWriter< ImageType > WriterType;
	typename WriterType::Pointer writer = WriterType::New();

	std::string filename = "testmasks.nii";
	writer->SetFileName(filename);
	// writer->SetImageIO(nifti_io);
	writer->SetInput(FarRegionMask);
	writer->Update();

}

void fMainWindow::LoadDrawing()
{
	//	SetActiveLandmarksType(NEARFAR_DRAWING_POINTS, drawingtype, 0);
	QString extensions = IMAGES_EXTENSIONS;
	extensions += ";;All Files (*)";
  QStringList files = QFileDialog::getOpenFileNames(this, tr("Load Masks"), mInputPathName, extensions);
	if (files.isEmpty())
		return;

	std::string filename = files[0].toStdString();
	itk::ImageIOBase::Pointer reader = itk::ImageIOFactory::CreateImageIO(filename.c_str(), itk::ImageIOFactory::ReadMode);
	if (reader)
	{
		typedef itk::Image<float, 3> ImageType;
		typedef itk::ImageFileReader<ImageType> ReaderType;
		ReaderType::Pointer reader = ReaderType::New();
		reader->SetFileName(filename);
		reader->ReleaseDataFlagOn();
		ImageType::Pointer inputImage = reader->GetOutput();
		inputImage->Update();

		std::vector<ImageType::IndexType> nearIndices;
		std::vector<ImageType::IndexType> farIndices;
		typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
		IteratorType maskIt(inputImage, inputImage->GetLargestPossibleRegion());
		maskIt.GoToBegin();
		while (!maskIt.IsAtEnd())
		{
			if (maskIt.Get() == NEAR_POINT_SAVE_LABEL)
				nearIndices.push_back(maskIt.GetIndex());
			else if (maskIt.Get() == FAR_POINT_SAVE_LABEL)
				farIndices.push_back(maskIt.GetIndex());
			++maskIt;
		}

		for (unsigned int i = 0; i < nearIndices.size(); i++)
		{
			float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)nearIndices[i][0], (int)nearIndices[i][1], (int)nearIndices[i][2]);
			*pData = 1;
		}
		for (unsigned int i = 0; i < farIndices.size(); i++)
		{
			float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)farIndices[i][0], (int)farIndices[i][1], (int)farIndices[i][2]);
			*pData = 2;
		}					
		this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
		this->mSlicerManagers[0]->Render();
	}
}
void fMainWindow::LoadSeedDrawing()
{
	QString extensions = IMAGES_EXTENSIONS;
	extensions += ";;All Files (*)";
	QStringList files = QFileDialog::getOpenFileNames(this, tr("Load Masks"), "/home", extensions);
	if (files.isEmpty())
		return;

	std::string filename = files[0].toStdString();
	itk::ImageIOBase::Pointer reader = itk::ImageIOFactory::CreateImageIO(filename.c_str(), itk::ImageIOFactory::ReadMode);
	if (reader)
	{
		typedef itk::Image<float, 3> ImageType;
		typedef itk::ImageFileReader<ImageType> ReaderType;
		ReaderType::Pointer reader = ReaderType::New();
		reader->SetFileName(filename);
		reader->ReleaseDataFlagOn();
		ImageType::Pointer inputImage = reader->GetOutput();
		inputImage->Update();

		std::vector<ImageType::IndexType> seedIndices;
		typedef itk::ImageRegionIteratorWithIndex <ImageType> IteratorType;
		IteratorType maskIt(inputImage, inputImage->GetLargestPossibleRegion());
		maskIt.GoToBegin();
		while (!maskIt.IsAtEnd())
		{
			if (maskIt.Get() == INIT_POINT_SAVE_LABEL)
				seedIndices.push_back(maskIt.GetIndex());
			++maskIt;
		}
		for (unsigned int i = 0; i < seedIndices.size(); i++)
		{
			float* pData = (float*)this->mSlicerManagers[0]->GetSlicer(0)->mMask->GetScalarPointer((int)seedIndices[i][0], (int)seedIndices[i][1], (int)seedIndices[i][2]);
			*pData = INIT_POINT_LABEL;
		}
		this->mSlicerManagers[0]->GetSlicer(0)->mMask->Modified();
		this->mSlicerManagers[0]->Render();
	}
}

void fMainWindow::UpdateBorderWidget(double startX,double startY,double endX, double endY)
{
	mBorderStartX = std::round(startX);
	mBorderStartY = std::round(startY);
	mBorderEndX = std::round(endX);
	mBorderEndY = std::round(endY);
}
void fMainWindow::UpdateBorderWidget(double startZ, double endZ)
{
	mBorderStartZ = std::round(startZ);
	mBorderStartZ = std::round(startZ);
}

void fMainWindow::overlayUseStateChanged(int state)
{
	if (state == 0) {
		overlayTable->setEnabled(false);
		overlaySlider->setEnabled(false);
		//
		for (int i = 0; i < (int)mSlicerManagers.size(); i++) {
			for (int j = 0; j < 3; j++) {
				mSlicerManagers[i]->mSlicers[j]->RemoveOverlay();
			}
		}
		//
		overlaySlider->setValue(5);
		//
		QList<QTableWidgetSelectionRange> ranges = overlayTable->selectedRanges();
		if (!ranges.empty()) {
			overlayTable->setRangeSelected(ranges[0], false);
		}
		//
		UpdateRenderWindows();
	}
	else {
		overlayTable->setEnabled(true);
		overlaySlider->setEnabled(true);
	}
}

void fMainWindow::overlaySliderChanged()
{
	static int value = -1;
	if (value == overlaySlider->value()) {
		return;
	}
	else {
		value = overlaySlider->value();
	}
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index >= 0 && index < (int)mSlicerManagers.size())
	{
		for (int i = 0; i < 3; i++)
		{
			mSlicerManagers[index]->GetSlicer(i)->SetOverlayOpacity((double)value / (10 + 1e-6));
		}
	}
}

void fMainWindow::overlayChanged()
{
	QList<QTableWidgetItem*> items = overlayTable->selectedItems();
	if (items.empty()) {
		return;
	}
	overlayChanged(items[0]);
}
void fMainWindow::overlayChanged(QTableWidgetItem *clickedItem)
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) {
		return;
	}
	int slicerManagerIndex = GetSlicerIndexFromItem(items[0]);
	if (slicerManagerIndex < 0 || slicerManagerIndex >= (int)mSlicerManagers.size())
	{
		return;
	}
	//
	int slicerManagerOverlayIndex = GetSlicerIndexFromItem(clickedItem);
	if (slicerManagerOverlayIndex < 0 || slicerManagerOverlayIndex >= (int)mSlicerManagers.size())
	{
		return;
	}
	for (unsigned int i = 0; i < mSlicerManagers.size(); i++)
	{
		if (i != static_cast<unsigned int>(slicerManagerIndex)) {
			for (int j = 0; j < 3; j++) {
				mSlicerManagers[i]->mSlicers[j]->RemoveOverlay();
			}
		}
		else
		{
			for (int j = 0; j < 3; j++) {
				mSlicerManagers[slicerManagerIndex]->mSlicers[j]->SetOverlay(mSlicerManagers[slicerManagerOverlayIndex]->mSlicers[j]->mImage);
				//
				double window = mSlicerManagers[slicerManagerOverlayIndex]->mSlicers[j]->GetColorWindow();
				double level = mSlicerManagers[slicerManagerOverlayIndex]->mSlicers[j]->GetColorLevel();
				vtkLookupTable* LUT = static_cast<vtkLookupTable*>(mSlicerManagers[slicerManagerOverlayIndex]->mSlicers[j]->GetWindowLevel()->GetLookupTable());
				if (LUT != NULL)
				{
					mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetWindow(window);
					mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetLevel(level);
					mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetLookupTable(LUT);
				}
				else
				{
					mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetLookupTable(NULL);
					mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetWindow(window);
					mSlicerManagers[slicerManagerIndex]->mSlicers[j]->mOverlayMapper->SetLevel(level);
				}
			}
		}
	}
	UpdateRenderWindows();
}
void fMainWindow::OpenDicomImages()
{
	std::vector<std::string> vector;
	std::string directoryname;
	QString directory = QFileDialog::getExistingDirectory(this, tr("Open Directory"), mInputPathName, QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks | QFileDialog::DontUseNativeDialog);
	if (directory.isNull())
		return;
	else
	{
		mInputPathName = itksys::SystemTools::GetFilenamePath(directory.toStdString()).c_str();
		directoryname = directory.toStdString();
		vector.push_back(directoryname);
	}
	OpenImages(IMAGE_DICOM,vector);
}
void fMainWindow::OpenNiftiImages()
{
	QString extensions = IMAGES_EXTENSIONS;
	extensions += ";;All Files (*)";
	
	QStringList files = QFileDialog::getOpenFileNames(this, tr("Load Images"), mInputPathName, extensions);
	if (files.isEmpty())
	{
		return;
	}
	mInputPathName = itksys::SystemTools::GetFilenamePath(files[0].toStdString()).c_str();
	std::vector<std::string> files_sorted;

	for (int i = 0; i < (int)files.size(); i++)
	{
		std::cout<<"File name = " <<files[i].toStdString()<<std::endl;
		std::string filename = cbica::getFilenameBase(files[i].toStdString());

		std::cout<<"File name = " <<filename<<std::endl;

		if (filename.find("t1ce") != std::string::npos || filename.find("T1CE") != std::string::npos ||
			filename.find("t1-ce") != std::string::npos || filename.find("T1-CE") != std::string::npos ||
			filename.find("t1c") != std::string::npos || filename.find("T1C") != std::string::npos || filename.find("T1c") != std::string::npos)
		{
			std::vector<std::string> vector;
			vector.push_back(files[i].toStdString());

			LoadSlicerImages(vector, IMAGE_NIFTI, IMAGE_TYPE_T1CE, IMAGE_TYPE_UNDEFINED);
			files.erase(files.begin() + i);
			break;
		}
	}
	for (int i = 0; i < (int)files.size(); i++)
	{
    std::string filename = cbica::getFilenameBase(files[i].toStdString());
		if ((filename.find("t1") != std::string::npos || filename.find("T1") != std::string::npos))
		{
			std::vector<std::string> vector;
			vector.push_back(files[i].toStdString());
			LoadSlicerImages(vector, IMAGE_NIFTI, IMAGE_TYPE_T1, IMAGE_TYPE_UNDEFINED);
			files.erase(files.begin() + i);
			break;
		}
	}
	for (int i = 0; i < (int)files.size(); i++)
	{
    std::string filename = cbica::getFilenameBase(files[i].toStdString());
		if (filename.find("t2") != std::string::npos || filename.find("T2") != std::string::npos)
		{
			std::vector<std::string> vector;
			vector.push_back(files[i].toStdString());
			LoadSlicerImages(vector, IMAGE_NIFTI, IMAGE_TYPE_T2, IMAGE_TYPE_UNDEFINED);
			files.erase(files.begin() + i);
			break;
		}
	}
	for (int i = 0; i < (int)files.size(); i++)
	{
    std::string filename = cbica::getFilenameBase(files[i].toStdString());
		if (filename.find("Flair") != std::string::npos || filename.find("flair") != std::string::npos || filename.find("FLAIR") != std::string::npos)
		{
			std::vector<std::string> vector;
			vector.push_back(files[i].toStdString());
			LoadSlicerImages(vector, IMAGE_NIFTI, IMAGE_TYPE_T2FLAIR, IMAGE_TYPE_UNDEFINED);
			files.erase(files.begin() + i);
			break;
		}
	}
	for (int i = 0; i < (int)files.size(); i++)
	{
    std::string filename = cbica::getFilenameBase(files[i].toStdString());
		if (filename.find("Recurrence") != std::string::npos || filename.find("recurrence") != std::string::npos)
		{
			std::vector<std::string> vector;
			vector.push_back(files[i].toStdString());
			LoadSlicerImages(vector, IMAGE_NIFTI, IMAGE_TYPE_RECURRENCE_OUTPUT, IMAGE_TYPE_UNDEFINED);
			files.erase(files.begin() + i);
			break;
		}
	}

	for (int i = 0; i < (int)files.size(); i++)
	{
    std::string filename = cbica::getFilenameBase(files[i].toStdString());
    if (filename.find("AXIAL") != std::string::npos || filename.find("axial") != std::string::npos || filename.find("Axial") != std::string::npos)
			LoadNonViewingImages(files[i].toStdString(), IMAGE_NIFTI, IMAGE_TYPE_DTI, RUNNING_MODE_NORMAL);
		else if (filename.find("FRACTIONAL") != std::string::npos || filename.find("fractional") != std::string::npos || filename.find("Fractional") != std::string::npos)
			LoadNonViewingImages(files[i].toStdString(), IMAGE_NIFTI, IMAGE_TYPE_DTI, RUNNING_MODE_NORMAL);
		else if (filename.find("RADIAL") != std::string::npos || filename.find("radial") != std::string::npos || filename.find("Radial") != std::string::npos)
			LoadNonViewingImages(files[i].toStdString(), IMAGE_NIFTI, IMAGE_TYPE_DTI, RUNNING_MODE_NORMAL);
		else if (filename.find("TRACE") != std::string::npos || filename.find("trace") != std::string::npos || filename.find("Trace") != std::string::npos)
			LoadNonViewingImages(files[i].toStdString(), IMAGE_NIFTI, IMAGE_TYPE_DTI, RUNNING_MODE_NORMAL);
    else if (filename.find("PERFUSION") != std::string::npos || filename.find("perfusion") != std::string::npos || filename.find("Perfusion") != std::string::npos || filename.find("perf") != std::string::npos)
    {
#ifdef USE_PROCESSDIALOG
      fProgressDialog progress("Loading " + files[i].toStdString(), 1);
      progress.SetProgress(0, 10);
      qApp->processEvents();
#endif
      LoadNonViewingImages(files[i].toStdString(), IMAGE_NIFTI, IMAGE_TYPE_PERFUSION, RUNNING_MODE_NORMAL);
#ifdef USE_PROCESSDIALOG
      progress.SetProgress(0, 10);
      qApp->processEvents();
#endif
    }
	}
}


void fMainWindow::ApplicationEGFR()
{
	StartEGFREstimate();
}
void fMainWindow::ApplicationRecurrence()
{

	recurrencePanel->exec();
}
void fMainWindow::ApplicationGeodesic()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) 
	{
		ShowMessage("Please specify an input image.");
		return;
	}
	int index = GetSlicerIndexFromItem(items[0]);
	if (index < 0 || index >= (int)mSlicerManagers.size()) 
	{
		ShowMessage("Please specify an input image.");
		return;
	}	
	VectorVectorDouble tumorPoints = FormulateDrawingPointsForTumorSegmentation();
	if (tumorPoints.size() == 0)
	{
		ShowMessage("Please draw initial seed points.");
		return;
	}
	QString extensions = IMAGES_EXTENSIONS;
	extensions += ";;All Files (*)";
	QString file;
	QFileDialog dialog(this, tr("Save Image"), mInputPathName, extensions);
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setAcceptMode(QFileDialog::AcceptSave);
	dialog.selectFile(QString(mSlicerManagers[index]->mFileName.c_str()));
	int ret = dialog.exec();
	if (ret == QDialog::Accepted)
	{
		file = dialog.selectedFiles()[0];
		std::string filename = file.toStdString();
		typedef itk::Image<float, 3> ImageType;
		typedef itk::Image<short, 3> ImageTypeGeodesic;

		ImageTypeGeodesic::Pointer Inp = ImageTypeGeodesic::New();
		ImageTypeGeodesic::Pointer tumorMask = ImageTypeGeodesic::New();
		ImageType::Pointer tumorMaskFinal = ImageType::New();

		typedef itk::CastImageFilter<ImageType, ImageTypeGeodesic> CastFilterType;
		CastFilterType::Pointer castFilter = CastFilterType::New();
		castFilter->SetInput(convertVtkToItkFloat(mSlicerManagers[index]->mImage));
		Inp = castFilter->GetOutput();
		Inp->Update();


		VectorVectorDouble tumorPoints = FormulateDrawingPointsForTumorSegmentation();
		tumorMask = mGeodesicSegmentation->GeodesicSegmentation3D<ImageTypeGeodesic>(Inp, tumorPoints);

		typedef itk::CastImageFilter<ImageTypeGeodesic, ImageType> CastFilterBackType;
		CastFilterBackType::Pointer castFilter2 = CastFilterBackType::New();
		castFilter2->SetInput(tumorMask);
		tumorMaskFinal = castFilter2->GetOutput();
		tumorMaskFinal->Update();

		typedef itk::ImageFileWriter<ImageType> WriterType;
		WriterType::Pointer writer = WriterType::New();
		writer->SetFileName(file.toStdString());
		writer->SetInput(tumorMaskFinal);
		writer->Update();
	}
}

void fMainWindow::ImageDenoising()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty()) 
		return;

	int index = GetSlicerIndexFromItem(items[0]);
	if (index < 0 || index >= (int)mSlicerManagers.size()) 
		return;

	QString extensions = IMAGES_EXTENSIONS;
	extensions += ";;All Files (*)";
	QString file;
	QFileDialog dialog(this, tr("Save Image"), mInputPathName, extensions);
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setAcceptMode(QFileDialog::AcceptSave);
	dialog.selectFile(QString(mSlicerManagers[index]->mFileName.c_str()));
	int ret = dialog.exec();
	if (ret == QDialog::Accepted)
	{
		typedef	itk::Image<float, 3> ImageType;
		// typedef ImageType::Pointer ImageTypePointer;
		file = dialog.selectedFiles()[0];

		typedef itk::ImageDuplicator<ImageType> DuplicatorType;
		DuplicatorType::Pointer duplicator = DuplicatorType::New();
		duplicator->SetInputImage(mSlicerManagers[index]->mITKImage);
		duplicator->Update();
		ImageType::Pointer inputImage = duplicator->GetOutput();
		
		statusBar()->showMessage(QString::fromUtf8("Susan noise removal in process......"));
		ImageType::Pointer outputImage = mPreprocessingObj->DenoisingMethodCall<ImageType>(inputImage);
		statusBar()->showMessage(QString::fromUtf8("Normal Mode"));

		typedef itk::ImageFileWriter<ImageType > WriterType;
		WriterType::Pointer writer = WriterType::New();
		writer->SetFileName(file.toStdString());
		writer->SetInput(outputImage);
		writer->Update();
	}
}
void fMainWindow::ImageBiasCorrection()
{
	QList<QTableWidgetItem*> items = imagesTable->selectedItems();
	if (items.empty())
		return;

	int index = GetSlicerIndexFromItem(items[0]);
	if (index < 0 || index >= (int)mSlicerManagers.size())
		return;

	QString extensions = IMAGES_EXTENSIONS;
	extensions += ";;All Files (*)";
	QString file;
	QFileDialog dialog(this, tr("Save Image"), mInputPathName, extensions);
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setAcceptMode(QFileDialog::AcceptSave);
	dialog.selectFile(QString(mSlicerManagers[index]->mFileName.c_str()));
	int ret = dialog.exec();
	if (ret == QDialog::Accepted)
	{
		typedef	itk::Image<float, 3> ImageType;
		// typedef ImageType::Pointer ImageTypePointer;
		file = dialog.selectedFiles()[0];

		typedef itk::ImageDuplicator<ImageType> DuplicatorType;
		DuplicatorType::Pointer duplicator = DuplicatorType::New();
		duplicator->SetInputImage(mSlicerManagers[index]->mITKImage);
		duplicator->Update();
		ImageType::Pointer inputImage = duplicator->GetOutput();
		statusBar()->showMessage(QString::fromUtf8("Bias correction in process......"));
		ImageType::Pointer outputImage = mPreprocessingObj->N3BiasCorrectionMethodCall<ImageType>(inputImage);
		statusBar()->showMessage(QString::fromUtf8("Normal Mode"));

		typedef itk::ImageFileWriter<ImageType > WriterType;
		WriterType::Pointer writer = WriterType::New();
		writer->SetFileName(file.toStdString());
		writer->SetInput(outputImage);
		writer->Update();


	}
}
void fMainWindow::ImageRegistration()
{
	registrationPanel->exec();
}
void fMainWindow::CustomPreprocessing()
{
}
void fMainWindow::ChangeBrushSize(int size)
{
	mDrawSize = size + 1;
	SetDrawMode(mDrawMode, mDrawSize);
}
void fMainWindow::ShowMessage(std::string message)
{
	QMessageBox box(this);
	box.setIcon(QMessageBox::Information);
	box.addButton(QMessageBox::Ok);
	box.setText(QString::fromStdString(message));
	box.setWindowTitle(tr("Missing Data"));
	box.setFont(mFont);
	box.exec();
}
void fMainWindow::Registration(std::string fixedFileName, std::vector<std::string> inputFileNames, std::vector<std::string> outputFileNames)
{
	typedef itk::Image<float, 3> ImageType;
	typedef ImageType::Pointer ImageTypePointer;
	typedef itk::Image<float, 3> InternalImageType;
	itk::MultiResolutionImageRegistrationMethod<ImageType, ImageType>::Pointer Registrar;


	ImageTypePointer TargetImage = mNifiDataManager->ReadNiftiImage(fixedFileName);
	for (unsigned int i = 0 ; i < inputFileNames.size(); i++)
	{
		ImageTypePointer SourceImage = mNifiDataManager->ReadNiftiImage(inputFileNames[i]);
		Registrar = mPreprocessingObj->Registration<ImageType, InternalImageType>(TargetImage, SourceImage);
		ImageTypePointer RegisteredImage = mPreprocessingObj->ResampleTransform<ImageType>(Registrar, TargetImage, SourceImage);
		mOutputManager->WriteImageWithGivenName<ImageType>(RegisteredImage, outputFileNames[i]);
	}
}
