#ifndef _fMainWindow_h_
#define _fMainWindow_h_

#include "itkCastImageFilter.h"

#include "CBICA_Viewer.h"
#include "Slicer.h"
#include "SlicerManager.h"
#include "NiftiDataManager.h"
#include "EGFRStatusPredictor.h"
#include "RecurrenceEstimator.h"
#include "ui_fMainWindow.h"
#include "fProgressDialog.h"
#include "InteractorStyleNavigator.h"
#include "QTablePushButton.h"
#include "SimpleImageManager.h"
#include "OutputWritingManager.h"
#include <PreprocessingPipelineClass.h>
#include <FeatureExtractionClass.h>
#include <FeatureReductionClass.h>
#include <FeatureScalingClass.h>
#include <SVMClassificationClass.h>
#include <GeodesicSegmentation.h>

#define TAB_IMAGES 0
#define TAB_TUMOR 1
#define TAB_LANDMARKS 2

#define TAB_IMAGES_COLUMN_CLOSE 0
#define TAB_IMAGES_COLUMN_TYPE 2
#define TAB_IMAGES_COLUMN_NAME 1
#define USE_PROCESSDIALOG

#define DRAW_MODE_NORMAL 0
#define DRAW_MODE_REC 1
#define DRAW_MODE_NREC 2
#define DRAW_MODE_SEED 3
#define DRAW_MODE_ERASE 4

class fMainWindow : public QMainWindow, private Ui::fMainWindow
{
	Q_OBJECT

public:
	fMainWindow();
	~fMainWindow();
	void LoadSlicerImages(std::vector<std::string> filenames, int imagetype,int imagesubtype, int loadingtype, bool bSkipDup = true );
	void LoadNonViewingImages(std::string directoryname,int imagetype,int imagesubtype,int runningmode);
	void RunInClusterMode();

	template<class InputPixelType, unsigned int VImageDimension>
	void LoadImage(std::string name, itk::Image<InputPixelType, VImageDimension>* image, vtkTransform* transform);
	void InitMask(vtkImageData* image);
	void SetDrawMode(int mode, int size);
	void InitDisplay();
	void InitSlicers();
	void DisplaySliders(int slicer, int window);
	int GetSlicerIndexFromItem(QTableWidgetItem* item);
	QTableWidgetItem* GetItemFromSlicerManager(SlicerManager* sm);
	QString Get4x4MatrixDoubleAsString(vtkSmartPointer<vtkMatrix4x4> matrix, const int precision = 3);

	template<class ImageType>
	void DisplayRecurrenceEstimationOutput(typename ImageType::Pointer RecurrenceProbabilityMap/*, typename ImageType::Pointer NonRecurrenceProbabilityMap, typename ImageType::Pointer LabelMap*/);

	template<class ImageType>
	void SetTransformedFarRegionPoints(typename ImageType::Pointer FarRegionMask);

	template<class ImageType>
	void FormulateNearFarPoints( std::vector<typename ImageType::IndexType> & nearIndices,  std::vector<typename ImageType::IndexType> & farIndices);

  template<class ImageType>
  std::vector< std::vector < typename ImageType::IndexType > > StartEGFREstimateDrawing();


	itk::Image<float, 3>::Pointer FlipGivenImage(itk::Image<float, 3>::Pointer image);
	itk::Image<float, 3>::Pointer RescaleImageIntensity(itk::Image<float, 3>::Pointer image);
	itk::Image<float, 4>::Pointer RescalePerfusionImage(itk::Image<float, 4>::Pointer image);

signals:
	void SelectedImageHasChanged(SlicerManager *);
	void LandmarksFocused(bool bFocused);
	void SeedPointsFocused(bool bFocused);
	void TissuePointsFocused(bool bFocused);

public slots:
	void UpdateNumberOfPointsInTable();
	void SetPresetComboBox();
	void StartEGFREstimate();



	void TestEGFRFunction();

	void StartRecurrenceEstimate(std::string outputdirectory, bool cbT1Data, bool cbT1ceData, bool cbT2Data, bool cbT2FlairData, bool cbDTIData, bool cbPerfData, bool cbDistData);
	void RecurrenceEstimateOnExistingModel(std::string modeldirectory, std::string inputdirectory, std::string outputdirectory, bool cbT1Data, bool cbT1ceData, bool cbT2Data, bool cbT2FlairData, bool cbDTIData, bool cbPerfData, bool cbDistData);
	void TrainNewModelOnGivenData(std::string directory, std::string outputdirectory, bool cbT1Data, bool cbT1ceData, bool cbT2Data, bool cbT2FlairData, bool cbDTIData, bool cbPerfData,bool cbDistData);

	int  GetFeatureVectorSize(bool useT1Data, bool useT1CEData, bool useT2FlairData, bool useT2Data, bool useDTIData, bool usePerfData, bool useDistData);
	void UpdateBorderWidget(double startX, double startY, double endX, double endY);
	void UpdateBorderWidget(double startZ, double endZ);

	itk::Image<float, 3>::Pointer GetFarRegionMask();
	
	
	void about();
  void help();
	void shortcuts();
	void OpenImages(int type,std::vector<std::string> directoryname);
	void OpenT1Images();
	void OpenT2Images();
	void OpenT1CEImages();
	void OpenT2FlairImages();
	void OpenPerfusionImages();
	void OpenDTIImages();
  void OpenRecurrenceImages();
	void OpenDicomImages();
	void OpenNiftiImages();
	void SaveImage();
	void BulkTestingNiftiData();

	void CurrentImageChanged(std::string id);
	void CurrentPickedImageChanged(std::string id);
	void ImageInfoChanged();
	void DisplayChanged();
	void DisplayChanged(QTableWidgetItem *item);
	void overlayUseStateChanged(int state);
	void overlayChanged();
	void overlayChanged(QTableWidgetItem *item);
	void overlaySliderChanged();

	void CloseImage(QTableWidgetItem* item);
	void CloseNonViewingT1Image(QTableWidgetItem* item);
	void CloseNonViewingT2Image(QTableWidgetItem* item);
	void CloseNonViewingPerfusionImage(QTableWidgetItem* item);
	void CloseNonViewingDTIImage(QTableWidgetItem* item);
	void MousePositionChanged(int visibility, double x, double y, double z, double X, double Y, double Z, double value);

	void UpdateSlice(int slicer, int slice);
	void UpdateSliceRange(int slicer, int min, int max);
	void WindowLevelChanged();
	void WindowLevelEdited();
	void SetWindowLevel(double w, double l);
	void UpdateWindowLevel();
	void thresholdSliderChanged();
	void UpdateLinkManager(std::string id, int slicer, double x, double y, double z);
	void UpdateLinkedNavigation(std::string id, SlicerManager *sm, Slicer* refSlicer);
	void AddLink(QString image1, QString image2);
	void RemoveLink(QString image1, QString image2);
	void ChangeImageWithOrder(SlicerManager *sm, int order);

	void NewClassificationRadioChecked();
	void ExistingClassificationRadioChecked();
	void CreateClassificationModelRadioChecked();

	void AxialViewSliderChanged();
	void CoronalViewSliderChanged();
	void SaggitalViewSliderChanged();

	void CloseImage();
	void ResetTransformationToIdentity();

	void UpdateRenderWindows();
	void resizeEvent(QResizeEvent *event);

	void SetActiveLandmarksType(int type, int row, int col);
	void SetActiveLandmarkTypeTab(int current);
	void MoveSlicerCursor(double x, double y, double z, int mode = 0);


	void EnableNearRegionDrawing();
	void EnableFarRegionDrawing();
	void EnableInitDrawing();
	void EscapeMode();
	void SaveDrawing();
	void LoadDrawing();
	void SaveSeedDrawing();
	void LoadSeedDrawing();

	VectorVectorDouble FormulateDrawingPointsForEdemaSegmentation();
	VectorVectorDouble FormulateDrawingPointsForTumorSegmentation();
	VectorVectorDouble GetFarRegionIndices();
	VectorVectorDouble GetNearRegionIndices();
	void ChangeBrushSize(int size);
	void ShowMessage(std::string message);

	void EraseAllTheDrawings();
	void EraseInitDrawing();
	void EraseNearDrawing();
	void EraseFarDrawing();
	void SetDrawingModeToEraserMode();
	void ResetNumberOfPoints();

	void SetNonViewingImagesTable();
	void SetMasksTable();
	void GetOutputDirectory();
	bool CheckCompletenessOfInputData(bool & t1DataPresent, bool & t2DataPresent, bool & t1ceDataPresent, bool & t2flairDataPresent, bool & perfusionDataPresent, bool & dtiDataPresent);
	bool CheckCompletenessOfInputDataForEGFR(bool & t1ceDataPresent, bool & t2flairDataPresent, bool & perfusionDataPresent);


	void ApplicationEGFR();
	void ApplicationRecurrence();
	void ApplicationGeodesic();
	void ImageDenoising();
	void ImageBiasCorrection();
	void ImageRegistration();
	void CustomPreprocessing();
	void Registration(std::string fixedfilename, std::vector<std::string> inputFileNames, std::vector<std::string> outputFileNames);

public:
	QFont mFont;
	std::vector<SlicerManager*> mSlicerManagers;
	std::vector<SimpleImageManager*> mNonViewingImageManager;

	QWidget* mMainWidget;
	QString mInputPathName;
	std::vector<QSlider*> verticalSliders;
	//
	std::string mCurrentSelectedImageId;
	std::string mCurrentPickedImageId;
    int mCurrentPickedImageIndex;
	//
	Landmarks* mLandmarks;
	Landmarks* mSeedPoints;
	Landmarks* mTissuePoints;
	vtkSmartPointer<vtkImageData> mMask;
	//
	int mDrawMode;
	int mDrawSize;
	std::string imagetype;
	int mCurrentNearPoints;
	int mCurrentFarPoints;
	int mCurrentInitPoints;


	int mBorderStartX;
	int mBorderStartY;
	int mBorderStartZ;

	int mBorderEndX;
	int mBorderEndY;
	int mBorderEndZ;


	PreprocessingPipelineClass	* mPreprocessingObj;
	FeatureExtractionClass		* mFeatureExtractionObj;
	FeatureReductionClass		* mFeatureReductionObj;
	FeatureScalingClass			* mFeatureScalingObj;
	SVMClassificationClass		* mClassificationObj;
	OutputWritingManager		* mOutputManager;
	NiftiDataManager			* mNifiDataManager;
	EGFRStatusPredictor			* mEGFRPredictor;
	RecurrenceEstimator			* mRecurrenceEstimator;
	GeodesicSegmentation		* mGeodesicSegmentation;

	


	std::string t1cePath;
	Ui::fMainWindow	m_mainWindowUI;
	
};



template<class ImageType>
std::vector< std::vector < typename ImageType::IndexType > > fMainWindow::StartEGFREstimateDrawing()
{
  bool t1ceDataPresent = false;
  bool t2flairDataPresent = false;
  bool perfusionDataPresent = false;
  std::vector< std::vector < typename ImageType::IndexType > > data;
  fProgressDialog progress("Pre-Processing", 1);
  progress.SetProgress(0, 100);
  qApp->processEvents();


  UpdateNumberOfPointsInTable();

  if (CheckCompletenessOfInputDataForEGFR(t1ceDataPresent, t2flairDataPresent, perfusionDataPresent) == false)
    return data;

  progress.SetProgress(5, 100);
  qApp->processEvents();

  //typedef itk::Image<float, 3> ImageType;
  typedef itk::Image<float, 4> PerfusionImageType;
  typedef itk::CastImageFilter< itk::Image<float, 3>, ImageType > CastFilterType;
  typename CastFilterType::Pointer castFilter = CastFilterType::New();
  typedef typename ImageType::Pointer ImagePointerType;
  ImagePointerType T1CEImagePointer;
  ImagePointerType T2FlairImagePointer;
  std::vector<ImagePointerType>	PerfusionImagePointer;
  typedef itk::Image<float, 3> InternalImageType;
  typename itk::MultiResolutionImageRegistrationMethod<ImageType, ImageType>::Pointer Registrar;
  std::vector<ImagePointerType> Perfusion_Registered;
  PerfusionImageType::Pointer perfusionImage = NULL;

  std::vector<typename ImageType::IndexType> nearIndices;
  std::vector<typename ImageType::IndexType> farIndices;
  FormulateNearFarPoints<ImageType>(nearIndices, farIndices);
  std::string imagetype = "";

  for (unsigned int index = 0; index < mSlicerManagers.size(); index++)
  {
    castFilter->SetInput(mSlicerManagers[index]->mITKImage);
    castFilter->Update();
    if (mSlicerManagers[index]->mImageSubType == IMAGE_TYPE_T1CE)
      T1CEImagePointer = castFilter->GetOutput();
    else
      T2FlairImagePointer = castFilter->GetOutput();
  }
  for (unsigned int index = 0; index < mNonViewingImageManager.size(); index++)
  {
    if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_PERFUSION && mNonViewingImageManager[index]->mImageType == IMAGE_DICOM)
    {
      for (unsigned int seriesindex = 0; seriesindex < mNonViewingImageManager[index]->mNVImageSeriesReaderPointer.size(); seriesindex++)
      {
        castFilter->SetInput(mNonViewingImageManager[index]->mNVImageSeriesReaderPointer[seriesindex]);
        castFilter->Update();        
        PerfusionImagePointer.push_back(castFilter->GetOutput());
      }
      imagetype = "dicom";
    }
    else if (mNonViewingImageManager[index]->mImageSubType == IMAGE_TYPE_PERFUSION && mNonViewingImageManager[index]->mImageType == IMAGE_NIFTI)
    {
      perfusionImage = mNonViewingImageManager[index]->mPerfusionImagePointer;
      imagetype = "nifti";
    }
  }
  Perfusion_Registered.resize(PerfusionImagePointer.size());
  if (imagetype == "dicom")
  {
    Registrar = mPreprocessingObj->Registration<ImageType, InternalImageType>(T1CEImagePointer, PerfusionImagePointer[0]);
    progress.SetProgress(10, 100);
    qApp->processEvents();

    for (unsigned int index = 0; index < PerfusionImagePointer.size(); index++)
    {
      Perfusion_Registered[index] = mPreprocessingObj->ResampleTransform<ImageType>(Registrar, T1CEImagePointer, PerfusionImagePointer[index]);
      progress.SetProgress((index + 1) * 2 + 10, 100);
      qApp->processEvents();
    }
  }
  VectorVectorDouble pNearIntensities;
  VectorVectorDouble pFarIntensities;
  std::vector<double> revisedNearIndices;
  std::vector<double> revisedFarIndices;

  std::vector<double> EGFRStatusParams;
  if (imagetype == "dicom")
    mEGFRPredictor->LoadPerfusionData<ImageType, PerfusionImageType>(perfusionImage, Perfusion_Registered, nearIndices, farIndices, pNearIntensities, pFarIntensities, IMAGE_DICOM);
  else
    mEGFRPredictor->LoadPerfusionData<ImageType, PerfusionImageType>(perfusionImage, Perfusion_Registered, nearIndices, farIndices, pNearIntensities, pFarIntensities, IMAGE_NIFTI);

  mEGFRPredictor->CalculateQualifiedIndicesForSaving(pNearIntensities, pFarIntensities, revisedNearIndices, revisedFarIndices);


  std::vector<typename ImageType::IndexType> revisedNearIndices3D;
  std::vector<typename ImageType::IndexType> revisedFarIndices3D;

  for (unsigned int index = 0; index < revisedNearIndices.size(); index++)
    revisedNearIndices3D.push_back(nearIndices[index]);

  for (unsigned int index = 0; index < revisedFarIndices.size(); index++)
    revisedFarIndices3D.push_back(farIndices[index]);

  std::vector< std::vector < typename ImageType::IndexType > > stats;
  stats.push_back(revisedNearIndices3D);
  stats.push_back(revisedFarIndices3D);
  return stats;
}




template<class InputPixelType, unsigned int VImageDimension>
void fMainWindow::LoadImage(std::string name, itk::Image<InputPixelType, VImageDimension>* image, vtkTransform* transform)
{
	bool bFirstLoad = false;
	if (mSlicerManagers.size() == 0) 
	{
		bFirstLoad = true;
	}

	QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

#ifdef USE_PROCESSDIALOG
	//init the progress events
	fProgressDialog progress("Loading " + name, 1);
	progress.SetProgress(0, 2);
	qApp->processEvents();
#endif

	SlicerManager* imageManager = new SlicerManager(3, mLandmarks, mSeedPoints /*INCORRECT, should be mNearPoints*/, mSeedPoints/*INCORRECT, should be mDrawingPoints*/);

	// convert image type to float
	imageManager->SetImage<InputPixelType, float, VImageDimension>(image, transform);

	//
	imageManager->SetMask(mMask);
	//
		
	int rowIndex = mSlicerManagers.size();
	imagesTable->setRowCount(rowIndex + 1);
	overlayTable->setRowCount(rowIndex + 1);

	mSlicerManagers.push_back(imageManager);

	QFileInfo fileinfo(name.c_str());
	QString id = name.c_str() + QString::number(mSlicerManagers.size()-1);
	//
	{
		QTableWidgetItem *item = new QTableWidgetItem(fileinfo.fileName());
		item->setData(Qt::UserRole, id.toStdString().c_str());
			
		QTablePushButton* cButton = new QTablePushButton;
		cButton->setItem(item);
		cButton->setText(tr("X"));
		cButton->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);
		cButton->setMaximumWidth(30);
		connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseImage(QTableWidgetItem*)));

		imagesTable->setCellWidget(rowIndex, TAB_IMAGES_COLUMN_CLOSE, cButton);
		imagesTable->setItem(rowIndex, TAB_IMAGES_COLUMN_NAME, item);
		imagesTable->resizeColumnToContents(TAB_IMAGES_COLUMN_CLOSE);
		imagesTable->resizeColumnToContents(TAB_IMAGES_COLUMN_NAME);
	}

	{
		QTableWidgetItem *item = new QTableWidgetItem(fileinfo.fileName());
		item->setData(Qt::UserRole, id.toStdString().c_str());
			
		QTablePushButton* cButton = new QTablePushButton;
		cButton->setItem(item);
		cButton->setText(tr("X"));
		cButton->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);
		cButton->setMaximumWidth(30);
		connect(cButton, SIGNAL(clickedInto(QTableWidgetItem*)), this, SLOT(CloseImage(QTableWidgetItem*)));

		overlayTable->setCellWidget(rowIndex, TAB_IMAGES_COLUMN_CLOSE, cButton);
		overlayTable->setItem(rowIndex, TAB_IMAGES_COLUMN_NAME, item);
		overlayTable->resizeColumnToContents(TAB_IMAGES_COLUMN_CLOSE);
		overlayTable->resizeColumnToContents(TAB_IMAGES_COLUMN_NAME);
	}

	mSlicerManagers.back()->SetId(id.toStdString());

	connect(mSlicerManagers.back(), SIGNAL(currentImageChanged(std::string)), 
		this,SLOT(CurrentImageChanged(std::string)));
	connect(mSlicerManagers.back(), SIGNAL(currentPickedImageChanged(std::string)),
		this, SLOT(CurrentPickedImageChanged(std::string)));
	connect(mSlicerManagers.back(), SIGNAL(UpdatePosition(int, double, double, double, double, double, double, double)),
		this,SLOT(MousePositionChanged(int,double, double, double, double, double, double, double)));
	connect(mSlicerManagers.back(), SIGNAL(WindowLevelChanged()),
		this,SLOT(WindowLevelChanged()));
	connect(mSlicerManagers.back(), SIGNAL(UpdateSlice(int,int)),
		this,SLOT(UpdateSlice(int,int)));
	connect(mSlicerManagers.back(), SIGNAL(UpdateSliceRange(int,int,int)),
		this,SLOT(UpdateSliceRange(int,int,int)));
	connect(mSlicerManagers.back(), SIGNAL(UpdateLinkManager(std::string,int,double,double,double)),
		this,SLOT(UpdateLinkManager(std::string,int,double,double,double)));
	connect(mSlicerManagers.back(), SIGNAL(UpdateLinkedNavigation(std::string,SlicerManager*,Slicer*)),
		this,SLOT(UpdateLinkedNavigation(std::string,SlicerManager*,Slicer*)));
	connect(mSlicerManagers.back(), SIGNAL(ChangeImageWithOrder(SlicerManager*,int)),
		this,SLOT(ChangeImageWithOrder(SlicerManager*,int)));

	connect(mSlicerManagers.back(), SIGNAL(SeedPointsAdded()), tumorPanel, SLOT(sAddPoint()));
	connect(mSlicerManagers.back(), SIGNAL(SeedPointsAdded(int, bool)), tumorPanel, SLOT(sAddPoint(int, bool)));
	connect(mSlicerManagers.back(), SIGNAL(TissuePointsAdded(int)), tumorPanel, SLOT(tAddPoint(int)));
	connect(mSlicerManagers.back(), SIGNAL(TissuePointsRemoved(int)), tumorPanel, SLOT(tRemovePoint(int)));


	InitSlicers();

	if (bFirstLoad) 
	{
		InitMask(mSlicerManagers.back()->mImage);
	}
	{
		for (int i = 0; i < (int)mSlicerManagers.back()->mSlicers.size(); i++) {
			mSlicerManagers.back()->mSlicers[i]->SetMask(mSlicerManagers.back()->GetMask());
		}
	}

#ifdef USE_PROCESSDIALOG
	progress.SetProgress(1, 2);
	qApp->processEvents();
#endif

	{
		AxialViewWidget->show();
		CoronalViewWidget->show();
		SaggitalViewWidget->show();
//		infoPanel->show();

		windowLabel->setEnabled(true);
		windowSpinBox->setEnabled(true);
		levelLabel->setEnabled(true);
		levelSpinBox->setEnabled(true);
		presetLabel->setEnabled(true);
		presetComboBox->setEnabled(true);

		if (bFirstLoad) {
			for (int i = 0; i < 3; i++) {
				mSlicerManagers.back()->GetSlicer(i)->SetInitPosition();
			}
			QTableWidgetItem* item = GetItemFromSlicerManager(mSlicerManagers.back());
			item->setSelected(true);
			DisplayChanged(item);
		} else {
			QTableWidgetItem* item = NULL;
 			for (int i = 0; i < (int)mSlicerManagers.size(); i++) {
				item = GetItemFromSlicerManager(mSlicerManagers[i]);
				if (item->isSelected()) {
					break;
				}
			}
			DisplayChanged(item);
		}

		if (mSlicerManagers.size() > 1) 
		{
			for (int i = 0; i < (int)mSlicerManagers.size(); i++ ) 
			{
				for (int j = i+1; j < (int)mSlicerManagers.size(); j++) 
				{
					AddLink(QString::fromStdString(mSlicerManagers[i]->GetId()), QString::fromStdString(mSlicerManagers[j]->GetId()));
				}
			}
		}

		if (bFirstLoad) {
			QTableWidgetItem* item = GetItemFromSlicerManager(mSlicerManagers.back());
			item->setSelected(true);
			DisplayChanged(item);
		} else {
			QTableWidgetItem* item = NULL;
 			for (int i = 0; i < (int)mSlicerManagers.size(); i++) {
				item = GetItemFromSlicerManager(mSlicerManagers[i]);
				if (item->isSelected()) {
					break;
				}
			}
			DisplayChanged(item);
		}
		
		InitDisplay();
	}

#ifdef USE_PROCESSDIALOG
	progress.SetProgress(2, 2);
	qApp->processEvents();
#endif

	QApplication::restoreOverrideCursor();
}

#endif
