///////////////////////////////////////////////////////////////////////////////////////
// fTumorPanel.cxx
//
// Copyright (c) 2015. All rights reserved.
// Section of Biomedical Image Analysis
// Center for Biomedical Image Computing and Analytics
// Department of Radiology
// Perelman School of Medicine
// University of Pennsylvania
//
// Contact details: sbia-software@uphs.upenn.edu
//
// License Agreement: https://www.cbica.upenn.edu/sbia/software/license.html
///////////////////////////////////////////////////////////////////////////////////////

#include "fTumorPanel.h"
#include "CBICA_Viewer.h"
#include "fMainWindow.h"
#include "SlicerManager.h"
#include "Landmarks.h"

class tEventFilter : public QObject
{
public:
	fTumorPanel * mTObj;
	tEventFilter(fTumorPanel * obj) :QObject()
	{
		mTObj = obj;
	};
	~tEventFilter(){};

	bool eventFilter(QObject* object, QEvent* event)
	{
		if (event->type() == QEvent::KeyPress)
		{
			QKeyEvent *keyEvent = static_cast<QKeyEvent *>(event);

      switch (keyEvent->key())
      {
      case Qt::Key_Return:
      {
        mTObj->HandleKeyPressingEventTTable();
        return true;
      }
      case Qt::Key_Down:
      {
        mTObj->HandleDownKeyEventTTable();
        return true;
      }
      case Qt::Key_Up:
      {
        mTObj->HandleUpKeyEventTTable();
        return true;
      }
      case Qt::Key_Delete:
      {
        mTObj->HandleDeleteKeyEventTTable();
        return true;
      }
      default:
      {
        std::cerr << "Undefined Key press.\n";
        //exit(EXIT_FAILURE); // probably not required
        return false;
      }
      }
		}
		else
		{
			return QObject::eventFilter(object, event);
		}
	}
};
class sEventFilter : public QObject
{
public:
	fTumorPanel * mTObj;
	sEventFilter(fTumorPanel * obj) :QObject()
	{
		mTObj = obj;
	};
	~sEventFilter(){};

	bool eventFilter(QObject* object, QEvent* event)
	{
		if (event->type() == QEvent::KeyPress)
		{
			QKeyEvent *keyEvent = static_cast<QKeyEvent *>(event);

      switch (keyEvent->key())
      {
      case Qt::Key_Return:
      {
        mTObj->HandleKeyPressingEventSTable();
        return true;
      }
      case Qt::Key_Down:
      {
        mTObj->HandleDownKeyEventSTable();
        return true;
      }
      case Qt::Key_Up:
      {
        mTObj->HandleUpKeyEventTTable();
        return true;
      }
      case Qt::Key_Delete:
      {
        mTObj->HandleDeleteKeyEventTTable();
        return true;
      }
      default:
      {
        std::cerr << "Undefined Key press.\n";
        //exit(EXIT_FAILURE); // probably not required
        return false;
      }
      }
		}
		else
		{
			return QObject::eventFilter(object, event);
		}
	}
};
//static fMainWindow* getMainWindow()
//{
//	QWidgetList widgets = qApp->topLevelWidgets();
//	for (QWidgetList::iterator i = widgets.begin(); i != widgets.end(); ++i)
//	{
//		if ((*i)->objectName() == "MainWindow")
//		{
//			return qobject_cast<fMainWindow*>(*i);
//		}
//	}
//	return qobject_cast<fMainWindow*>(qApp->activeWindow());
//}
fTumorPanel::fTumorPanel(QWidget * parent) : QWidget(parent)
{
	setupUi(this);
	mTissueType = 0;
	mCurrentSPoints = NULL;
	mCurrentTPoints = NULL;
	sTableWidget->setSelectionBehavior(QAbstractItemView::SelectRows);
	sTableWidget->setSelectionMode(QAbstractItemView::SingleSelection);

	sLoadButton->setEnabled(false);
	sSaveButton->setEnabled(false);
	sRemoveButton->setEnabled(false);
	sRemoveAllButton->setEnabled(false);
	tLoadButton->setEnabled(false);
	tSaveButton->setEnabled(false);
	tRemoveButton->setEnabled(false);
	tRemoveAllButton->setEnabled(false);

	connect(sLoadButton, SIGNAL(clicked()), this, SLOT(sLoad()));
	connect(sSaveButton, SIGNAL(clicked()), this, SLOT(sSave()));
	connect(sRemoveButton, SIGNAL(clicked()), this, SLOT(sRemoveSelectedPoints()));
	connect(sRemoveAllButton, SIGNAL(clicked()), this, SLOT(sRemoveAllPoints()));
	connect(tLoadButton, SIGNAL(clicked()), this, SLOT(tLoad()));
	connect(tSaveButton, SIGNAL(clicked()), this, SLOT(tSave()));
	connect(tRemoveButton, SIGNAL(clicked()), this, SLOT(tRemoveSelectedPoints()));
	connect(tRemoveAllButton, SIGNAL(clicked()), this, SLOT(tRemoveAllPoints()));
	connect(sTableWidget, SIGNAL(cellClicked(int, int)), this, SLOT(sTableClicked(int, int)));
	connect(tTableWidget, SIGNAL(cellClicked(int, int)), this, SLOT(tTableClicked(int, int)));
	connect(sTableWidget, SIGNAL(cellDoubleClicked(int, int)), this, SLOT(sTableDoubleClicked(int, int)));
	connect(tTableWidget, SIGNAL(cellDoubleClicked(int, int)), this, SLOT(tTableDoubleClicked(int, int)));

	connect(type1RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType1()));
	connect(type2RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType2()));
	connect(type3RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType3()));
	connect(type4RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType4()));
	connect(type5RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType5()));
	connect(type6RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType6()));
	connect(type7RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType7()));
	connect(type8RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType8()));
	connect(type9RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType9()));
	connect(type10RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType10()));
	connect(type11RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType11()));
	connect(type12RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType12()));
	connect(type13RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType13()));
	connect(type14RadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType14()));

	connect(seedTypeRadioButton, SIGNAL(toggled(bool)), this, SLOT(SetSeedType()));
	connect(tissueTypeRadioButton, SIGNAL(toggled(bool)), this, SLOT(SetTissueType()));

	tTableWidget->installEventFilter(new tEventFilter(this));
	sTableWidget->installEventFilter(new sEventFilter(this));

	QFont font;
	font.setFamily(QString::fromUtf8("Calibri"));
	sTableWidget->setFont(font);
	tTableWidget->setFont(font);

	sTableWidget->resizeColumnsToContents();
	sTableWidget->resizeRowsToContents();
	tTableWidget->resizeColumnsToContents();
	tTableWidget->resizeRowsToContents();

	this->SetTissueType1();
	emit SetTissueCounter(mType);
	type1RadioButton->setChecked(true);
	tissueTypeRadioButton->setChecked(true);

	QFont font1;
	font1.setFamily(QString::fromUtf8("Calibri"));
	font1.setBold(true);
	tissueCounterLabel->setFont(font1);
	tissueTypeLabel->setFont(font1);
}

void fTumorPanel::Clear()
{
	sTableWidget->clearContents();
	tTableWidget->clearContents();
	sLoadButton->setEnabled(false);
	sSaveButton->setEnabled(false);
	sRemoveButton->setEnabled(false);
	sRemoveAllButton->setEnabled(false);
	tLoadButton->setEnabled(false);
	tSaveButton->setEnabled(false);
	tRemoveButton->setEnabled(false);
	tRemoveAllButton->setEnabled(false);
}

void fTumorPanel::sLoad()
{
	QString file;
	QFileDialog dialog(this, tr("Load Seed Points"), mCurrentPath.c_str(), tr("Seed Points (*.txt)"));
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setAcceptMode(QFileDialog::AcceptOpen);

	dialog.selectFile("init_seed.txt");

	int ret = dialog.exec();
	if (ret == QDialog::Accepted) 
	{
		file = dialog.selectedFiles()[0];
		//*/
		FILE* fp;
		char str_tmp[1024];
		double scan_seeds_info[MaxNumberOFTumorSeeds][4];
		int scan_seeds_num;
		fp = fopen(file.toStdString().c_str(), "r");
		if (fp == NULL) {
			printf("Cannot open %s\n", file.toStdString().c_str());
			return;
		}
		fscanf(fp, "%s", str_tmp);
		scan_seeds_num = atoi(str_tmp);
		if (scan_seeds_num > MaxNumberOFTumorSeeds) {
			printf("scan_seeds_num = %d > %d\n", scan_seeds_num, MaxNumberOFTumorSeeds);
			fclose(fp);
			return;
		}
		printf("scan_seeds_num = %d\n", scan_seeds_num);
		mCurrentSPoints->Clear();
		for (int i = 0; i < scan_seeds_num; i++) {
			fscanf(fp, "%lf %lf %lf %lf", &scan_seeds_info[i][0], &scan_seeds_info[i][1], &scan_seeds_info[i][2], &scan_seeds_info[i][3]);
			//
#ifdef USE_LANDMARKS_LPS_COORD
			mCurrentSPoints->AddLandmark(scan_seeds_info[i][0], scan_seeds_info[i][1], scan_seeds_info[i][2], scan_seeds_info[i][3] / 2.0, 0, i);
#endif
#ifdef USE_LANDMARKS_RAS_COORD
			mCurrentSPoints->AddLandmark(-scan_seeds_info[i][0], -scan_seeds_info[i][1], scan_seeds_info[i][2], scan_seeds_info[i][3] / 2.0, 0, i);
#endif
			//
			printf("seed %d: %f %f %f %f\n", i, scan_seeds_info[i][0], scan_seeds_info[i][1], scan_seeds_info[i][2], scan_seeds_info[i][3]);
		}
		fclose(fp);

	}
	SetCurrentSPoints(mCurrentSPoints);
	emit UpdateRenderWindows();
}
void fTumorPanel::sSave()
{
	QString file;
	QFileDialog dialog(this, tr("Save Seed Points"), mCurrentPath.c_str(), tr("Seed Points (*.txt)"));
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setAcceptMode(QFileDialog::AcceptSave);
	dialog.selectFile("init_seed.txt");

	int ret = dialog.exec();
	if (ret == QDialog::Accepted) 
	{
		file = dialog.selectedFiles()[0];
		//*/
		FILE* fp;
		fp = fopen(file.toStdString().c_str(), "w");
		if (fp == NULL) {
			printf("Cannot open %s\n", file.toStdString().c_str());
			return;
		}
		fprintf(fp, "%d\n", mCurrentSPoints->GetNumberOfPoints());
		for (int i = 0; i < (int)mCurrentSPoints->GetNumberOfPoints(); i++) 
		{
#ifdef USE_LANDMARKS_LPS_COORD
			fprintf(fp, "%f %f %f %f\n", mCurrentSPoints->mLandmarks[i].coordinates[0], mCurrentSPoints->mLandmarks[i].coordinates[1], mCurrentSPoints->mLandmarks[i].coordinates[2], mCurrentSPoints->mLandmarks[i].radius * 2.0);
#endif
#ifdef USE_LANDMARKS_RAS_COORD
			fprintf(fp, "%f %f %f %f\n", -mCurrentSPoints->mLandmarks[i].coordinates[0], -mCurrentSPoints->mLandmarks[i].coordinates[1], mCurrentSPoints->mLandmarks[i].coordinates[2], mCurrentSPoints->mLandmarks[i].radius * 2.0);
#endif
		}
		fclose(fp);
	}
}
//
void fTumorPanel::tLoad()
{
	QString file;
	QFileDialog dialog(this, tr("Load Tissue Points"), mCurrentPath.c_str(), tr("Tissue Points (*.txt)"));
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setAcceptMode(QFileDialog::AcceptOpen);
	dialog.selectFile("init_point.txt");

	int ret = dialog.exec();
	if (ret == QDialog::Accepted)
	{
		file = dialog.selectedFiles()[0];
		FILE* fp;
		unsigned int i;
    int j;
		fp = fopen(file.toStdString().c_str(), "r");
		if (fp == NULL)
		{
			printf("Cannot open %s\n", file.toStdString().c_str());
			return;
		}
		mCurrentTPoints->Clear();
		while (TRUE)
		{
			int res;
			char class_id[1024];
			double fx, fy, fz;
			//
			res = fscanf(fp, "%s %lf %lf %lf", class_id, &fx, &fy, &fz);
			if (res == 0 || res == EOF) { break; }
				for (j = 0; j < NumberOfPriorChannels; j++)
				{
					if (strcmp(class_id, labels[j]) == 0)
					{
						if (j == 0)
						{
							// skip BG
						}
						else
						{
#ifdef USE_LANDMARKS_LPS_COORD
							mCurrentTPoints->AddLandmark(fx, fy, fz, 0, 0, j);
#endif
#ifdef USE_LANDMARKS_RAS_COORD
							mCurrentTPoints->AddLandmark(-fx, -fy, fz, 0, 0, j);
#endif
						}
						break;
					}
				}
				if (j == NumberOfPriorChannels)
				{
					printf("class id is wrong\n");
					mCurrentTPoints->Clear();
					fclose(fp);
					return;
				}
		}
			for (i = 0; i < mCurrentTPoints->GetNumberOfPoints(); i++)
			{
				if (mCurrentTPoints->mLandmarks[i].bValid)
				{
					printf("point %d: %s %f %f %f\n", i, labels[mCurrentTPoints->mLandmarks[i].id], mCurrentTPoints->mLandmarks[i].coordinates[0], mCurrentTPoints->mLandmarks[i].coordinates[1], mCurrentTPoints->mLandmarks[i].coordinates[2]);
				}
			}
		fclose(fp);

	}
	SetCurrentTPoints(mCurrentTPoints);
	emit UpdateRenderWindows();
}
void fTumorPanel::tSave()
{
	QString file;
	QFileDialog dialog(this, tr("Save Tissue Points"), mCurrentPath.c_str(), tr("Tissue Points (*.txt)"));
	dialog.setFileMode(QFileDialog::AnyFile);
	dialog.setAcceptMode(QFileDialog::AcceptSave);
	dialog.selectFile("init_point.txt");


	int ret = dialog.exec();
	if (ret == QDialog::Accepted)
	{
		file = dialog.selectedFiles()[0];
		//*/
		FILE* fp;
		fp = fopen(file.toStdString().c_str(), "w");
		if (fp == NULL)
		{
			printf("Cannot open %s\n", file.toStdString().c_str());
			return;
		}
		for (int i = 0; i < NumberOfPriorChannels; i++)
		{
			for (unsigned int j = 0; j < mCurrentTPoints->GetNumberOfPoints(); j++)
			{
				if (mCurrentTPoints->mLandmarks[j].id == i && mCurrentTPoints->mLandmarks[j].bValid)
				{
#ifdef USE_LANDMARKS_LPS_COORD
					fprintf(fp, "%s\n%f %f %f\n", labels[i], mCurrentTPoints->mLandmarks[j].coordinates[0], mCurrentTPoints->mLandmarks[j].coordinates[1], mCurrentTPoints->mLandmarks[j].coordinates[2]);
#endif
#ifdef USE_LANDMARKS_RAS_COORD
					fprintf(fp, "%s\n%f %f %f\n", labels[i], -mCurrentTPoints->mLandmarks[j].coordinates[0], -mCurrentTPoints->mLandmarks[j].coordinates[1], mCurrentTPoints->mLandmarks[j].coordinates[2]);
#endif
				}
			}
		}
		fclose(fp);
	}
}

void fTumorPanel::sRemoveSelectedPoints()
{
	QList<QTableWidgetItem*> items = sTableWidget->selectedItems();
	if (items.empty())
		return;

	int rowIndex = items[0]->row();
	std::vector<int> rowIndices;
	rowIndices.push_back(rowIndex);

	for (int i = 0; i < items.size(); i++)
	{
		if (std::find(rowIndices.begin(), rowIndices.end(), items[i]->row()) != rowIndices.end())
			rowIndices.push_back(items[i]->row());
	}
	for (unsigned int j = 0; j < rowIndices.size(); j++)
		sRemoveCurrentIndexPoint(rowIndex);
}

void fTumorPanel::tRemoveCurrentIndexPoint(unsigned int rowIndex)
{
	mCurrentTPoints->RemoveLandmark(rowIndex);
	tTableWidget->removeRow(rowIndex);
	emit UpdateRenderWindows();

	if (rowIndex > mCurrentTPoints->GetNumberOfPoints())
		return;

	tTableWidget->selectRow(rowIndex);
	QTableWidgetItem * item2 = tTableWidget->item(rowIndex, 0);
	tTableWidget->setItemSelected(item2, true);
	UpdateCurrentPoints();
}

void fTumorPanel::sRemoveCurrentIndexPoint(unsigned int rowIndex)
{
	mCurrentSPoints->RemoveLandmark(rowIndex);
	sTableWidget->removeRow(rowIndex);

	emit UpdateRenderWindows();

	if (rowIndex>mCurrentSPoints->GetNumberOfPoints())
		return;

	sTableWidget->selectRow(rowIndex);
	QTableWidgetItem * item2 = sTableWidget->item(rowIndex, 0);
	sTableWidget->setItemSelected(item2, true);
}

void fTumorPanel::tRemoveSelectedPoints()
{
	QList<QTableWidgetItem*> items = tTableWidget->selectedItems();
	if (items.empty())
		return;

	int rowIndex = items[0]->row();
	std::vector<int> rowIndices;
	rowIndices.push_back(rowIndex);

	for (int i = 0; i < items.size(); i++)
	{
		if (std::find(rowIndices.begin(), rowIndices.end(), items[i]->row()) != rowIndices.end())
			rowIndices.push_back(items[i]->row());
	}
	for (unsigned int j = 0; j < rowIndices.size(); j++)
		tRemoveCurrentIndexPoint(rowIndex);
	UpdateCurrentPoints();
}

void fTumorPanel::sAddPoint()
{
	sAddPoint(mCurrentSPoints->GetNumberOfPoints() - 1,false);
}

void fTumorPanel::sAddPoint(int landmarksIndex,bool update)
{
	int rowIndex = landmarksIndex;// mCurrentSPoints->GetNumberOfPoints() - 1; //sTableWidget->rowCount() //landmarksIndex;
	float x, y, z;
	// double val;
	bool bValid;
	float r;

	//sTableWidget->setRowCount(rowIndex+1);
	//sTableWidget->setRowHeight(rowIndex, 20);

	if (update==false)
		sTableWidget->setRowCount(rowIndex + 1);


#ifdef USE_LANDMARKS_LPS_COORD
	x = mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[0];
	y = mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[1];
	z = mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[2];
#endif
#ifdef USE_LANDMARKS_RAS_COORD
	x = -mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[0];
	y = -mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[1];
	z = mCurrentSPoints->mLandmarks[landmarksIndex].coordinates[2];
#endif
	// val = mCurrentSPoints->mLandmarks[landmarksIndex].pixel_value;
	bValid = mCurrentSPoints->mLandmarks[landmarksIndex].bValid;
	r = mCurrentSPoints->mLandmarks[landmarksIndex].radius;

	if (bValid) {
		QTableWidgetItem* xItem = new QTableWidgetItem(QString::number(x, 'f', 3));
		//xItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
		xItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
		sTableWidget->setItem(rowIndex, 0, xItem);

		QTableWidgetItem* yItem = new QTableWidgetItem(QString::number(y, 'f', 3));
		//yItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
		yItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
		sTableWidget->setItem(rowIndex, 1, yItem);

		QTableWidgetItem* zItem = new QTableWidgetItem(QString::number(z, 'f', 3));
		//zItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
		zItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
		sTableWidget->setItem(rowIndex, 2, zItem);

		QTableWidgetItem* rItem = new QTableWidgetItem(QString::number(r, 'f', 3));
		//rItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
		rItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
		sTableWidget->setItem(rowIndex, 3, rItem);
	}
}

void fTumorPanel::tAddPoint(int rowIndex)
{
	float x, y, z;
	// double val;
	bool bValid;

	int idx = mCurrentTPoints->GetNumberOfPoints() - 1;
	tTableWidget->setRowCount(mCurrentTPoints->GetNumberOfPoints());

#ifdef USE_LANDMARKS_LPS_COORD
	x = mCurrentTPoints->mLandmarks[idx].coordinates[0];
	y = mCurrentTPoints->mLandmarks[idx].coordinates[1];
	z = mCurrentTPoints->mLandmarks[idx].coordinates[2];
#endif
#ifdef USE_LANDMARKS_RAS_COORD
	x = -mCurrentTPoints->mLandmarks[idx].coordinates[0];
	y = -mCurrentTPoints->mLandmarks[idx].coordinates[1];
	z = mCurrentTPoints->mLandmarks[idx].coordinates[2];
#endif
	// val = mCurrentTPoints->mLandmarks[idx].pixel_value;
	bValid = mCurrentTPoints->mLandmarks[idx].bValid;

	std::string tissue_label = labels[mCurrentTPoints->mLandmarks[idx].id];

	if (bValid)
	{

		QTableWidgetItem* labelItem = new QTableWidgetItem(QString::fromStdString(tissue_label));
		//xItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
		labelItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
		tTableWidget->setItem(idx, 0, labelItem);

		QTableWidgetItem* xItem = new QTableWidgetItem(QString::number(x, 'f', 3));
		//xItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
		xItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
		tTableWidget->setItem(idx, 1, xItem);

		QTableWidgetItem* yItem = new QTableWidgetItem(QString::number(y, 'f', 3));
		//yItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
		yItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
		tTableWidget->setItem(idx, 2, yItem);

		QTableWidgetItem* zItem = new QTableWidgetItem(QString::number(z, 'f', 3));
		//zItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
		zItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
		tTableWidget->setItem(idx, 3, zItem);

		emit UpdateRenderWindows();
		tTableWidget->scrollToItem(labelItem, QAbstractItemView::PositionAtTop);
	}
	UpdateCurrentPoints();
}

void fTumorPanel::SetCurrentSPoints(Landmarks* lm)
{
	sLoadButton->setEnabled(true);
	sSaveButton->setEnabled(true);
	sRemoveButton->setEnabled(true);
	sRemoveAllButton->setEnabled(true);

	mCurrentSPoints = lm;
	sTableWidget->clearContents();
	for (unsigned int i = 0; i < mCurrentSPoints->GetNumberOfPoints(); i++)
	{
		sAddPoint(i,false);
	}
	sTableWidget->resizeColumnsToContents();
}
void fTumorPanel::SetCurrentTPoints(Landmarks* lm)
{
	tLoadButton->setEnabled(true);
	tSaveButton->setEnabled(true);
	tRemoveButton->setEnabled(true);
	tRemoveAllButton->setEnabled(true);
	mCurrentTPoints = lm;

	tTableWidget->clearContents();


	float x, y, z;
	// double val;
	bool bValid;
	tTableWidget->setRowCount(mCurrentTPoints->GetNumberOfPoints());
	for (unsigned int i = 0; i < mCurrentTPoints->GetNumberOfPoints(); i++)
	{
#ifdef USE_LANDMARKS_LPS_COORD
		x = mCurrentTPoints->mLandmarks[i].coordinates[0];
		y = mCurrentTPoints->mLandmarks[i].coordinates[1];
		z = mCurrentTPoints->mLandmarks[i].coordinates[2];
#endif
#ifdef USE_LANDMARKS_RAS_COORD
		x = -mCurrentTPoints->mLandmarks[i].coordinates[0];
		y = -mCurrentTPoints->mLandmarks[i].coordinates[1];
		z = mCurrentTPoints->mLandmarks[i].coordinates[2];
#endif
		// val = mCurrentTPoints->mLandmarks[i].pixel_value;
		bValid = mCurrentTPoints->mLandmarks[i].bValid;

		std::string tissue_label = labels[mCurrentTPoints->mLandmarks[i].id];


		if (bValid)
		{

			QTableWidgetItem* labelItem = new QTableWidgetItem(QString::fromStdString(tissue_label));
			//xItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
			labelItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
			tTableWidget->setItem(i, 0, labelItem);

			QTableWidgetItem* xItem = new QTableWidgetItem(QString::number(x, 'f', 3));
			//xItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
			xItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
			tTableWidget->setItem(i, 1, xItem);

			QTableWidgetItem* yItem = new QTableWidgetItem(QString::number(y, 'f', 3));
			//yItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
			yItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
			tTableWidget->setItem(i, 2, yItem);

			QTableWidgetItem* zItem = new QTableWidgetItem(QString::number(z, 'f', 3));
			//zItem->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
			zItem->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);
			tTableWidget->setItem(i, 3, zItem);
		}
	}
	emit UpdateRenderWindows();
	//	
	//	for (unsigned int i = 0; i < tTableWidget->rowCount(); i++) 
	//{
	//	tAddPoint(i);
	//}
	tTableWidget->resizeColumnsToContents();
	UpdateCurrentPoints();
}

void fTumorPanel::sTableClicked(int row, int col)
{
	//emit SetActiveLandmarksType(LANDMARK_TYPE_SEED_POINTS, row, col);
}
void fTumorPanel::tTableClicked(int row, int col)
{
	//	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, row, col);
}

void fTumorPanel::sTableDoubleClicked(int row, int col)
{
	if (mCurrentSPoints == NULL)
		return;
	if (mCurrentSPoints->GetNumberOfPoints() == 0)
		return;

	emit SetActiveLandmarksType(LANDMARK_TYPE_SEED_POINTS, row, col);

	if (row >= 0 && row < (int)mCurrentSPoints->GetNumberOfPoints()) 
	{
		if (mCurrentSPoints->mLandmarks[row].bValid) 
		{
			double x, y, z;
			x = mCurrentSPoints->mLandmarks[row].coordinates[0];
			y = mCurrentSPoints->mLandmarks[row].coordinates[1];
			z = mCurrentSPoints->mLandmarks[row].coordinates[2];
			emit MoveSlicerCursor(x, y, z);
		}
	}
}
void fTumorPanel::tTableDoubleClicked(int row, int col)
{
	if (mCurrentTPoints == NULL)
		return;
	if (mCurrentTPoints->GetNumberOfPoints() == 0)
		return;

	mTissueType = mCurrentTPoints->mLandmarks[row].id - 1;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, col);
	emit SetTissueTableIndex(row);
	
	if (mTissueType == 0)
		type1RadioButton->setChecked(true);
	else if (mTissueType == 1)
		type2RadioButton->setChecked(true);
	else if (mTissueType == 2)
		type3RadioButton->setChecked(true);
	else if (mTissueType == 3)
		type4RadioButton->setChecked(true);
	else if (mTissueType == 4)
		type5RadioButton->setChecked(true);
	else if (mTissueType == 5)
		type6RadioButton->setChecked(true);
	else if (mTissueType == 6)
		type7RadioButton->setChecked(true);
	else if (mTissueType == 7)
		type8RadioButton->setChecked(true);
	else if (mTissueType == 8)
		type9RadioButton->setChecked(true);
	else if (mTissueType == 9)
		type10RadioButton->setChecked(true);
	else if (mTissueType == 10)
		type11RadioButton->setChecked(true);
	else if (mTissueType == 11)
		type12RadioButton->setChecked(true);
	else if (mTissueType == 12)
		type13RadioButton->setChecked(true);
	else if (mTissueType == 13)
		type14RadioButton->setChecked(true);

	double x, y, z;
	x = mCurrentTPoints->mLandmarks[row].coordinates[0];
	y = mCurrentTPoints->mLandmarks[row].coordinates[1];
	z = mCurrentTPoints->mLandmarks[row].coordinates[2];
	emit MoveSlicerCursor(x, y, z);
}

void fTumorPanel::sTableFocused(bool bFocused)
{
	//QFont font;
	//font.setFamily(QString::fromUtf8("Calibri"));
	////font.setPointSize(9);
	//font.setBold(bFocused);
	//sLabel->setFont(font);
}

void fTumorPanel::tTableFocused(bool bFocused)
{
	//QFont font;
	//font.setFamily(QString::fromUtf8("Calibri"));
	////font.setPointSize(9);
	//font.setBold(bFocused);
	//tLabel->setFont(font);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
void fTumorPanel::SetTissueType1()
{
	mTissueType = 0;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 0, 0);
}
void fTumorPanel::SetTissueType2()
{
	mTissueType = 1;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 1, 0);
}
void fTumorPanel::SetTissueType3()
{
	mTissueType = 2;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 2, 0);
}
void fTumorPanel::SetTissueType4()
{
	mTissueType = 3;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 3, 0);
}
void fTumorPanel::SetTissueType5()
{
	mTissueType = 4;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 4, 0);
}
void fTumorPanel::SetTissueType6()
{
	mTissueType = 5;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 5, 0);
}
void fTumorPanel::SetTissueType7()
{
	mTissueType = 6;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 6, 0);
}
void fTumorPanel::SetTissueType8()
{
	mTissueType = 7;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 7, 0);
}
void fTumorPanel::SetTissueType9()
{
	mTissueType = 8;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 8, 0);
}
void fTumorPanel::SetTissueType10()
{
	mTissueType = 9;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 9, 0);
}
void fTumorPanel::SetTissueType11()
{
	mTissueType = 10;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 10, 0);
}
void fTumorPanel::SetTissueType12()
{
	mTissueType = 11;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 11, 0);
}
void fTumorPanel::SetTissueType13()
{
	mTissueType = 12;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 12, 0);
}
void fTumorPanel::SetTissueType14()
{
	mTissueType = 13;
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, 13, 0);
}


void fTumorPanel::SetCurrentSelectedTissueType()
{
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::tRemoveAllPoints()
{
	int rowCount = mCurrentTPoints->GetNumberOfPoints();
	for (int rowIndex = rowCount - 1; rowIndex >= 0; rowIndex--)
	{
		mCurrentTPoints->RemoveLandmark(rowIndex);
		tTableWidget->removeRow(rowIndex);
	}
	emit UpdateRenderWindows();
	UpdateCurrentPoints();
}
void fTumorPanel::sRemoveAllPoints()
{
	int rowCount = mCurrentSPoints->GetNumberOfPoints();
	for (int rowIndex = rowCount - 1; rowIndex >= 0; rowIndex--)
	{
		mCurrentSPoints->RemoveLandmark(rowIndex);
		sTableWidget->removeRow(rowIndex);
	}
	emit UpdateRenderWindows();
}
void fTumorPanel::HighlightCurrentSelctedPoints(double x, double y, double z, double X, double Y, double Z, double value)
{
	if (mCurrentTPoints == NULL)
		return;

	int rowCount = mCurrentTPoints->GetNumberOfPoints();
	for (int rowIndex = 0; rowIndex<rowCount; rowIndex++)
	{
		float roundX = floorf(x * 1000) / 1000;
		float roundY = floorf(y * 1000) / 1000;
		float roundZ = floorf(z * 1000) / 1000;

		float roundCurrentX = floorf(mCurrentTPoints->mLandmarks[rowIndex].coordinates[0] * 1000) / 1000;
		float roundCurrentY = floorf(mCurrentTPoints->mLandmarks[rowIndex].coordinates[1] * 1000) / 1000;
		float roundCurrentZ = floorf(mCurrentTPoints->mLandmarks[rowIndex].coordinates[2] * 1000) / 1000;

		if (roundX == roundCurrentX && roundY == roundCurrentY && roundZ == roundCurrentZ)
		{
			tTableWidget->selectRow(rowIndex);
			QTableWidgetItem * item = tTableWidget->item(rowIndex, 0);
			tTableWidget->scrollToItem(item, QAbstractItemView::PositionAtTop);
			tTableWidget->setItemSelected(item, true);
			break;
		}
	}
}
void fTumorPanel::HandleKeyPressingEventTTable()
{
	if (mCurrentTPoints == NULL)
		return;

	if (mCurrentTPoints->GetNumberOfPoints() == 0)
		return;

	QList<QTableWidgetItem*> items = tTableWidget->selectedItems();
	if (items.empty())
		return;

	int row = items[0]->row();

	double x, y, z;
	x = mCurrentTPoints->mLandmarks[row].coordinates[0];
	y = mCurrentTPoints->mLandmarks[row].coordinates[1];
	z = mCurrentTPoints->mLandmarks[row].coordinates[2];
	emit MoveSlicerCursor(x, y, z);
}
void fTumorPanel::HandleDownKeyEventTTable()
{
	if (mCurrentTPoints == NULL)
		return;

	if (mCurrentTPoints->GetNumberOfPoints() == 0)
		return;

	QList<QTableWidgetItem*> items = tTableWidget->selectedItems();
	if (items.empty())
		return;

	unsigned int rowindex = items[0]->row() + 1;
	if (rowindex > mCurrentTPoints->GetNumberOfPoints())
		return;

	for (unsigned int i = 0; i < mCurrentTPoints->GetNumberOfPoints(); i++)
	{
		QTableWidgetItem * item3 = tTableWidget->item(i, 0);
		tTableWidget->setItemSelected(item3, false);
	}
	tTableWidget->selectRow(rowindex);
	QTableWidgetItem * item2 = tTableWidget->item(rowindex, 0);
	tTableWidget->scrollToItem(item2, QAbstractItemView::PositionAtTop);
	tTableWidget->setItemSelected(item2, true);

}
void fTumorPanel::HandleUpKeyEventTTable()
{
	if (mCurrentTPoints == NULL)
		return;

	if (mCurrentTPoints->GetNumberOfPoints() == 0)
		return;

	QList<QTableWidgetItem*> items = tTableWidget->selectedItems();
	if (items.empty())
		return;

	int rowindex = items[0]->row() - 1;
	if (rowindex < 0)
		return;
	for (unsigned int i = 0; i < mCurrentTPoints->GetNumberOfPoints(); i++)
	{
		QTableWidgetItem * item3 = tTableWidget->item(i, 0);
		tTableWidget->setItemSelected(item3, false);
	}
	tTableWidget->selectRow(rowindex);
	QTableWidgetItem * item2 = tTableWidget->item(rowindex, 0);
	tTableWidget->scrollToItem(item2, QAbstractItemView::PositionAtTop);
	tTableWidget->setItemSelected(item2, true);
}

void fTumorPanel::HandleDeleteKeyEventTTable()
{
	tRemoveSelectedPoints();
}


void fTumorPanel::HandleKeyPressingEventSTable()
{
	if (mCurrentSPoints == NULL)
		return;

	if (mCurrentSPoints->GetNumberOfPoints() == 0)
		return;

	QList<QTableWidgetItem*> items = sTableWidget->selectedItems();
	if (items.empty())
		return;

	int row = items[0]->row();

	double x, y, z;
	x = mCurrentSPoints->mLandmarks[row].coordinates[0];
	y = mCurrentSPoints->mLandmarks[row].coordinates[1];
	z = mCurrentSPoints->mLandmarks[row].coordinates[2];
	emit MoveSlicerCursor(x, y, z);
}
void fTumorPanel::HandleDownKeyEventSTable()
{
	if (mCurrentSPoints == NULL)
		return;

	if (mCurrentSPoints->GetNumberOfPoints() == 0)
		return;

	QList<QTableWidgetItem*> items = sTableWidget->selectedItems();
	if (items.empty())
		return;

	unsigned int rowindex = items[0]->row() + 1;
	if (rowindex > mCurrentSPoints->GetNumberOfPoints())
		return;



	for (unsigned int i = 0; i < mCurrentSPoints->GetNumberOfPoints(); i++)
	{
		QTableWidgetItem * item3 = sTableWidget->item(i, 0);
		sTableWidget->setItemSelected(item3, false);
	}
	sTableWidget->selectRow(rowindex);
	QTableWidgetItem * item2 = sTableWidget->item(rowindex, 0);
	sTableWidget->scrollToItem(item2, QAbstractItemView::PositionAtTop);
	sTableWidget->setItemSelected(item2, true);

}
void fTumorPanel::HandleUpKeyEventSTable()
{
	if (mCurrentSPoints == NULL)
		return;

	if (mCurrentSPoints->GetNumberOfPoints() == 0)
		return;

	QList<QTableWidgetItem*> items = sTableWidget->selectedItems();
	if (items.empty())
		return;

	int rowindex = items[0]->row() - 1;
	if (rowindex < 0)
		return;

	for (unsigned int i = 0; i < mCurrentSPoints->GetNumberOfPoints(); i++)
	{
		QTableWidgetItem * item3 = sTableWidget->item(i, 0);
		sTableWidget->setItemSelected(item3, false);
	}
	sTableWidget->selectRow(rowindex);
	QTableWidgetItem * item2 = sTableWidget->item(rowindex, 0);
	sTableWidget->scrollToItem(item2, QAbstractItemView::PositionAtTop);
	sTableWidget->setItemSelected(item2, true);
}

void fTumorPanel::HandleDeleteKeyEventSTable()
{
	sRemoveSelectedPoints();
}

void fTumorPanel::SetSeedType()
{
	if (mCurrentSPoints == NULL)
	{
		emit SetActiveLandmarksType(LANDMARK_TYPE_SEED_POINTS, 0, 0);
		return;
	}
	else
		emit SetActiveLandmarksType(LANDMARK_TYPE_SEED_POINTS, mCurrentSPoints->GetNumberOfPoints(), 0);

}
void fTumorPanel::SetTissueType()
{
	emit SetActiveLandmarksType(LANDMARK_TYPE_TISSUE_POINTS, mTissueType, 0);
}
void fTumorPanel::UpdateCurrentPoints()
{
	if (mCurrentTPoints == NULL)
		return;

	int csfCounter = 0;
	int gmCounter = 0;
	int wmCounter = 0;
	int vsCounter = 0;
	int edCounter = 0;
	int ncrCounter = 0;
	int tuCounter = 0;
	int neCounter = 0;
	int cbCounter = 0;
	int vtCounter = 0;
	int canCounter = 0;
	int caeCounter = 0;
	int rtnCounter = 0;
	int rteCounter = 0;

	for (unsigned int i = 0; i < mCurrentTPoints->GetNumberOfPoints(); i++)
	{
    switch (mCurrentTPoints->mLandmarks[i].id)
    {
    case 1:
      csfCounter++;
    case 2:
      gmCounter++;
    case 3:
      wmCounter++;
    case 4:
      vsCounter++;
    case 5:
      edCounter++;
    case 6:
      ncrCounter++;
    case 7:
      tuCounter++;
    case 8:
      neCounter++;
    case 9:
      cbCounter++;
    case 10:
      vtCounter++;
    case 11:
      canCounter++;
    case 12:
      caeCounter++;
    case 13:
      rtnCounter++;
    case 14:
      rteCounter++;
    default:
      std::cerr << "Undefined mCurrentTPoints->mLandmarks\n";
      exit(EXIT_FAILURE);
    }    
	}
	PointsLabel1->setText(QString::number(csfCounter));
	PointsLabel2->setText(QString::number(gmCounter));
	PointsLabel3->setText(QString::number(wmCounter));
	PointsLabel4->setText(QString::number(vsCounter));
	PointsLabel5->setText(QString::number(edCounter));
	PointsLabel6->setText(QString::number(ncrCounter));
	PointsLabel7->setText(QString::number(tuCounter));
	PointsLabel8->setText(QString::number(neCounter));
	PointsLabel9->setText(QString::number(cbCounter));
	PointsLabel10->setText(QString::number(vtCounter));
	PointsLabel11->setText(QString::number(canCounter));
	PointsLabel12->setText(QString::number(caeCounter));
	PointsLabel13->setText(QString::number(rtnCounter));
	PointsLabel14->setText(QString::number(rteCounter));


}
