/*!
 * \file DeformImage.c
 *
 * Copyright (c) 2011 University of Pennsylvania. All rights reserved.
 * See COPYING file or https://www.rad.upenn.edu/sbia/software/license.html.
 *
 * Contact: SBIA Group <sbia-software at uphs.upenn.edu>
 */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>

#ifndef max
#define max(a,b) ((a)>(b)?(a):(b))
#define min(a,b) ((a)<(b)?(a):(b))
#endif
#define PI 3.141592653589793324
#define GeoZeroVec(v) ((v).x = (v).y = (v).z = 0.0)
#define GeoMultVec(a,b,c) \ 
  do {(c).x = a*(b).x; (c).y = a*(b).y;	(c).z = a*(b).z; } while (0)
#define Geo_Vet(a,b,c) \ 
  do {(c).x = (b).x-(a).x; (c).y = (b).y-(a).y; (c).z = (b).z-(a).z;} while (0)

typedef double Rdouble;
typedef float  Rfloat;
typedef struct _GeoPoint { Rfloat x, y, z; } GeoPoint;

int image_size_x, image_size_y, image_size_z;
float res_x, res_y, res_z;

void Main(int argc,char *argv[]) ;
void show_usage() ;


/*=========================  Geometrical Procedures  ======================= */

Rdouble GeoDotProd ( GeoPoint *vec0, GeoPoint *vec1 )
{
 return ( vec0->x * vec1->x + vec0->y * vec1->y + vec0->z * vec1->z );
}

void GeoCrossProd ( GeoPoint *in0, GeoPoint *in1, GeoPoint *out )
{
 out->x = (in0->y * in1->z) - (in0->z * in1->y);
 out->y = (in0->z * in1->x) - (in0->x * in1->z);
 out->z = (in0->x * in1->y) - (in0->y * in1->x);
}

Rdouble GeoTripleProd ( GeoPoint *vec0, GeoPoint *vec1, GeoPoint *vec2 )
{
 GeoPoint tmp;

 GeoCrossProd ( vec0, vec1, &tmp );
 return ( GeoDotProd( &tmp, vec2 ) );
}

Rdouble GeoVecLen ( GeoPoint *vec )
{
 return sqrt ( GeoDotProd ( vec, vec ) );
}

int GeoPolyNormal ( int	n_verts, GeoPoint *verts, GeoPoint *n ) 
{
 int      i;
 Rfloat	  n_size;		
 GeoPoint v0, v1, p; 

 GeoZeroVec ( *n );
 Geo_Vet ( verts[0], verts[1], v0 );
 for ( i = 2; i < n_verts; i++ )
     {
      Geo_Vet ( verts[0], verts[i], v1 );
      GeoCrossProd ( &v0, &v1, &p );
      n->x += p.x; n->y += p.y; n->z += p.z;
      v0 = v1;
     }

 n_size = GeoVecLen ( n );
 if ( n_size > 0.0 )
    {
     GeoMultVec ( 1/n_size, *n, *n );
     return 1;
    }
 else
     return 0;
}

/*=========================  geo_solid_angle  =========================*/
/* 
  Calculates the solid angle given by the spherical projection of 
  a 3D plane polygon
*/

Rdouble geo_solid_angle ( 
        int      n_vert,  /* number of vertices */
        GeoPoint *verts,  /* vertex coordinates list */
        GeoPoint *p )     /* point to be tested */
{
 int      i;
 Rdouble  area = 0.0, ang, s, l1, l2;
 GeoPoint p1, p2, r1, a, b, n1, n2;
 GeoPoint plane;

 if ( n_vert < 3 ) return 0.0;

 GeoPolyNormal ( n_vert, verts, &plane );
 
 /* 
    WARNING: at this point, a practical implementation should check
    whether p is too close to the polygon plane. If it is, then
    there are two possibilities: 
      a) if the projection of p onto the plane is outside the 
         polygon, then area zero should be returned;
      b) otherwise, p is on the polyhedron boundary.
 */ 

 p2 = verts[n_vert-1];  /* last vertex */
 p1 = verts[0];         /* first vertex */
 Geo_Vet ( p1, p2, a ); /* a = p2 - p1 */  

 for ( i = 0; i < n_vert; i++ )
     {
      Geo_Vet(*p, p1, r1); 
      p2 = verts[(i+1)%n_vert];
      Geo_Vet ( p1, p2, b );
      GeoCrossProd ( &a, &r1, &n1 );
      GeoCrossProd ( &r1, &b, &n2 );
    
      l1 = GeoVecLen ( &n1 );
      l2 = GeoVecLen ( &n2 );
      s  = GeoDotProd ( &n1, &n2 ) / ( l1 * l2 );
      ang = acos ( max(-1.0,min(1.0,s)) );
      s = GeoTripleProd( &b, &a, &plane );
      area += s > 0.0 ? PI - ang : PI + ang;
     
      GeoMultVec ( -1.0, b, a );
      p1 = p2;
     }

 area -= PI*(n_vert-2);

 return ( GeoDotProd ( &plane, &r1 ) > 0.0 ) ? -area : area; 
}

/* ======================  main ========================== */

int PointInsideHexahedron(GeoPoint *verts,  /* vertex coordinates list */
        		  		  GeoPoint p )     /* point to be tested */
{
 
 int      nv,inside;
 GeoPoint vertsf1[4],vertsf2[4],vertsf3[4],vertsf4[4],vertsf5[4],vertsf6[4];
 Rdouble  Area =0.0;
 
 nv=4;//number of vertices per each face of the polyhedron (here hexahedron)
 
 //face1, ordered (Hari)
 vertsf1[0]=verts[0];
 vertsf1[1]=verts[1];
 vertsf1[2]=verts[3];
 vertsf1[3]=verts[2];
 
  //face2, ordered (Hari)
 vertsf2[0]=verts[1];
 vertsf2[1]=verts[5];
 vertsf2[2]=verts[7];
 vertsf2[3]=verts[3];
 
  //face3, ordered (Hari)
 vertsf3[0]=verts[7];
 vertsf3[1]=verts[5];
 vertsf3[2]=verts[4];
 vertsf3[3]=verts[6];
 
  //face4, ordered (Hari)
 vertsf4[0]=verts[0];
 vertsf4[1]=verts[2];
 vertsf4[2]=verts[6];
 vertsf4[3]=verts[4];
 
  //face5, ordered (Hari)
 vertsf5[0]=verts[0];
 vertsf5[1]=verts[4];
 vertsf5[2]=verts[5];
 vertsf5[3]=verts[1];
 
  //face6, ordered (Hari)
 vertsf6[0]=verts[2];
 vertsf6[1]=verts[3];
 vertsf6[2]=verts[7];
 vertsf6[3]=verts[6];
 
     
 Area = geo_solid_angle ( nv, vertsf1, &p )+geo_solid_angle ( nv, vertsf2, &p )+\
 	 geo_solid_angle ( nv, vertsf3, &p )+geo_solid_angle ( nv, vertsf4, &p )+\
 	  geo_solid_angle ( nv, vertsf5, &p )+geo_solid_angle ( nv, vertsf6, &p ); 
      

 if ( (Area > 2*PI) || (Area < -2*PI) )
 	inside=1;
 else 
 	inside=0;
 
 return inside;
}


#undef __FUNCT__
#define __FUNCT__ "DeformImage"

void DeformImage(char InputImageFile[80], char InputDefFile[80], int image_size_x, int image_size_y, int image_size_z, float res_x, float res_y, float res_z, char OutputImageFile[80])
{
	
  int xm,ym,zm,ne,nsd;
  float Lx,Ly,Lz,Hx,Hy,Hz;
  int count, counti, countj, countk, countv, i, j, k, ie, indcornerleft, inc, incmax, missing, ind;
  int ii,jj,ielocal,indi,indj,indk, indx1, indx2, indy1, indy2, indz1,indz2, inside;
  int ic[8],knl[8];
  int iee[6];
  GeoPoint vundef[8];//element vertices, undeformed (regular) mesh
  GeoPoint vdef[8];//element vertices, deformed (iregular) mesh
  GeoPoint ce; //element centroid coord., undeformed mesh
  float xvmin, xvmax, yvmin, yvmax, zvmin, zvmax;
  unsigned char *grayscaleorig,*grayscaledef;
  float	*totaldef;
  float tmp;
  FILE *fb, *fb1;
  
  xm=image_size_x+1;
  ym=image_size_y+1;
  zm=image_size_z+1;


  Lx = res_x*image_size_x; Ly = res_y*image_size_y; Lz = res_z*image_size_z; //work in *mm* and *not in meters*!!
  Hx = res_x; Hy = res_y; Hz = res_z;
  ne=image_size_x*image_size_y*image_size_z; 
  
  //!!Note: if det(F)>0, then the number of elements ne should be the same in the undeformed and the deformed configuration
 
  
  nsd=3;
  
  count=0;
  
  //read nodal displacements from file
  
  totaldef=malloc(nsd*xm*ym*zm*sizeof(float));
 
  fb=fopen(InputDefFile,"rb"); //test case
  
  for(jj=0;jj<xm*ym*zm;jj++) 
  	for(ii=0; ii<3;ii++) {
  	fread(&tmp,sizeof(float),1,fb); 
  	totaldef[jj*3+ii]=tmp;
  	}
  		
  fclose(fb);

 /*temporary* fix for the z=0 slice that returns garbage from catVF for now!*/
  
  counti = 1; countj = 1; countk = 1;
 
 for (inc =0;inc<xm*ym*zm;inc++){
 			
        ind = inc * nsd;
        i = counti-1; j = countj-1; k = countk-1;
        
        
        if (i==0 || i==1 || j==0 || j==1 || k==0 || k==1) {totaldef[ind]=0.0; totaldef[ind+1]=0.0; totaldef[ind+2]=0.0;} //temporary fix for the z=0 slice that returns garbage from catVF for now! Manually set displacements to 0.
       	if (i==xm-1 || i==xm-2 || j==ym-1 || j==ym-2 || k==zm-1 || k==zm-2) {totaldef[ind]=0.0; totaldef[ind+1]=0.0; totaldef[ind+2]=0.0;} //temporary fix
	
	//if(totaldef[ind]>1.0) printf("i j k totaldef %d %d %d %g\n", i,j,k,totaldef[ind]);
       	
       	counti++;
        if (counti==xm+1){ countj++; counti=1; }
        if (countj==ym+1){ counti=1; countj=1; countk++;}
        
 }


  printf("image_size_x %d\n",image_size_x);
  printf("image_size_y %d\n",image_size_y);
  printf("image_size_z %d\n",image_size_z);
  
  printf("-Hx %g\n",Hx);
  printf("-Hy %g\n",Hy);
  printf("-Hz %g\n",Hz);
  
  printf("-Lx %g\n",Lx);
  printf("-Ly %g\n",Ly);
  printf("-Lz %g\n",Lz);
  
  
  fb  = fopen(InputImageFile,"r");//initial image, gray scale
  fb1  = fopen(OutputImageFile,"w");//deformed image, gray scale
 
  /*initialize deformed image labels to the original (undeformed) image */
  grayscaleorig=malloc(ne*sizeof(char));
  grayscaledef=malloc(ne*sizeof(char));
   
  fread(grayscaleorig,sizeof(char),ne,fb);
  
  
  for (ie=0;ie<ne; ie++){grayscaledef[ie]=255;}//initialize with a distinctive label
        
  
  /*modify the deformed image labels, by using a combined "bounding box" + "point inside convex polyhedron" approach */
  counti = 1; countj = 1; countk = 1; 
  
  for (ie=0;ie<ne; ie++){
  
  	//printf("ie grayscaleorig %d %d\n",ie,grayscaleorig[ie]);
        
        i = counti-1; j = countj-1; k = countk-1;  
        
        //printf("ie %d\n",ie);
       	
       	//element vertices, undeformed mesh; xyz ordering
       	
       	indcornerleft=i+j*xm+k*xm*ym; 
       	
       	
       	ic[0]=indcornerleft;
        ic[1]=ic[0]+1;
        ic[2]=ic[0]+xm;
        ic[3]=ic[2]+1;
        
        ic[4]=indcornerleft+xm*ym;
        ic[5]=ic[4]+1;
        ic[6]=ic[4]+xm;
        ic[7]=ic[6]+1;
        
        vundef[0].x=i*Hx;
        vundef[1].x=vundef[0].x+Hx;
        vundef[2].x=vundef[0].x;
        vundef[3].x=vundef[1].x;
        vundef[4].x=vundef[0].x;
        vundef[5].x=vundef[1].x;
        vundef[6].x=vundef[2].x;
        vundef[7].x=vundef[3].x;
        
        vundef[0].y=j*Hy;
        vundef[1].y=vundef[0].y;
        vundef[2].y=vundef[0].y+Hy;
        vundef[3].y=vundef[2].y;
        vundef[4].y=vundef[0].y;
        vundef[5].y=vundef[1].y;
        vundef[6].y=vundef[2].y;
        vundef[7].y=vundef[3].y;
        
        vundef[0].z=k*Hz;
        vundef[1].z=vundef[0].z;
        vundef[2].z=vundef[0].z;
        vundef[3].z=vundef[0].z;
        vundef[4].z=vundef[0].z+Hz;
        vundef[5].z=vundef[4].z;
        vundef[6].z=vundef[4].z;
        vundef[7].z=vundef[4].z;
           
        	
        for (countv=0;countv<=7;countv++){
        
        	knl[countv] = ic[countv] * nsd;
        	      
		vdef[countv].x=vundef[countv].x+totaldef[knl[countv]];
			
		vdef[countv].y=vundef[countv].y+totaldef[knl[countv]+1];
	
		vdef[countv].z=vundef[countv].z+totaldef[knl[countv]+2];
		

	 	} //end for node vertices
		
		
	/*find xmin,xmax,ymin,ymax and zmin,zmax for the 8 deformed vertices*/
	
	xvmin=100000.0;
  	yvmin=100000.0;
  	zvmin=100000.0;
  	xvmax=-100000.0;
  	yvmax=-100000.0;
  	zvmax=-100000.0;
	
	for (countv=0;countv<=7;countv++){
	
		xvmin=min(xvmin,vdef[countv].x);	
		yvmin=min(yvmin,vdef[countv].y);	
		zvmin=min(zvmin,vdef[countv].z);	
		
		xvmax=max(xvmax,vdef[countv].x);	
		yvmax=max(yvmax,vdef[countv].y);	
		zvmax=max(zvmax,vdef[countv].z);	
		
	}
		
	
	indx1=(int)(xvmin/Hx+0.5);
	indx2=(int)(xvmax/Hx+0.5);
	indy1=(int)(yvmin/Hy+0.5);
	indy2=(int)(yvmax/Hy+0.5);
	indz1=(int)(zvmin/Hz+0.5);
	indz2=(int)(zvmax/Hz+0.5);



	for (indk=indz1;indk<=indz2;indk++)
		for (indj=indy1;indj<=indy2;indj++)	
			for (indi=indx1;indi<=indx2;indi++) {
			

        		ce.x = (indi+1.0/2.0) * Hx; /* element centroids, undeformed mesh*/
        		ce.y = (indj+1.0/2.0) * Hy;
        		ce.z = (indk+1.0/2.0) * Hz;
        
        		
        		ielocal=indi+indj*(xm-1)+indk*(xm-1)*(ym-1);
        		
        		/*test if center point ielocal inside the hexahedron (deformed cube) ie*/
        		
        		inside=PointInsideHexahedron(vdef,ce);
        		
        	
        		if (inside==1 && grayscaledef[ielocal]==255){ 
        		
        			grayscaledef[ielocal]=grayscaleorig[ie]; 
        		       	
        		
        		}
 			
 			}
 
        counti++;
        if (counti==xm){ countj++; counti=1; }
        if (countj==ym){ counti=1; countj=1; countk++; }
        
  }
  
  
  missing=0;
  for(ie=0;ie<ne;ie++){if (grayscaleorig[ie]!=255 && grayscaledef[ie]==255) {
  								printf("ie grayscaleorig grayscaledef %d %d %d\n",ie,grayscaleorig[ie],grayscaledef[ie]);
  								missing=1;
  								}
  								}
  
  inc=0;
  incmax=3;								
  
  while(missing && inc<incmax){
  
  printf("Final cleaning: 6 nearest neighbors inc %d\n",inc);
  
  //test if all the voxels remaped; if not - check closest neighbors
  counti = 1; countj = 1; countk = 1; 
  
  for (ie=0;ie<ne; ie++){
 
        i = counti-1; j = countj-1; k = countk-1; 
       
        if (grayscaleorig[ie]!=255 && grayscaledef[ie]==255) {
		
			iee[0]=i-1+j*(xm-1)+k*(xm-1)*(ym-1);
			iee[1]=i+1+j*(xm-1)+k*(xm-1)*(ym-1);
			iee[2]=i+(j-1)*(xm-1)+k*(xm-1)*(ym-1);
			iee[3]=i+(j+1)*(xm-1)+k*(xm-1)*(ym-1);
			iee[4]=i+j*(xm-1)+(k-1)*(xm-1)*(ym-1);
			iee[5]=i+j*(xm-1)+(k+1)*(xm-1)*(ym-1);
			
			
			
				if (grayscaledef[iee[0]]==grayscaledef[iee[1]] && grayscaledef[iee[1]]==grayscaledef[iee[2]] && grayscaledef[iee[2]]==grayscaledef[iee[3]]){
					grayscaledef[ie]=grayscaledef[iee[0]];
					}
				
				if (grayscaledef[iee[0]]==grayscaledef[iee[1]] && grayscaledef[iee[1]]==grayscaledef[iee[2]] && grayscaledef[iee[2]]==grayscaledef[iee[4]]){
					grayscaledef[ie]=grayscaledef[iee[0]];
					}
					
				if (grayscaledef[iee[0]]==grayscaledef[iee[1]] && grayscaledef[iee[1]]==grayscaledef[iee[2]] && grayscaledef[iee[2]]==grayscaledef[iee[5]]){
					grayscaledef[ie]=grayscaledef[iee[0]];
					}
					
				if (grayscaledef[iee[0]]==grayscaledef[iee[1]] && grayscaledef[iee[1]]==grayscaledef[iee[3]] && grayscaledef[iee[3]]==grayscaledef[iee[4]]){
					grayscaledef[ie]=grayscaledef[iee[0]];
					}
					
				if (grayscaledef[iee[0]]==grayscaledef[iee[1]] && grayscaledef[iee[1]]==grayscaledef[iee[3]] && grayscaledef[iee[3]]==grayscaledef[iee[5]]){
					grayscaledef[ie]=grayscaledef[iee[0]];
					}
					
				if (grayscaledef[iee[0]]==grayscaledef[iee[1]] && grayscaledef[iee[1]]==grayscaledef[iee[4]] && grayscaledef[iee[4]]==grayscaledef[iee[5]]){
					grayscaledef[ie]=grayscaledef[iee[0]];
					}
					
				if (grayscaledef[iee[0]]==grayscaledef[iee[2]] && grayscaledef[iee[2]]==grayscaledef[iee[3]] && grayscaledef[iee[3]]==grayscaledef[iee[4]]){
					grayscaledef[ie]=grayscaledef[iee[0]];
					}
					
				if (grayscaledef[iee[0]]==grayscaledef[iee[2]] && grayscaledef[iee[2]]==grayscaledef[iee[3]] && grayscaledef[iee[3]]==grayscaledef[iee[5]]){
					grayscaledef[ie]=grayscaledef[iee[0]];
					}
					
				if (grayscaledef[iee[0]]==grayscaledef[iee[2]] && grayscaledef[iee[2]]==grayscaledef[iee[4]] && grayscaledef[iee[4]]==grayscaledef[iee[5]]){
					grayscaledef[ie]=grayscaledef[iee[0]];
					}
					
				if (grayscaledef[iee[0]]==grayscaledef[iee[3]] && grayscaledef[iee[3]]==grayscaledef[iee[4]] && grayscaledef[iee[4]]==grayscaledef[iee[5]]){
					grayscaledef[ie]=grayscaledef[iee[0]];
					}
					
				if (grayscaledef[iee[1]]==grayscaledef[iee[2]] && grayscaledef[iee[2]]==grayscaledef[iee[3]] && grayscaledef[iee[3]]==grayscaledef[iee[4]]){
					grayscaledef[ie]=grayscaledef[iee[1]];
					}
					
				if (grayscaledef[iee[1]]==grayscaledef[iee[2]] && grayscaledef[iee[2]]==grayscaledef[iee[3]] && grayscaledef[iee[3]]==grayscaledef[iee[5]]){
					grayscaledef[ie]=grayscaledef[iee[1]];
					}
					
				if (grayscaledef[iee[1]]==grayscaledef[iee[2]] && grayscaledef[iee[2]]==grayscaledef[iee[4]] && grayscaledef[iee[4]]==grayscaledef[iee[5]]){
					grayscaledef[ie]=grayscaledef[iee[1]];
					}
					
				if (grayscaledef[iee[1]]==grayscaledef[iee[3]] && grayscaledef[iee[3]]==grayscaledef[iee[4]] && grayscaledef[iee[4]]==grayscaledef[iee[5]]){
					grayscaledef[ie]=grayscaledef[iee[1]];
					}
					
				if (grayscaledef[iee[2]]==grayscaledef[iee[3]] && grayscaledef[iee[3]]==grayscaledef[iee[4]] && grayscaledef[iee[4]]==grayscaledef[iee[5]]){
					grayscaledef[ie]=grayscaledef[iee[2]];
					}
									
				
			}//end if grayscaledef[ie]==255
			


        counti++;
        if (counti==xm){ countj++; counti=1; }
        if (countj==ym){ counti=1; countj=1; countk++; }
        
        
  } //end for ie
  
  missing=0;
  for(ie=0;ie<ne;ie++){if (grayscaleorig[ie]!=255 && grayscaledef[ie]==255) {
  								printf("ie grayscaleorig grayscaledef %d %d %d\n",ie,grayscaleorig[ie],grayscaledef[ie]);
  								missing=1;
  								}
  					   }
  if(missing==1) inc=inc+1;
  
   }//end while
   
   
  printf("missing %d\n",missing);
 
  if (missing==1){
    
  /*final cleaning, by using a "bounding box" approach - instead of the exact, "point inside convex polyhedron" one*/
  printf("Final cleaning: bounding box approach");
  counti = 1; countj = 1; countk = 1; 
  
  for (ie=0;ie<ne; ie++){
        
        i = counti-1; j = countj-1; k = countk-1;  
	
       	
       	/*element vertices, undeformed mesh; xyz ordering*/
       	
       	indcornerleft=i+j*xm+k*xm*ym; 
       	
       	
       	ic[0]=indcornerleft;
        ic[1]=ic[0]+1;
        ic[2]=ic[0]+xm;
        ic[3]=ic[2]+1;
        
        ic[4]=indcornerleft+xm*ym;
        ic[5]=ic[4]+1;
        ic[6]=ic[4]+xm;
        ic[7]=ic[6]+1;
        
        vundef[0].x=i*Hx;
        vundef[1].x=vundef[0].x+Hx;
        vundef[2].x=vundef[0].x;
        vundef[3].x=vundef[1].x;
        vundef[4].x=vundef[0].x;
        vundef[5].x=vundef[1].x;
        vundef[6].x=vundef[2].x;
        vundef[7].x=vundef[3].x;
        
        vundef[0].y=j*Hy;
        vundef[1].y=vundef[0].y;
        vundef[2].y=vundef[0].y+Hy;
        vundef[3].y=vundef[2].y;
        vundef[4].y=vundef[0].y;
        vundef[5].y=vundef[1].y;
        vundef[6].y=vundef[2].y;
        vundef[7].y=vundef[3].y;
        
        vundef[0].z=k*Hz;
        vundef[1].z=vundef[0].z;
        vundef[2].z=vundef[0].z;
        vundef[3].z=vundef[0].z;
        vundef[4].z=vundef[0].z+Hz;
        vundef[5].z=vundef[4].z;
        vundef[6].z=vundef[4].z;
        vundef[7].z=vundef[4].z;
           
        	
        for (countv=0;countv<=7;countv++){
        
        	knl[countv] = ic[countv] * nsd;
        	
		vdef[countv].x=vundef[countv].x+totaldef[knl[countv]];
			
		vdef[countv].y=vundef[countv].y+totaldef[knl[countv]+1];
	
		vdef[countv].z=vundef[countv].z+totaldef[knl[countv]+2];
		
		
	 	} //end for node vertices
		
		
	//find xmin,xmax,ymin,ymax and zmin,zmax for the 8 deformed vertices
	
	xvmin=100000.0;
  	yvmin=100000.0;
  	zvmin=100000.0;
  	xvmax=-100000.0;
  	yvmax=-100000.0;
  	zvmax=-100000.0;
	
	for (countv=0;countv<=7;countv++){
	
		xvmin=min(xvmin,vdef[countv].x);	
		yvmin=min(yvmin,vdef[countv].y);	
		zvmin=min(zvmin,vdef[countv].z);	
		
		xvmax=max(xvmax,vdef[countv].x);	
		yvmax=max(yvmax,vdef[countv].y);	
		zvmax=max(zvmax,vdef[countv].z);	
		
	}
		
	
	indx1=(int)(xvmin/Hx+0.5);
	indx2=(int)(xvmax/Hx+0.5);
	indy1=(int)(yvmin/Hy+0.5);
	indy2=(int)(yvmax/Hy+0.5);
	indz1=(int)(zvmin/Hz+0.5);
	indz2=(int)(zvmax/Hz+0.5);



	for (indk=indz1;indk<=indz2;indk++)
		for (indj=indy1;indj<=indy2;indj++)	
			for (indi=indx1;indi<=indx2;indi++) {
			

        		ce.x = (indi+1.0/2.0) * Hx; /* element centroids, undeformed mesh*/
        		ce.y = (indj+1.0/2.0) * Hy;
        		ce.z = (indk+1.0/2.0) * Hz;
        
        	
        		ielocal=indi+indj*(xm-1)+indk*(xm-1)*(ym-1);
        		
        		if ((ce.x>=xvmin && ce.x<=xvmax) && (ce.y>=yvmin && ce.y<=yvmax) && (ce.z>=zvmin && ce.z<=zvmax) && grayscaledef[ielocal]==255){ 
        		
        		grayscaledef[ielocal]=grayscaleorig[ie]; 
        		       	
        		}
        		       	
 			}
 			
 
        counti++;
        if (counti==xm){ countj++; counti=1; }
        if (countj==ym){ counti=1; countj=1; countk++; }
        
  }
  
  }//end if missing
  
  
  for(ie=0;ie<ne;ie++){if (grayscaleorig[ie]!=255 && grayscaledef[ie]==255) {
  								printf("ie grayscaleorig grayscaledef %d %d %d\n",ie,grayscaleorig[ie],grayscaledef[ie]);
  								missing=1;}
  					   }
  					   
  
  
  fwrite(grayscaledef,sizeof(char),ne,fb1);
  
  fclose(fb);
  fclose(fb1);
  
  free(totaldef);
  free(grayscaleorig);
  free(grayscaledef);

}

main(int argc,char *argv[])
{
  Main(argc,argv);    
}


void Main(int argc,char *argv[])
{
  int           i,j,k,c,num;
  FILE          *fp;
  extern char   *optarg;


  num=3;

  if(argc<num)
    show_usage() ;


  /*  input  size */
  image_size_x= 256;
  image_size_y= 256;
  image_size_z= 124;
  
  res_x=0.9375;
  res_y=0.9375;
  res_z=1.5;
  
  while((c=getopt(argc-3,argv+3,"v:r:")) != -1)
    {
      switch(c)
	{
	case 'v':
	  sscanf(optarg, "%d,%d,%d", &image_size_x, &image_size_y, &image_size_z ) ; /* original input size */
	  break ;
	  
	case 'r':
	  sscanf(optarg, "%g,%g,%g", &res_x, &res_y, &res_z ) ; /* original input resolution */
	  break ;

	default:
	  break;
	}
    }
  
  DeformImage(argv[1],argv[2], image_size_x, image_size_y, image_size_z, res_x, res_y, res_z, argv[3]);
  

}


void show_usage()
{
  printf("USAGE: DeformImage <input Original Image> <input Deformation Field, float xyz> <output Deformed Image>\n\
\t -v <int,int,int>             : x,y,z size in input image, default : 256,256,124  \n\
\t -r <float,float,float>       : x,y,z physical resolution in input image, default : 0.9385,0.9375,1.5  \n\
");
  exit(1);
}


