/*=========================================================================

  Program:         Level-Set Segmentation for Slicer3
  Module:          BubbleMakerWithFloodFilling.cxx
  Language:        C++
  Date:            2008-07-21 16:24:54
  Version:         1.00
  Author:	       Carlos S. Mendoza - Universidad de Sevilla, Spain
  E-mail:          carlos.sanchez.mendoza@gmail.com
  Acknowledgments: This work is funded by a scholarship for research 
                   personnel instruction from the University of 
				   Sevilla, Spain.
				   
				   This work has been developed under the supervision of
				   Mr. Steve Pieper Ph.D. and Ron Kikinis M.D., during 
				   an internship in the Surgical Planning Laboratory in 
				   Harvard Medical School and Brigham and Women's Hospital.

=========================================================================*/

/*-------------------------------------------------------------------------

The purpose of this program is the creation of a label map made of spheres with different
centers and radii. The typical usage will be the seeding of a segmentation algorithm, 
specially  level-set evolution algorithms.

-------------------------------------------------------------------------*/

#if defined(_MSC_VER)
#pragma warning ( disable : 4786 )
#endif

#ifdef __BORLANDC__
#define ITK_LEAN_AND_MEAN
#endif

#include "BubbleMakerCLP.h"
#include "itkOrientedImage.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkSphereSpatialFunction.h" 
#include "itkIndex.h"
#include "itkFloodFilledSpatialFunctionConditionalIterator.h"
#include <iostream>

int main( int argc, char *argv[] )
{	
	
	PARSE_ARGS;
	const unsigned int Dimension = 3;
	typedef float InputPixelType;
	typedef unsigned short OutputPixelType;
	typedef itk::OrientedImage<OutputPixelType,Dimension> OutputImageType;
	typedef itk::OrientedImage<InputPixelType,Dimension> InputImageType;
	typedef itk::ImageFileReader<InputImageType> InputReaderType;
	typedef itk::ImageFileWriter<OutputImageType> OutputWriterType;
	typedef itk::SphereSpatialFunction<Dimension,OutputImageType::PointType> BinarySphereGenerator;
	typedef BinarySphereGenerator::Pointer SpherePointerType;
	typedef itk::FloodFilledSpatialFunctionConditionalIterator<OutputImageType,BinarySphereGenerator> ConditionalIterator;
	
	// Check that the number of centers equals that of radii.

	if (bubbleCenters.size()!= bubbleRadii.size())
	{	
		std::cout << "Failure. Number of radii should equal number of fiducials"; 
		return EXIT_FAILURE;
	}
	
	SpherePointerType spheres[100];
	
	// For every bubble center
	
	for (unsigned int i=0 ; i<bubbleCenters.size(); i++)
	{
		// A bynary sphere spatial funtion is created
		
		spheres[i] = BinarySphereGenerator::New();			
		itk::Index<Dimension> index = {{0,0,0}};
		InputImageType::PointType *point= new InputImageType::PointType();
		
		// The fiducial from Slicer is converted from Slicer RAS to ITK LPS 

		point->SetElement(0,-bubbleCenters[i][0]);
		point->SetElement(1,-bubbleCenters[i][1]);
		point->SetElement(2,bubbleCenters[i][2]);
		
		// The sphere center and radius get set
		
		spheres[i]->SetCenter(*point);
		spheres[i]->SetRadius(bubbleRadii[i]);
		delete point;
	}
	
	// The volume of reference provides the spacing, origin etc for the spheres label image
	
	InputReaderType::Pointer inputReader = InputReaderType::New();
	inputReader->SetFileName(inputReference.c_str());
	inputReader->Update();
	
	OutputImageType::Pointer results = OutputImageType::New();
	
	results->SetRegions(inputReader->GetOutput()->GetLargestPossibleRegion());
	results->SetDirection(inputReader->GetOutput()->GetDirection());
	results->SetOrigin(inputReader->GetOutput()->GetOrigin());
	results->SetSpacing(inputReader->GetOutput()->GetSpacing());
	
	results->Allocate();
	results->FillBuffer(0);

	// The resulting image is already allocated and filled with zeros, then for every bubble

	for (unsigned int i=0; i<bubbleCenters.size();i++)
	{
		OutputImageType::IndexType index;
		results->TransformPhysicalPointToIndex(spheres[i]->GetCenter(),index);
		
		// Starting from the center of the bubble we conditionally visit the inside of the bubble setting on the corresponding output pixel
		// When we get out of the bubble the iterator is finnished (very efficient)

		// The label value is taken from the Slicer GUI
		
		ConditionalIterator iterator(results,spheres[i],index);
		for( ; !( iterator.IsAtEnd() ); ++iterator)
		{
			iterator.Set(outputLabelValue);
		}
		
	}
	
	OutputWriterType::Pointer writer = OutputWriterType::New();
	writer->SetFileName(outputLabel.c_str());
	writer->SetInput(results);
	writer->Update();
	
	std::cout << "Success"; 
	return EXIT_SUCCESS;
}
















