/*=========================================================================

  Program:         Level-Set Segmentation for Slicer3
  Module:          RegionSelector.cxx
  Language:        C++
  Date:            2008-08-04 12:55:30
  Version:         1.00
  Author:	       Carlos S. Mendoza - Universidad de Sevilla, Spain
  E-mail:          carlos.sanchez.mendoza@gmail.com
  Acknowledgments: This work is funded by a scholarship for research 
                   personnel instruction from the University of 
				   Sevilla, Spain.
				   
				   This work has been developed under the supervision of
				   Mr. Steve Pieper Ph.D. and Ron Kikinis M.D., during an 
				   internship in the Surgical Planning Laboratory in 
				   Harvard Medical School and Brigham and Women's Hospital.

 
=========================================================================*/


/*-------------------------------------------------------------------------

The purpose of this program is to provide a simple mechanism for the 
selection of a region of interest. The arguments for the region definition
are four points in each of the three coordinate planes.

The resulting volume traces on every coordinate plane stack will be the 
smallest rectangle containing all four points defined in that direction.

The user should place the corners in such a way as to leave the object 
of interest inside the imaginary rectangle described roughly by them.

-------------------------------------------------------------------------*/



#if defined(_MSC_VER)
#pragma warning ( disable : 4786 )
#endif

#ifdef __BORLANDC__
#define ITK_LEAN_AND_MEAN
#endif

#include "RegionSelectorCLP.h"
#include "itkOrientedImage.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkImageRegion.h"
#include "itkRegionOfInterestImageFilter.h"
#include <iostream>

int main( int argc, char *argv[] )
{	
	PARSE_ARGS;
	const unsigned int Dimension = 3;
	typedef float PixelType;
	typedef itk::OrientedImage<PixelType,Dimension> ImageType;
	typedef itk::ImageFileReader<ImageType> ReaderType;
	typedef itk::ImageFileWriter<ImageType> WriterType;
	typedef itk::RegionOfInterestImageFilter<ImageType,ImageType> ClipperType;
	typedef ImageType::RegionType RegionType;
	typedef ImageType::IndexType IndexType;
	typedef ImageType::SizeType SizeType;
	typedef ImageType::PointType PointType;
	
	// First thing is to check we got the right number of corners
	
	bool failure=0;

	if (axialCorners.size()!= 4)
	{	
		std::cout << "Failure. Four axial fiducials needed"; 
		failure=1;
	}

	if (coronalCorners.size()!= 4)
	{	
		std::cout << "Failure. Four coronal fiducials needed"; 
		failure=1;
	}

	if (saggitalCorners.size()!= 4)
	{	
		std::cout << "Failure. Four saggital fiducials needed"; 
		failure=1;
	}

	if (failure)
	{
		std::cout << "Failure. Wrong arguments";
		return EXIT_FAILURE;
	}
	
	// We read the image to be cropped
	
	ReaderType::Pointer reader = ReaderType::New();
	reader->SetFileName(input.c_str());
	reader->Update();

	// We are using a region that will computed from the corners

	RegionType *region = new RegionType();
	IndexType corners[12];
	PointType *point= new PointType();

	// The points provided from Slicer need to be converted from RAS to LPS 
	// and from physical points to indexes in the image.

	for (int i=0;i<4;++i)
	{	
		point->SetElement(0,-axialCorners[i][0]);
		point->SetElement(1,-axialCorners[i][1]);
		point->SetElement(2,axialCorners[i][2]);
		reader->GetOutput()->TransformPhysicalPointToIndex(*point,corners[i]);

		point->SetElement(0,-coronalCorners[i][0]);
		point->SetElement(1,-coronalCorners[i][1]);
		point->SetElement(2,coronalCorners[i][2]);
		reader->GetOutput()->TransformPhysicalPointToIndex(*point,corners[i+4]);

		point->SetElement(0,-saggitalCorners[i][0]);
		point->SetElement(1,-saggitalCorners[i][1]);
		point->SetElement(2,saggitalCorners[i][2]);
		reader->GetOutput()->TransformPhysicalPointToIndex(*point,corners[i+8]);	
	}
	
	delete point;
	
	long i_min = corners[0][0];
	long i_max = corners[0][0];
	long j_min = corners[0][1];
	long j_max = corners[0][1];
	long k_min = corners[0][2];
	long k_max = corners[0][2];
	
	// Then we specify that our volume is the smallest square parallelepiped containing all points

	for (int i=0;i<12;++i)
	{
		if (corners[i][0]<i_min) i_min=corners[i][0];
		if (corners[i][0]>i_max) i_max=corners[i][0];
		if (corners[i][1]<j_min) j_min=corners[i][1];
		if (corners[i][1]>j_max) j_max=corners[i][1];
		if (corners[i][2]<k_min) k_min=corners[i][2];
		if (corners[i][2]>k_max) k_max=corners[i][2];
	
		std::cout << i_min << " ";
		std::cout << i_max << " ";
		std::cout << j_min << " ";
		std::cout << j_max << " ";
		std::cout << k_min << " ";
		std::cout << k_max << " ";
		std::cout << "\n";
	
	}

	delete corners;

	region->SetIndex(0,i_min);
	region->SetIndex(1,j_min);
	region->SetIndex(2,k_min);

	region->SetSize(0,(i_max-i_min));
	region->SetSize(1,(j_max-j_min));
	region->SetSize(2,(k_max-k_min));
	
	// We then feed the region to a region selecting filter

	ClipperType::Pointer clipper = ClipperType::New();
	clipper->SetInput(reader->GetOutput());
	clipper->SetRegionOfInterest(*region);
	clipper->Update();

	// And write back the resulting image

	WriterType::Pointer writer = WriterType::New();
	writer->SetInput(clipper->GetOutput());
	writer->SetFileName(output.c_str());
		
	writer->Update();

	delete region;

	std::cout << "Success"; 
	return EXIT_SUCCESS;
}
















