/*=========================================================================

Program:   Insight Segmentation & Registration Toolkit
Module:    $RCSfile: itkHammerRegistrationFilter.h,v $
Language:  C++
Date:      $Date: 2009/01/15 15:06:35 $
Version:   $Revision: 1.1 $

Copyright (c) Insight Software Consortium. All rights reserved.
See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for
details.

This program is developed under NIH NCBC collaboration grant
R01 EB006733, "Development and Dissemination of Robust Brain MRI
Measurement Tools". 

This software is distributed WITHOUT ANY WARRANTY; without even 
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkHammerRegistrationFilter_h
#define __itkHammerRegistrationFilter_h

#include "itkPDEDeformableRegistrationFilter.h"
#include "itkHammerRegistrationFunction.h"
#include "itkImageToImageFilter.h"

namespace itk {

/** \class HammerRegistrationFilter
 * \brief Deformably register two images using the demons algorithm.
 *
 * HammerRegistrationFilter implements the demons deformable algorithm that 
 * register two images by computing the deformation field which will map a 
 * moving image onto a fixed image.
 *
 * A deformation field is represented as a image whose pixel type is some
 * vector type with at least N elements, where N is the dimension of
 * the fixed image. The vector type must support element access via operator
 * []. It is assumed that the vector elements behave like floating point
 * scalars.
 *
 * This class is templated over the fixed image type, moving image type
 * and the deformation field type.
 *
 * The input fixed and moving images are set via methods SetFixedImage
 * and SetMovingImage respectively. An initial deformation field maybe set via
 * SetInitialDeformationField or SetInput. If no initial field is set,
 * a zero field is used as the initial condition.
 *
 * The algorithm has one parameters: the number of iteration to be performed.
 *
 * The output deformation field can be obtained via methods GetOutput
 * or GetDeformationField.
 *
 * This class make use of the finite difference solver hierarchy. Update
 * for each iteration is computed in HammerRegistrationFunction.
 *
 * \warning This filter assumes that the fixed image type, moving image type
 * and deformation field type all have the same number of dimensions.
 * 
 * \sa HammerRegistrationFunction 
 * \ingroup DeformableImageRegistration MultiThreaded
 */
template<class TFixedImage, class TMovingImage, class TDeformationField, class TAVFilterType>
class ITK_EXPORT HammerRegistrationFilter : 
    public PDEDeformableRegistrationFilter< TFixedImage, TMovingImage,
                                            TDeformationField>
{
public:
  /** Standard class typedefs. */
  typedef HammerRegistrationFilter                  Self;
  typedef PDEDeformableRegistrationFilter<
    TFixedImage, TMovingImage,TDeformationField>    Superclass;
  typedef SmartPointer<Self>                        Pointer;
  typedef SmartPointer<const Self>                  ConstPointer;

  /** Method for creation through the object factory. */
  itkNewMacro(Self);

  /** Run-time type information (and related methods). */
  itkTypeMacro( HammerRegistrationFilter, 
                PDEDeformableRegistrationFilter );

  /** Inherit types from superclass. */
  typedef typename Superclass::TimeStepType  TimeStepType;

  /** FixedImage image type. */
  typedef typename Superclass::FixedImageType     FixedImageType;
  typedef typename Superclass::FixedImagePointer  FixedImagePointer;

  /** MovingImage image type. */
  typedef typename Superclass::MovingImageType     MovingImageType;
  typedef typename Superclass::MovingImagePointer  MovingImagePointer;
  
  /** Deformation field type. */
  typedef typename Superclass::DeformationFieldType 
  DeformationFieldType;
  typedef typename Superclass::DeformationFieldPointer  
  DeformationFieldPointer;

  /** FiniteDifferenceFunction type. */
  typedef typename Superclass::FiniteDifferenceFunctionType
  FiniteDifferenceFunctionType;

  /** HammerRegistrationFilterFunction type. */
  typedef HammerRegistrationFunction<FixedImageType,MovingImageType,
                                     DeformationFieldType>  HammerRegistrationFunctionType;
  
  /** Attribute vector filter type. */
  typedef TAVFilterType AttributeVectorFilterType;
  typedef typename AttributeVectorFilterType::Pointer  AttributeVectorFilterPointerType;

  /** Attribute vector type. */
  // This may need some implementation. Should be straightforward
  // typedef typename AttributeVectorFilterType::AttributeVectorType AttributeVectorType;

  /** Get the metric value. The metric value is the mean square difference 
   * in intensity between the fixed image and transforming moving image 
   * computed over the the overlapping region between the two images. 
   * This is value is only available for the previous iteration and 
   * NOT the current iteration. */
  virtual double GetMetric() const;

  /** Set/Get AttributeVectorFilter */
  itkSetObjectMacro( AttributeVectorFilter, AttributeVectorFilterType );
  itkGetConstObjectMacro( AttributeVectorFilter, AttributeVectorFilterType );

  /** Set/Get parameters */
  itkSetMacro( NumberOfMaximumDrivingVoxels, unsigned long );
  itkGetMacro( NumberOfMaximumDrivingVoxels, unsigned long );

protected:
  HammerRegistrationFilter();
  ~HammerRegistrationFilter() {}
  void PrintSelf(std::ostream& os, Indent indent) const;

  /** Initialize the state of filter and equation before each iteration. */
  virtual void InitializeIteration();

  /** Apply update. */
  virtual void ApplyUpdate(TimeStepType dt);

private:
  HammerRegistrationFilter(const Self&); //purposely not implemented
  void operator=(const Self&); //purposely not implemented

  unsigned long m_NumberOfMaximumDrivingVoxels;
  AttributeVectorFilterPointerType m_AttributeVectorFilter;

};


} // end namespace itk

#ifndef ITK_MANUAL_INSTANTIATION
#include "itkHammerRegistrationFilter.txx"
#endif

#endif
