/*=========================================================================

Program:   Insight Segmentation & Registration Toolkit
Module:    $RCSfile: itkHammerRegistrationFunction.h,v $
Language:  C++
Date:      $Date: 2009/01/15 15:06:35 $
Version:   $Revision: 1.1 $

Copyright (c) Insight Software Consortium. All rights reserved.
See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for
details.

This program is developed under NIH NCBC collaboration grant
R01 EB006733, "Development and Dissemination of Robust Brain MRI
Measurement Tools". 

This software is distributed WITHOUT ANY WARRANTY; without even 
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkHammerRegistrationFunction_h
#define __itkHammerRegistrationFunction_h

#include "itkPDEDeformableRegistrationFunction.h"
#include "itkPoint.h"
#include "itkCovariantVector.h"
#include "itkInterpolateImageFunction.h"
#include "itkLinearInterpolateImageFunction.h"
#include "itkCentralDifferenceImageFunction.h"
#include "itkHammerTissueAttributeVectorImageFilter.h"
#include "itkHammerTissueAttributeVector.h"
#include "itkAffineTransform.h"

#include "itkVector.h"

#define ITER_THRD  0.85
#define MAX_SIGMA  100
#define MAX_LEVEL  500
#define EDGE  3

namespace itk {

/**
 * \class HammerRegistrationFunction
 *
 * This class encapsulate the PDE which drives the demons registration
 * algorithm. It is used by DemonsRegistrationFilter to compute the
 * output deformation field which will map a moving image onto a
 * a fixed image.
 *
 * Non-integer moving image values are obtained by using
 * interpolation. The default interpolator is of type
 * LinearInterpolateImageFunction. The user may set other
 * interpolators via method SetMovingImageInterpolator. Note that the input
 * interpolator must derive from baseclass InterpolateImageFunction.
 *
 * This class is templated over the fixed image type, moving image type,
 * and the deformation field type.
 *
 * \warning This filter assumes that the fixed image type, moving image type
 * and deformation field type all have the same number of dimensions.
 *
 * \sa DemonsRegistrationFilter
 * \ingroup FiniteDifferenceFunctions
 */
template<class TFixedImage, class TMovingImage, class TDeformationField>
class ITK_EXPORT HammerRegistrationFunction :
    public PDEDeformableRegistrationFunction< TFixedImage,
                                              TMovingImage, 
                                              TDeformationField>
{
public:
  /** Extract dimension from input image. */
  itkStaticConstMacro(InputImageDimension, unsigned int,
		      TFixedImage::ImageDimension);

  /** Standard class typedefs. */
  typedef HammerRegistrationFunction    Self;
  typedef PDEDeformableRegistrationFunction< TFixedImage,
                                             TMovingImage, TDeformationField > Superclass;
  typedef SmartPointer<Self>            Pointer;
  typedef SmartPointer<const Self>      ConstPointer;

  /** Method for creation through the object factory. */
  itkNewMacro(Self);

  /** Run-time type information (and related methods). */
  itkTypeMacro( HammerRegistrationFunction,
                PDEDeformableRegistrationFunction );

  /** MovingImage image type. */
  typedef typename Superclass::MovingImageType     MovingImageType;
  typedef typename Superclass::MovingImagePointer  MovingImagePointer;

  /** FixedImage image type. */
  typedef typename Superclass::FixedImageType     FixedImageType;
  typedef typename Superclass::FixedImagePointer  FixedImagePointer;
  typedef typename FixedImageType::IndexType      IndexType;
  typedef typename FixedImageType::SizeType       SizeType;
  typedef typename FixedImageType::SpacingType    SpacingType;
  typedef typename FixedImageType::PointType      PointType;
  typedef typename FixedImageType::RegionType     RegionType;

  /** Deformation field type. */
  typedef typename Superclass::DeformationFieldType  DeformationFieldType;
  typedef typename Superclass::DeformationFieldTypePointer
                                                     DeformationFieldTypePointer;

  typedef typename DeformationFieldType::PixelType   DeformationVectorType;

  /** Inherit some enums from the superclass. */
  itkStaticConstMacro(ImageDimension, unsigned int,Superclass::ImageDimension);

  /** Set the const marco in hammer registration function **/
  /** Marcos defined in HAMMER **/

 	
  /** Inherit some enums from the superclass. */
  typedef typename Superclass::PixelType         PixelType;
  typedef typename Superclass::RadiusType        RadiusType;
  typedef typename Superclass::NeighborhoodType  NeighborhoodType;
  typedef typename NeighborhoodType::OffsetType  NeighborOffsetType;
  typedef typename Superclass::FloatOffsetType   FloatOffsetType;
  typedef typename Superclass::TimeStepType      TimeStepType;

  /** Interpolator type. */
  typedef double                                     CoordRepType;
  typedef InterpolateImageFunction<MovingImageType,CoordRepType> 
                                                     InterpolatorType;
  typedef typename InterpolatorType::Pointer         InterpolatorPointer;
  typedef LinearInterpolateImageFunction<MovingImageType,CoordRepType>
                                                     DefaultInterpolatorType;

  /** Image attrbute type */
  typedef itk::HammerTissueAttributeVector           AttributeVectorType;
  typedef itk::Image<AttributeVectorType, 3>         ImageAttributeType;
  typedef typename ImageAttributeType::Pointer       ImageAttributePointerType;

  /** Affine Transform Type */
  typedef itk::AffineTransform<double, 3>            AffineTransformType;
  typedef typename AffineTransformType::Pointer      AffineTransformPointerType;

  typedef std::vector<DeformationVectorType> VectorArrayType;

  virtual void InitializeIteration();

  /** This method is called by a finite difference solver image filter at
   * each pixel that does not lie on a data set boundary */
  virtual PixelType  ComputeUpdate(const NeighborhoodType &neighborhood,
                                   void *globalData,
                                   const FloatOffsetType &offset = FloatOffsetType(0.0));

  /** This class uses a constant timestep of 1. */
  virtual TimeStepType ComputeGlobalTimeStep(void * itkNotUsed(GlobalData)) 
    const
  { return m_TimeStep; }

  /** set fixed and moving image attribute */
  void SetFixedAttributeImage (ImageAttributeType * fAVec)
  {
    this->m_FixedAttributeImage = fAVec;
    return;
  }

  void SetMovingAttributeImage (ImageAttributeType * mAVec)
  {
    this->m_MovingAttributeImage = mAVec;
    return;
  }

  /** set points to be evaluated on fixed image */
  void SetPickedPointsOnFixedImage( std::vector<IndexType> & pts )
  {
    for (int k = 0; k < pts.size(); k++)
      {
      m_PickedPointsOnFixedImage.push_back( pts[k] ); 
      }
    return;
  }

  void SetPickedPointsOnMovingImage( std::vector<IndexType> & pts )
  {
    for (int k = 0; k < pts.size(); k++)
      {
      m_PickedPointsOnMovingImage.push_back( pts[k] ); 
      }
    return;
  }

  const AffineTransformType * GetGlobalAffineTransform(void) const
    { return m_GlobalAffineTransform; }



  /** Return a pointer to a global data structure that is passed to
   * this object from the solver at each calculation.  */
  virtual void *GetGlobalDataPointer() const
    {
    GlobalDataStruct *global = new GlobalDataStruct();
    global->m_SumOfSquaredDifference  = 0.0;
    global->m_NumberOfPixelsProcessed = 0L;
    global->m_SumOfSquaredChange      = 0;
    return global;
    }

  /** Release memory for global data structure. */
  virtual void ReleaseGlobalDataPointer( void *GlobalData ) const;

protected:
  HammerRegistrationFunction();
  ~HammerRegistrationFunction() {}
  void PrintSelf(std::ostream& os, Indent indent) const;

  /** FixedImage image neighborhood iterator type. */
  typedef ConstNeighborhoodIterator<FixedImageType> 
    FixedImageNeighborhoodIteratorType;

  struct GlobalDataStruct
    {
    double          m_SumOfSquaredDifference;
    unsigned long   m_NumberOfPixelsProcessed;
    double          m_SumOfSquaredChange;
    };

private:
  HammerRegistrationFunction(const Self&); //purposely not implemented
  void operator=(const Self&); //purposely not implemented

  /** The global timestep. */
  TimeStepType                    m_TimeStep;
  
  ImageAttributePointerType m_FixedAttributeImage;
  ImageAttributePointerType m_MovingAttributeImage;

  RadiusType                m_Radius;
  
  std::vector<IndexType>    m_PickedPointsOnFixedImage;
  std::vector<IndexType>    m_PickedPointsOnMovingImage;
  std::vector<AttributeVectorType>    m_AVectorPickedPointsOnFixedImage;
  std::vector<AttributeVectorType>    m_AVectorPickedPointsOnMovingImage;

  // indices in the spherical neighborhood
  std::vector<NeighborOffsetType> m_OffsetInSphericalNeighborhood;

  /** The metric value is the mean square difference in intensity between
   * the fixed image and transforming moving image computed over the 
   * the overlapping region between the two images. */
  mutable double                  m_Metric;
  mutable double                  m_SumOfSquaredDifference;
  mutable unsigned long           m_NumberOfPixelsProcessed;
  mutable double                  m_RMSChange;
  mutable double                  m_SumOfSquaredChange;

  /** parameters for HAMMER registration */
  VectorArrayType m_PointMatchingNeighborhood; 
  VectorArrayType m_SubvolumeNeighborhood;   
  VectorArrayType m_ModelDrivingPoint;
  VectorArrayType m_SubjectDrivingPoint;
  VectorArrayType m_ModelDrivingPointDisplacement;
  mutable VectorArrayType m_InverseDisplacement;
  DeformationFieldTypePointer m_PreviouseDeformationField;
  std::vector<NeighborOffsetType> m_SearchNeighbor;
  float	m_GuassianAtLevelSigma[MAX_SIGMA][MAX_LEVEL];

  /** parameters need to set by user in registration filter**/
  int         m_SearchRadius;
  int         m_PointMatchRadius;
  int         m_SubvolumeRadius;
  int         m_SubsampleDrivingPointStepSize;
  float       m_GaussianLookupTable;
  float       m_DeformRate;
  float       m_IterationRatio;
  float       m_SubvolumnSimilarityThreshold;
  float       m_PointMatchingThreshold;
  float       m_SubvolumnSimilarityDegree;
  bool        m_IsBigVN;
  int         m_NeighborhoodStep;
  bool        m_IsYoungBrain;
  float       m_SmoothFactor;
  float       m_DeformationFieldSigma;

  AffineTransformPointerType m_GlobalAffineTransform;

  std::vector<float> m_DrivingVoxelQualification;

  /** Mutex lock to protect modification to metric. */
  mutable SimpleFastMutexLock     m_MetricCalculationLock;

public:
	/************************************************************************/
	/* 
	used to set up the neighborhood in subvolumn matching (m_PointMatchingNeighborhood)
	*/
	/************************************************************************/
	void CreatePointMatchingNeighbor(int Radius);
	/************************************************************************/
	/*
	used to set the neighborhood in warp subvolumn according to tentative displacement (m_SubvolumnNeighborhood)
	*/
	/************************************************************************/
	void CreateSubvolumnNeighbor(int Radius);
	/************************************************************************/
	/*
	used to calculate the offset of searching neighbor
	*/
	void CreateSearchNeighbor(int Radius);
	/************************************************************************/
	/************************************************************************/
	/* 
	the method to record the location each neighborhood point from inside to outside. neet to convert the code to ITK style
	*/
	/************************************************************************/
	void CalculateNeighborhoodbyIncreasingRadius(VectorArrayType &Neighbor, int Radius)  throw (InvalidRequestedRegionError);
	/************************************************************************/
	/*
	the metric function to calculate the distance of two attribute vector
	*/
	/************************************************************************/
	float SimilarityBetweenTwoImgAttribute(AttributeVectorType Template_Feature, AttributeVectorType Subject_Feature) const;
	/************************************************************************/
	/* 
	core function of HAMMER which give out the correspondence in searching area 
	by not only compare the similarity of points but also the overall similarity of subvolumns
	NOTE: the input is the index in m_ModelDrivingPoint. 
	the output the candidate deformation vector and the subvolumn matching degree.
	*/
	/************************************************************************/
	float DetermineCorrespondenceOnOneDrivingVoxel(const int &DrivingPointIndex, DeformationVectorType &Deformation_Update) const;
	/************************************************************************/
	/*
	the procedure perform subvolumn matching, return the overall similarity degree of matching result
	*/
	/************************************************************************/
	float SubVolumnMatching(IndexType &ImageIndex, DeformationVectorType TentativeWarp, VectorArrayType CertainNeighborhood, int NeighborhoodStep) const;
	/************************************************************************/
	/*
	compute the magnitude of deformation vector
	*/
	/************************************************************************/
	float ComputeVectorMagnitude(DeformationVectorType &Deform_Vector);
	/************************************************************************/
	/* 
	obtain the inverse force from subject to template image
	*/
	/************************************************************************/
	void FindingInverseForceFromSubject() const;
	/************************************************************************/
	/*
	Warp the subvolumn w.r.t. tentative deform vector in gaussian way
	*/
	/************************************************************************/
	void DisseminateDeformation(const int &DrivingPointIndex, DeformationVectorType TentativeWarp, VectorArrayType CertainNeighborhood, int NeighborhoodSize, int GaussianSigma);
  
  /** determine driving voxel with given parameters */
  void IdentifyDrivingVoxelsInFixedImage( );
  void IdentifyDrivingVoxelsInMovingImage( );
  int IdentifyDrivingVoxels( ImageAttributePointerType avPointer, std::vector<PointType> & drivingVoxels );

  /** Gaussian smoothing of the deformationfield */
  void SmoothDeformationField();

  /** fit the deformation field with a global affine transform */
  void FitGlobalAffineTransform();

};


} // end namespace itk

#ifndef ITK_MANUAL_INSTANTIATION
#include "itkHammerRegistrationFunction.txx"
#endif

#endif
