/*=========================================================================

Program:   Insight Segmentation & Registration Toolkit
Module:    $RCSfile: itkHammerDeformableRegistrationImageFilter.h,v $
Language:  C++
Date:      $Date: 2009/01/13 20:19:20 $
Version:   $Revision: 1.4 $

Copyright (c) Insight Software Consortium. All rights reserved.
See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for
details.

This program is developed under NIH NCBC collaboration grant
R01 EB006733, "Development and Dissemination of Robust Brain MRI
Measurement Tools". 

This software is distributed WITHOUT ANY WARRANTY; without even 
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkHammerDeformableRegistrationImageFilter_h
#define __itkHammerDeformableRegistrationImageFilter_h

#include "itkImageToImageFilter.h"
#include "itkOrientedImage.h"
#include "itkConstNeighborhoodIterator.h"

#include "itkPoint.h"
#include "itkCovariantVector.h"
#include "itkInterpolateImageFunction.h"
#include "itkLinearInterpolateImageFunction.h"
#include "itkCentralDifferenceImageFunction.h"
#include "itkHammerTissueAttributeVectorImageFilter.h"
//#include "itkHammerTissueAttributeVector.h"
#include "itkAffineTransform.h"

#include "itkVector.h"
#include "itkImageRegionIterator.h"
#include "itkImageRegionConstIterator.h"
#include <vnl/vnl_matrix.h>
#include <vnl/algo/vnl_svd.h>
#include <vcl_iostream.h>

#define MAX_SIGMA 100
#define MAX_LEVEL 500
#define ITER_THRD 0.85
#define STEPPNT 3
//#define DEBUG_MODE
#ifdef DEBUG_MODE
#define DUMP_DEFORMATIONFIELD
#define DUMP_GMI_FEATURE
#endif

namespace itk
{
/** \class HammerDeformableRegistrationImageFilter
  * \brief Computes the deformation vector field using HAMMER
  *
  */
template <class TInputImage, class TOutputImage>
class ITK_EXPORT HammerDeformableRegistrationImageFilter :
    public ImageToImageFilter< TInputImage, TOutputImage >
{
public:
  /** Extract dimension from input image. */
  itkStaticConstMacro(InputImageDimension, unsigned int,
		      TInputImage::ImageDimension);
  itkStaticConstMacro(OutputImageDimension, unsigned int,
		      TInputImage::ImageDimension);

  /** Standard class typedefs. */
  typedef HammerDeformableRegistrationImageFilter Self;

  /** Convenient typedefs for simplifying declarations. */
  typedef TInputImage                       InputImageType;
  typedef typename InputImageType::Pointer  InputImagePointer;
  typedef typename InputImageType::ConstPointer    InputImageConstPointer;

  typedef TOutputImage                      OutputImageType;
  typedef typename OutputImageType::Pointer OutputImagePointer;

  /** Deformation field type. */
  typedef TOutputImage                               DeformationFieldType;
  typedef typename DeformationFieldType::Pointer     DeformationFieldPointer;
  typedef typename DeformationFieldType::PixelType   DeformationVectorType;

  typedef typename InputImageType::IndexType      IndexType;
  typedef typename InputImageType::SizeType       SizeType;
  typedef typename InputImageType::SpacingType    SpacingType;
  typedef typename InputImageType::PointType      PointType;
  typedef typename InputImageType::RegionType     RegionType;

  /** Neighborhood radius type */
  typedef typename ConstNeighborhoodIterator<TInputImage>::RadiusType RadiusType;


  /** Interpolator type. */
  typedef double                                     CoordRepType;
  typedef InterpolateImageFunction<InputImageType,   CoordRepType> 
                                                     InterpolatorType;
  typedef typename InterpolatorType::Pointer         InterpolatorPointer;
  typedef LinearInterpolateImageFunction<InputImageType,CoordRepType>
                                                     DefaultInterpolatorType;

  /** types for neighborhood iterator */
  typedef ConstNeighborhoodIterator<InputImageType>     NeighborhoodIteratorType;
  typedef typename NeighborhoodIteratorType::OffsetType NeighborOffsetType;


  /** Standard class typedefs. */
  typedef ImageToImageFilter< InputImageType, OutputImageType> Superclass;
  typedef SmartPointer<Self>                                   Pointer;
  typedef SmartPointer<const Self>                             ConstPointer;

  /** Method for creation through the object factory. */
  itkNewMacro(Self);

  /** Run-time type information (and related methods). */
  itkTypeMacro(HammerDeformableRegistrationImageFilter, ImageToImageFilter);

  /** Image typedef support. */
  typedef typename InputImageType::PixelType   InputPixelType;
  typedef typename OutputImageType::PixelType  OutputValueType;
  typedef typename OutputImageType::RegionType OutputImageRegionType;

  typedef typename itk::ImageRegionIterator <OutputImageType> DeformationFieldIteratorType;
  typedef typename itk::ImageRegionConstIterator <OutputImageType> ConstDeforamtionFieldIteratorType;
  typedef typename itk::ImageRegionIterator<InputImageType> InputImageIteratorType;
  typedef typename itk::ImageRegionConstIterator<InputImageType> InputImageConstIteratorType;
  


  /** Image attrbute type */
  typedef itk::HammerTissueAttributeVector                 AttributeVectorType;
  //typedef itk::OrientedImage<AttributeVectorType, 3>       ImageAttributeType;
  typedef itk::Image<AttributeVectorType, 3>       ImageAttributeType;
  typedef ImageAttributeType::Pointer                      ImageAttributePointerType;
  typedef std::vector<AttributeVectorType>                 AttributeVectorArrayType;
  typedef typename itk::ImageRegionIterator<ImageAttributeType> AttributeImageIteratorType;
  typedef typename itk::ImageRegionConstIterator<ImageAttributeType> AttributeImageConstIteratorType;

  /** Affine Transform Type */
  typedef itk::AffineTransform<double, 3>            AffineTransformType;
  typedef typename AffineTransformType::Pointer      AffineTransformPointer;

  typedef std::vector<DeformationVectorType>         DeforamtionVectorArrayType;
  typedef std::vector<IndexType>					 IndexArrayType;

  itkSetMacro( NumberOfIterations, unsigned int);
  itkGetMacro( NumberOfIterations, unsigned int);

  itkSetMacro( PointMatchingThreshold, float);
  itkGetMacro( PointMatchingThreshold, float);

  itkSetMacro( AffineInitialization, bool);
  itkGetMacro( AffineInitialization, bool);

  itkSetMacro( SubvolumnSimilarityThreshold, float);
  itkGetMacro( SubvolumnSimilarityThreshold, float);

  itkSetMacro( SearchRadius, int);
  itkGetMacro( SearchRadius, int);

  itkSetMacro( DeformRate, float);
  itkGetMacro( DeformRate, float);

  //itkSetObjectMacro(FixedAttributeImage, ImageAttributeType);
  //itkSetObjectMacro(MovingAttributeImage, ImageAttributeType);


  /** HammerDeformableRegistrationImageFilter needs a larger input requested region than
    * the output requested region.  As such, GradientImageFilter needs
    * to provide an implementation for GenerateInputRequestedRegion()
    * in order to inform the pipeline execution model.
    *
    * \sa ImageToImageFilter::GenerateInputRequestedRegion() */
  virtual void GenerateInputRequestedRegion() throw(InvalidRequestedRegionError);

  virtual void SetFixedImage( InputImageType* image )
  {
    this->SetInput( 0, image );
    this->m_FixedImage = this->GetInput( 0 );
  }

  virtual void SetMovingImage( InputImageType* image )
  {
    this->SetInput( 1, image );
    this->m_MovingImage = this->GetInput( 1 );
  }

  std::vector<float> GetGMIThreshold()
  {
    return this->m_GMIThreshold;  
  }

  void SetGMIThreshold( std::vector<float> &vec)
  {
    this->m_GMIThreshold.clear();
    for (unsigned int k = 0; k < vec.size(); k++)
    {
      this->m_GMIThreshold.push_back( vec[k] );
    }
    return;
  }

  void SetRadius(const RadiusType &r)
    { m_Radius = r; }

protected:
  HammerDeformableRegistrationImageFilter();
  virtual ~HammerDeformableRegistrationImageFilter();
  void PrintSelf(std::ostream& os, Indent indent) const;
  void GenerateData();

private:

  HammerDeformableRegistrationImageFilter(const Self&); //purposely not implemented
  void operator=(const Self&); //purposely not implemented

  InputImageConstPointer                m_FixedImage;
  InputImageConstPointer                m_MovingImage;

//   InputImagePointer                     m_CurrFixedImage;
//   InputImagePointer                     m_CurrMovingImage;

  //ImageAttributePointerType            m_FixedAttributeImage;
  //ImageAttributePointerType            m_MovingAttributeImage;

  //DeformationFieldPointer				m_OutputDeformFld;
  DeformationFieldPointer				m_FinalOutputDeformFld;
  //DeformationFieldPointer				m_OutputDeformFld_Copy;

  unsigned int                          m_NumberOfIterations;
  RadiusType                            m_Radius;

  IndexArrayType				        m_PickedPointsOnFixedImage;
  IndexArrayType				        m_PickedPointsOnMovingImage;

  std::vector<float>                    m_GMIThreshold;

  // indices in the spherical neighborhood
  std::vector<NeighborOffsetType>       m_OffsetInSphericalNeighborhood;

  /** The metric value is the mean square difference in intensity between
   * the fixed image and transforming moving image computed over the 
   * the overlapping region between the two images. */
  mutable double                        m_Metric;
  mutable double                        m_SumOfSquaredDifference;
  mutable unsigned long                 m_NumberOfPixelsProcessed;
  mutable double                        m_RMSChange;
  mutable double                        m_SumOfSquaredChange;

  /** parameters for HAMMER registration */
  IndexArrayType                        m_SubvolumeNeighborhood;   
  IndexArrayType						m_SmoothNeighbor;   
  DeforamtionVectorArrayType			m_ModelDrivingPointDisplacement;
  mutable DeforamtionVectorArrayType    m_InverseDisplacement;
  IndexArrayType						m_SearchNeighbor;
  float									m_GuassianAtLevelSigma[MAX_SIGMA][MAX_LEVEL];
  AttributeVectorArrayType				m_ModelAttributeVector;

  /** parameters need to set by user in registration filter**/
  int         m_SearchRadius;
  int         m_SubvolumeRadius; 
  float       m_DeformRate;
  float       m_IterationRatio;
  float       m_SubvolumnSimilarityThreshold;
  float       m_PointMatchingThreshold;
  float       m_SubvolumnSimilarityDegree;
  bool        m_IsBigVN;
  bool        m_IsYoungBrain;
  bool        m_AffineInitialization;

  float       m_SmoothFactor;
  float       m_DeformationFieldSigma;
  int	      m_StartSearchPoint;
  bool		  m_InitialDeformationField;
  float	      m_DfmSmthCoeff;
  int		  m_SmoothTime;
  int         m_NeighborhoodStep;

  vnl_matrix<double> GlobalAffineMatrix;
  AffineTransformType::Pointer m_AffineTransform;
  int		  Start_Search_Point;

  /** Standard deviation for Gaussian smoothing */
  double                   m_StandardDeviations[OutputImageDimension];

  /** Maximum error for Gaussian operator approximation. */
  double                    m_MaximumError;

  /** Limits of Guassian kernel width. */
  unsigned int              m_MaximumKernelWidth;

  /*Modified by GR*/
  std::vector<float>         m_FixedImageDrivingVoxelQualification;
  std::vector<float>         m_MovingImageDrivingVoxelQualification;

  /** Set/Get the desired maximum error of the Guassian kernel approximate. 
  * \sa GaussianOperator. */
  itkSetMacro( MaximumError, double );
  itkGetMacro( MaximumError, double );

  /** Set/Get the desired limits of the Gaussian kernel width.
  * \sa GaussianOperator. */
  itkSetMacro( MaximumKernelWidth, unsigned int );
  itkGetMacro( MaximumKernelWidth, unsigned int );
public:

  /** compute the initial affine transform using a registration step */
  void LinearInitialization();

  /** used to set up the neighborhood in subvolumn matching
    * (m_PointMatchingNeighborhood)  */
  void CreatePointMatchingNeighbor(IndexArrayType &Neighbor, int Radius);
	
  /** used to set the neighborhood in warp subvolumn according to
    * tentative displacement (m_SubvolumnNeighborhood)  */  
  void CreateSubvolumnNeighbor(IndexArrayType &Neighbor, int Radius);
	
  /** used to calculate the offset of searching neighbor */
  void CreateSearchNeighbor(IndexArrayType &Neighbor, int Radius);

  /** the method to record the location each neighborhood point from
    * inside to outside. neet to convert the code to ITK style  */
  void CalculateNeighborhoodbyIncreasingRadius(IndexArrayType &Neighbor, int Radius)  throw (InvalidRequestedRegionError);
	
  /** the metric function to calculate the distance of two attribute
    * vector  */
  float SimilarityBetweenTwoImgAttribute(AttributeVectorType Template_Feature, AttributeVectorType Subject_Feature) const;
	
  /** core function of HAMMER which give out the correspondence in
    * searching area by not only compare the similarity of points but
    * also the overall similarity of subvolumns  
    * NOTE: the input is the index in m_ModelDrivingPoint. the output the
    * candidate deformation vector and the subvolumn matching degree. */
  float DetermineCorrespondenceOnOneDrivingVoxel(ImageAttributePointerType &FixedAttributeImage, ImageAttributePointerType &MovingAttributeImage, DeformationFieldPointer &DeformFld, int DrivingPointIndex, DeformationFieldPointer DeformFld_Last, DeformationVectorType &DeformationUpdate, int SearchRadius, int Step) const;

  /** the procedure perform subvolumn matching, return the overall
    * similarity degree of matching result */
  float SubVolumnMatching(ImageAttributePointerType &FixedAttributeImage, ImageAttributePointerType &MovingAttributeImage, DeformationFieldPointer &DeformFld, IndexType &ImageIndex, DeformationVectorType TentativeWarp, IndexArrayType CertainNeighborhood, int NeighborhoodStep, float *MinDist, float MinDist_Threshold) const;
	
  /** Compute the Eclidian distance of certain vector
  **/
  float ComputeVectorMagnitude(DeformationVectorType Deform_Vector) const;
	
  /** obtain the inverse force from subject to template image */
  void FindingInverseForceFromSubject(ImageAttributePointerType &FixedAttributeImage, ImageAttributePointerType &MovingAttributeImage, DeformationFieldPointer &DeformFld, int SearchRadius) const;

  /** Warp the subvolumn w.r.t. tentative deform vector in gaussian
    * way */
  void DisseminateDeformation(DeformationFieldPointer &DeformFld, const int &DrivingPointIndex, DeformationVectorType TentativeWarp, IndexArrayType CertainNeighborhood, int NeighborhoodSize, int GaussianSigma);
  
  /** determine driving voxel with given parameters */
  void IdentifyDrivingVoxelsInFixedImage(ImageAttributePointerType &FixedAttributeImage, std::vector<float> &FixedImageCriteria);
  void IdentifyDrivingVoxelsInMovingImage(ImageAttributePointerType &MovingAttributeImage, std::vector<float> &MovingImageCriteria);
  unsigned int IdentifyDrivingVoxels( ImageAttributePointerType avPointer, std::vector<IndexType> &drivingVoxels, std::vector<float> &Criteria);
  unsigned int IdentifyDrivingVoxels_GR( ImageAttributePointerType avPointer, std::vector<IndexType> &drivingVoxels, std::vector<float> &Criteria);

  /** Gaussian smoothing of the deformationfield */
  void SmoothDeformationField(ImageAttributePointerType &FixedAttributeImage, DeformationFieldPointer DeformFld, DeformationFieldPointer DeformFld_Last, float LocalRatio);

  /** fit the deformation field with a global affine transform */
  void FitGlobalAffineTransform(DeforamtionVectorArrayType CurrentDeformationOnDrivingPoint, DeforamtionVectorArrayType PreviousDeformationOnDrivingPoint);

  /** Set the standard deviations for gaussian smoothing **/
  void SetStandardDeviations( double value );

  /** Smooth the deforomation field once **/
  void SmoothDeformation_OneTime(ImageAttributePointerType &FixedAttributeImage, DeformationFieldPointer DeformFld, int Time);

  /** Edge preserving Smooth the deforomation field once **/
  void EdgePreserveSmoothDeformation_OneTime(ImageAttributePointerType &FixedAttributeImage, DeformationFieldPointer DeformFld, int Time);

  /** serious HAMMER registration in a loop **/
  void HAMMERRegistrationOneRound(ImageAttributePointerType &FixedAttributeImage, ImageAttributePointerType &MovingAttributeImage, int resolution, float ratio_iteration, float LocalRatio, DeformationFieldPointer DeformFld);
  /* the main loop of HAMMER algorithm*/
  void HAMMERMainLoop(ImageAttributePointerType &FixedAttributeImage, ImageAttributePointerType &MovingAttributeImage, DeformationFieldPointer &DeformFld);
};

} // end namespace itk

#ifndef ITK_MANUAL_INSTANTIATION
#include "itkHammerDeformableRegistrationImageFilter.txx"
#endif

#endif
