/*=========================================================================

  Program:   HAMMER Deformable Registration
  Module:    $RCSfile: itkHammerDeformationImageFunction.h,v $
  Language:  C++
  Date:      $Date: 2007/01/30 20:56:09 $
  Version:   $Revision: 1.9 $

=========================================================================*/
#ifndef __itkHammerDeformationImageFunction_h
#define __itkHammerDeformationImageFunction_h
#define MAX_SIGMA 100
#define MAX_LEVEL 500
#include "itkImageFunction.h"
#include "itkNumericTraits.h"
#include "itkVector.h"
#include <vector>
#include "itkImage.h"
#include "itkPointSet.h"
#include "itkImageRegionIterator.h"
#include "itkImageRegionConstIterator.h"
namespace itk
{

// TInputImage is the type of GMI attribute vector images. When
// finding deformation for a point, we do not need the original
// images. 

template <class TInputImage, class TCoordRep = float >
class ITK_EXPORT HammerDeformationImageFunction :
	public ImageFunction< TInputImage, ITK_TYPENAME NumericTraits<typename TInputImage::PixelType>::RealType, TCoordRep >
{
public:
  /** Standard class typedefs. */
  typedef HammerDeformationImageFunction Self;
  typedef ImageFunction<TInputImage, ITK_TYPENAME NumericTraits<typename TInputImage::PixelType>::RealType,
	  TCoordRep > Superclass;

  typedef SmartPointer<Self> Pointer;
  typedef SmartPointer<const Self>  ConstPointer;

  typedef TInputImage                       InputImageType;
  typedef typename InputImageType::Pointer  InputImagePointer;
  typedef typename InputImageType::PixelType InputAttributeType;
  typedef typename InputImageType::IndexType InputAttributIndexType;
 
  typedef itk::Vector<float, 3> DisplacementType;
  typedef itk::Image<DisplacementType, 3> DeformationFiledType;
  
  typedef itk::PointSet<float, 3> PointDisplacementType;
  typedef itk::Image<int, 3> LocationImageType; 
  typedef LocationImageType::Pointer LocationImagePointType;
  typedef PointDisplacementType::PointsContainer DrivingPointContainer;
  typedef DrivingPointContainer::Iterator DrivingPointIterator;
  typedef itk::ImageRegionIterator <LocationImageType> LocationIteratorType;
  typedef itk::ImageRegionConstIterator <LocationImageType> ConstLocationIteratorType;

  /** types for neighborhood iterator */
  //typedef itk::NeighborhoodIterator<InputImageType> NeighborhoodIteratorType;
  //typedef typename NeighborhoodIteratorType::OffsetType NeighborOffsetType;

  /** Run-time type information (and related methods). */
  itkTypeMacro(HammerDeformationImageFunction, ImageFunction);

  /** Method for creation through the object factory. */
  itkNewMacro(Self);

  itkStaticConstMacro(InputImageDimension, unsigned int, TInputImage::ImageDimension);

  /** InputImageType typedef support. */
  typedef TInputImage InputImageType;
  typedef typename Superclass::InputImageConstPointer InputImageConstPointer;

  /** OutputType typdef support. */
  typedef typename Superclass::OutputType OutputType;

  /** Index typedef support. */
  typedef typename Superclass::IndexType IndexType;
  
  /** ContinuousIndex typedef support. */
  typedef typename Superclass::ContinuousIndexType ContinuousIndexType;

  /** Point typedef support. */
  typedef typename Superclass::PointType PointType;

  /** Dimension of the underlying image. */
  //itkStaticConstMacro(ImageDimension, unsigned int, InputImageType::ImageDimension);

  /** Datatype used for the mean */
  //typedef itk::Vector<float, 3> RealType;
  typedef typename NumericTraits<typename InputImageType::PixelType>::RealType
	  RealType;

  virtual void GenerateSearchNeighborhood()  throw(InvalidRequestedRegionError) ;
  virtual void GenerateSubvolumnMatchingNeighborhood()  throw(InvalidRequestedRegionError) ;
  void ObtainTemplateDrivingPoint() const;
  float SimilarityBetweenTwoImgAttribute(InputAttributeType Template_Feature, InputAttributeType Subject_Feature) const;

  /** Evalulate the function at specified index */
  virtual RealType EvaluateAtIndex( const IndexType& index ) const;
  
  /** Evaluate the function at non-integer positions */
  
  virtual RealType Evaluate( const PointType& point ) const
    { 
      IndexType index;
      this->ConvertPointToNearestIndex( point, index );
      return this->EvaluateAtIndex( index ); 
    }

  virtual RealType EvaluateAtContinuousIndex( 
    const ContinuousIndexType& cindex ) const
    { 
      IndexType index;
      this->ConvertContinuousIndexToNearestIndex( cindex, index );
      return this->EvaluateAtIndex( index ) ; 
    }
  
  /** Get/Set the radius of the neighborhood over which the
      statistics are evaluated */
  itkSetMacro( NeighborhoodRadius, unsigned int );
  itkGetConstReferenceMacro( NeighborhoodRadius, unsigned int );

  /** Get/Set parameters */
  
  itkSetMacro( VoxelSimilarityThreshold, float );
  itkGetMacro( VoxelSimilarityThreshold, float );
  
  itkSetMacro( NeighborhoodSimilarityThreshold, float );
  itkGetMacro( NeighborhoodSimilarityThreshold, float );

  itkSetMacro( Sigma, float );
  itkGetMacro( Sigma, float );

  itkSetMacro( SubvolumnRadius, int);
  itkGetMacro (SubvolumnRadius, int);

  itkSetMacro( NeighborhoodStep, int);
  itkGetMacro( NeighborhoodStep, int);

  itkSetMacro( IsBigVN, bool);
  itkGetMacro( IsBigVN, bool);

  itkSetMacro( IsYoungBrain, bool);
  itkGetMacro( IsYoungBrain, bool);

  /** The second input image is stored as a member image */
  void SetSubjectImage( const InputImageType * ptr )
  {  m_SubjectImage = ptr; }
    
  const InputImageType * GetSubjectImage() const
  {  return  m_SubjectImage.GetPointer(); }
  
  void SetDeformationField( const DeformationFiledType *ptr)
  {
	  m_DeformationFiled = ptr;
  }

  const DeformationFiledType *GetDeformationField() const
  {
	  return m_DeformationFiled.GetPointer();
  }

  void SetTemplateDrivingPointArray(const DisplacementType *ptr)
  {
	  m_TemplateDrivingPointArray = ptr;
  }

  const DisplacementType *GetTemplateDrivingPointArray() const
  {
	  return m_TemplateDrivingPointArray;
  }

  void SetTemplateDrivingPointDisplacement(const DisplacementType *ptr)
  {
	  m_TemplateDrivingPointDisplacement = ptr;
  }

  const DisplacementType *GetTemplateDrivingPointDisplacement() const
  {
	  return m_TemplateDrivingPointDisplacement;
  }


protected:
  HammerDeformationImageFunction();
  ~HammerDeformationImageFunction(){};
  void PrintSelf(std::ostream& os, Indent indent) const;

  bool m_UseImageSpacing;
  bool m_UseImageDirection;
  float m_Scale;
  // indices in the spherical neighborhood
  typedef itk::Offset<3> NeighborOffsetType;
  std::vector<NeighborOffsetType> m_OffsetInSphericalNeighborhood;
  std::vector<NeighborOffsetType> m_SubvolumnMatchingNeighborhood;
								    
private:
  HammerDeformationImageFunction( const Self& ); //purposely not implemented
  void operator=( const Self& ); //purposely not implemented

  InputImageConstPointer m_SubjectImage;   // this is the
                                           // subject image in
                                           // HAMMER's term 
  LocationImagePointType m_TemplateDrivingPointLocation;   //this image is used to record the index of template drivinging point in image space
  DeformationFiledType::Pointer m_DeformationFiled ;  // this is the current deformation field
  PointDisplacementType::Pointer m_TemplateDrivingPointDisplacement;  //this is the current displacement on each template driving point
  DrivingPointContainer::Pointer m_TemplateDrivingPointArray;      //this array is used to record the coordinate of the template driving point

  unsigned int   m_NeighborhoodRadius;
  float          m_VoxelSimilarityThreshold;
  float          m_NeighborhoodSimilarityThreshold;
  float          m_Sigma;
  int			 m_SubvolumnRadius;
  bool			 m_TemplateDrivingPointLocationReady;
  int			 m_NeighborhoodStep;
  float			 GuassianAtLevelSigma[MAX_SIGMA][MAX_LEVEL];
  bool           m_IsBigVN;
  bool           m_IsYoungBrain;
};

} // end namespace itk

#ifndef ITK_MANUAL_INSTANTIATION
# include "itkHammerDeformationImageFunction.txx"
#endif

#endif

