/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: HammerTPS.h,v $
  Language:  C++
  Date:      $Date: 2007-12-20 17:10:46 $
  Version:   $Revision: 1.31 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkHammerTPS_h
#define __itkHammerTPS_h

#include "itkKernelTransform.h"

namespace itk
{
/** \class HammerTPS
 *
 * \ingroup Transforms
 */
template <class TScalarType,         // Data type for scalars (float or double)
          unsigned int NDimensions = 3>          // Number of dimensions
class ITK_EXPORT HammerTPS : 
                public KernelTransform<   TScalarType, NDimensions>
{
public:
  /** Standard class typedefs. */
  typedef HammerTPS Self;
  typedef KernelTransform<    TScalarType, NDimensions>   Superclass;
  typedef SmartPointer<Self>        Pointer;
  typedef SmartPointer<const Self>  ConstPointer;
  
  /** New macro for creation of through a Smart Pointer */
  itkNewMacro( Self );

  /** Run-time type information (and related methods). */
  itkTypeMacro( HammerTPS, KernelTransform );

  /** Scalar type. */
  typedef typename Superclass::ScalarType  ScalarType;

  /** Parameters type. */
  typedef typename Superclass::ParametersType  ParametersType;

  /** Jacobian Type */
  typedef typename Superclass::JacobianType  JacobianType;

  /** Dimension of the domain space. */
  itkStaticConstMacro(SpaceDimension, unsigned int,Superclass::SpaceDimension);


                              
  /** These (rather redundant) typedefs are needed because on SGI, typedefs
   * are not inherited */
  typedef typename Superclass::InputPointType  InputPointType;
  typedef typename Superclass::OutputPointType  OutputPointType;
  typedef typename Superclass::InputVectorType InputVectorType;
  typedef typename Superclass::OutputVectorType OutputVectorType;
  typedef typename Superclass::InputCovariantVectorType InputCovariantVectorType;
  typedef typename Superclass::OutputCovariantVectorType OutputCovariantVectorType;
  typedef typename Superclass::PointsIterator PointsIterator;

    /*
    Below is the member variables and functions added by Guorong
    */
  vnl_matrix<double> TPS_Psi, TPS_Kernel, TPS_Param, TPS_Affine;
  vnl_matrix<double> FixedPoint, MovingPoint;
  vnl_matrix<double> Q1, Q2, R, inverse_R;
  
  double ComputePointDistance(InputPointType &Point1, InputPointType &Point2);
  //used in the first step of TPS, to compute the distance matrix
  void PrepareTPSKernel();
  //used in the second step of TPS, to compute the TPS parameters and affine matrix
  double UpdateTPSParameter();
  /** Compute the position of point in the new space */

  OutputPointType TransformPoint(const InputPointType& thisPoint) const;
    /** Compute the contribution of the landmarks weighted by the kernel funcion
      to the global deformation of the space  */
  void ComputeTPSContribution( const InputPointType & inputPoint,
                                                     OutputPointType & result );

  itkSetMacro( Lemda, double);
  itkGetMacro( Lemda, double);
  itkSetMacro( Lemda2, double);
  itkGetMacro( Lemda2, double);
  
protected:
  HammerTPS()
  {
	  //MultiThreader::Pointer m_Threader = MultiThreader::New();
	  //int m_NumberOfThreads = m_Threader->GetNumberOfThreads();
	  //m_Threader->SetNumberOfThreads(1);
  };
  virtual ~HammerTPS() {}
  
  /** These (rather redundant) typedefs are needed because on SGI, typedefs
   * are not inherited. */
  typedef typename Superclass::GMatrixType GMatrixType;
  
  /** Compute G(x)
   * For the thin plate spline, this is:
   * G(x) = r(x)*I
   * \f$ G(x) = r(x)*I \f$
   * where
   * r(x) = Euclidean norm = sqrt[x1^2 + x2^2 + x3^2]
   * \f[ r(x) = \sqrt{ x_1^2 + x_2^2 + x_3^2 }  \f]
   * I = identity matrix. */
  virtual void ComputeG(const InputVectorType& landmarkVector, GMatrixType & gmatrix) const;

 private:
  HammerTPS(const Self&); //purposely not implemented
  void operator=(const Self&); //purposely not implemented
  double m_Lemda, m_Lemda2;
};

} // namespace itk

// Define instantiation macro for this template.
#define ITK_TEMPLATE_HammerTPS(_, EXPORT, x, y) namespace itk { \
  _(2(class EXPORT HammerTPS< ITK_TEMPLATE_2 x >)) \
  namespace Templates { typedef HammerTPS< ITK_TEMPLATE_2 x > \
                                                  HammerTPS##y; } \
  }

#if ITK_TEMPLATE_EXPLICIT
# include "Templates/HammerTPS+-.h"
#endif

#if ITK_TEMPLATE_TXX
# include "itkHammerTPS.txx"
#endif

#endif // __HammerTPS_h
