/*=========================================================================

Program:   Insight Segmentation & Registration Toolkit
Module:    $RCSfile: itkHammerDeformableRegistrationImageFilter.txx,v $
Language:  C++
Date:      $Date: 2009/01/14 21:46:50 $
Version:   $Revision: 1.6 $

Copyright (c) Insight Software Consortium. All rights reserved.
See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for
details.

This program is developed under NIH NCBC collaboration grant
R01 EB006733, "Development and Dissemination of Robust Brain MRI
Measurement Tools". 

This software is distributed WITHOUT ANY WARRANTY; without even 
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkHammerDeformableRegistrationImageFilter_txx
#define __itkHammerDeformableRegistrationImageFilter_txx
#include "itkHammerDeformableRegistrationImageFilter.h"

#include "itkImageRegionIterator.h"
#include "itkNeighborhoodAlgorithm.h"
#include "itkZeroFluxNeumannBoundaryCondition.h"
#include "itkProgressReporter.h"
#include "itkGaussianOperator.h"
#include "itkVectorNeighborhoodOperatorImageFilter.h"

#include "itkResampleImageFilter.h"
#include "itkImageRegistrationMethod.h"
#include "itkMattesMutualInformationImageToImageMetric.h"
#include "itkRegularStepGradientDescentOptimizer.h"
#include "itkNearestNeighborInterpolateImageFunction.h"
#include "itkCenteredTransformInitializer.h"
#include "itkVectorResampleImageFilter.h"
#include "itkIdentityTransform.h"
#include "itkVectorLinearInterpolateImageFunction.h"
#include "itkMattesMutualInformationImageToImageMetric.h"
#include "itkImageRegistrationMethod.h"
#include "itkImageFileWriter.h"


//#define DUMP_DEFORMATIONFIELD
//#define DUMP_GMI_FEATURE

#ifdef DUMP_DEFORMATIONFIELD
typedef struct FVECTOR3D
{
  float x;
  float y;
  float z;
}Fvector3d;


Fvector3d ***Fvector3dalloc3d(int i_size,int j_size,int k_size)
{
  Fvector3d ***array;
  int i,k;

  array=(Fvector3d ***) calloc(k_size,sizeof(Fvector3d **));

  for(k=0;k<k_size;k++)
    array[k]=(Fvector3d **) calloc(i_size,sizeof(Fvector3d *));

  for(k=0;k<k_size;k++)
    for(i=0;i<i_size;i++)
      array[k][i]=(Fvector3d *) calloc(j_size,sizeof(Fvector3d ));
	
  return(array);
}


void Fvector3dfree3d(Fvector3d ***array,int k_size,int i_size)
{
  int k,i;

  for(k=0;k<k_size;k++)
    for(i=0;i<i_size;i++)
      free(array[k][i]);

  for(k=0;k<k_size;k++)
    free(array[k]);

  free(array);
}
#endif

namespace itk
{

class CommandIterationUpdate : public itk::Command 
{
public:
  typedef  CommandIterationUpdate   Self;
  typedef  itk::Command             Superclass;
  typedef itk::SmartPointer<Self>  Pointer;
  itkNewMacro( Self );

protected:
  CommandIterationUpdate() {};
  itk::ProcessObject::Pointer m_Registration;
  
public:
  typedef itk::RegularStepGradientDescentOptimizer  OptimizerType;
  typedef   const OptimizerType   *    OptimizerPointer;

  void SetRegistration( itk::ProcessObject *p)
  {
    m_Registration = p;
  }
  
  void Execute(itk::Object *caller, const itk::EventObject & event)
  {
    Execute( (const itk::Object *)caller, event);
  }

  void Execute(const itk::Object * object, const itk::EventObject & event)
  {
    OptimizerPointer optimizer = 
      dynamic_cast< OptimizerPointer >( object );
    if( !(itk::IterationEvent().CheckEvent( &event )) )
      {
      return;
      }
      
    std::cout << optimizer->GetCurrentIteration() << "   ";
    std::cout << optimizer->GetCurrentStepLength() << "   ";
    std::cout << optimizer->GetValue() << std::endl;
    if (m_Registration)
      {
      m_Registration->UpdateProgress( 
        static_cast<double>(optimizer->GetCurrentIteration()) /
        static_cast<double>(optimizer->GetNumberOfIterations()));
      }
  }
};


//
// Constructor
//
template <class TInputImage, class TOutputImage>
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::HammerDeformableRegistrationImageFilter()
{ 
	RadiusType r;
	for(unsigned int j = 0; j < InputImageDimension; j++ )
	{
		r[j] = 0;
	}
	this->SetRadius(r);

	//set up the default parameter once the registration function is intialized
	m_IsBigVN = true;
	m_NeighborhoodStep = 3;
	m_DeformRate = 0.05;
	m_IsYoungBrain = false;
	m_SmoothFactor = 0.1;
	m_StartSearchPoint = 0;
	m_MaximumError = 0.1;
	m_MaximumKernelWidth = 30;
	m_InitialDeformationField = true;
	m_NumberOfIterations = 50;

	m_DeformRate = 0.05;
	m_PointMatchingThreshold = 0.8;
	m_SubvolumnSimilarityThreshold = 0.6;
	m_SearchRadius = 12;
	m_DfmSmthCoeff = 0.5;
	m_SmoothTime = 1;

	CreateSearchNeighbor(m_SmoothNeighbor, 1);

	for(int l=0; l<MAX_LEVEL; l++)
	{
		m_GuassianAtLevelSigma[0][l] = 0 ;
	}

	for(int s=1; s<MAX_SIGMA; s++)
	{
		for(int l=0; l<MAX_LEVEL; l++)
		{
			m_GuassianAtLevelSigma[s][l] = exp(-l*l/(2.0*s*s)) ;
		}
	}

	this->m_SubvolumeNeighborhood.clear();
	this->m_ModelDrivingPointDisplacement.clear();
	this->m_InverseDisplacement.clear();
	this->m_SearchNeighbor.clear();
	this->m_FixedImageDrivingVoxelQualification.resize(6);
	this->m_MovingImageDrivingVoxelQualification.resize(6);

	this->m_AffineTransform = AffineTransformType::New();
	this->m_AffineTransform->SetIdentity();

	this->m_AffineInitialization = true;
}


//
// Destructor
//
template <class TInputImage, class TOutputImage>
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::~HammerDeformableRegistrationImageFilter()
{

  // release memory if variables are not itk SmartPointer's

}


template <class TInputImage, class TOutputImage>
void
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::GenerateInputRequestedRegion() throw (InvalidRequestedRegionError)
{
  // call the superclass' implementation of this method
  Superclass::GenerateInputRequestedRegion();

}


template <class TInputImage, class TOutputImage>
void
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::GenerateData()
{
	int CurrentLevel;
	bool m_UpsampleDeformationField;

	// ************
	// initialize
	// 1. check is FixedImage and MovingImage are set
	if (!m_FixedImage)
	{
		itkExceptionMacro(<<"FixedImage is not present");
	}

	if( !m_MovingImage )
	{
		itkExceptionMacro(<<"MovingImage is not present");
	}

	// do linear initialization if asked to do so
	if (this->m_AffineInitialization)
	{
		//this->LinearInitialization();
	}


	this->AllocateOutputs();
	this->m_FinalOutputDeformFld = this->GetOutput( 0 );


	/*
	Step 1. Downsample the moving and fixed image
	Step 2. Calculate the GMIs w.r.t. to current resolution
	Step 3. Initialize the deforaiton field or upsapmle from the last iteration
	Step 4. Initialize the driving voxels
	Step 5. HAMMER_Mainloop
	Step 6. Restore the deformation field
	*/
	for(CurrentLevel=0;CurrentLevel<3;CurrentLevel++)
	{
		InputImagePointer                     m_CurrFixedImage;
		InputImagePointer                     m_CurrMovingImage;
		ImageAttributePointerType             m_CurrFixedImageAttribute;
		ImageAttributePointerType	            m_CurrMovingImageAttribute;
		DeformationFieldPointer               m_CurrDeformationField;
		typedef itk::HammerTissueAttributeVectorImageFilter<InputImageType, ImageAttributeType> AttributeFilterType;
		typename AttributeFilterType::Pointer attributeFilter = AttributeFilterType::New();
		double scale = 3;

		m_UpsampleDeformationField = true;
		if(CurrentLevel==0)
		{
			std::cout<<"Processing in low resolution"<<std::endl;
			m_UpsampleDeformationField = false;
			m_NumberOfIterations = 50;
      m_SearchRadius = 12;
			scale = 3;
		}
		else if(CurrentLevel==1)
		{
			std::cout<<"Processing in mid resolution"<<std::endl;
			scale = 4;
      m_SearchRadius = 10;
			m_NumberOfIterations = 30;
		}
		else if(CurrentLevel==2)
		{
			std::cout<<"Processing in high resolution"<<std::endl;
			scale = 7;
      m_SearchRadius = 8;
			m_NumberOfIterations = 20;
		}

		attributeFilter->SetBGValue( 0 );
		attributeFilter->SetGMValue( 150 );
		attributeFilter->SetWMValue( 250 );
		attributeFilter->SetVNValue( 50 );
		attributeFilter->SetCSFValue( 10 );

		attributeFilter->SetNumberOfThreads( 1 );
		attributeFilter->SetStrength( 1 );
		attributeFilter->SetScale( scale );

		
		//for downsample the image
		typedef itk::ResampleImageFilter<InputImageType,InputImageType> DownSampleFilterType;
		typename DownSampleFilterType::Pointer DownSampleFilter = DownSampleFilterType::New();

		typedef itk::AffineTransform< double, 3 >  TransformType;
		TransformType::Pointer transform = TransformType::New();
		transform->SetIdentity();
		DownSampleFilter->SetTransform( transform );
		typedef itk::NearestNeighborInterpolateImageFunction<InputImageType, double >  InterpolatorType;
		DownSampleFilter->SetDefaultPixelValue( 0 );
		DownSampleFilter->SetOutputSpacing( m_FixedImage->GetSpacing() );
		DownSampleFilter->SetOutputOrigin( m_FixedImage->GetOrigin() );

		DownSampleFilter->SetOutputDirection( m_FixedImage->GetDirection() );

		//for upsample the deformation field
		typedef itk::VectorResampleImageFilter<OutputImageType, OutputImageType >  DeformationUpSampleFilterType;

		typename DeformationUpSampleFilterType::Pointer DeformationUpSampleFilter = DeformationUpSampleFilterType::New();
		typedef itk::VectorLinearInterpolateImageFunction< 
			OutputImageType, double >  DeformationFieldInterpolatorType;
		typename DeformationFieldInterpolatorType::Pointer DeformationFieldInterpolator = DeformationFieldInterpolatorType::New();
		DeformationUpSampleFilter->SetInterpolator( DeformationFieldInterpolator );
		typedef itk::IdentityTransform< double, InputImageDimension >  IdentityTransformType;
		typename IdentityTransformType::Pointer IdentityTransform = IdentityTransformType::New();
		DeformationUpSampleFilter->SetTransform( IdentityTransform );
		DeformationVectorType BlackValue;
		BlackValue.Fill(0);
		DeformationUpSampleFilter->SetDefaultPixelValue( BlackValue );
		DeformationUpSampleFilter->SetOutputSpacing( m_FixedImage->GetSpacing() );
		DeformationUpSampleFilter->SetOutputOrigin( m_FixedImage->GetOrigin() );
		DeformationUpSampleFilter->SetOutputDirection(m_FixedImage->GetDirection());

		//Step 1. Downsample the moving and fixed image
		SizeType FullSize = m_FixedImage->GetLargestPossibleRegion().GetSize();
		for(int s=0;s<InputImageDimension;s++)
			FullSize[s] = (((int)FullSize[s])>>(2-CurrentLevel));
		DownSampleFilter->SetSize( FullSize );

		// modify spacing accordingly
		typename InputImageType::SpacingType spacing = m_FixedImage->GetSpacing();
		double factor = pow(2.0, 2.0-CurrentLevel);
		for (int k = 0; k < 3; k++)
		{
			spacing[k] *= factor;
		}
		DownSampleFilter->SetOutputSpacing( spacing );

		std::cout << "Size: " << FullSize << std::endl;
		std::cout << "Spacing: " << spacing << std::endl;

		DownSampleFilter->SetOutputOrigin( m_FixedImage->GetOrigin() );

		DownSampleFilter->SetInput(m_FixedImage);		
		DownSampleFilter->Update();
		m_CurrFixedImage = DownSampleFilter->GetOutput();
		m_CurrFixedImage->DisconnectPipeline();

		//m_AffineTransform->Print( std::cout );
		DownSampleFilter->SetInput(m_MovingImage);
		DownSampleFilter->SetTransform( m_AffineTransform );
		DownSampleFilter->Update();
		m_CurrMovingImage = DownSampleFilter->GetOutput();
		m_CurrMovingImage->DisconnectPipeline();

		//std::cout << m_CurrFixedImage->GetLargestPossibleRegion() << std::endl;
		//std::cout << m_CurrMovingImage->GetLargestPossibleRegion() << std::endl;

		//m_CurrMovingImage->Print( std::cout );
#ifdef DEBUG_MODE
		typename itk::ImageFileWriter<InputImageType>::Pointer dummywriter = itk::ImageFileWriter<InputImageType>::New();
		dummywriter->SetFileName( "fixed.hdr" );
		dummywriter->SetInput( m_CurrFixedImage );
		dummywriter->Update();

		dummywriter->SetFileName( "moving.hdr" );
		dummywriter->SetInput( m_CurrMovingImage );
		dummywriter->Update();
#endif
		//Step 2. Calculate the GMIs w.r.t. to current resolution
		//std::cout << m_CurrFixedImage->GetLargestPossibleRegion() << std::endl;
		attributeFilter->SetInput( m_CurrFixedImage );
		attributeFilter->Update();
		//std::cout << m_CurrFixedImage->GetLargestPossibleRegion() << std::endl;
		m_CurrFixedImageAttribute =  attributeFilter->GetOutput() ;
		m_CurrFixedImageAttribute->DisconnectPipeline();

		attributeFilter->SetInput( m_CurrMovingImage );
		attributeFilter->Update();
		m_CurrMovingImageAttribute = attributeFilter->GetOutput();
		m_CurrMovingImageAttribute->DisconnectPipeline();

		//dump the image attribute
#ifdef DUMP_GMI_FEATURE
		InputImagePointer dummyImage = InputImageType::New();
		dummyImage->CopyInformation(m_CurrFixedImage);
		dummyImage->SetRegions(m_CurrFixedImage->GetLargestPossibleRegion());
		dummyImage->Allocate();
		InputImageIteratorType dummy_image_iterator (dummyImage, dummyImage->GetLargestPossibleRegion());
		AttributeImageIteratorType dummy_attribute_Iterator (m_CurrFixedImageAttribute, m_CurrFixedImageAttribute->GetLargestPossibleRegion());
		for(dummy_attribute_Iterator.GoToBegin(),dummy_image_iterator.GoToBegin();!dummy_image_iterator.IsAtEnd();++dummy_image_iterator,++dummy_attribute_Iterator)
		{
			dummy_image_iterator.Set(dummy_attribute_Iterator.Get()[1]);
		}
		dummywriter->SetFileName( "Tiss.hdr" );
		dummywriter->SetInput(dummyImage);
		dummywriter->Update();
		for(dummy_attribute_Iterator.GoToBegin(),dummy_image_iterator.GoToBegin();!dummy_image_iterator.IsAtEnd();++dummy_image_iterator,++dummy_attribute_Iterator)
		{
			dummy_image_iterator.Set(dummy_attribute_Iterator.Get()[0]);
		}
		dummywriter->SetFileName( "Edge.hdr" );
		dummywriter->SetInput(dummyImage);
		dummywriter->Update();
		for(dummy_attribute_Iterator.GoToBegin(),dummy_image_iterator.GoToBegin();!dummy_image_iterator.IsAtEnd();++dummy_image_iterator,++dummy_attribute_Iterator)
		{
			dummy_image_iterator.Set(dummy_attribute_Iterator.Get()[2]);
		}
		dummywriter->SetFileName( "Geom.hdr" );
		dummywriter->SetInput(dummyImage);
		dummywriter->Update();
		for(dummy_attribute_Iterator.GoToBegin(),dummy_image_iterator.GoToBegin();!dummy_image_iterator.IsAtEnd();++dummy_image_iterator,++dummy_attribute_Iterator)
		{
			dummy_image_iterator.Set(dummy_attribute_Iterator.Get()[3]);
		}
		dummywriter->SetFileName( "Vln.hdr" );
		dummywriter->SetInput(dummyImage);
		dummywriter->Update();
		for(dummy_attribute_Iterator.GoToBegin(),dummy_image_iterator.GoToBegin();!dummy_image_iterator.IsAtEnd();++dummy_image_iterator,++dummy_attribute_Iterator)
		{
			dummy_image_iterator.Set(dummy_attribute_Iterator.Get()[4]);
		}
		dummywriter->SetFileName( "CSF.hdr" );
		dummywriter->SetInput(dummyImage);
		dummywriter->Update();
#endif

		std::cout << "Step 3. Initialize the deformaiton field or upsapmle from the last iteration\n";

		//Step 3. Initialize the deformaiton field or upsapmle from the last iteration
		RegionType dummyRegion = m_CurrFixedImage->GetLargestPossibleRegion();
		std::cout << dummyRegion << std::endl;
		m_CurrDeformationField = DeformationFieldType::New();
		m_CurrDeformationField->CopyInformation( m_CurrFixedImage );
		m_CurrDeformationField->SetRegions(dummyRegion);
		m_CurrDeformationField->Allocate();
		if(m_UpsampleDeformationField == true && m_InitialDeformationField == true)
		{
			//upsample the deformation field from last resolution
			printf("Upsample...\n");
			DeformationFieldPointer m_CurrDeformFld_Copy = DeformationFieldType::New();
			RegionType dummydeformfieldregion = m_CurrFixedImage->GetLargestPossibleRegion();
			SizeType dummysize=dummydeformfieldregion.GetSize();
			for(int s=0;s<InputImageDimension;s++)
				dummysize[s]/=2;
			dummydeformfieldregion.SetSize(dummysize);
			m_CurrDeformFld_Copy->CopyInformation(m_CurrFixedImage);
			m_CurrDeformFld_Copy->SetRegions(dummydeformfieldregion);
			DeformationUpSampleFilter->SetSize(FullSize);
			m_CurrDeformFld_Copy->Allocate();
			DeformationFieldIteratorType DeformFld_Iter_Copy(m_CurrDeformFld_Copy, m_CurrDeformFld_Copy->GetRequestedRegion());
			DeformationFieldIteratorType DeformFld_Iter(m_FinalOutputDeformFld, m_FinalOutputDeformFld->GetRequestedRegion());
			for(DeformFld_Iter.GoToBegin(),DeformFld_Iter_Copy.GoToBegin();!DeformFld_Iter_Copy.IsAtEnd();++DeformFld_Iter,++DeformFld_Iter_Copy)
			{	
				DeformFld_Iter_Copy.Set(DeformFld_Iter.Get());
			}

			DeformationUpSampleFilter->SetInput(m_CurrDeformFld_Copy);
			DeformationUpSampleFilter->Update();
			m_CurrDeformationField = DeformationUpSampleFilter->GetOutput();
			m_CurrDeformationField->DisconnectPipeline();
		}
		else
		{
			//Fill 0
			printf("Initilize with zeros...\n");
			DeformationFieldIteratorType DeformFld_Iter(m_CurrDeformationField, m_CurrDeformationField->GetRequestedRegion());
			DeformationVectorType Deform_Zero;
			for(int k=0;k<InputImageDimension;k++)
				Deform_Zero[k] = 0;
			for(DeformFld_Iter.GoToBegin();!DeformFld_Iter.IsAtEnd();++DeformFld_Iter)
			{
				DeformFld_Iter.Set(Deform_Zero);
			}
		}

		//Step 4. Initialize the driving voxels
		m_FixedImageDrivingVoxelQualification[0] = (0.9*255);
		m_FixedImageDrivingVoxelQualification[1] = (0.4*255);
		m_FixedImageDrivingVoxelQualification[2] = (0.4*255/1.1);
		m_FixedImageDrivingVoxelQualification[3] = (255/12);
		m_FixedImageDrivingVoxelQualification[4] = (0);
		m_FixedImageDrivingVoxelQualification[5] = (255/4);

		IdentifyDrivingVoxelsInFixedImage(m_CurrFixedImageAttribute, m_FixedImageDrivingVoxelQualification);
		std::cout<<"Driving voxel in model image is "<<m_PickedPointsOnFixedImage.size()<<std::endl;


		m_MovingImageDrivingVoxelQualification[0] = (0.9*255);
		m_MovingImageDrivingVoxelQualification[1] = (0.4*255);
		m_MovingImageDrivingVoxelQualification[2] = (0.4*255/1.1);
		m_MovingImageDrivingVoxelQualification[3] = (255/4);
		m_MovingImageDrivingVoxelQualification[4] = (0);
		m_MovingImageDrivingVoxelQualification[5] = (255/4);

		IdentifyDrivingVoxelsInMovingImage(m_CurrMovingImageAttribute, m_MovingImageDrivingVoxelQualification);
		std::cout<<"Driving voxel in subject image is "<<m_PickedPointsOnMovingImage.size()<<std::endl;
		m_InverseDisplacement.clear();
		m_ModelDrivingPointDisplacement.clear();
		m_ModelAttributeVector.clear();
		m_InverseDisplacement.resize(m_PickedPointsOnFixedImage.size());
		m_ModelDrivingPointDisplacement.resize(m_PickedPointsOnFixedImage.size());

		for(unsigned int k=0;k<m_PickedPointsOnFixedImage.size();k++)
		{
			//DeformationVectorType dfm = m_CurrDeformationField->GetPixel(m_PickedPointsOnFixedImage[k]);
			for(int s=0;s<InputImageDimension;s++)
			{	
				m_ModelDrivingPointDisplacement[k][s] = m_PickedPointsOnFixedImage[k][s];// + dfm[s];
				m_InverseDisplacement[k][s] = 0;
			}
			AttributeVectorType Feature = m_CurrFixedImageAttribute->GetPixel(m_PickedPointsOnFixedImage[k]);
			m_ModelAttributeVector.push_back(Feature);
		}

		//Step 5. HAMMER_Mainloop
		HAMMERMainLoop(m_CurrFixedImageAttribute, m_CurrMovingImageAttribute, m_CurrDeformationField);
		

		//Step 6. Restore the deformation field		
		DeformationFieldIteratorType DeformFld_Iter(m_CurrDeformationField, m_CurrDeformationField->GetRequestedRegion());
		DeformationFieldIteratorType DeformFld_Iter_Copy(m_FinalOutputDeformFld, m_FinalOutputDeformFld->GetRequestedRegion());
		for(DeformFld_Iter.GoToBegin(),DeformFld_Iter_Copy.GoToBegin();!DeformFld_Iter.IsAtEnd();++DeformFld_Iter,++DeformFld_Iter_Copy)
		{
			DeformFld_Iter_Copy.Set(DeformFld_Iter.Get());
		}
	}//end of Cur_Level

#ifdef DEBUG_MODE
	printf("Generate the result\n");
	typedef itk::ImageFileWriter<InputImageType> ImageWriterType;
	ImageWriterType::Pointer writer = ImageWriterType::New();
	InputImagePointer WarpedImage = InputImageType::New();
	WarpedImage->CopyInformation(m_FixedImage);
	WarpedImage->SetRegions(m_FixedImage->GetLargestPossibleRegion());
	WarpedImage->Allocate();
	//warp the moving image
	DeformationFieldIteratorType DeformFld_Iter_Dump(m_FinalOutputDeformFld, m_FinalOutputDeformFld->GetRequestedRegion());
	InputImageIteratorType WarpedImage_Iter(WarpedImage, WarpedImage->GetLargestPossibleRegion());
	DeformFld_Iter_Dump.GoToBegin();
	WarpedImage_Iter.GoToBegin();
	SizeType ImageSize = m_FinalOutputDeformFld->GetRequestedRegion().GetSize();
	int x_size = ImageSize[0];
	int y_size = ImageSize[1];
	int z_size = ImageSize[2];
	IndexType index, targetindex;
	InputPixelType MovingValue; 
	DeformationVectorType dumpdfm;
	for(int k=0;k<z_size;k++)
		for(int i=0;i<x_size;i++)
			for(int j=0;j<y_size;j++)
			{				
				dumpdfm = DeformFld_Iter_Dump.Get();
				index = DeformFld_Iter_Dump.GetIndex();
				for(int s=0;s<InputImageDimension;s++)
				{
					targetindex[s] = index[s]+(int)(dumpdfm[s]+0.5);
					if(targetindex[s]<0)
						targetindex[s]=0;
					if(targetindex[s]>=ImageSize[s])
						targetindex[s]=ImageSize[s]-1;
				}
				MovingValue = m_MovingImage->GetPixel(targetindex);
				WarpedImage_Iter.Set(MovingValue);
				//Deform[k][i][j].x = dumpdfm[1];
				//Deform[k][i][j].y = dumpdfm[0];
				//Deform[k][i][j].z = dumpdfm[2];
				++DeformFld_Iter_Dump;
				++WarpedImage_Iter;
			}

	writer->SetFileName("warpresult.hdr");
	writer->SetInput(WarpedImage);
	writer->Update();
#endif
	return;
}

/**
* Standard "PrintSelf" method
*/
template <class TInputImage, class TOutputImage>
void
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::PrintSelf(std::ostream& os, Indent indent) const
{
	Superclass::PrintSelf( os, indent );
}

/**
* Create the neighborhood in point matching
**/
template <class TInputImage, class TOutputImage>
void 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::CreatePointMatchingNeighbor(IndexArrayType &Neighbor, int Radius)
{
	//CalculateNeighborhoodbyIncreasingRadius(m_PointMatchingNeighborhood, Radius);
	Neighbor.clear();
	CreateSearchNeighbor(Neighbor, Radius);
}

/**
* Create the neighborhood in subvolumn deform
*/
template <class TInputImage, class TOutputImage>
void 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::CreateSubvolumnNeighbor(IndexArrayType &Neighbor, int Radius)
{
	Neighbor.clear();
	CalculateNeighborhoodbyIncreasingRadius(Neighbor, Radius);	
}

/**
* Create the neighbor from center to outside
**/
template <class TInputImage, class TOutputImage>
void
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::CalculateNeighborhoodbyIncreasingRadius(IndexArrayType &Neighbor, int Radius)  throw (InvalidRequestedRegionError) 
{
	// pad input requested region by 1, so the simple edge detection
	// works on the entire image domain

	IndexType dummyIdx;
	int x, y, z;
	int Half_Radius = Radius/2;

	for(int s=0;s<InputImageDimension;s++)
		dummyIdx[s] = 0;
	Neighbor.push_back(dummyIdx);
	for(int r=1;r<=Half_Radius;r++)
	{
		for(z=-r; z<=r; z+=2*r)
		{  
			for(x=-r; x<=r; x++)
			{
				for(y=-r; y<=r; y++)
				{
					dummyIdx[1] = x;
					dummyIdx[0] = y;
					dummyIdx[2] = z;
					//std::cout<<dummyIdx<<std::endl;
					Neighbor.push_back(dummyIdx);
				}
			}
		}

		for(x=-r; x<=r; x+=2*r)
		{
			for(z=-r+1; z<=r-1; z++) 
			{
				for(y=-r; y<=r; y++)
				{
					dummyIdx[1] = x;
					dummyIdx[0] = y;
					dummyIdx[2] = z;
					Neighbor.push_back(dummyIdx);
				}
			}
		}

		for(y=-r; y<=r; y+=2*r)
		{
			for(z=-r+1; z<=r-1; z++) 
			{
				for(x=-r+1; x<=r-1; x++) 
				{
					dummyIdx[1] = x;
					dummyIdx[0] = y;
					dummyIdx[2] = z;
					Neighbor.push_back(dummyIdx);
				}
			}
		}
	}
}

/**
*  Create the search neighborhood
**/
template <class TInputImage, class TOutputImage>
void
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::CreateSearchNeighbor(IndexArrayType &Neighbor, int Radius)
{  
	SpacingType inputSpacing;
	Size<InputImageDimension> sphereRadius;
	for (int i = 0; i < InputImageDimension; ++i)
	{
		//sphereRadius[i] = static_cast<unsigned long>( this->m_Scale/inputSpacing[i] );
		sphereRadius[i] = static_cast<unsigned long>(Radius);
	}

	// compute spherical neighborhood for geometrical attribute
	// computation
	InputImagePointer dummyImage = InputImageType::New();
	RegionType dummyRegion;
	PointType dummyOrigin;
	IndexType dummyStart;
	SizeType dummySize;

	for(int s=0;s<InputImageDimension;s++)
		inputSpacing[s] = 1;
	dummyImage->SetSpacing( inputSpacing );
	for (int k = 0; k < InputImageDimension; k++)
	{
		dummySize[k] = sphereRadius[k]+sphereRadius[k]+1;
		dummyStart[k] = -sphereRadius[k];
		dummyOrigin[k] = 0;
	}
	dummyRegion.SetIndex( dummyStart );
	dummyRegion.SetSize( dummySize );
	dummyImage->SetRegions( dummyRegion );
	dummyImage->SetOrigin( dummyOrigin );

	itk::ImageRegionIteratorWithIndex<InputImageType> it( dummyImage, dummyRegion );
	for (it.GoToBegin(); !it.IsAtEnd(); ++it)
	{
		Neighbor.push_back(it.GetIndex());                
	}  
}


/**
 * the metric function
**/
template <class TInputImage, class TOutputImage>
float
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::SimilarityBetweenTwoImgAttribute(AttributeVectorType Template_Feature, AttributeVectorType Subject_Feature) const
{
	return Template_Feature.ComputeSimilarity(Subject_Feature);
}

/**
 * compute the magnitude of vector
**/
template <class TInputImage, class TOutputImage>
float 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>  
::ComputeVectorMagnitude(DeformationVectorType Deform_Vector) const
{
	return Deform_Vector.GetNorm();
}

/**
 * the core function which determine the correspondence on model driving voxel array
*/
template <class TInputImage, class TOutputImage>
float 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::DetermineCorrespondenceOnOneDrivingVoxel(ImageAttributePointerType &FixedAttributeImage, ImageAttributePointerType &MovingAttributeImage, DeformationFieldPointer &DeformFld, int DrivingPointIndex, DeformationFieldPointer DeformFld_Last, DeformationVectorType &DeformationUpdate, int SearchRadius, int Step) const
{
	int i, j, k;
	float PointSimilarity, NeighborhoodSimilarity, MaxSimilarityDegree;
	float DisplacementMagnitude;
	RegionType dummyRegion = FixedAttributeImage->GetLargestPossibleRegion();
	AttributeVectorType TemplateFeature, SubjectFeature;
	DeformationVectorType DeformationOnParticularPoint, TentativeWarp;
	DeformationUpdate.Fill( 0 );
	float MinDistance, MinDistThresh;


	int SmplStep =  m_SubvolumeNeighborhood.size()/20;
	IndexType ImageIndex;                             
	
	ImageIndex = m_PickedPointsOnFixedImage[DrivingPointIndex];

	//Step 1: check whether the the input point has the inverse force
	float Displacement_Magnitude = ComputeVectorMagnitude(m_InverseDisplacement[DrivingPointIndex]);
	if(Displacement_Magnitude>0 && m_IterationRatio<=ITER_THRD)
	{    
		DeformationVectorType CurrentDeformation, PreviousDeformation;
		CurrentDeformation = DeformFld->GetPixel(ImageIndex);
		PreviousDeformation = DeformFld_Last->GetPixel(ImageIndex);
		
		DeformationUpdate = PreviousDeformation - CurrentDeformation + m_InverseDisplacement[DrivingPointIndex]*(1.0+(1.0 + m_IterationRatio))/2.0;

		DisplacementMagnitude = ComputeVectorMagnitude(DeformationUpdate);
		if(DisplacementMagnitude > (m_SubvolumeRadius>>1))
		{
			DeformationUpdate = DeformationUpdate/DisplacementMagnitude*(m_SubvolumeRadius>>1);
		}
		return 10000.0 ;
	}

	//Step 2: find the correspondence of particular point
	MaxSimilarityDegree = 0;
	TentativeWarp.Fill(0);
	TemplateFeature = FixedAttributeImage->GetPixel(ImageIndex);
	DeformationOnParticularPoint = DeformFld->GetPixel(ImageIndex);
	MinDistThresh = 10000.0;

	for(i=-SearchRadius;i<=SearchRadius;i+=Step)
		for(j=-SearchRadius;j<=SearchRadius;j+=Step)
			for(k=-SearchRadius;k<=SearchRadius;k+=Step)
			{
				IndexType SubjIdx;
				SubjIdx[0] = int(DeformationOnParticularPoint[0] + ImageIndex[0] + j + 0.5);
				SubjIdx[1] = int(DeformationOnParticularPoint[1] + ImageIndex[1] + i + 0.5);
				SubjIdx[2] = int(DeformationOnParticularPoint[2] + ImageIndex[2] + k + 0.5);

				if(!dummyRegion.IsInside(SubjIdx))
					continue;
				SubjectFeature = MovingAttributeImage->GetPixel(SubjIdx);
				PointSimilarity = SimilarityBetweenTwoImgAttribute(TemplateFeature, SubjectFeature);
				//Step 2.2: compare the similarity between two neighborhood
				if(PointSimilarity>m_PointMatchingThreshold || (m_IsBigVN== true && TemplateFeature.GetVentricleVolume()>0 && SubjectFeature.GetVentricleVolume()>0))
				{      
					TentativeWarp[0] = j;
					TentativeWarp[1] = i;
					TentativeWarp[2] = k;
					NeighborhoodSimilarity = SubVolumnMatching(FixedAttributeImage, MovingAttributeImage, DeformFld, ImageIndex, TentativeWarp, m_SubvolumeNeighborhood, SmplStep, &MinDistance, MinDistThresh);
					if(NeighborhoodSimilarity>MaxSimilarityDegree)
					{      
						DeformationUpdate = TentativeWarp;
						MaxSimilarityDegree = NeighborhoodSimilarity;
						MinDistThresh = MinDistance;
					}

				}
			}
	return ( MaxSimilarityDegree );
}
template <class TInputImage, class TOutputImage>
float 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::SubVolumnMatching(ImageAttributePointerType &FixedAttributeImage, ImageAttributePointerType &MovingAttributeImage, DeformationFieldPointer &DeformFld, IndexType &ImageIndex, DeformationVectorType TentativeWarp, IndexArrayType CertainNeighborhood, int NeighborhoodStep, float *MinDist, float MinDist_Threshold) const
{
	float NeighborhoodSimilarity, CurrentSimilarity, DisplacementMagnitude;
	int RealSize = 0;
	AttributeVectorType SubvolumnTemplateFeature, SubvolumnSubjectFeature;
	IndexType MdlIdx, SubjIdx;
	DeformationVectorType DeformationOnParticularPoint, CurrLocation;
	RegionType dummyFixedImageRegion, dummyMovingImageRegion;
	float DistSeg, Overall_Count;

	NeighborhoodSimilarity = 0;
	DistSeg = 0;
	Overall_Count = CertainNeighborhood.size()/NeighborhoodStep;
	dummyFixedImageRegion = FixedAttributeImage->GetLargestPossibleRegion();
	dummyMovingImageRegion = MovingAttributeImage->GetLargestPossibleRegion();
	for(unsigned int SubvolumnIter=0; SubvolumnIter < CertainNeighborhood.size(); SubvolumnIter += NeighborhoodStep)
	{
		DisplacementMagnitude = 0;
		
		for(int s=0;s<InputImageDimension;s++)
		{
			MdlIdx[s] = ImageIndex[s] + CertainNeighborhood[SubvolumnIter][s];

			DisplacementMagnitude += abs(CertainNeighborhood[SubvolumnIter][s]);
		}
		DisplacementMagnitude /= InputImageDimension;
		if(!dummyFixedImageRegion.IsInside(MdlIdx))
		{
			continue;
		}

		SubvolumnTemplateFeature = FixedAttributeImage->GetPixel(MdlIdx);        
		DeformationOnParticularPoint = DeformFld->GetPixel(MdlIdx);

		for(int s=0;s<InputImageDimension;s++)
		{
			CurrLocation[s] =  MdlIdx[s]+ DeformationOnParticularPoint[s] + TentativeWarp[s]*m_GuassianAtLevelSigma[m_SubvolumeRadius][(int)DisplacementMagnitude];
			SubjIdx[s] = (int)(CurrLocation[s] + 0.5);
		}

		if(dummyMovingImageRegion.IsInside(SubjIdx)==false)
		{
			CurrentSimilarity = 0;
		}
		else
		{
			SubvolumnSubjectFeature = MovingAttributeImage->GetPixel(SubjIdx);

			CurrentSimilarity = SimilarityBetweenTwoImgAttribute(SubvolumnSubjectFeature, SubvolumnTemplateFeature);
			if(CurrentSimilarity < 0.6 && m_IsYoungBrain == true && SubvolumnTemplateFeature.GetEdge() == 0)
			{
				CurrentSimilarity = 0;
			}

			RealSize += 1.0;
			if( SubvolumnTemplateFeature.GetEdge()>0 ) /* June 6, 2001*/
			{        CurrentSimilarity *= 1.2 ;
			RealSize += (1.2-1.0) ;
			}			
		}
		DistSeg += (1-CurrentSimilarity)/Overall_Count;
		if( DistSeg>MinDist_Threshold) 
			break ; /* no need to continue on this selected deformation */
		NeighborhoodSimilarity += CurrentSimilarity;		
	}
	if(RealSize>0)
	{
		NeighborhoodSimilarity /= RealSize;
	}
	*MinDist = DistSeg;
	return NeighborhoodSimilarity;  
}


template <class TInputImage, class TOutputImage>
void 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::FindingInverseForceFromSubject(ImageAttributePointerType &FixedAttributeImage, ImageAttributePointerType &MovingAttributeImage, DeformationFieldPointer &DeformFld, int SearchRadius) const
{
	typedef itk::Image<int, InputImageDimension> ModelDrivingPointImageType;
	typename ModelDrivingPointImageType::Pointer ModelDrivingPointImage = ModelDrivingPointImageType::New();
	float MinDistThresh;
	RegionType dummyRegion;
	unsigned int PointID;

	int s, SmplStep;
	IndexType SubjIndex;
	AttributeVectorType TemplateFeature, SubjectFeature;
	DeformationVectorType TentativeWarp, MaxDeform;


	unsigned int MdlPickedPointNum;
	std::vector<float> Multiple;
	int incre;

	incre = 2*SearchRadius/6 ; 
	if(incre<1)
		incre=1 ;  

	dummyRegion = FixedAttributeImage->GetLargestPossibleRegion();
	ModelDrivingPointImage->SetRegions(dummyRegion);
	ModelDrivingPointImage->Allocate();

	typedef itk::ImageRegionIterator<ModelDrivingPointImageType> IteratorType;
	IteratorType Itor(ModelDrivingPointImage, dummyRegion);
	for(Itor.GoToBegin();!Itor.IsAtEnd();++Itor)
	{
		Itor.Set(-1);
	}

	SmplStep = m_SubvolumeNeighborhood.size()/20;

	MdlPickedPointNum = m_PickedPointsOnFixedImage.size();
	Multiple.resize(MdlPickedPointNum);
	for( PointID=0; PointID < MdlPickedPointNum; PointID++)
	{
		
		for(s=0;s<InputImageDimension;s++)
		{
			SubjIndex[s] = (int)(m_ModelDrivingPointDisplacement[PointID][s]);
			m_InverseDisplacement[PointID][s] = 0;
		}
		if(dummyRegion.IsInside(SubjIndex)==false)
			continue;
		ModelDrivingPointImage->SetPixel(SubjIndex, PointID);
		Multiple[PointID]= 0;
	}

	unsigned int TotalSamples = m_PickedPointsOnFixedImage.size()/2;
	if(TotalSamples>m_PickedPointsOnMovingImage.size()) 
		TotalSamples = m_PickedPointsOnMovingImage.size() ;
	if(TotalSamples>20000)  
		TotalSamples = 20000 ;
	float StepDesign = (float)(m_PickedPointsOnMovingImage.size())/(float)TotalSamples ; 
	MinDistThresh = 10000.0;

	int PointMatched = 0;
	PointID = 0;

	for(float l=0; PointID < m_PickedPointsOnMovingImage.size(); l += StepDesign, PointID=(int)l)
	{    
		float MaxSimilarityDegree = 0;
		MaxDeform.Fill(0);
		SubjectFeature = MovingAttributeImage->GetPixel(m_PickedPointsOnMovingImage[PointID]);

		int dfm_s = 0;    
		for(int i=-SearchRadius;i<=SearchRadius;i+=incre)
		{
			for(int j=-SearchRadius;j<=SearchRadius;j+=incre)
			{
				for(int k=-SearchRadius;k<=SearchRadius;k+=incre)
				{
					SubjIndex[0] = static_cast<int>(m_PickedPointsOnMovingImage[PointID][0]) + j;
					SubjIndex[1] = static_cast<int>(m_PickedPointsOnMovingImage[PointID][1]) + i;
					SubjIndex[2] = static_cast<int>(m_PickedPointsOnMovingImage[PointID][2]) + k;

					if(dummyRegion.IsInside(SubjIndex) == false)
						continue;

					int MdlPickedPointID = ModelDrivingPointImage->GetPixel(SubjIndex);
					if( MdlPickedPointID >= 0)
					{
						//TemplateFeature = m_FixedAttributeImage->GetPixel(MdlIndex);
						if(MdlPickedPointID==70 && PointID==177)
							printf("");
						TemplateFeature = m_ModelAttributeVector[MdlPickedPointID];
						float PointSimilarity = SimilarityBetweenTwoImgAttribute(TemplateFeature, SubjectFeature);
						if(PointSimilarity>m_PointMatchingThreshold)
						{
							TentativeWarp[0] = -j;
							TentativeWarp[1] = -i;
							TentativeWarp[2] = -k;
							typename InputImageType::IndexType MdlIndex = m_PickedPointsOnFixedImage[MdlPickedPointID];
							float MinDistance = 0;
							float SubvolumnSimilarity = SubVolumnMatching(FixedAttributeImage, MovingAttributeImage, DeformFld, MdlIndex, TentativeWarp, m_SubvolumeNeighborhood, SmplStep, &MinDistance, MinDistThresh);

							if(SubvolumnSimilarity>MaxSimilarityDegree)
							{
								MaxSimilarityDegree = SubvolumnSimilarity;
								for(s=0;s<InputImageDimension;s++)
								{
									MaxDeform[s] = TentativeWarp[s];
								}
								dfm_s = MdlPickedPointID;
							}
						}
					}
				}
			}
		}

		if(MaxSimilarityDegree>m_SubvolumnSimilarityThreshold)
		{				
			PointMatched++;
			for(s=0;s<InputImageDimension;s++)
				m_InverseDisplacement[dfm_s][s] += MaxDeform[s];
			Multiple[dfm_s]+=1.0;
		}
	}
	std::cout<<PointMatched<<" Matched."<<std::endl;

	int PickupNum = 0;
	for(PointID=0;PointID<MdlPickedPointNum;PointID++)
	{
		if(Multiple[PointID]>0)
		{			
			for(s=0;s<InputImageDimension;s++)
			{
				m_InverseDisplacement[PointID][s] /= Multiple[PointID];
			}		
			PickupNum++;
		}		
	}
	std::cout<<"Inverse force ratio = "<<PickupNum<<"/"<<m_InverseDisplacement.size()<<std::endl;
}

template <class TInputImage, class TOutputImage>
void 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::DisseminateDeformation(DeformationFieldPointer &DeformFld, const int &DrivingPointIndex, DeformationVectorType TentativeWarp, IndexArrayType CertainNeighborhood, int NeighborhoodSize, int GaussianSigma)
{
	int PointID, s;
	float CenterRate, Delta;
	IndexType CenterPointIndex, StudyPointIndex, SurroundingPointIndex;
	float Distance, Weight;
	DeformationVectorType MeanDeform, DeformationOnCertainPoint, DeformationOnCenterPoint, DeformUpdate;
	RegionType dummyRegion;

	dummyRegion = DeformFld->GetLargestPossibleRegion();
	CenterRate = m_SmoothFactor; 
	Delta = 0.005*m_IterationRatio ; /* June 6, 2001*/
	

	CenterPointIndex = m_PickedPointsOnFixedImage[DrivingPointIndex];
	
	for(PointID=0;PointID<NeighborhoodSize;PointID++)
	{    
		Distance = 0;		
		for(s=0;s<InputImageDimension;s++)
		{
			MeanDeform[s] = 0;
			StudyPointIndex[s] = CenterPointIndex[s] + CertainNeighborhood[PointID][s];
			Distance += abs(CertainNeighborhood[PointID][s]);
		}
		
		Distance /= InputImageDimension;
		if(!dummyRegion.IsInside(StudyPointIndex))
		{
			continue;
		}

		Weight = m_GuassianAtLevelSigma[GaussianSigma][(int)Distance];
		float TempNum = 0;
		for(int n=1;n<27;n++) //need to find better way 
		{
			for(s=0;s<InputImageDimension;s++)
			{
				SurroundingPointIndex[s] = StudyPointIndex[s] + m_SubvolumeNeighborhood[n][s];
			}

			if(!dummyRegion.IsInside(SurroundingPointIndex))
			{
				continue;
			}

			DeformationOnCertainPoint = DeformFld->GetPixel(SurroundingPointIndex);
			for(s=0;s<InputImageDimension;s++)
			{
				MeanDeform[s] += DeformationOnCertainPoint[s];
			}

			TempNum += 1.0;
		}
		if(TempNum>0)
		{
			for(s=0;s<InputImageDimension;s++)
				MeanDeform[s] /= TempNum;
		}
		DeformationOnCenterPoint = DeformFld->GetPixel(StudyPointIndex);
		
		for(s=0;s<InputImageDimension;s++)
		{
			MeanDeform[s] = (MeanDeform[s]-DeformationOnCenterPoint[s])*CenterRate;
			DeformUpdate[s] = (MeanDeform[s]+TentativeWarp[s]*Weight)*(m_DeformRate+Delta);
			DeformationOnCenterPoint[s] += DeformUpdate[s];
		}
		
		DeformFld->SetPixel(StudyPointIndex, DeformationOnCenterPoint);
	}
}

template <class TInputImage, class TOutputImage>
void 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::FitGlobalAffineTransform(DeforamtionVectorArrayType CurrentDeformationOnDrivingPoint, DeforamtionVectorArrayType PreviousDeformationOnDrivingPoint)
{
	vnl_matrix<double> ModelMatrix, ObjectMatrix, FittingMatrix, TmpM1, TmpM2, InverseMatrix;

	/* create  metrices*/
	ModelMatrix.set_size(4, CurrentDeformationOnDrivingPoint.size());
	ObjectMatrix.set_size(4, CurrentDeformationOnDrivingPoint.size());
	FittingMatrix.set_size(4, CurrentDeformationOnDrivingPoint.size());

	/* get ModelMatrix->data[4][MdlVer_Num] */
	for(unsigned int j=0; j<CurrentDeformationOnDrivingPoint.size(); j++)
	{
		//std::cout<<CurrentDeformationOnDrivingPoint[j]<<std::endl;
		ModelMatrix[0][j] = PreviousDeformationOnDrivingPoint[j][1] ;
		ModelMatrix[1][j] = PreviousDeformationOnDrivingPoint[j][0] ;
		ModelMatrix[2][j] = PreviousDeformationOnDrivingPoint[j][2] ;
		ModelMatrix[3][j] = 1 ;
	}

	/* get ObjectMatrix->data[4][MdlVer_Num] */
	for(unsigned int j=0; j<CurrentDeformationOnDrivingPoint.size(); j++)
	{
		//std::cout<<PreviousDeformationOnDrivingPoint[j]<<std::endl;
		ObjectMatrix[0][j] = CurrentDeformationOnDrivingPoint[j][1] ;
		ObjectMatrix[1][j] = CurrentDeformationOnDrivingPoint[j][0] ;
		ObjectMatrix[2][j] = CurrentDeformationOnDrivingPoint[j][2] ;
		ObjectMatrix[3][j] = 1 ;
	}


	/*get matrix 4X4 TmpM1*/
	TmpM1 = ModelMatrix*ModelMatrix.transpose();
	
	/*get matrix 4X4 TmpM2 */
	TmpM2 = ObjectMatrix*ModelMatrix.transpose();
	
	vnl_svd<double> svd(TmpM1);
	InverseMatrix = svd.inverse();
	
	GlobalAffineMatrix = TmpM2*InverseMatrix;			
}


template <class TInputImage, class TOutputImage>
void 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::SmoothDeformationField(ImageAttributePointerType &FixedAttributeImage, DeformationFieldPointer DeformFld, DeformationFieldPointer DeformFld_Last, float LocalRatio)
{
	SizeType size = DeformFld->GetRequestedRegion().GetSize();
	ConstDeforamtionFieldIteratorType DeformFldLast_Iter(DeformFld_Last, DeformFld_Last->GetLargestPossibleRegion());
	DeformationFieldIteratorType DeformFld_Iter(DeformFld, DeformFld->GetLargestPossibleRegion());
	DeformationVectorType Global, Deform_Update, Crnt;	
	
	//GlobalAffineMatrix.set_identity();

	for(DeformFldLast_Iter.GoToBegin(),DeformFld_Iter.GoToBegin();!DeformFld_Iter.IsAtEnd();++DeformFld_Iter, ++DeformFldLast_Iter)
	{
		IndexType idx = DeformFldLast_Iter.GetIndex();
		Crnt = DeformFldLast_Iter.Get();
		Deform_Update = DeformFld_Iter.Get();
		
		for(int k=0;k<OutputImageDimension;k++)
			Crnt[k] += idx[k];
		
		Global[1] = GlobalAffineMatrix[0][0]*Crnt[1] + GlobalAffineMatrix[0][1]*Crnt[0] + GlobalAffineMatrix[0][2]*Crnt[2] + GlobalAffineMatrix[0][3] - idx[1];
		Global[0] = GlobalAffineMatrix[1][0]*Crnt[1] + GlobalAffineMatrix[1][1]*Crnt[0] + GlobalAffineMatrix[1][2]*Crnt[2] + GlobalAffineMatrix[1][3] - idx[0];
		Global[2] = GlobalAffineMatrix[2][0]*Crnt[1] + GlobalAffineMatrix[2][1]*Crnt[0] + GlobalAffineMatrix[2][2]*Crnt[2] + GlobalAffineMatrix[2][3] - idx[2];
		for(int k=0;k<OutputImageDimension;k++)	
		{			
			Deform_Update[k] = Global[k] + (Deform_Update[k]-Global[k])*LocalRatio/1.2;
			if(Deform_Update[k]+idx[k]<0)
				Deform_Update[k] = -idx[k];
			if(Deform_Update[k]+idx[k]>=size[k]-1)
				Deform_Update[k] = size[k]-1-idx[k];
		}		
		DeformFld_Iter.Set(Deform_Update);
	}

	//somoothing
	for(int ntime=0;ntime<m_SmoothTime;ntime++)
	{
		SmoothDeformation_OneTime(FixedAttributeImage, DeformFld, m_SmoothTime-ntime);
	}
	if(m_SmoothTime>0)
	{
		for(int ntime=0;ntime<m_SmoothTime;ntime++)
		{
			EdgePreserveSmoothDeformation_OneTime(FixedAttributeImage, DeformFld, m_SmoothTime-ntime);
		}
	}
}


/**
 * Set the standard deviations.
 */
template <class TInputImage, class TOutputImage>
void
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::SetStandardDeviations(
  double value )
{

  unsigned int j;
  for( j = 0; j < 3; j++ )
    {
    if( value != m_StandardDeviations[j] )
      {
      break;
      }
    }
  if( j < 3 )
    {
    this->Modified();
    for( j = 0; j < InputImageDimension; j++ )
      {
      m_StandardDeviations[j] = value;
      }
    }

}

template <class TInputImage, class TOutputImage>
void
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::SmoothDeformation_OneTime(ImageAttributePointerType &FixedAttributeImage, DeformationFieldPointer DeformFld, int Time)
{
	SizeType size = DeformFld->GetRequestedRegion().GetSize();
	float LocalRatio;
	DeformationFieldIteratorType DeformFld_Iter(DeformFld, DeformFld->GetLargestPossibleRegion());
	DeformationVectorType PointDeform, CurrentDeform, MeanDeform;	
	IndexType CurrentIndex, PointIndex;
	AttributeVectorType TemplateFeature, CurrentFeature;
	RegionType dummyRegion = DeformFld->GetLargestPossibleRegion();

	LocalRatio = m_DfmSmthCoeff;
	for(DeformFld_Iter.GoToBegin();!DeformFld_Iter.IsAtEnd();++DeformFld_Iter)
	{
		float Num = 0;
		PointIndex = DeformFld_Iter.GetIndex();
		PointDeform = DeformFld_Iter.Get();
		TemplateFeature = FixedAttributeImage->GetPixel(PointIndex);
		for(int s=0;s<InputImageDimension;s++)
			MeanDeform[s] = 0;
		for(int k=1;k<27;k++)
		{
			for(int s=0;s<InputImageDimension;s++)
				CurrentIndex[s] = PointIndex[s] + m_SubvolumeNeighborhood[k][s];
			if(!dummyRegion.IsInside(CurrentIndex))
				continue;
			CurrentDeform = DeformFld->GetPixel(CurrentIndex);
			for(int s=0;s<InputImageDimension;s++)
				MeanDeform[s] += CurrentDeform[s];
			Num += 1;
		}//end of k
		for(int s=0;s<InputImageDimension;s++)
			MeanDeform[s] /= Num;
		if(Time<=1 || TemplateFeature.GetEdge()!=100 && TemplateFeature.GetEdge()!=180 ) 
		{
			for(int s=0;s<InputImageDimension;s++)
				PointDeform[s] += (MeanDeform[s]-PointDeform[s])*LocalRatio;
		}
		else
		{
			for(int s=0;s<InputImageDimension;s++)
				CurrentDeform[s] = (MeanDeform[s]-PointDeform[s])*LocalRatio;
			float mag = ComputeVectorMagnitude(CurrentDeform);
			if( mag>3.0 ) /* 2.0 before June 22 2004 */
			{
				for(int s=0;s<InputImageDimension;s++)
				{
					CurrentDeform[s] /= mag/3.0;
				}
			}
			for(int s=0;s<InputImageDimension;s++)
				PointDeform[s] += CurrentDeform[s];
		}
		for(int s=0;s<InputImageDimension;s++)
		{
			if(PointDeform[s]+PointIndex[s]<0)
				PointDeform[s] = - PointIndex[s];
			if(PointDeform[s]+PointIndex[s]>=size[s]-1)
			{
				PointDeform[s] = size[s]-1-PointIndex[s];
			}
		}
		DeformFld_Iter.Set(PointDeform);
	}
}

template <class TInputImage, class TOutputImage>
void
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::EdgePreserveSmoothDeformation_OneTime(ImageAttributePointerType &FixedAttributeImage, DeformationFieldPointer DeformFld, int Time)
{
	SizeType size = DeformFld->GetRequestedRegion().GetSize();
	float LocalRatio;
	DeformationFieldIteratorType DeformFld_Iter(DeformFld, DeformFld->GetLargestPossibleRegion());
	DeformationVectorType PointDeform, CurrentDeform, MeanDeform;	
	IndexType CurrentIndex, PointIndex;
	AttributeVectorType TemplateFeature, CurrentFeature;
	RegionType dummyRegion = FixedAttributeImage->GetLargestPossibleRegion();

	LocalRatio = m_DfmSmthCoeff;
	for(DeformFld_Iter.GoToBegin();!DeformFld_Iter.IsAtEnd();++DeformFld_Iter)
	{
		float Num = 0;
		PointIndex = DeformFld_Iter.GetIndex();
		PointDeform = DeformFld_Iter.Get();
		TemplateFeature = FixedAttributeImage->GetPixel(PointIndex);
		for(int s=0;s<InputImageDimension;s++)
			MeanDeform[s] = 0;
		for(int k=1;k<27;k++)
		{
			for(int s=0;s<InputImageDimension;s++)
				CurrentIndex[s] = PointIndex[s] + m_SubvolumeNeighborhood[k][s];
			if(!dummyRegion.IsInside(CurrentIndex))
				continue;
			CurrentDeform = DeformFld->GetPixel(CurrentIndex);
			for(int s=0;s<InputImageDimension;s++)
				MeanDeform[s] += CurrentDeform[s];
			Num += 1;
			if(Time>1 && TemplateFeature.GetEdge()==100 && TemplateFeature.GetEdge()==180 ) 
			{
				for(int s=0;s<InputImageDimension;s++)
					MeanDeform[s] += CurrentDeform[s];
				Num += 1;
			}
		}
		for(int s=0;s<InputImageDimension;s++)
			MeanDeform[s] /= Num;
		if(Time<=1 || TemplateFeature.GetEdge()!=100 && TemplateFeature.GetEdge()!=180 ) 
		{
			for(int s=0;s<InputImageDimension;s++)
				PointDeform[s] += (MeanDeform[s]-PointDeform[s])*LocalRatio;
		}
		else
		{
			for(int s=0;s<InputImageDimension;s++)
				PointDeform[s] += (MeanDeform[s]-PointDeform[s])*LocalRatio/100.0;
		}
		for(int s=0;s<InputImageDimension;s++)
		{
			if(PointDeform[s]+PointIndex[s]<0)
				PointDeform[s] = - PointIndex[s];
			if(PointDeform[s]+PointIndex[s]>=size[s]-1)
			{
				PointDeform[s] = size[s]-1-PointIndex[s];
			}
		}
		DeformFld_Iter.Set(PointDeform);
	}
}

template <class TInputImage, class TOutputImage>
void
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::HAMMERRegistrationOneRound(ImageAttributePointerType &FixedAttributeImage, ImageAttributePointerType &MovingAttributeImage, int resolution, float ratio_iteration, float LocalRatio, DeformationFieldPointer DeformFld)
{
	int vv;
	DeformationVectorType dfm, dfm_prev;
	float MaxDegree;
	IndexType idx;
	DeforamtionVectorArrayType m_LastDeformationList;
	//Step1: copy the previous deformation and previous displacement on driving voxel
	DeformationFieldPointer DeformFieldLast = DeformationFieldType::New();
	// copy field to m_DeformationFieldLast
	DeformFieldLast->CopyInformation( DeformFld );
	DeformFieldLast->SetRegions( DeformFld->GetLargestPossibleRegion() );
	DeformFieldLast->Allocate();


	DeformationFieldIteratorType DeformFldLast_Iter(DeformFieldLast, DeformFieldLast->GetRequestedRegion());
	DeformationFieldIteratorType DeformFld_Iter(DeformFld, DeformFld->GetRequestedRegion());
	for(DeformFldLast_Iter.GoToBegin(),DeformFld_Iter.GoToBegin();!DeformFld_Iter.IsAtEnd();++DeformFld_Iter, ++DeformFldLast_Iter)
	{
		DeformFldLast_Iter.Set(DeformFld_Iter.Get());
	}
	//copy the vector

	m_LastDeformationList.resize(m_ModelDrivingPointDisplacement.size());
	for(unsigned int k=0;k<m_ModelDrivingPointDisplacement.size();k++)
	{
		for(int s=0;s<InputImageDimension;s++)
			m_LastDeformationList[k][s] = m_ModelDrivingPointDisplacement[k][s];
	}

	//Step2: Initilzation before the feature matching 
	int SHIFT = resolution/2 ; 
	if(SHIFT<1)
		SHIFT=1; 

	int incre = 2*SHIFT/6; 
	if(incre<1) 
		incre=1 ;

	m_SubvolumeRadius = (SHIFT+1)*2+1; /*nb_size should be odd number */ 

	CreatePointMatchingNeighbor(m_SearchNeighbor, SHIFT);
  m_SubvolumeNeighborhood.clear();
	CreateSubvolumnNeighbor(m_SubvolumeNeighborhood, m_SubvolumeRadius);

	int Subvolumn_Matching_Size = m_SubvolumeNeighborhood.size();

	//Step2.1 get the forces from the landmarks 
	if(ratio_iteration<=ITER_THRD) 
		FindingInverseForceFromSubject(FixedAttributeImage, MovingAttributeImage, DeformFld, SHIFT+1);

	std::cout << "Step 3 Start the correspondence matching\n";
	//Step 3 Start the correspondence matching 
	int MdlVer_Num = m_PickedPointsOnFixedImage.size();
	for(vv=0; vv<MdlVer_Num; vv+=STEPPNT) 
	{
		int Template_Driving_Point_Index = (vv + m_StartSearchPoint)%MdlVer_Num ; 

		MaxDegree = DetermineCorrespondenceOnOneDrivingVoxel(FixedAttributeImage, MovingAttributeImage, DeformFld, Template_Driving_Point_Index, DeformFieldLast, dfm, SHIFT, incre);
		//std::cout<<Template_Driving_Point_Index<<":"<<MaxDegree<<"-->"<<dfm<<std::endl;	
		if( MaxDegree>m_SubvolumnSimilarityThreshold)
		{					
			int UsedVNum = Subvolumn_Matching_Size;
			int UsedGuassianSigma = m_SubvolumeRadius ;

			DisseminateDeformation(DeformFld, Template_Driving_Point_Index, dfm, m_SubvolumeNeighborhood, UsedVNum, UsedGuassianSigma);			
		}//end of else

	}//end of vv loop	

	m_StartSearchPoint = vv ;
#ifdef DUMP_DEFORMATIONFIELD
//  	IndexType tmpidx;
//  	DeformationVectorType tmpdfm;
//  	tmpidx[2] = 15;
//  	printf("Dump Presmoothing Data\n");
//  	int t1=0;
//  	for(int i=0;i<64;i++)
//  		for(int j=0;j<64;j++)
//  		{
//  			tmpidx[0] = i;
//  			tmpidx[1] = j;
//  			tmpdfm = DeformFld->GetPixel(tmpidx);
// 			printf("%d %f %f %f\n", t1, tmpdfm[0], tmpdfm[1], tmpdfm[2]);
//  			//std::cout<<t1<<tmpdfm<<std::endl;
//  			++t1;
// 		}
#endif
#ifdef DUMP_DEFORMATIONFIELD

	FILE *fp = fopen("Dump_Deform.img", "wb");
	DeformationFieldIteratorType DeformFld_Iter_Dump(DeformFld, DeformFld->GetRequestedRegion());
	DeformFld_Iter_Dump.GoToBegin();
	SizeType ImageSize = DeformFld->GetRequestedRegion().GetSize();
	int x_size = ImageSize[0];
	int y_size = ImageSize[1];
	int z_size = ImageSize[2];
	Fvector3d ***Deform = Fvector3dalloc3d(x_size, y_size, z_size);
	for(int k=0;k<z_size;k++)
		for(int i=0;i<x_size;i++)
			for(int j=0;j<y_size;j++)
			{
				Deform[k][i][j].x = DeformFld_Iter_Dump.Get()[1];
				Deform[k][i][j].y = DeformFld_Iter_Dump.Get()[0];
				Deform[k][i][j].z = DeformFld_Iter_Dump.Get()[2];
				++DeformFld_Iter_Dump;
			}
	fwrite(&x_size, sizeof(int), 1, fp);
	fwrite(&z_size, sizeof(int), 1, fp);
	for(int k=0;k<z_size;k++)
		for(int i =0;i<x_size;i++)
			fwrite(Deform[k][i], sizeof(Fvector3d), y_size, fp);
	fclose(fp);
	Fvector3dfree3d(Deform, z_size, x_size);
#endif
	//Smoothing
	
	for(unsigned int k=0;k<m_PickedPointsOnFixedImage.size();k++)
	{
		idx = m_PickedPointsOnFixedImage[k];
		dfm = DeformFld->GetPixel(idx);
		for(int j=0;j<InputImageDimension;j++)
			m_ModelDrivingPointDisplacement[k][j] = dfm[j]+idx[j];		
	}
	FitGlobalAffineTransform(m_ModelDrivingPointDisplacement, m_LastDeformationList);
	SmoothDeformationField(FixedAttributeImage, DeformFld, DeformFieldLast, LocalRatio);
#ifdef DUMP_DEFORMATIONFIELD
// 	   printf("Dump after smooth data\n");
// 	   tmpidx[2] = 15;
// 	   for(int i=0;i<64;i++)
// 	 	  for(int j=0;j<64;j++)
// 	 	  {
// 	 		  tmpidx[0] = i;
// 	 		  tmpidx[1] = j;
// 	 		  tmpdfm = DeformFld->GetPixel(tmpidx);
// 	 		  printf("%f %f %f\n", tmpdfm[0], tmpdfm[1], tmpdfm[2]);
// 	 	  }

	fp = fopen("Dump_Deform_Smooth.img", "wb");


	DeformFld_Iter_Dump.GoToBegin();
	ImageSize = DeformFld->GetRequestedRegion().GetSize();
	x_size = ImageSize[0];
	y_size = ImageSize[1];
	z_size = ImageSize[2];
	Deform = Fvector3dalloc3d(x_size, y_size, z_size);
	IndexType index;
	DeformationVectorType dumpdfm;
	for(int k=0;k<z_size;k++)
		for(int i=0;i<x_size;i++)
			for(int j=0;j<y_size;j++)
			{				
				dumpdfm = DeformFld_Iter_Dump.Get();
				Deform[k][i][j].x = dumpdfm[1];
				Deform[k][i][j].y = dumpdfm[0];
				Deform[k][i][j].z = dumpdfm[2];
				++DeformFld_Iter_Dump;
			}
	fwrite(&x_size, sizeof(int), 1, fp);
	fwrite(&z_size, sizeof(int), 1, fp);
	for(int k=0;k<z_size;k++)
		for(int i =0;i<x_size;i++)
			fwrite(Deform[k][i], sizeof(Fvector3d), y_size, fp);
	fclose(fp);
	Fvector3dfree3d(Deform, z_size, x_size);
#endif
}

template <class TInputImage, class TOutputImage>
void 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::HAMMERMainLoop(ImageAttributePointerType &FixedAttributeImage, ImageAttributePointerType &MovingAttributeImage, DeformationFieldPointer &DeformFld)
{
	m_IterationRatio = 0;
	float PointMathingThreshold = m_PointMatchingThreshold;

	for(unsigned int iter=0;iter<m_NumberOfIterations;iter++)
	{
		std::cout<<"########## ITER "<<iter<<" ########################"<<std::endl;
		if(m_IterationRatio>0.5)
		{
			m_FixedImageDrivingVoxelQualification[0] = (0.9*(1-m_IterationRatio)*255);
			m_FixedImageDrivingVoxelQualification[1] = (0.4*(2.0*m_IterationRatio)*255);
			m_FixedImageDrivingVoxelQualification[2] = (m_FixedImageDrivingVoxelQualification[1]/1.1);
			m_FixedImageDrivingVoxelQualification[3] = (255/4);
			m_FixedImageDrivingVoxelQualification[4] = (0);
			m_FixedImageDrivingVoxelQualification[5] = (255/4);

			IdentifyDrivingVoxelsInFixedImage(FixedAttributeImage, m_FixedImageDrivingVoxelQualification);
			std::cout << m_PickedPointsOnFixedImage.size() << std::endl;
			m_InverseDisplacement.clear();
			m_ModelDrivingPointDisplacement.clear();
			m_ModelAttributeVector.clear();
			m_InverseDisplacement.resize(m_PickedPointsOnFixedImage.size());
			m_ModelDrivingPointDisplacement.resize(m_PickedPointsOnFixedImage.size());
			for(unsigned int k=0;k<m_PickedPointsOnFixedImage.size();k++)
			{
				IndexType index = m_PickedPointsOnFixedImage[k];
				DeformationVectorType Deform = DeformFld->GetPixel(index);
				for(int s=0;s<InputImageDimension;s++)
				{
					m_ModelDrivingPointDisplacement[k][s] = index[s] + Deform[s];
					m_InverseDisplacement[k][s] = 0;
				}
				AttributeVectorType Feature = FixedAttributeImage->GetPixel(index);
				m_ModelAttributeVector.push_back(Feature);
			}
		}// m_IterationRatio is smaller than  0.5
		m_IterationRatio = (float)iter/(float)(m_NumberOfIterations) ;

		int resolution = static_cast<int>( m_SearchRadius*exp(-m_IterationRatio*m_IterationRatio/2./0.36) );
		if(resolution<1) 
			resolution=1 ;

		m_PointMatchingThreshold = PointMathingThreshold*(1-m_IterationRatio)+0.001 ; /* 0.8 */ 
		float LocalRatio = 0.25 + 0.65*exp(-(m_IterationRatio-1.)*(m_IterationRatio-1.)/2./.25/.25) ;
		//HAMMER_ONE_LOOP + SMOOTHING

		HAMMERRegistrationOneRound(FixedAttributeImage, MovingAttributeImage, resolution, m_IterationRatio, LocalRatio, DeformFld);
		printf("print MdlVer\n");
		for(unsigned int k=0;k<m_PickedPointsOnFixedImage.size();k++)
		{
			IndexType index = m_PickedPointsOnFixedImage[k];
			DeformationVectorType Deform = DeformFld->GetPixel(index);
			for(int s=0;s<InputImageDimension;s++)
			{
				m_ModelDrivingPointDisplacement[k][s] = index[s] + Deform[s];								
			}
		}
	}
}


template <class TInputImage, class TOutputImage>
void 
HammerDeformableRegistrationImageFilter<TInputImage, TOutputImage>
::IdentifyDrivingVoxelsInFixedImage(ImageAttributePointerType &FixedAttributeImage, std::vector<float> &FixedImageCriteria)
{
  m_PickedPointsOnFixedImage.clear();
  IdentifyDrivingVoxels_GR( FixedAttributeImage, m_PickedPointsOnFixedImage, FixedImageCriteria );
}

template <class TFixedImage, class TMovingImage>
void 
HammerDeformableRegistrationImageFilter<TFixedImage, TMovingImage>
::IdentifyDrivingVoxelsInMovingImage(ImageAttributePointerType &MovingAttributeImage, std::vector<float> &MovingImageCriteria )
{
  m_PickedPointsOnMovingImage.clear();
  IdentifyDrivingVoxels_GR( MovingAttributeImage, m_PickedPointsOnMovingImage, MovingImageCriteria );
}
  
template <class TFixedImage, class TMovingImage>
unsigned int 
HammerDeformableRegistrationImageFilter<TFixedImage, TMovingImage>
::IdentifyDrivingVoxels( ImageAttributePointerType avPointer, std::vector<IndexType> & drivingVoxels, std::vector<float> &Criteria)
{
  // Generate a binary image that indicates voxels already been selected
  typename InputImageType::Pointer mask = InputImageType::New();
  mask->CopyInformation( this->m_FixedAttributeImage );
  mask->SetRegions( mask->GetLargestPossibleRegion() );
  mask->Allocate();
  mask->FillBuffer( 0 );

  unsigned int nVoxels = drivingVoxels.size();
  for (unsigned int k = 0; k < nVoxels; k++)
    {
    typename InputImageType::IndexType idx = drivingVoxels[k];
    mask->SetPixel( idx, 1 );
    }

  itk::ImageRegionIteratorWithIndex<ImageAttributeType> 
    itAV( avPointer, avPointer->GetLargestPossibleRegion() );

  itk::ImageRegionIteratorWithIndex<InputImageType> 
    itMask( mask, mask->GetLargestPossibleRegion() );

  for ( itAV.GoToBegin(), itMask.GoToBegin(); !itAV.IsAtEnd(); ++itAV, ++itMask )
    {
    if ( itMask.Get() == 1 )   // voxel already a driving voxel
      {
      continue;
      }
    AttributeVectorType a = itAV.Get();
    if ( a.IsQualifiedDrivingVoxel( Criteria ) )
      {
      drivingVoxels.push_back( itAV.GetIndex() );
      nVoxels ++;
      }
    }

  return nVoxels;
}

 
template <class TFixedImage, class TMovingImage>
unsigned int 
HammerDeformableRegistrationImageFilter<TFixedImage, TMovingImage>
::IdentifyDrivingVoxels_GR( ImageAttributePointerType avPointer, std::vector<IndexType> & drivingVoxels, std::vector<float> &Criteria)
{
  std::cout << "attribute:\n" << avPointer->GetLargestPossibleRegion() << std::endl;
  // Generate a binary image that indicates voxels already been selected
  int nVoxels = 0;
  int nEdge = 0;
  itk::ImageRegionIteratorWithIndex<ImageAttributeType> 
    itAV( avPointer, avPointer->GetLargestPossibleRegion() );

  typename InputImageType::Pointer mask = InputImageType::New();
  mask->CopyInformation( avPointer );
  mask->SetRegions( mask->GetLargestPossibleRegion() );
  mask->Allocate();
  mask->FillBuffer( 0 );

  //std::cout << "attribute:\n" << avPointer->GetLargestPossibleRegion() << std::endl;
  //std::cout << "mask:\n" << mask->GetLargestPossibleRegion() << std::endl;

  drivingVoxels.clear();
  for ( itAV.GoToBegin(); !itAV.IsAtEnd(); ++itAV)
    {
    AttributeVectorType a = itAV.Get();
    if ( a.IsQualifiedDrivingVoxel_GR( Criteria ) )
      {
      drivingVoxels.push_back( itAV.GetIndex() );
      nVoxels ++;
      }
    if(a.GetEdge()>0)
      nEdge++;
    }
	
  for(unsigned int k=0;k<drivingVoxels.size();k++)
    {
    IndexType idx = drivingVoxels[k];
    mask->SetPixel(idx, 250);
    }
  typedef itk::ImageFileWriter< InputImageType > WriterType;
  typename WriterType::Pointer writer = WriterType::New();
  writer->SetInput(mask);
  writer->SetFileName("DVImg.hdr");
  writer->Update();

  std::cout<<"DV Number is "<<nVoxels<<"Edge Number is "<<nEdge<<std::endl;
  return nVoxels;
}

template <class TFixedImage, class TMovingImage>
void 
HammerDeformableRegistrationImageFilter<TFixedImage, TMovingImage>
::LinearInitialization( )
{
  // do serious registration
  typedef itk::MattesMutualInformationImageToImageMetric<InputImageType, InputImageType> MetricType;
  typedef itk::RegularStepGradientDescentOptimizer OptimizerType;
  typedef itk::NearestNeighborInterpolateImageFunction<InputImageType, double> InterpolatorType;
  typedef itk::ImageRegistrationMethod<InputImageType, InputImageType> RegistrationType;

  typename MetricType::Pointer metric                 = MetricType::New();
  typename OptimizerType::Pointer optimizer           = OptimizerType::New();
  typename InterpolatorType::Pointer interpolator     = InterpolatorType::New();
  typename RegistrationType::Pointer registration     = RegistrationType::New();

  //  metric->SetUsePhysicalSpaceSampling( true );
  registration->SetMetric(        metric        );
  registration->SetOptimizer(     optimizer     );
  registration->SetInterpolator(  interpolator  );

  registration->SetTransform( m_AffineTransform );
  registration->SetFixedImage(  m_FixedImage  );
  registration->SetMovingImage( m_MovingImage  );

  metric->ReinitializeSeed( 31415926 );
  metric->SetNumberOfSpatialSamples( 10000 );
  metric->SetNumberOfHistogramBins( 50 );

  typename OptimizerType::ScalesType scales;
  scales.SetSize( m_AffineTransform->GetNumberOfParameters() );
  scales.Fill( 1 );
  for (::size_t k = 9; k < scales.GetSize(); k++)
    {
    scales[k] = 0.1;
    }  

  optimizer->SetScales( scales );

  typename InputImageType::RegionType fixedRegion = m_FixedImage->GetLargestPossibleRegion();
  
  std::cout << "Region used for registration: \n" << fixedRegion << std::endl;
  registration->SetFixedImageRegion( fixedRegion );

  std::cout << m_AffineTransform->GetParameters() << std::endl;

  registration->SetInitialTransformParameters( m_AffineTransform->GetParameters() );
  optimizer->SetNumberOfIterations ( 200 );
  optimizer->SetMinimumStepLength ( .0005 );
  optimizer->SetMaximumStepLength ( 0.2 );
  optimizer->SetMinimize ( true );

  typename CommandIterationUpdate::Pointer observer = CommandIterationUpdate::New();
  optimizer->AddObserver( itk::IterationEvent(), observer );
  observer->SetRegistration( registration );
  
  try 
    { 
    registration->StartRegistration(); 
    } 
  catch( itk::ExceptionObject & err ) 
    { 
    std::cerr << "ExceptionObject caught !" << std::endl; 
    std::cerr << err << std::endl; 
    return ;
    } 
  m_AffineTransform->SetParameters( registration->GetLastTransformParameters() );
  m_AffineTransform->Print( std::cout );

  typename itk::ResampleImageFilter<InputImageType, InputImageType>::Pointer resample = 
    itk::ResampleImageFilter<InputImageType, InputImageType>::New();
    
  resample->SetInput ( m_MovingImage ); 
  resample->SetOutputOrigin( m_FixedImage->GetOrigin() );
  resample->SetOutputSpacing( m_FixedImage->GetSpacing() );
  resample->SetSize( m_FixedImage->GetLargestPossibleRegion().GetSize() );
  resample->SetOutputStartIndex( m_FixedImage->GetLargestPossibleRegion().GetIndex() );
  resample->SetOutputDirection( m_FixedImage->GetDirection() );

  resample->SetTransform ( m_AffineTransform );
  resample->SetInterpolator ( interpolator );
  resample->Update();

  this->SetMovingImage( resample->GetOutput() );

  typename itk::ImageFileWriter<InputImageType>::Pointer affineWriter = itk::ImageFileWriter<InputImageType>::New();
  affineWriter->SetFileName( "movingAffine.mha" );
  affineWriter->SetInput( m_MovingImage );
  affineWriter->Update();

  return;
}

} // end namespace itk


#endif
