
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string>
#include <vector>

#include "itkFixedArray.h"
#include "itkMatrix.h"
#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkShrinkImageFilter.h"

#include "itkVector.h"
#include "../../../utilities/tclap/CmdLine.h"

#include "../plugin/itkAttributeVectorImageFilter.h"
#include "../plugin/itkHammerDeformationImageFunction.h"
#include "../plugin/itkHammerRegistrationFunction.h"

using namespace TCLAP;

typedef itk::Vector<unsigned char, 5> ImgAttribute;

typedef float FloatType;
typedef itk::Vector<FloatType, 3>       ITKFvector3d;
typedef itk::Vector<int, 3>             Ivector3d;
typedef itk::Matrix<FloatType, 4, 4>    Matrix;
typedef itk::Image<unsigned char, 3>    ImageType;
typedef itk::Image<ITKFvector3d, 3>     DeformationFieldType;
typedef itk::Image<ImgAttribute, 3>     AttributeImageType;

int main(int argc,char *argv[])  
{
	float XYZres = 1. ;
	int nThreads = 1;
	std::string fixedFilename;
	std::string movingFilename;

	itk::OStringStream msg;

	try
	{
		CmdLine cl ( "HAMMER Attribute Vector Test: Heirarchical Attribute Matching Mechanism for Elastic Registration, NA-MIC Kit",
			' ',
			"$Revision: 1.2 $" );

		msg.str ( "" ); msg << "Number of threads (default: " << nThreads << ")";
		ValueArg<int> ThreadArg ( "T", "Thread", msg.str(), false, nThreads, "int", cl );
		msg.str ( "" ); msg << "sample rate (default: " << XYZres << ")";
		ValueArg<float> SampleRateArg ( "s", "Sample", msg.str(), false, XYZres, "float", cl );

		UnlabeledValueArg<std::string> FixedImageArg ( "fixed", "Fixed image filename", 1, fixedFilename, "string", cl );
		UnlabeledValueArg<std::string> MovingImageArg ( "moving", "Moving image filename", 1, movingFilename, "string", cl );

		cl.parse ( argc, argv );
		nThreads = ThreadArg.getValue();
		XYZres =  SampleRateArg.getValue();

		fixedFilename = FixedImageArg.getValue();
		movingFilename = MovingImageArg.getValue();
	} 
	catch ( ArgException exception ) {
		std::cerr << "error: " << exception.error() << " for arg " << exception.argId() << std::endl;
		exit ( EXIT_FAILURE );
	}

	double scale = 7*(1./XYZres) ; 
	if(scale<3) 
		scale=3; 
	printf("scale=%f\n", scale) ;

	/***** Model image, segmented *****/
	/*Img_XY = 256 ;*/	 

	printf("\nmodel image : %s\n", fixedFilename.c_str()) ;  
	printf("subject image : %s\n", movingFilename.c_str()) ;  

	/*** Load in fixed image and compute the attribute vectors ***/
	itk::ImageFileReader<ImageType>::Pointer ImgReader = itk::ImageFileReader<ImageType>::New();
	ImgReader->SetFileName( fixedFilename.c_str() );
	try
	{
		ImgReader->Update();
	}
	catch( itk::ExceptionObject *ex )
	{
		std::cerr << ex << std::endl;
	}

	typedef itk::ShrinkImageFilter<ImageType, ImageType> DownSampleType;
	DownSampleType::Pointer downsample = DownSampleType::New();
	downsample->SetInput( ImgReader->GetOutput() );
	for (int k = 0; k < 3; k++)
	{
		downsample->SetShrinkFactor( k, static_cast<int> (XYZres) );
	}
	downsample->Update();

	ImageType::Pointer Img = downsample->GetOutput();
	Img->DisconnectPipeline();
	std::cout << "Fixed image file read in\n";

	typedef itk::AttributeVectorImageFilter<ImageType, AttributeImageType> AttributeFilterType;
	AttributeFilterType::Pointer modleAttributeFilter = AttributeFilterType::New();
	modleAttributeFilter->SetInput( Img );
	modleAttributeFilter->SetBGValue( 0 );
	modleAttributeFilter->SetGMValue( 150 );
	modleAttributeFilter->SetWMValue( 250 );
	modleAttributeFilter->SetVNValue( 50 );
	modleAttributeFilter->SetCSFValue( 10 );

	modleAttributeFilter->SetNumberOfThreads( nThreads );
	modleAttributeFilter->SetStrength( 1 );
	modleAttributeFilter->SetScale( scale );
	modleAttributeFilter->Update();

	AttributeImageType::Pointer fixedAVec = modleAttributeFilter->GetOutput();
	fixedAVec->DisconnectPipeline();

	/*** Load in moving image and compute the attribute vectors ***/
	ImgReader->SetFileName( movingFilename.c_str() );
	try
	{
		ImgReader->Update();
	}
	catch( itk::ExceptionObject *ex )
	{
		std::cerr << ex << std::endl;
	}

	downsample->SetInput( ImgReader->GetOutput() );
	for (int k = 0; k < 3; k++)
	{
		downsample->SetShrinkFactor( k, static_cast<int> (XYZres) );
	}
	downsample->Update();

	ImageType::Pointer mImg = downsample->GetOutput();
	mImg->DisconnectPipeline();
	std::cout << "Moving image file read in\n";

	AttributeFilterType::Pointer subjectAttributeFilter = AttributeFilterType::New();
	subjectAttributeFilter->SetInput( mImg );
	subjectAttributeFilter->SetBGValue( 0 );
	subjectAttributeFilter->SetGMValue( 150 );
	subjectAttributeFilter->SetWMValue( 250 );
	subjectAttributeFilter ->SetVNValue( 50 );
	subjectAttributeFilter->SetCSFValue( 10 );

	subjectAttributeFilter->SetNumberOfThreads( nThreads );
	subjectAttributeFilter->SetStrength( 1 );
	subjectAttributeFilter->SetScale( scale );
	subjectAttributeFilter->Update();

	AttributeImageType::Pointer movingAVec = subjectAttributeFilter->GetOutput();
	movingAVec->DisconnectPipeline();

	// define HammerDeformationImageFunction
	typedef itk::HammerDeformationImageFunction<AttributeImageType, float> DeformationImageFunctionType;
	DeformationImageFunctionType::Pointer deformation = DeformationImageFunctionType::New();
	deformation->SetInputImage( fixedAVec );
	deformation->SetSubjectImage( movingAVec );

	// the following lines set deformation parameters
	deformation->SetNeighborhoodRadius( 3 );
	deformation->SetVoxelSimilarityThreshold( 0.8 );
	deformation->SetNeighborhoodSimilarityThreshold( 0.6 );
	deformation->SetSigma( 1 );

	typedef itk::HammerRegistrationFunction<AttributeImageType, AttributeImageType, DeformationFieldType> HAMMERFunctionType;
	HAMMERFunctionType::Pointer Hammer = HAMMERFunctionType::New();

	// ************************************************
	// 1. identify driving voxels in the fixedImage (Model image)
	// please refer to HammerPointPickingTest.cxx

	AttributeImageType::IndexType idx;
	idx [0] = 100;   // just a place holder, the value of idx should be
	// filled using the results of driving voxel identification
	idx [1] = 100;
	idx [2] = 100;

	DeformationImageFunctionType::RealType vec;
	vec = deformation->EvaluateAtIndex( idx );

	return 0;
}



