
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string>
#include <vector>

#include "itkFixedArray.h"
#include "itkMatrix.h"
#include "itkImage.h"
#include "itkImageFileReader.h"
#include "itkImageFileWriter.h"
#include "itkImageRegionIteratorWithIndex.h"
#include "itkShrinkImageFilter.h"
#include "itkWarpImageFilter.h"
#include "itkVectorLinearInterpolateImageFunction.h" 
#include "itkNearestNeighborInterpolateImageFunction.h" 

#include "itkSimpleFilterWatcher.h"
#include "itkPluginUtilities.h"

#include "itkVector.h"
#include "../../../utilities/tclap/CmdLine.h"

#include "../plugin/itkHammerTissueAttributeVectorImageFilter.h"
#include "../plugin/itkHammerDeformableRegistrationImageFilter.h"
#include "time.h"

using namespace TCLAP;

typedef itk::HammerTissueAttributeVector ImgAttribute;

typedef float FloatType;
typedef itk::Vector<FloatType, 3>       ITKFvector3d;
typedef itk::Vector<int, 3>             Ivector3d;
typedef itk::Matrix<FloatType, 4, 4>    Matrix;
typedef itk::OrientedImage<unsigned char, 3>    ImageType;
typedef itk::ImageFileReader<ImageType> ReaderType;
typedef itk::ImageFileWriter<ImageType> WriterType;

typedef itk::OrientedImage<unsigned short, 3>    UShortImageType;
typedef itk::ImageFileReader<UShortImageType> USReaderType;
typedef itk::ImageFileWriter<UShortImageType> USWriterType;

typedef itk::OrientedImage<ITKFvector3d, 3>     DeformationFieldType;
typedef itk::OrientedImage<ImgAttribute, 3>     AttributeImageType;
typedef itk::HammerTissueAttributeVectorImageFilter<ImageType, AttributeImageType> AttributeFilterType;

unsigned char Geom_UP = static_cast<unsigned char>(0.9*255) ;
unsigned char Geom_DN = static_cast<unsigned char>(0.4*255) ;
unsigned char VNvlm_UP = static_cast<unsigned char>(255/12) ;
unsigned char VNvlm_DN = 170 ; /* ~65% added on Dec 2006 */
unsigned char CSFBG_UP = static_cast<unsigned char>(255/4) ;

unsigned char Geom_DownsUp = static_cast<unsigned char>(Geom_DN/1.1) ; /*1.5*/

void
splitString (const std::string &text,
             const std::string &separators,
             std::vector<std::string> &words)
{
  const std::string::size_type n = text.length();
  std::string::size_type start = text.find_first_not_of(separators);
  while (start < n)
    {
    std::string::size_type stop = text.find_first_of(separators, start);
    if (stop > n) stop = n;
    words.push_back(text.substr(start, stop - start));
    start = text.find_first_not_of(separators, stop+1);
    }
}

void DownSample( ImageType::Pointer image,
                ImageType::Pointer oImage,
                float downSampleFactor)
{

  oImage->CopyInformation( image );

  ImageType::RegionType region = oImage->GetLargestPossibleRegion();
  ImageType::SpacingType spacing = oImage->GetSpacing();

  for (int k = 0; k < 3; k++)
  {
    float extent = static_cast<float>(region.GetSize(k))*spacing[k];
    spacing[k] *= downSampleFactor;
    int nPixel = static_cast<int>(extent/spacing[k]);
    region.SetSize(k, nPixel);
  }

  oImage->SetSpacing( spacing );
  oImage->SetRegions( region );
  oImage->Allocate();
  oImage->FillBuffer( 0 );

  itk::NearestNeighborInterpolateImageFunction< ImageType, double >::Pointer interpolator = 
    itk::NearestNeighborInterpolateImageFunction< ImageType, double >::New();

  interpolator->SetInputImage( image );
  
  itk::ImageRegionIteratorWithIndex<ImageType> itOut( oImage, region );
  for (itOut.GoToBegin(); !itOut.IsAtEnd(); ++itOut)
  {
    ImageType::IndexType idx = itOut.GetIndex();
    ImageType::PointType pt;
    
    oImage->TransformIndexToPhysicalPoint( idx, pt );
    itk::ContinuousIndex<double, 3> cIdx;
    image->TransformPhysicalPointToContinuousIndex( pt, cIdx );

    if( image->GetLargestPossibleRegion().IsInside(cIdx) )
    {
      itOut.Set( static_cast<ImageType::PixelType> (interpolator->Evaluate(pt)) );
    }
  }
}

int main(int argc,char *argv[])  
{
	float XYZres = 1. ;
	int nThreads = 1;
	std::string fixedFilename;
	std::string movingFilename;
	std::string resampledFilename;
	std::string deformationFieldname;

  std::string movingGrayFilename; 
  std::string resampledGrayFilename; 

  std::string iterationasstring = "50,50,50";
  unsigned int iterations[3];
  iterations[0] = 50; iterations[1] = 50; iterations[2] = 50;

  std::string tissueLabelString = "10,150,250"; 
  unsigned int tissueLabel[3];
  tissueLabel[0] = 10; tissueLabel[1] = 150; tissueLabel[2] = 250; 

	itk::OStringStream msg;

	try
	{
		CmdLine cl ( "HAMMER Registration Test: Heirarchical Attribute Matching Mechanism for Elastic Registration, NA-MIC Kit",
			' ',
			"$Revision: 1.2 $" );

		// parse registration parameters from command line arguments
		msg.str ( "" ); msg << "Number of threads (default: " << nThreads << ")";
		ValueArg<int> ThreadArg ( "T", "Thread", msg.str(), false, nThreads, "int", cl );
		msg.str ( "" ); msg << "sample rate (default: " << XYZres << ")";
		ValueArg<float> SampleRateArg ( "s", "Sample", msg.str(), false, XYZres, "float", cl );
		msg.str ( "" ); msg << "Number of iterations (default: 50,50,50)";
    ValueArg<std::string> IterationArg ( "i", "iterations", msg.str(), false, "50,50,50", "std::vector<unsigned int>", cl );

    msg.str("");msg << "Tissue label for csf, gm, and wm (in this order) in the input images. (default: " << tissueLabelString << ")"; \
    TCLAP::ValueArg<std::string > tissueLabelArg("l", "tissuelabel", msg.str(), 0, tissueLabelString, "std::vector<int>", cl); \


		UnlabeledValueArg<std::string> FixedImageArg ( "fixed", "Fixed image filename", 1, fixedFilename, "string", cl );
		UnlabeledValueArg<std::string> MovingImageArg ( "moving", "Moving image filename", 1, movingFilename, "string", cl );
		UnlabeledValueArg<std::string> MovingGrayImageArg ( "movinggray", "Moving intensity image filename", 1, movingGrayFilename, "string", cl );
		UnlabeledValueArg<std::string> ResampleImageArg ( "resampled", "Resampled image filename", 1, resampledFilename, "string", cl );
		UnlabeledValueArg<std::string> ResampleGrayImageArg ( "resampledgray", "Resampled intensity image filename", 1, resampledGrayFilename, "string", cl );
		UnlabeledValueArg<std::string> DeformationFieldArg ( "deformation", "Deformation field filename", 1, deformationFieldname, "string", cl );

		cl.parse ( argc, argv );

		// parse registration parameters from command line arguments
		nThreads = ThreadArg.getValue();
		XYZres =  SampleRateArg.getValue();
    iterationasstring = IterationArg.getValue();
    tissueLabelString = tissueLabelArg.getValue();

    std::vector<std::string> words; 
    std::string sep(","); 
    splitString(iterationasstring, sep, words); 

    if (words.size() >= 3)
    {
      for (unsigned int _j = 0; _j <3; ++_j)
      {
        iterations[_j] = atoi(words[_j].c_str()); 
      }
    }
    else
    {
      for (unsigned int _j = 0; _j <words.size(); ++_j)
      {
        iterations[_j] = atoi(words[_j].c_str()); 
      }
    }
    
    std::vector<std::string> words2; 
    splitString(tissueLabelString, sep, words2); 
    for (unsigned int _j = 0; _j < 3; ++_j)
    {
      tissueLabel[_j] = atoi(words2[_j].c_str()); 
    }

		fixedFilename = FixedImageArg.getValue();
		movingFilename = MovingImageArg.getValue();
		movingGrayFilename = MovingGrayImageArg.getValue();
		resampledFilename = ResampleImageArg.getValue();
		resampledGrayFilename = ResampleGrayImageArg.getValue();
		deformationFieldname = DeformationFieldArg.getValue();
	} 
	catch ( ArgException exception ) {
		std::cerr << "error: " << exception.error() << " for arg " << exception.argId() << std::endl;
		exit ( EXIT_FAILURE );
	}

	double scale = 7*(1./XYZres) ; if(scale<3) scale=3; printf("scale=%f\n", scale) ;

	/***** Model image, segmented *****/
	/*Img_XY = 256 ;*/	 

	printf("\nmodel image : %s\n", fixedFilename.c_str()) ;  
	printf("subject image : %s\n", movingFilename.c_str()) ;  

	/*** Load in fixed image and compute the attribute vectors ***/
	ReaderType::Pointer ImgReader = ReaderType::New();
	ImgReader->SetFileName( fixedFilename.c_str() );
	try
	{
		ImgReader->Update();
	}
	catch( itk::ExceptionObject *ex )
	{
		std::cerr << ex << std::endl;
	}
	ImageType::Pointer fImg0 = ImgReader->GetOutput();
	fImg0->DisconnectPipeline();
	std::cout << "Fixed image file read in\n";

	/*** Load in moving image and compute the attribute vectors ***/
	ImgReader->SetFileName( movingFilename.c_str() );
	try
	{
		ImgReader->Update();
	}
	catch( itk::ExceptionObject *ex )
	{
		std::cerr << ex << std::endl;
	}
	ImageType::Pointer mImg0 = ImgReader->GetOutput();
	mImg0->DisconnectPipeline();
	std::cout << "Moving image file read in\n";

  // convert to 10.150,250 label 
  int csfLabel = tissueLabel[0];
  int gmLabel = tissueLabel[1];
  int wmLabel = tissueLabel[2];

  itk::ImageRegionIterator<ImageType> itFixed( fImg0, fImg0->GetLargestPossibleRegion() );
  for (itFixed.GoToBegin(); !itFixed.IsAtEnd(); ++itFixed)
    {
    ImageType::PixelType p = itFixed.Get();
    if (p == csfLabel)
      {
      itFixed.Set( 10 );
      }
    if (p == gmLabel)
      {
      itFixed.Set( 150 );
      }
    if (p == wmLabel)
      {
      itFixed.Set( 250 );
      }
    }

  itk::ImageRegionIterator<ImageType> itMoving( mImg0, mImg0->GetLargestPossibleRegion() );
  for (itMoving.GoToBegin(); !itMoving.IsAtEnd(); ++itMoving)
    {
    ImageType::PixelType p = itMoving.Get();
    if (p == csfLabel)
      {
      itMoving.Set( 10 );
      }
    if (p == gmLabel)
      {
      itMoving.Set( 150 );
      }
    if (p == wmLabel)
      {
      itMoving.Set( 250 );
      }
    }

  //ImageType::Pointer fImg = ImageType::New();
	//ImageType::Pointer mImg = ImageType::New();

	//DownSample( fImg0, fImg, XYZres );
	//DownSample( mImg0, mImg, XYZres );

	typedef itk::HammerDeformableRegistrationImageFilter<
		ImageType,
		DeformationFieldType> RegistrationFilterType;
	RegistrationFilterType::Pointer hammer = RegistrationFilterType::New();

	// hammer only need to know the fixed image, moving image, and 
	// how attribute vectors are computed (also similarity computation)
	// to perform registration

	hammer->SetFixedImage( fImg0 );
	hammer->SetMovingImage( mImg0 );

	// ----------------------------------------------------------
	// define an attribute vector filter object and set it to the 
	// hammer registration filter so the registration know   
	// 1. How to compute attribute vectors
	// 2. How to compute similarity between attribute vectors
	AttributeFilterType::Pointer avFilter = AttributeFilterType::New();
	// hammer->SetAttributeVectorFilter( avFilter );
	// ----------------------------------------------------------


	// ----------------------------------------------------------
	// the following lines set parameters for HAMMER registration
	// all these parameters should have default values
  hammer->SetIterations( iterations[0], iterations[1], iterations[2] );
  hammer->SetDeformRate(0.05);
  hammer->SetPointMatchingThreshold(0.8);
  hammer->SetSubvolumnSimilarityThreshold(0.6);
  hammer->SetSearchRadius(12);
  // hammer->SetNumberOfMaximumDrivingVoxels( 5000 );
  // hammer->SetSigma ( 2 ); 
  // ----------------------------------------------------------
  
  itk::PluginFilterWatcher watchHammer(hammer,   "Hammer");
 
  clock_t Start = clock();
 	hammer->Update();
 	clock_t End = clock();

	std::cout<<"Execution time = "<<(End-Start)/CLOCKS_PER_SEC<<"s"<<std::endl;

  typedef itk::WarpImageFilter<
		ImageType, 
		ImageType,
		DeformationFieldType  >     WarperType;

	typedef itk::NearestNeighborInterpolateImageFunction<
		ImageType,
		double >  InterpolatorType;

	WarperType::Pointer warper = WarperType::New();
	InterpolatorType::Pointer interpolator = InterpolatorType::New();

	warper->SetInput( mImg0 );
	warper->SetInterpolator( interpolator );
	warper->SetOutputDirection( fImg0->GetDirection() );
	warper->SetOutputSpacing( fImg0->GetSpacing() );
	warper->SetOutputOrigin( fImg0->GetOrigin() );
  warper->SetDeformationField(hammer->GetOutput());

	WriterType::Pointer      writer =  WriterType::New();
	writer->SetFileName( resampledFilename.c_str() );
	writer->SetInput( warper->GetOutput() );
	writer->Update();

  USReaderType::Pointer usReader = USReaderType::New();
  usReader->SetFileName( movingGrayFilename.c_str() );

  typedef itk::WarpImageFilter<
    UShortImageType, 
    UShortImageType,
    DeformationFieldType  >     UShortWarperType;

  UShortWarperType::Pointer warper2 = UShortWarperType::New();
  warper2->SetInput( usReader->GetOutput() );
  warper2->SetOutputDirection( fImg0->GetDirection() );
  warper2->SetOutputSpacing( fImg0->GetSpacing() );
  warper2->SetOutputOrigin( fImg0->GetOrigin() );
  warper2->SetDeformationField(hammer->GetOutput());

  typedef itk::LinearInterpolateImageFunction< UShortImageType, double >  LinearInterpolatorType;
  LinearInterpolatorType::Pointer lInterpolator = LinearInterpolatorType::New();

  warper2->SetInterpolator( lInterpolator );

  USWriterType::Pointer writer2 = USWriterType::New();
  writer2->SetFileName( resampledGrayFilename.c_str() );
  writer2->SetInput( warper2->GetOutput() );

  try
  {
    writer2->Update();
  }
  catch( itk::ExceptionObject *ex )
  {
    std::cerr << ex << std::endl;
  }


	return 0;
}



