#!/usr/bin/perl

###################################################################################################################
#                                                                                                                 #
#    This is the main file to generate SVM model for WML segmentation using multispectral MRI (T1/T2/PD/FLAIR)    #
#    last modified: Oct 6, 2007                                                                                   #
#    Author: Zhiqiang Lao                                                                                         #
#    Comments: Zhiqiang.Lao@uphs.upenn.edu                                                                        #
#                                                                                                                 #
###################################################################################################################

$argc = @ARGV;
&show_usage if ( $argc < 1 );


########################################
#       Reading input parameters       #
########################################
@default_dim       = (256,256,46); @dim = ();
@dim               = &my_get_string_list_with_default("-d", *ARGV, *default_dim);

@default_res       = (0.9375,0.9375,3.0); @res = ();
@res               = &my_get_string_list_with_default("-v", *ARGV, *default_res);

@default_weight    = (1.0,1.0,1.0,1.0); @w = ();
@w                 = &my_get_string_list_with_default("-w", *ARGV, *default_weight);

@default_threshold = (0.9,0.1);
@threshold         = &my_get_string_list_with_default("-t", *ARGV, *default_threshold);

$radius            = &my_get_string_with_default("-r", *ARGV, "2");
$max_iter          = &my_get_string_with_default("-i", *ARGV, "10");
$use_premask       = &my_get_string_with_default("-M", *ARGV, "0");
$temp_path         = &my_get_string_with_default("-P", *ARGV, "\/tmp\/");
$suffix1	   = &my_get_string_with_default("-s", *ARGV, "byte"); # "byte.match.smooth.img"  # "byte.cbq.match.smooth.img
$suffix2	   = &my_get_string_with_default("-S", *ARGV, "coreg.smooth.brain.correct.match"); # "byte.match.smooth.img"  # "byte.cbq.match.smooth.img

my $current_dir = `pwd`; chomp($current_dir); 
$current_dir = $current_dir."\/";
print "current dir: $current_dir\n";
$t1path           = &my_get_string_with_default("-q", *ARGV, $current_dir);

print "training image dimension:  ($dim[0], $dim[1], $dim[2])\n";
print "training image resolution: ($res[0], $res[1], $res[2])\n";
print "neighbor size: $radius\n";
print "max iter number: $max_iter\n";
print "weight: ($w[0], $w[1], $w[2], $w[3])\n";
print "use_premask: $use_premask\n";
print "path to put temporary result: $temp_path\n";
print "path to load the training data: $t1path\n";

$max_iter = $max_iter - 1;
$t2path                        = $t1path;
$pdpath                        = $t1path;
$flpath                        = $t1path;
$manual_segmentation_path      = $t1path;
$manual_segmentation_open_path = $t1path;
$lesion_premask_path           = $t1path;

print "t1path:  $t1path\nt2path:  $t2path\npdpath:  $pdpath\nflpath:  $flpath\n";
print "manual_segmentation_path:  $manual_segmentation_path\nmanual_segmentation_open_path:  $manual_segmentation_open_path\n";
print "lesion_premask_path:  $lesion_premask_path\n";

#@subs = (); $ind = 0;
#open LIST, "$ARGV[0]";
#while ($rec = <LIST>) {
#  chop($rec);
#  $subs[$ind] = $rec;
#  $ind = $ind + 1;
#}
#close LIST;

@subs = readdata($ARGV[0]);
$ind = $#subs+1; 
print "\nsubs: @subs\n\n";

$temp_folder = $temp_path;
if ($temp_path=~ m/\/tmp\//) {
	$temp_sub_folder = int(100000*rand ());
	$temp_folder = "$temp_path\/$temp_sub_folder";
}
if (!-e "$temp_folder") {;
	mkdir($temp_folder, 0755) or die "Cannot mkdir $temp_folder: $!"; 
}
chdir("$temp_folder") or die "Could not change to $temp_folder: $!";

#############################################
#       Read input & define variables       #
#############################################
@T1Files = ();@T2Files = ();@PDFiles = ();@FLFiles = ();
@ManualMaskFile = ();@ManualMaskFile_open = ();@LesionPremaskFile = ();
$DimPara2D = "-d$dim[0],$dim[1]";$DimPara3D = "-d$dim[0],$dim[1],$dim[2]";
$WeightPara= "-w$w[0],$w[1],$w[2],$w[3]";

$suffix	= "$suffix1.$suffix2.img";
for ($i=0; $i<$ind; $i++) {
  $T1Files[$i] = "$subs[$i].T1.$suffix";  
  $T2Files[$i] = "$subs[$i].T2.$suffix";  
  $PDFiles[$i] = "$subs[$i].PD.$suffix";   
  $FLFiles[$i] = "$subs[$i].FL.$suffix";    
  $ManualMaskFile[$i] = "$subs[$i].manual.mask.img";	# $ManualMaskFile[$i] = "$subs[$i].lesion.mask.img";
  $ManualMaskFile_open[$i]  = "$subs[$i].manual.mask.open.img";# $ManualMaskFile_open[$i]  = "$subs[$i].lesion.mask.open.img";  
  $LesionPremaskFile[$i] = "$subs[$i].FL.$suffix1.lesion.premask.img";
  print "$t1path/$T1Files[$i]\n";   
  system("cp $t1path/$T1Files[$i] .")==0 or die "copy failed: $?";		 
  system("cp $t2path/$T2Files[$i] .")==0 or die "copy failed: $?";		 
  system("cp $pdpath/$PDFiles[$i] .")==0 or die "copy failed: $?";
  system("cp $flpath/$FLFiles[$i] .")==0 or die "copy failed: $?";		 
  system("cp $manual_segmentation_path/$ManualMaskFile[$i] .")==0 or die "copy failed: $?";           
  system("cp $manual_segmentation_open_path/$ManualMaskFile_open[$i] .")==0 or die "copy failed: $?"; 
  system("cp $lesion_premask_path/$LesionPremaskFile[$i] .")==0 or die "copy failed: $?";             
}
print "\ntotal train subject number: $ind\n";
print "T1Files: @T1Files\nT2Files: @T2Files\nPDFiles: @PDFiles\nFLFiles: @FLFiles\nManualMaskFile: @ManualMaskFile\n";
print "ManualMaskFile_open: @ManualMaskFile_open\nLesionPremaskFile: @LesionPremaskFile\n";

$average_manual_segmentation_voxel_number = &get_manual_segmentation_voxel_number(*ManualMaskFile, *dim);
print "average_manual_segmentation_voxel_number: $average_manual_segmentation_voxel_number\n";

#########################################################
#  create intermedicate folder prepared for processing  #
#########################################################

open MANUAL_MASK, ">manual_mask.lst";
open MANUAL_MASK_OPEN, ">manual_mask_dilate.lst";
for ($i=0; $i<$ind; $i++) {
  print MANUAL_MASK "$ManualMaskFile[$i]\n";
  print MANUAL_MASK_OPEN "$ManualMaskFile_open[$i]\n";
}
close MANUAL_MASK;
close MANUAL_MASK_OPEN;

##################################
#   Build up initial SVM model   #
##################################
@cmd_lesion = ();@cmd_nonlesion = ();
for ($i=0; $i<$ind; $i++) {
  $ImgFiles = "$T1Files[$i] $T2Files[$i] $PDFiles[$i] $FLFiles[$i]";
  $cmd = "wmlgetselectedfeature $ImgFiles $ManualMaskFile[$i] $subs[$i].lesion.vec.0 $DimPara2D -t0 -m1 -n$average_manual_segmentation_voxel_number $WeightPara -r$radius";
  system($cmd) == 0 or die "$cmd failed: $?";
  $cmd = "wmlgetselectedfeature $ImgFiles $ManualMaskFile[$i] $subs[$i].nonlesion.vec.0 $DimPara2D -t1 -m1 -n$average_manual_segmentation_voxel_number $WeightPara -r$radius";
  system($cmd) == 0 or die "$cmd failed: $?";
}
print "total subject number: $ind\n";

$iter = 0; $iterPlus1 = $iter + 1;
&create_training_file(*subs, $iterPlus1);	# it takes as input $lsub[$i].(non)lesion.vec.0 and creates the train.example to be used in svm
$cmd = "svm_torch -v -ae -e 0.0002 -t 2 -st 6000 train.example $ARGV[1].0";
system($cmd) == 0 or die "$cmd failed: $?";
print "$ARGV[1].0 created\n";

#############################################################################
#                       Iteratively refine SVM model                        #
#############################################################################
if ($max_iter>2) { $max_iter = 2; }

open PROGRESS, ">progress.dat";print PROGRESS "threshold\tTPVF\tFPVF\n=================================\n";close PROGRESS;
$accuracy = 0.9;$svm_error = 0.005;
while (1) {

  $pre_iter = $iter;  $iter++;
  print "*****************************iteration $iter******************************************\n";
#############################################################################
#     Generate evaluation map using SVM model from previous iteration       #
#############################################################################
## In WMLTestSingleImageKeepAllFunctionValue_NoSmooth it classifies only the voxels in the lesion.premask (option -M) and saves the emap: $subs[$i].WML.mask.img.allvalue.$pre_iter
  for ($i=0; $i<$ind; $i++) {
    $ImgFiles = "$T1Files[$i] $T2Files[$i] $PDFiles[$i] $FLFiles[$i]";
    $cmd = "wmltestsingleimagekeepallfunctionvaluenosmooth $ImgFiles $ARGV[1].$pre_iter $subs[$i].WML.mask.img.allvalue.$pre_iter $DimPara2D -m1 $WeightPara -r$radius -M$LesionPremaskFile[$i]";
    system($cmd) == 0 or die "$cmd failed: $?";
    ##  lesion=-1 and non-lesion=1 so the small probabilities are for lesion => invert emap to have the large probabilities for lesion
    $cmd = "invertimgfloat $subs[$i].WML.mask.img.allvalue.$pre_iter $subs[$i].WML.mask.img.allvalue.invert.$pre_iter $DimPara2D";
    system($cmd) == 0 or die "$cmd failed: $?";
  }
  print "after get emap based on SVM model from previous iteration\n";

#############################################################################
#         threshold evaluation map to get binary lesion masks               #
#############################################################################
  $selected_threshold = 0.0;  $it = 1.0;  @err = (0.0,0.0,0.0);
  while (($err[0] < $accuracy) && ($it>-0.9)) {

    print "threshold: $it\n";
    for ($i=0; $i<$ind; $i++) {
      $cmd = "oneimgop $subs[$i].WML.mask.img.allvalue.invert.$pre_iter $subs[$i].WML.mask.img.$pre_iter -t$it -m0 -f1";
      system($cmd) == 0 or die "$cmd failed: $?";
    } 
    
    $it = $it - 0.1;

    open AUTO_MASK_LST, ">auto_mask.lst";
    for ($i=0; $i<$ind; $i++) { print AUTO_MASK_LST "$subs[$i].WML.mask.img.$pre_iter\n"; }
    $cmd = "wmlgetautomanualmaskdiff manual_mask.lst auto_mask.lst err.dat $DimPara3D -r$res[0],$res[1],$res[2]";
    system($cmd) == 0 or die "$cmd failed: $?";
    
    open ERR, "err.dat";    $rec=<ERR>;    close ERR;
    @err = split(' ', $rec);
  } ## end while (($err[0] < $accuracy)...
  $accuracy = $accuracy - 0.02;
  print "after threshold evaluation map to get binary lesion masks\n";
  
  $selected_threshold = $it + 0.1;
  open PROGRESS, ">>progress.dat";  print PROGRESS "$selected_threshold\t$err[0]\t$err[1]\n";
  close PROGRESS;
  print "after writing PROGRESS\n";

  if (($err[0]>$threshold[0] && $err[1]<$threshold[1]) || ($iter>$max_iter)) {
    print "going to exit...\n";
    $iter = $iter - 1;	# added 23Dec07 because it existed in version1 of svn 
    $cmd = "cp $temp_folder\/$ARGV[1].$iter $current_dir\/$ARGV[1]"; system($cmd) == 0 or die "$cmd failed: $?";
    clean_intermediate_files($current_dir, $temp_folder, $ARGV[1]);
    exit(1);
  }

  for ($i=0; $i<$ind; $i++) {
    $ImgFiles = "$T1Files[$i] $T2Files[$i] $PDFiles[$i] $FLFiles[$i]";
    $cmd = "openingcloseingregiongrowing3Dbinaryimgspecial $subs[$i].WML.mask.img.$pre_iter $subs[$i].WML.mask.img.$pre_iter.rg -t2 -v$res[2] -n1 -m1 -g1658";
    system($cmd) == 0 or die "$cmd failed: $?";
    
    # the next option calculates the overlap for image WML.mask and is effective also if the image is in gray scale, i.e. if the emap is used instead of the binary segmentation
    # -T0.08 shows the percentage of WML.mask voxels with label i (>0) that are inside the ManualMaskFile
    # for binary WML.masks: if TP > 0.08 then remove this subject from training (WML.mask.nonlesion.diff=empty), else keep this subject in training (WML.mask.nonlesion.diff=WML.mask.img)
    $cmd = "maskoutimg $subs[$i].WML.mask.img.$pre_iter.rg $ManualMaskFile_open[$i] $subs[$i].WML.mask.nonlesion.diff $DimPara3D "; 
    $cmd = "$cmd -t0";	# $cmd = "$cmd -t2 -T0.08"; # this was until Jan 15
    system($cmd) == 0 or die "$cmd failed: $?";
    $cmd = "wmlgetselectedfeature $ImgFiles $subs[$i].WML.mask.nonlesion.diff $subs[$i].nonlesion.vec.$iter $DimPara2D -t0 -m1 $WeightPara -r$radius -S";	# the number of voxels inside the lesion is small 
    system($cmd) == 0 or die "$cmd failed: $?";
  }

  for ($i=0; $i<$ind; $i++) {
    system("cat $subs[$i].nonlesion.vec.$iter >> delta.nonlesion.dat");
  }
  $current_total_nonlesion_number = &get_file_linenumber("delta.nonlesion.dat");
  system("\\rm delta.nonlesion.dat") == 0 or die "rm failed: $?";

  print "current_total_nonlesion_number: $current_total_nonlesion_number\n";
  if ($current_total_nonlesion_number < 50) {
    print "no further nonlesion tissue need to be selected\n";
    $iter = $iter - 1;	# added 23Dec07 because it existed in version1 of svn and also when you run it, the "$iter - 1" is the latest existing model
    $cmd = "cp $temp_folder\/$ARGV[1].$iter $current_dir\/$ARGV[1]"; system($cmd) == 0 or die "$cmd failed: $?";
    clean_intermediate_files($current_dir, $temp_folder, $ARGV[1]);
    exit(2);
  }

  $iterPlus1 = $iter + 1;
  &create_training_file(*subs, $iterPlus1);
  $cmd = "svm_torch -v -ae -e 0.0005 -t 2 -st 6000 train.example $ARGV[1].$iter";
  system($cmd) == 0 or die "$cmd failed: $?";
  unlink("train.example");
  $svm_error = $svm_error + 0.005;

  print "$ARGV[1].$iter created\n";
}  ## end while (1) ...


# ***************************************************************************
# call format: clean_intermediate_files($current_path, $temp_path, $model_prefix)
# ***************************************************************************
sub clean_intermediate_files
{
  local($lcurrent_path) = $_[0];
  local($ltemp_path)    = $_[1];
  local($lmdl_prefix)   = $_[2];

  my($lcmd);
  
  $lcmd = "$lcurrent_path\/"; system($cmd) == 0 or die "$lcmd failed: $?";
  $lcmd = "\\rm -fr $ltemp_path/*.img"; system($cmd) == 0 or die "$lcmd failed: $?"; # $lcmd = "\\rm -fr $ltemp_path"; system($cmd) == 0 or die "$lcmd failed: $?";
}


# ***************************************************************************
# call format: create_training_file(*sub, $iter)
# ***************************************************************************
sub create_training_file
  {
    local(*lsub) = $_[0];
    local($liter) = $_[1];
    
    my($i, $t, $total_line_number, $feature_length, $rec);
   
    my $allfiles = " ";        
    for ($i=0; $i<$#lsub+1; $i++) {
    	$allfiles = "$allfiles  $lsub[$i].lesion.vec.0 ";      
    }
    $cmd = "cat $allfiles > temp.example.lesion"; print "About to run:\n $cmd \n"; 
    system($cmd) == 0 or die "$cmd failed: $?";

    $allfiles = "temp.example.lesion ";  
    for ($t=0; $t<$liter; $t++) {        
      for ($i=0; $i<$#lsub+1; $i++) {
      	$allfiles = "$allfiles  $lsub[$i].nonlesion.vec.$t ";   
      }
    }
    $cmd = "cat $allfiles > temp.example"; print "About to run:\n $cmd \n"; 
    system($cmd) == 0 or die "$cmd failed: $?";   

    system("removeduplicateelementfromlists.pl temp.example temp.example.nodup");
    $total_line_number = &get_file_linenumber("temp.example.nodup");
    $feature_length = &get_file_columnnumber("temp.example.nodup");

    open TRAIN_EXAMPLE_HERE, ">train.example.line1";
    print TRAIN_EXAMPLE_HERE "$total_line_number        $feature_length\n";
    close TRAIN_EXAMPLE_HERE;
    $cmd = "cat train.example.line1  temp.example.nodup > train.example"; system($cmd) == 0 or die "$cmd failed: $?";  
      	

    unlink "temp.example.lesion","temp.example","temp.example.nodup","train.example.line1" ;
  }



# ***************************************************************************
# call format: get_manual_segmentation_voxel_number(*mask_list, *dim)
# ***************************************************************************
sub get_manual_segmentation_voxel_number
  {
    (*lmask_list) = $_[0];
    (*ldim) = $_[1];

    my($tmp_vol, $total_vol);
    $total_vol = 0;
    for ($i=0; $i<=$#lmask_list; $i++) {
      system("getwholeimagevolume $lmask_list[$i] tmp.vol -d$ldim[0],$ldim[1],$ldim[2]");
      open TTT, "tmp.vol";
      $tmp_vol = <TTT>;
      close TTT;
      $total_vol = $total_vol + $tmp_vol;
    }

    unlink("tmp.vol");
    return ($total_vol/($#lmask_list+1));
  }

# ***************************************************************************
# call format: exec_cmd_parallel(*cmd, $number_process)
# ***************************************************************************
sub exec_cmd_parallel
  {
    my($number_process);
    (*llcmd) = $_[0];
    ($number_process) = $_[1];
    print "number of process: $number_process\n";

    for ($t=0; $t<1; $t++) {
      print "$llcmd[$t]\n";
    }

    my(@lkidpid,$p);
    @lkidpid = ();
    for ($p=0; $p<$number_process; $p++) {
      if (!defined($lkidpid[$p] = fork())) {
	die "cannot fork: $!";
      } elsif ($lkidpid[$p] == 0) {
	exec("$llcmd[$p]");
	exit;
      }
    }

    for ($p=0; $p<$number_process; $p++) {
      waitpid($lkidpid[$p], 0);
    }
  }

# ******************************************************************************
# call format: exec_cmd_parallel_limited_cpu(*cmd, $number_process, $number_cpu)
# ******************************************************************************
sub exec_cmd_parallel_limited_cpu  {

  my($number_process,$number_cpu);

  (*lcmd)         = $_[0];
  $number_process = $_[1];
  $number_cpu     = $_[2];
  
  my($number_process_remain);
  $number_process_remain = $number_process % $number_cpu;
  $number_task = ($number_process - $number_process_remain) / $number_cpu;
  
  my($i,$j);
  @cmd_here = ();
  for ($i=0; $i<$number_task; $i++) {
    for ($j=0; $j<$number_cpu; $j++) {
      $cmd_here[$j] = $lcmd[$i*$number_cpu+$j];
    }
    &exec_cmd_parallel(*cmd_here, $number_cpu);
  }

  if ($number_process_remain != 0) {
    for ($j=0; $j<$number_process_remain; $j++) {
      $cmd_here[$j] = $lcmd[$number_process-$number_process_remain+$j];
    }
    &exec_cmd_parallel(*cmd_here, $number_process_remain);
  }
}


# ***************************************************************************
# call format: my_get_string_with_default($option, *argument_list, $default)
# ***************************************************************************
sub my_get_string_with_default
  {
    $option = $_[0];
    (*arg_list) = $_[1];
    $default = $_[2];

    local($count) = 0;
    while ($count < ($#arg_list + 1))
      {
        if (index($arg_list[$count], $option) != -1)
          {
            return (substr($arg_list[$count], 2));
          }
        $count = $count + 1;
      }
    return $default;
  }

# **********************************************************************************
# call format: my_get_string($option, *argument_list)
# **********************************************************************************
sub my_get_string
{
   $option = $_[0];
   (*arg_list) = $_[1];

   local($count) = 0;
   while ($count < ($#arg_list + 1))
   {
      if (index($arg_list[$count], $option) != -1)
      {
         return (substr($arg_list[$count], 2));
      }
      $count = $count + 1;
   }
}

# **********************************************************************************
# call format: my_get_string_list_with_default($option, *argument_list, *default)
# **********************************************************************************
sub my_get_string_list_with_default
  {
   $option = $_[0];
   (*arg_list) = $_[1];
   (*default) = $_[2];

   local($count) = 0;
   while ($count < ($#arg_list + 1)) 
     {
       if (index($arg_list[$count], $option) != -1)
	 {
	   $tmp_str = substr($arg_list[$count], 2);
	   @lst = split(/,/, substr($arg_list[$count], 2));
	   return (@lst);
	 }
       $count = $count + 1;
     }
   return (@default);
 }

# **********************************************************************************
# call format: get_file_linenumber(file)
# **********************************************************************************
sub get_file_linenumber
  {
    $file = $_[0];

    open TMP, "$file";
    local($tmp_count) = 0;
    while ($record = <TMP>)
      {
	$tmp_count = $tmp_count + 1;
      }
    close TMP;

    return $tmp_count;
  }

# **********************************************************************************
# call format: get_file_columnnumber(file)
# **********************************************************************************
sub get_file_columnnumber
  {
    local($file) = $_[0];

    open TMP, "$file";
    local($col_rec) = <TMP>;
    close TMP;
    @ttmp = split(' ', $col_rec);

    return ($#ttmp+1);
  }
  
 sub readdata {
## It reads data from a text file, splits them according to white space or new line 
## and saves them in an array. First it strips out the comments (marked with #). 
## Example: 
##	20 5 4 3 #hello
##	# this is an empty line
##	1 2
## Output array: 20 5 4 3 1 2  

	my $datafile = $_[0];
	my (@list, @tmp);
	
	if ( !(open(DATA,"$datafile")))	{
		print("Could not open file $datafile for reading: $!");
		return(0);
	}

	while(<DATA>)
	{		
		next if (/^\s*#/); 	# skip comments that are after text (not whitespace)	
		$_=~s/#.*//;		# strip comments
		chomp;                  # strip newline
    		$_=~s/^\s+//;           # strip leading whitespace
    		$_=~s/\s+$//;           # strip trailing whitespace
    		next unless length;     # anything left?
		
		@tmp = split(m/\s+/, $_);# @tmp = split(m/[\s]/, $_);
		push(@list, @tmp);		
	}
	close DATA;
	# print "finished reading: "; for (my $i=0; $i<=$#list; ++$i) { print $list[$i]." "; } print "\n";
	return(@list);        
} 
  

sub show_usage
{
  print "This program creates model file from a set of training set\n";
  print "for White Matter Lesion (WML) detection via adaBoost\n";
  print "\n\nUsage: $0 train_list model [options]\n\n";
  print " -d<int>,<int>,<int>  : dimension of input images in XY plane (default: 256,256,46)\n";
  print " -v<float>,<float>,<float>  : image resolution (default: 0.9375,0.9375,3.0)\n";
  print " -t<float>,<float>    : threshold for TPVF and FPVF (default: 0.9,0.1)\n";
  print " -i<int>              : maximum iteration number (default: 10)\n";
  print " -r<int>              : neighbor size (default: 2)\n";
  print " -M<int>              : use premask or not 0: not use 1: use (default: not use)/* THIS OPTION IS NOT TAKEN INTO ACCOUNT IN THE SCRIPT; THE MASK IS ALWAYS USED */\n";
  #print " -p<string>           : path where this package installed\n";
  print " -P<string>           : path where to put intermediate result (default: /tmp)\n";
  print " -q<string>           : path where the training data reside (default: current)\n";
  print " -w <float>,<float>,<float>,<float> : weighting parameters for t1, t2, pd, flair respectively (default: 1.0,1.0,1.0,1.0)\n";
  print " -s<string>           : prefix after ID.SequenceName (default: byte)\n";
  print " -S<string>           : prefix indicating the preprocessing steps applied (default: coreg.smooth.brain.correct.match) \n";
  print "\n Example: With the default options for the prefixes, the FL image has the filename 301D00262.FL.byte.coreg.smooth.brain.correct.match.img\n";
  print "            and the premask 301D00262.FL.byte.lesion.premask.img\n";
  print "\n contact: Zhiqiang.Lao\@uphs.upenn.edu\n";
  exit(1);
}
