#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <pgstd.h>
/*-----------------------------------------------------------------
:   File: anslice.c
:   Reslice an image using Unser's spline interpolation.  No
:   antialiasing is performed.  Also creates an analyze header.
:----------------------------------------------------------------*/

/* Constants */
#define USHORT 0
#define SHORT 1
#define BYTE 2
#define MAX_STRING_SIZE 512
#define MAX_BYTE 255
#define MAX_SHORT 32767
#define MAX_USHORT 65535
#define PI 3.1415927 
#define PIX_SIZE 0.9375
#define SLICE_SIZE 1.5

/* Main processing function declaration */
int affineTransform (double transform[][4],double origin[],float *inPtr,
		     float *outPtr,long nx,long ny,long nz,
		     int interpolationOrder,float background, double voxel[3]);
void make_hdr(char *fname,int width,int height,int depth,int numvol,
	      char *datatype,int max,int min,float voxel[3],int flipped);

/* Option arguments variables */
int getopts(int argc,char *const *argv,char *optstring);/**/
int optsind=1;
char optsarg[MAX_STRING_SIZE];

char *usage[] = {
"                                                 ",
"anslice - reslices 3D volume, given translation, rotation and scaling",
"          parameters and generates ANALYZE header (Version 2.0a)",
"Usage: anslice [options] input_file output_file",
"       options: -N %d %d %d size of X,Y,Z dimensions, usually ",
"                            columns,rows,slices (default (256,256,auto))",
"                -o %f %f %f origin (default 0.5 x dimensions of volume)",
"                -c %f %f %f scaling along x,y,z-axes (default (1,1,1))",
"                -t %f %f %f translation (default (0,0,0))",
"                -r %f %f %f Rx,Ry,Rz in degs (default (0,0,0))",
"                            Rx is Pitch, Ry is Yaw, Rz is Roll",
"                -n %d spline order 0-7 (default 3)",
"                -B %d background value (default 0)",
"                -V %f %f %f voxel size (default 0.9375 x 0.9375 x 1.5)",
"                -b byte data type",
"                -s short data type",
"                -8 truncate ouput as bytes, regardless of input data type",
"                -e rescale output as bytes, regardless of input data type",
"                -u unsigned short data type (default)",
"                -v verbose mode on (default)",
"                -q verbose mode off",
"                           ",
NULL
};

void ShowUsage(void)
{
  char buf[BUFSIZ];
  int i;

  setbuf(stderr, buf);
  for (i = 0; usage[i] != NULL; i++)
    fprintf(stderr, "%s\n", usage[i]);
  exit(-1);
}

/* Function to multiply 4x4 matrices.  One of the input matrices
   can be equal to the output matrix. */
void MatrixMultiply(double inMatrix1[4][4],
		    double inMatrix2[4][4],
		    double outMatrix[4][4])
{
  int i,j,k;
  double tempMatrix[4][4];

  /* Perform multiplication */
  for (i=0;i<4;i++)
    for (j=0;j<4;j++) {
      tempMatrix[i][j] = 0.0;
      for (k=0;k<4;k++)
	tempMatrix[i][j] += inMatrix1[i][k]*inMatrix2[k][j];
    }
  /* Now recopy to output matrix */
  for (i=0;i<4;i++)
    for (j=0;j<4;j++)
      outMatrix[i][j] = tempMatrix[i][j];

  return;
}

int main(int argc,
	  char *argv[])
{
  PGbyte *Bbuffer;
  PGshort *Sbuffer;
  PGushort *Usbuffer;
  PGfloat *in_data,*out_data;
  char *in_fn,*out_fn;
  char filename[MAX_STRING_SIZE];
  char data_string[MAX_STRING_SIZE];
  int X=256,Y=256,Z=0;
  int length;
  int NewX,NewY,NewZ;
  double scaleMatrix[4][4],
         transMatrix[4][4],
         rotxMatrix[4][4],
         rotyMatrix[4][4],
         rotzMatrix[4][4],
         compMatrix[4][4],
         invMatrix[4][4];
  double origin[3]={-1.0,0.0,0.0};
  double scale[3]={1.0,1.0,1.0};
  double trans[3]={0.0,0.0,0.0};
  double roll=0.0,yaw=0.0,pitch=0.0;
  double dtemp;
  double xvector[3],yvector[3],zvector[3];
  int background = 0;
  int n=3;
  int data_type=USHORT;
  int opt,errflg;
  int verbose=PG_TRUE;
  int i,j;
  int retval;
  int write_byte=PG_FALSE;
  int rescale=PG_FALSE;
  int flipped=PG_TRUE;
  double voxel_orig[3]={PIX_SIZE,PIX_SIZE,SLICE_SIZE};
  float voxel[3];
  int max,min;
  int fmax;

    /* Parse command line */
  if (argc == 1) {
    ShowUsage();
    exit(1);
  }
  errflg = PG_FALSE;
  while ((opt = getopts(argc,argv, "bsu8evqB:n:r:t:c:o:N:V:")) != EOF)
    switch (opt) {
    case 'v':
      verbose = PG_TRUE;
      break;
    case 'q':
      verbose = PG_FALSE;
      break;
    case '8':
      write_byte = PG_TRUE;
      rescale = PG_FALSE;
      break;
    case 'e':
      write_byte = PG_TRUE;
      rescale = PG_TRUE;
      break;
    case 'b':
      data_type = BYTE;
      break;
    case 's':
      data_type = SHORT;
      break;
    case 'u':
      data_type = USHORT;
      break;
    case 'n':
      n = atoi(optsarg);
      if (n > 7 || n < 0) errflg = PG_TRUE;
      break;
    case 'B':
      background = atoi(optsarg);
      break;
    case 'r':
      pitch = atof(argv[optsind-1]);
      yaw = atof(argv[optsind++]);
      roll = atof(argv[optsind++]);
      break;
    case 't':
      trans[0] = atof(argv[optsind-1]);
      trans[1] = atof(argv[optsind++]);
      trans[2] = atof(argv[optsind++]);
      break;
    case 'c':
      scale[0] = atof(argv[optsind-1]);
      scale[1] = atof(argv[optsind++]);
      scale[2] = atof(argv[optsind++]);
      break;
    case 'o':
      origin[0] = atof(argv[optsind-1]);
      origin[1] = atof(argv[optsind++]);
      origin[2] = atof(argv[optsind++]);
      if (origin[0] < 0 || origin[1] < 0 || origin[2] < 0) errflg=PG_TRUE;
      break;
    case 'N':
      X = atoi(argv[optsind-1]);
      Y = atoi(argv[optsind++]);
      Z = atoi(argv[optsind++]);
      if (X <= 0 || Y <= 0 || Z <= 0) errflg=PG_TRUE;
      break;
    case 'V':
      voxel_orig[0] = atof(argv[optsind-1]);
      voxel_orig[1] = atof(argv[optsind++]);
      voxel_orig[2] = atof(argv[optsind++]);
      if (voxel_orig[0] <= 0 || voxel_orig[1] <= 0 || 
	  voxel_orig[2] <= 0) errflg=PG_TRUE;
      break;
    }
  if (errflg) {
    printf("Error: unknown option or parameter out of range.\n");
    ShowUsage();
    exit(1);
  }
  if (optsind < argc - 1) {
    in_fn = argv[optsind++];
    out_fn = argv[optsind];
  }
  else {
    ShowUsage();
    exit(1);
  }

  /* Read input image */
  if (Z == 0) {
    retval = pgFileSize(in_fn);
    if (retval == PG_ERROR) {
      printf("I/O Error: could not read file '%s'.\n",in_fn);
      exit(1);
    } 
    if (data_type == SHORT || data_type == USHORT)
      retval = retval / 2;
    Z = (int)retval / (X*Y); 
  }
  length = X*Y*Z;

  /* Allocate arrays and read image file */
  in_data = pgFvector(0,length-1);
  if (verbose)
    printf("Reading %d x %d x %d image '%s'.\n",X,Y,Z,in_fn);
  switch (data_type) {
  case BYTE:
    Bbuffer = pgBvector(0,length-1);
    retval = pgReadByte(&Bbuffer[0],length,0,in_fn);
    if (retval == PG_ERROR) {
      printf("I/O Error: could not read file '%s'.\n",in_fn);
      exit(1);
    } 
    for (i=0;i<length;i++)
          in_data[i] = (float)Bbuffer[i];
    pgFreeBvector(Bbuffer,0,length-1);
    break;
  case SHORT:
    Sbuffer = pgSvector(0,length-1);
    retval = pgReadShort(&Sbuffer[0],length,0,in_fn);
    if (retval == PG_ERROR) {
      printf("I/O Error: could not read file '%s'.\n",in_fn);
      exit(1);
    } 
    for (i=0;i<length;i++)
          in_data[i] = (float)Sbuffer[i];
    pgFreeSvector(Sbuffer,0,length-1);
    break;
  case USHORT:
    Usbuffer = pgUsvector(0,length-1);
    retval = pgReadUshort(&Usbuffer[0],length,0,in_fn);
    if (retval == PG_ERROR) {
      printf("I/O Error: could not read file '%s'.\n",in_fn);
      exit(1);
    } 
    for (i=0;i<length;i++)
          in_data[i] = (float)Usbuffer[i];
    pgFreeUsvector(Usbuffer,0,length-1);
    break;
  }
  
  /* Determine origin if necessary */
  if (origin[0] == -1) {
    origin[0] = (double)X / 2.0 - 0.5;
    origin[1] = (double)Y / 2.0 - 0.5;
    origin[2] = (double)Z / 2.0 - 0.5;
  }
  if (verbose) {
    printf("Using orgin of (%f, %f, %f).\n",origin[0],origin[1],origin[2]);
    printf("      spline interpolation order %d.\n",n);
    printf("      background value of %d.\n",background);
  }

  /*************************************************/
  /* Now compute homogeneous transformation matrix */
  /*************************************************/

  /* Set up translation matrix */
  for (i=0;i<4;i++)
    for (j=0;j<4;j++) {
      if (i==j) transMatrix[i][j] = 1.0;
      else if (j==3) transMatrix[i][j] = trans[i];
      else transMatrix[i][j] = 0.0;
    }
  /* Set up scaling matrix */
  for (i=0;i<4;i++)
    for (j=0;j<4;j++) {
      if (i==3 && j==3) scaleMatrix[i][j] = 1.0;
      else if (i==j && i !=3) scaleMatrix[i][j] = scale[i];
      else scaleMatrix[i][j] = 0.0;
    }
  /* Now convert all angles to radians */
  roll = roll * PI / 180.0;
  yaw = yaw * PI / 180.0;
  pitch = pitch * PI / 180.0;

  /* Set up rotation matrices */
  for (i=0;i<4;i++)
    for (j=0;j<4;j++) {
      rotzMatrix[i][j] = 0.0;
      rotyMatrix[i][j] = 0.0;
      rotxMatrix[i][j] = 0.0; 
    }
  rotxMatrix[1][1] = cos(pitch); rotxMatrix[1][2] = -sin(pitch);
  rotxMatrix[2][1] = sin(pitch); rotxMatrix[2][2] = cos(pitch);
  rotxMatrix[0][0] = rotxMatrix[3][3] = 1.0;

  rotyMatrix[2][2] = cos(yaw); rotyMatrix[2][0] = -sin(yaw);
  rotyMatrix[0][2] = sin(yaw); rotyMatrix[0][0] = cos(yaw);
  rotyMatrix[1][1] = rotyMatrix[3][3] = 1.0;

  rotzMatrix[0][0] = cos(roll); rotzMatrix[0][1] = -sin(roll);
  rotzMatrix[1][0] = sin(roll); rotzMatrix[1][1] = cos(roll);
  rotzMatrix[2][2] = rotzMatrix[3][3] = 1.0;

  /* Now compute composite matrix */
  /* modified by Zhiqiang Lao 3/28/2006 */
  /* to fit parameters from volrot */
  //  MatrixMultiply(rotxMatrix,rotyMatrix,compMatrix); 
  //  MatrixMultiply(compMatrix,rotzMatrix,compMatrix); 
  MatrixMultiply(rotzMatrix,rotyMatrix,compMatrix); 
  MatrixMultiply(compMatrix,rotxMatrix,compMatrix); 
  MatrixMultiply(compMatrix,scaleMatrix,compMatrix); 
  MatrixMultiply(compMatrix,transMatrix,compMatrix); 

  if (verbose) {
    printf("Computed composite transformation matrix:\n");
    for (i=0;i<4;i++) {
      for (j=0;j<4;j++)
	printf("\t%f",compMatrix[i][j]);
      printf("\n");
    }
  }

  /* Compute inverse matrix for voxel size computation */
  scaleMatrix[0][0] = 1.0/scaleMatrix[0][0];
  scaleMatrix[1][1] = 1.0/scaleMatrix[1][1];
  scaleMatrix[2][2] = 1.0/scaleMatrix[2][2];
  dtemp = rotxMatrix[1][2]; rotxMatrix[1][2] = rotxMatrix[2][1];
  rotxMatrix[2][1] = dtemp;
  dtemp = rotyMatrix[0][2]; rotyMatrix[0][2] = rotyMatrix[2][0];
  rotyMatrix[2][0] = dtemp;
  dtemp = rotzMatrix[0][1]; rotzMatrix[0][1] = rotzMatrix[1][0];
  rotzMatrix[1][0] = dtemp;
  MatrixMultiply(scaleMatrix,rotzMatrix,invMatrix);
  MatrixMultiply(invMatrix,rotyMatrix,invMatrix);
  MatrixMultiply(invMatrix,rotxMatrix,invMatrix);
  /*  if (verbose) {
    printf("Computed inverse composite transformation matrix:\n");
    for (i=0;i<4;i++) {
      for (j=0;j<4;j++)
	printf("\t%f",invMatrix[i][j]);
      printf("\n");
    }
  }
  */

  /* Run Unser's code */
  out_data=pgFvector(0,length-1);
  if (verbose) {
    printf("Working...");fflush(stdout);
  }
  affineTransform(compMatrix,origin,(float *)in_data,(float *)out_data,
		  (long)X,(long)Y,(long)Z,n,(float)background,voxel_orig);
  if (verbose) printf("done.\n");
  
  /* Write output file */
  sprintf(filename,"%s.img",out_fn);
  if (verbose)
    printf("Writing %d x %d x %d image '%s'.\n",X,Y,Z,filename);
  if (write_byte) {
    data_type = BYTE;
    if (verbose) printf("  Using 8 bit output format.\n");
    if (rescale) {
      if (verbose) printf("  Rescaling output.\n");
      /* Determine maximum */
      max = 0.0;
      for (i=0;i<length;i++)
	if (out_data[i] > max) max = out_data[i];
      /* Rescale accordingly */
      for (i=0;i<length;i++)
	out_data[i] = out_data[i] * (float)MAX_BYTE / max;
    }
  }
  switch (data_type) {
  case BYTE:
    Bbuffer = pgBvector(0,length-1);
    for (i=0;i<length;i++)
      if (out_data[i] < 0.0) 
	Bbuffer[i] = 0;
      else if (out_data[i] > (float)MAX_BYTE)
	Bbuffer[i] = MAX_BYTE;
      else
	Bbuffer[i] = (PGbyte)(out_data[i] + 0.5);
    retval = pgWriteByte(&Bbuffer[0],length,filename);
    pgFreeBvector(Bbuffer,0,length-1);
    strcpy(data_string,"CHAR");
    max = MAX_BYTE; min = 0;
    break;
  case SHORT:
    Sbuffer = pgSvector(0,length-1);
    for (i=0;i<length;i++)
      if (out_data[i] < 0.0) 
	Sbuffer[i] = 0;
      else if (out_data[i] > (float)MAX_SHORT)
	Sbuffer[i] = MAX_SHORT;
      else
	Sbuffer[i] = (PGshort)(out_data[i] + 0.5);
    retval = pgWriteShort(&Sbuffer[0],length,filename);
    pgFreeSvector(Sbuffer,0,length-1);
    strcpy(data_string,"SHORT");
    max = MAX_SHORT; min = 0;
    break;
  case USHORT:
    Usbuffer = pgUsvector(0,length-1);
    for (i=0;i<length;i++)
      if (out_data[i] < 0.0) 
	Usbuffer[i] = 0;
      else if (out_data[i] > (float)MAX_USHORT)
	Usbuffer[i] = MAX_USHORT;
      else
	Usbuffer[i] = (PGushort)(out_data[i] + 0.5);
    retval = pgWriteUshort(&Usbuffer[0],length,filename);
    pgFreeUsvector(Usbuffer,0,length-1);
    strcpy(data_string,"SHORT");
    max = MAX_USHORT; min = 0;
    break;
  }
  if (retval == PG_ERROR) {
    printf("I/O Error: could not write file '%s'.\n",filename);
    exit(1);
  } 

  /* Create Analyze header */
  sprintf(filename,"%s.hdr",out_fn);
  voxel[0] = (float)voxel_orig[0];
  voxel[1] = (float)voxel_orig[1];
  voxel[2] = (float)voxel_orig[2];
  if (verbose) {
    printf("Creating header file '%s'.\n",filename);
    printf("   Voxel dimensions are %f x %f x %f.\n",voxel_orig[0],
	   voxel[1], voxel[2]);
  }
  make_hdr(filename,X,Y,Z,1,data_string,max,min,voxel,flipped);

  /* Clean up */
  pgFreeFvector(in_data,0,length-1);
  pgFreeFvector(out_data,0,length-1);

  return 0;
}
