#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <math.h>
#include <strings.h>
#include <mvcd.h>
#include <cres.h>
#include <matrixSHEN.h>  /*by SHEN*/

#define MaxOfUC  255 
#define YYES     1
#define NNO      0
#define X_PLN    11
#define Y_PLN    22
#define Z_PLN    33
#define ALL_PLN  1658

typedef struct TRANSFORM_PARA {
  float scale;
  float shift;
} TRANSFORM_PARA;

typedef struct SLICE_INFO {
  int ind;
  float m;
  float v;
} SLICE_INFO;

typedef struct BOUNDINGBOX {
  Ivector3d ll, ur;
} BOUNDINGBOX;

typedef struct HISTOGRAM {
  float h[MaxOfUC+1];
} HISTOGRAM;

void           show_usage(int argc, char **argv);
void           LinearlyInterpolateHistogram(float *histo);
void           smooth_histogram_byMedianFilter(float *histo);
float          medianvalue(float *vec, int len);
void           mysort(float *vec, int len);
void           PrintVector(float *vec, int len);
void           PrintHistogram(HISTOGRAM his);
void           MatchingTwoImages3D(int argc, char *argv[], Ivector3d dim, int step, int op);
float          vector_distance(float *vec1, float *vec2, int len);
void           vector_transform(float *vec, int len, float s, int t);
void           vector_InverseTransform(float *vec, int len, float s, int t);
float          GetMax(float *vec, int len) ;
void           Vector_Normalization(float *vec, int len) ;
int            GetImgZDim(char file[1000], Ivector3d dim);
void           ReadImgUC(char filename[80], unsigned char ***vol, Ivector3d dim);
void           WriteImgUC(char filename[80], unsigned char ***data, Ivector3d dim);
TRANSFORM_PARA MatchingTwoHistograms(float *histoA, float *histoB, Ivector3d dim);
TRANSFORM_PARA MatchingTwoHistograms_special(float *histoA, float *histoB, Ivector3d dim, int slice_no);
TRANSFORM_PARA MatchingTwoHistograms_old(float *histoA, float *histoB, Ivector3d dim);
void           InhomoCorrectionImages2D(int argc, char *argv[], Ivector3d dim, int op, int threshold);
void           InhomoCorrectionImages2D_improve(int argc, char *argv[], Ivector3d dim, int op, int threshold, int iter);
void           InhomoCorrectionImages3D(int argc, char *argv[], Ivector3d dim, int threshold, int OtherImg, char OtherImgName[1000]);
int            PickSlice2Start(unsigned char ***img, Ivector3d dim, int op, int start_id, int end_id);
float          GetMean(float *vec, int len);
float          GetVariance(float *vec, int len);
void           ApplyTransformation(unsigned char ***img, Ivector3d dim, TRANSFORM_PARA si, int start_no, int end_no, int direction);
void           GetHistogram(unsigned char ***img, Ivector3d dim, float *histo, int start_no, int end_no, int direction);
float          Entropy(Ivector3d pos, HISTOGRAM **LH);
float          MutualInformation(unsigned char **im1, unsigned char **im2, Ivector3d pos, Ivector3d dim, HISTOGRAM **LH_source, HISTOGRAM **LH_target);
float          JointEntropy(unsigned char **im1, unsigned char **im2, Ivector3d dim);
void           PrintElement2D(unsigned char **ele, Ivector3d dim);
void           CreateWeightingSlice(unsigned char ***img, Ivector3d dim, int ind_source, int ind_target, int wsz, float *histo_source, float *histo_target, float ratio, int cost_function);
void           SliceDifference(unsigned char **s1, unsigned char **s2, Ivector3d dim, float **diff);
float          ElementDifference(Ivector3d pos, int wsz, float **diff);
void           InhomoCorrectionImages3D_improve(int argc, char *argv[], Ivector3d dim, int threshold, int OtherImg, char OtherImgName[1000],int focus_slice_no, int iter);
void           InhomoCorrectionImages2D_improve_again(int argc, char *argv[], Ivector3d dim, int op, int threshold, int iter, int wsz, int cost_function);
float          GetWeightThreshold(float **weighting, Ivector3d dim, float ratio);
HISTOGRAM      **HISTOGRAMalloc2d(int i_size,int j_size);
void           HISTOGRAMfree2d(HISTOGRAM **array,int i_size);
void           ComputingLocalHistogramSlice(unsigned char **img, Ivector3d dim, int wsz, HISTOGRAM **LH);
void           ComputingLocalHistogramSlice_fast(unsigned char **img, Ivector3d dim, int wsz, HISTOGRAM **LH);
unsigned char  **UCalloc2d_test(int i_size,int j_size);
void           UCinitialization2d(unsigned char **arr, int xsz, int ysz);
void           Finitialization2d(float **arr, int xsz, int ysz);
void           HISTOGRAMinitialization2d(HISTOGRAM **arr, int xsz, int ysz);
void           WriteImgF(char filename[80], float **data, Ivector3d dim);

int main(int argc,char *argv[])
{
  int            c, num, step, op, threshold, OtherImg, focus_slice_number, iter, wsz, cost_function;
  Ivector3d      dim;
  extern char    *optarg;
  char           OtherImgName[1000];

  num = 2;
  if(argc < num) show_usage(argc, argv);

  dim.x = dim.y = 256;
  op = 33;
  threshold = 0;
  OtherImg = NNO;
  focus_slice_number = 1;
  wsz = 24;
  iter = 1;
  cost_function = 0;
  while((c=getopt(argc-2, argv+2,"d:t:l:o:n:i:w:c:")) != -1) 
    {
      switch(c) 
	{
	case 'd':
	  sscanf(optarg, "%d,%d", &dim.x, &dim.y);
	  break;
	  
	case 't':
	  op = atoi(optarg);
	  break;
	  
	case 'l':
	  threshold = atoi(optarg);
	  break;
      
	case 'o':
	  OtherImg = YYES;
	  sscanf(optarg, "%s", OtherImgName);
	  break;

	case 'n':
	  focus_slice_number = atoi(optarg);
	  break;

	case 'i':
	  iter = atoi(optarg);
	  break;

	case 'w':
	  wsz = atoi(optarg);
	  break;

	case 'c':
	  cost_function = atoi(optarg);
	  break;

	default:
	  break;
	}
    }
  
  printf("XY dim: (%d, %d)\n", dim.x, dim.y);
  printf("operation: %d\n", op);
  printf("threshold: %d\n", threshold);
  printf("number of focusing continuous slices: %d\n", focus_slice_number);
  printf("iteration number: %d\n", iter);
  printf("window size: %d\n", wsz);
  printf("cost function: %d\n", cost_function);
  
  if (OtherImg == YYES) printf("Other image: %s\n", OtherImgName);

  if ((op == X_PLN) || (op == Y_PLN) || (op == Z_PLN)) InhomoCorrectionImages2D_improve_again(argc, argv, dim, op, threshold, iter, wsz, cost_function);
  if (op == ALL_PLN) InhomoCorrectionImages3D_improve(argc, argv, dim, threshold, OtherImg, OtherImgName, focus_slice_number, iter);

  return 0;
}


void show_usage(int argc, char **argv)
{
  printf("USAGE: %s <input.img> <modified.input.img>\n\
\t -d <int>,<int>         : XY dimensions of input image (default: 256,256)\n\
\t -t <int>               : corrent inhomogeneity along which plane (default: 1658)\n\
\t                          X_PLN: 11\n\
\t                          Y_PLN: 22\n\
\t                          Z_PLN: 33\n\
\t                          ALL_PLN: 1658\n\
\t -l <int>               : threshold, to ignore the intesities less than this (default: 0)\n\
\t -n <int>               : # of continue slices to focus (default: 1)\n\
\t -i <int>               : # of iteration (default: 1)\n\
\t -w <int>               : window size (default: 24)\n\
\t -o <string>            : apply to other image\n\
\t -c <int>               : type of cost function 0: MI for focusing only (default: 0)\n\
\t                                                1: Difference for focusing only\n\
\t                                                2: MI*Difference for focusing only\n\
\t                                                3: MI for histogram computing\n\
\t                                                4: Difference for histogram computing\n\
\t                                                5: MI*Difference for histogram computing\n\
", argv[0]);
  exit(1);
}


void InhomoCorrectionImages2D(int argc, char *argv[], Ivector3d dim, int op, int threshold)
{
  unsigned char        ***img;
  int                  i, j, k, t, shift, temp, start_id, end_id, focus_id, *sum;
  float                scale, mindist, s, dist, distA, distB, **histo;
  FILE                 *fp, *fpp;
  float                *histoA, *histoB, *histoB_trans, *histoA_trans;
  double               TotalIntensityA, TotalIntensityB, NonZeroPnts ;
  char                 filename[180] ;
  TRANSFORM_PARA       si;

  dim.z = GetImgZDim(argv[1], dim);
  printf("\n\ndimA: (%d, %d, %d)\n", dim.x, dim.y, dim.z);

  img = UCalloc3d(dim.x, dim.y, dim.z);
  ReadImgUC(argv[1], img, dim);
  printf("%s read ok\n", argv[1]);

  /* To ignore background, by thresholding */
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) 
      for (j=0; j<dim.y; j++)
    	if( img[k][i][j]<threshold ) img[k][i][j]=0 ;

  switch (op) {
  case X_PLN:
    sum = Ialloc1d(dim.x);
    for (i=0; i<dim.x; i++)
      for (k=0; k<dim.z; k++)
	for (j=0; j<dim.y; j++)
	  if (img[k][i][j] != 0) sum[i]++;

    for (i=0; i<dim.x; i++)
      if ((sum[i] == 0) && (sum[i+1] != 0)) {
	start_id = i+1;
	break;
      }
    for (i=dim.x-1; i>=0; i--)
      if ((sum[i] == 0) && (sum[i-1] != 0)) {
	end_id = i-1;
	break;
      }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, op, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);

    histo = Falloc2d(dim.x, 256);
    for (i=focus_id; i>start_id-2; i--) {
      
      for (k=0; k<dim.z; k++)
	for (j=0; j<dim.y; j++) histo[i][img[k][i][j]] += 1.0;
      for (k=0; k<dim.z; k++)
	for (j=0; j<dim.y; j++) histo[i-1][img[k][i-1][j]] += 1.0;
      printf("slice: %d ", i-1);
      si = MatchingTwoHistograms(histo[i-1], histo[i], dim);
      
      for (k=0; k<dim.z; k++)
	for (j=0; j<dim.y; j++)
	  if (img[k][i-1][j] != 0) {
	    temp = (int)((img[k][i-1][j] - si.shift)*si.scale + 0.5);
	    if (temp < 0)   temp = 1; 
	    if (temp > MaxOfUC) temp = MaxOfUC;
	    
	    img[k][i-1][j] = (unsigned char)temp;
	  }
    }

    for (i=focus_id; i<end_id-2; i++) {

      for (k=0; k<dim.z; k++)
	for (j=0; j<dim.y; j++) histo[i][img[k][i][j]] += 1.0;
      for (k=0; k<dim.z; k++)
	for (j=0; j<dim.y; j++) histo[i+1][img[k][i+1][j]] += 1.0;
      printf("slice: %d ", i+1);
      si = MatchingTwoHistograms(histo[i+1], histo[i], dim);

      for (k=0; k<dim.z; k++)
	for (j=0; j<dim.y; j++) 
	  if (img[k][i+1][j] != 0) {
	    temp = (int)((img[k][i+1][j] - si.shift)*si.scale + 0.5);
	    if (temp < 0)   temp = 1; 
	    if (temp > MaxOfUC) temp = MaxOfUC;
	    
	    img[k][i+1][j] = (unsigned char)temp;
	  }
    }
    free(sum);
    Ffree2d(histo, dim.x);
    break;

  case Y_PLN:
    sum = Ialloc1d(dim.y);
    for (j=0; j<dim.y; j++) 
      for (k=0; k<dim.z; k++)
	for (i=0; i<dim.x; i++)
	  if (img[k][i][j] != 0) sum[j]++;

    for (j=0; j<dim.y; j++)
      if ((sum[j] == 0) && (sum[j+1] != 0)) {
	start_id = j+1;
	break;
      }
    for (j=dim.y-1; j>=0; j--)
      if ((sum[j] == 0) && (sum[j-1] != 0)) {
	end_id = j-1;
	break;
      }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, op, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);

    histo = Falloc2d(dim.y, 256);
    for (j=focus_id; j>start_id; j--) {
      
      for (k=0; k<dim.z; k++)
	for (i=0; i<dim.x; i++) histo[j][img[k][i][j]] += 1.0;
      for (k=0; k<dim.z; k++)
	for (i=0; i<dim.x; i++) histo[j-1][img[k][i][j-1]] += 1.0;
      printf("slice: %d ", j-1);
      si = MatchingTwoHistograms(histo[j-1], histo[j], dim);

      for (k=0; k<dim.z; k++)
	for (i=0; i<dim.x; i++)
	  if (img[k][i][j-1] != 0) {
	    temp = (int)((img[k][i][j-1] - si.shift)*si.scale + 0.5);
	    if (temp < 0)   temp = 1; 
	    if (temp > MaxOfUC) temp = MaxOfUC;
	    
	    img[k][i][j-1] = (unsigned char)temp;
	  }
    }

    for (j=focus_id; j<end_id; j++) {
      
      for (k=0; k<dim.z; k++)
	for (i=0; i<dim.x; i++) histo[j][img[k][i][j]] += 1.0;
      for (k=0; k<dim.z; k++)
	for (i=0; i<dim.x; i++) histo[j+1][img[k][i][j+1]] += 1.0;
      printf("slice: %d ", j-1);
      si = MatchingTwoHistograms(histo[j+1], histo[j], dim);

      for (k=0; k<dim.z; k++)
	for (i=0; i<dim.x; i++) 
	  if (img[k][i][j+1] != 0) {
	    temp = (int)((img[k][i][j+1] - si.shift)*si.scale + 0.5);
	    if (temp < 0)   temp = 1; 
	    if (temp > MaxOfUC) temp = MaxOfUC;
	    
	    img[k][i][j+1] = (unsigned char)temp;
	  }
    }
    free(sum);
    Ffree2d(histo, dim.z);
    break;

  case Z_PLN:
    sum = Ialloc1d(dim.z);
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
    	for (j=0; j<dim.y; j++) 
	  if (img[k][i][j] != 0) sum[k]++;
    //    PrintVector((float *)sum, 256);exit(1);

    for (k=0; k<dim.z; k++) {
      if (sum[k] != 0) {
	start_id = k;
	break;
      }
      if ((sum[k] == 0) && (sum[k+1] != 0)) {
	start_id = k+1;
	break;
      }
    }
    for (k=dim.z-1; k>=0; k--) {
      if (sum[k] != 0) {
	end_id = k;
	break;
      }
      if ((sum[k] == 0) && (sum[k-1] != 0)) {
	end_id = k-1;
	break;
      }
    }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, op, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);

    histo = Falloc2d(dim.z, 256);
    for (k=focus_id; k>start_id; k--) {
      
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) histo[k][img[k][i][j]] += 1.0;
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) histo[k-1][img[k-1][i][j]] += 1.0;
      printf("slice: %d ", k-1);
      si = MatchingTwoHistograms(histo[k-1], histo[k], dim);

      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) 
	  if (img[k-1][i][j] != 0) {
	    temp = (int)((img[k-1][i][j] - si.shift)*si.scale + 0.5);
	    if (temp < 0)   temp = 1; 
	    if (temp > MaxOfUC) temp = MaxOfUC;
	    
	    img[k-1][i][j] = (unsigned char)temp;
	  }
    }

    for (k=focus_id; k<end_id; k++) {
      
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) histo[k][img[k][i][j]] += 1.0;
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) histo[k+1][img[k+1][i][j]] += 1.0;
      printf("slice: %d ", k+1);
      si = MatchingTwoHistograms(histo[k+1], histo[k], dim);

      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) 
	  if (img[k+1][i][j] != 0) {
	    temp = (int)((img[k+1][i][j] - si.shift)*si.scale + 0.5);
	    if (temp < 0)   temp = 1; 
	    if (temp > MaxOfUC) temp = MaxOfUC;
	    
	    img[k+1][i][j] = (unsigned char)temp;
	  }
    }
    free(sum);
    Ffree2d(histo, dim.z);
    break;

  default:
    break;
  }
      
  /* save result */
  WriteImgUC(argv[2], img, dim);
  UCfree3d(img, dim.z, dim.x);
}


void InhomoCorrectionImages2D_improve(int argc, char *argv[], Ivector3d dim, int op, int threshold, int iter)
{
  unsigned char        ***img;
  int                  i, j, k, l, t, shift, temp, start_id, end_id, focus_id, *sum;
  float                scale, mindist, s, dist, distA, distB, **histo;
  FILE                 *fp, *fpp;
  float                *histoA, *histoB, *histoB_trans, *histoA_trans;
  double               TotalIntensityA, TotalIntensityB, NonZeroPnts ;
  char                 filename[180] ;
  TRANSFORM_PARA       si;

  dim.z = GetImgZDim(argv[1], dim);
  printf("\n\ndimA: (%d, %d, %d)\n", dim.x, dim.y, dim.z);

  img = UCalloc3d(dim.x, dim.y, dim.z);
  ReadImgUC(argv[1], img, dim);
  printf("%s read ok\n", argv[1]);

  /* To ignore background, by thresholding */
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) 
      for (j=0; j<dim.y; j++)
    	if( img[k][i][j]<threshold ) img[k][i][j]=0 ;

  switch (op) {
  case X_PLN:
    sum = Ialloc1d(dim.x);
    for (i=0; i<dim.x; i++)
      for (k=0; k<dim.z; k++)
	for (j=0; j<dim.y; j++)
	  if (img[k][i][j] != 0) sum[i]++;

    for (i=0; i<dim.x; i++)
      if ((sum[i] == 0) && (sum[i+1] != 0)) {
	start_id = i+1;
	break;
      }
    for (i=dim.x-1; i>=0; i--)
      if ((sum[i] == 0) && (sum[i-1] != 0)) {
	end_id = i-1;
	break;
      }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, op, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);

    histo = Falloc2d(dim.x, 256);
    for (l=0; l<iter; l++) {
      for (i=focus_id; i>start_id-2; i--) {
	
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++) histo[i][img[k][i][j]] += 1.0;
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++) histo[i-1][img[k][i-1][j]] += 1.0;
	printf("slice: %d ", i-1);
	si = MatchingTwoHistograms(histo[i-1], histo[i], dim);
	
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++)
	    if (img[k][i-1][j] != 0) {
	      temp = (int)((img[k][i-1][j] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k][i-1][j] = (unsigned char)temp;
	    }
      }
      
      for (i=focus_id; i<end_id-2; i++) {
	
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++) histo[i][img[k][i][j]] += 1.0;
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++) histo[i+1][img[k][i+1][j]] += 1.0;
	printf("slice: %d ", i+1);
	si = MatchingTwoHistograms(histo[i+1], histo[i], dim);
	
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++) 
	    if (img[k][i+1][j] != 0) {
	      temp = (int)((img[k][i+1][j] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k][i+1][j] = (unsigned char)temp;
	    }
      }
    }
    free(sum);
    Ffree2d(histo, dim.x);
    break;

  case Y_PLN:
    sum = Ialloc1d(dim.y);
    for (j=0; j<dim.y; j++) 
      for (k=0; k<dim.z; k++)
	for (i=0; i<dim.x; i++)
	  if (img[k][i][j] != 0) sum[j]++;

    for (j=0; j<dim.y; j++)
      if ((sum[j] == 0) && (sum[j+1] != 0)) {
	start_id = j+1;
	break;
      }
    for (j=dim.y-1; j>=0; j--)
      if ((sum[j] == 0) && (sum[j-1] != 0)) {
	end_id = j-1;
	break;
      }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, op, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);

    histo = Falloc2d(dim.y, 256);
    for (l=0; l<iter; l++) {
      for (j=focus_id; j>start_id; j--) {
	
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++) histo[j][img[k][i][j]] += 1.0;
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++) histo[j-1][img[k][i][j-1]] += 1.0;
	printf("slice: %d ", j-1);
	si = MatchingTwoHistograms(histo[j-1], histo[j], dim);
	
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++)
	    if (img[k][i][j-1] != 0) {
	      temp = (int)((img[k][i][j-1] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k][i][j-1] = (unsigned char)temp;
	    }
      }
      
      for (j=focus_id; j<end_id; j++) {
	
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++) histo[j][img[k][i][j]] += 1.0;
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++) histo[j+1][img[k][i][j+1]] += 1.0;
	printf("slice: %d ", j-1);
	si = MatchingTwoHistograms(histo[j+1], histo[j], dim);
	
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++) 
	    if (img[k][i][j+1] != 0) {
	      temp = (int)((img[k][i][j+1] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k][i][j+1] = (unsigned char)temp;
	    }
      }
    }
    free(sum);
    Ffree2d(histo, dim.z);
    break;

  case Z_PLN:
    sum = Ialloc1d(dim.z);
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
    	for (j=0; j<dim.y; j++) 
	  if (img[k][i][j] != 0) sum[k]++;

    for (k=0; k<dim.z; k++) {
      if (sum[k] != 0) {
	start_id = k;
	break;
      }
      if ((sum[k] == 0) && (sum[k+1] != 0)) {
	start_id = k+1;
	break;
      }
    }
    for (k=dim.z-1; k>=0; k--) {
      if (sum[k] != 0) {
	end_id = k;
	break;
      }
      if ((sum[k] == 0) && (sum[k-1] != 0)) {
	end_id = k-1;
	break;
      }
    }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, op, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);

    histo = Falloc2d(dim.z, 256);
    for (l=0; l<iter; l++) {
      for (k=focus_id; k>start_id; k--) {
	
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) histo[k][img[k][i][j]] += 1.0;
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) histo[k-1][img[k-1][i][j]] += 1.0;
	printf("slice: %d ", k-1);
	si = MatchingTwoHistograms(histo[k-1], histo[k], dim);
	
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) 
	    if (img[k-1][i][j] != 0) {
	      temp = (int)((img[k-1][i][j] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k-1][i][j] = (unsigned char)temp;
	    }
      }
      
      for (k=focus_id; k<end_id; k++) {
	
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) histo[k][img[k][i][j]] += 1.0;
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) histo[k+1][img[k+1][i][j]] += 1.0;
	printf("slice: %d ", k+1);
	si = MatchingTwoHistograms(histo[k+1], histo[k], dim);
	
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) 
	    if (img[k+1][i][j] != 0) {
	      temp = (int)((img[k+1][i][j] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k+1][i][j] = (unsigned char)temp;
	    }
      }
    }
    free(sum);
    Ffree2d(histo, dim.z);
    break;

  default:
    break;
  }
      
  /* save result */
  WriteImgUC(argv[2], img, dim);
  UCfree3d(img, dim.z, dim.x);
}

void InhomoCorrectionImages2D_improve_again(int argc, char *argv[], Ivector3d dim, int op, int threshold, int iter, int wsz, int cost_function)
{
  unsigned char        ***img;
  int                  i, j, k, l, t, shift, temp, start_id, end_id, focus_id, *sum, min_value, max_value, h;
  float                scale, mindist, s, dist, distA, distB, **histo, ratio, SliceVol1, SliceVol2, tmp_value, *histo1, *histo2;
  FILE                 *fp, *fpp;
  float                *histoA, *histoB, *histoB_trans, *histoA_trans;
  double               TotalIntensityA, TotalIntensityB, NonZeroPnts ;
  char                 filename[180] ;
  TRANSFORM_PARA       si;

  dim.z = GetImgZDim(argv[1], dim);
  printf("\n\ndimA: (%d, %d, %d)\n", dim.x, dim.y, dim.z);

  img = UCalloc3d(dim.x, dim.y, dim.z);
  ReadImgUC(argv[1], img, dim);
  printf("%s read ok\n", argv[1]);

  /* To ignore background, by thresholding */
  min_value = 999; max_value = -999;
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) 
      for (j=0; j<dim.y; j++)
    	if( img[k][i][j]<threshold ) img[k][i][j]=0;

  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) 
      for (j=0; j<dim.y; j++) {
	if (img[k][i][j] > max_value) max_value = img[k][i][j];
	if (img[k][i][j] < min_value) min_value = img[k][i][j];
      }

  /* remapping image to the range of [0, 180] added Aug 29, 2006*/
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) 
      for (j=0; j<dim.y; j++) 
	if (img[k][i][j] > 0) {
	  tmp_value = (int)((float)180*(img[k][i][j] - min_value) / (max_value - min_value) + 0.5);
	  if (tmp_value > 255) tmp_value = 255;
	  if (tmp_value < 0)   tmp_value = 0;
	  img[k][i][j] = (unsigned char)tmp_value;
      }

  switch (op) {
  case X_PLN:
    sum = Ialloc1d(dim.x);
    for (i=0; i<dim.x; i++)
      for (k=0; k<dim.z; k++)
	for (j=0; j<dim.y; j++)
	  if (img[k][i][j] != 0) sum[i]++;

    for (i=0; i<dim.x; i++)
      if ((sum[i] == 0) && (sum[i+1] != 0)) {
	start_id = i+1;
	break;
      }
    for (i=dim.x-1; i>=0; i--)
      if ((sum[i] == 0) && (sum[i-1] != 0)) {
	end_id = i-1;
	break;
      }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, op, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);

    histo = Falloc2d(dim.x, 256);
    for (l=0; l<iter; l++) {
      for (i=focus_id; i>start_id-2; i--) {
	
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++) histo[i][img[k][i][j]] += 1.0;
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++) histo[i-1][img[k][i-1][j]] += 1.0;
	printf("slice: %d ", i-1);
	si = MatchingTwoHistograms(histo[i-1], histo[i], dim);
	
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++)
	    if (img[k][i-1][j] != 0) {
	      temp = (int)((img[k][i-1][j] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k][i-1][j] = (unsigned char)temp;
	    }
      }
      
      for (i=focus_id; i<end_id-2; i++) {
	
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++) histo[i][img[k][i][j]] += 1.0;
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++) histo[i+1][img[k][i+1][j]] += 1.0;
	printf("slice: %d ", i+1);
	si = MatchingTwoHistograms(histo[i+1], histo[i], dim);
	
	for (k=0; k<dim.z; k++)
	  for (j=0; j<dim.y; j++) 
	    if (img[k][i+1][j] != 0) {
	      temp = (int)((img[k][i+1][j] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k][i+1][j] = (unsigned char)temp;
	    }
      }
    }
    free(sum);
    Ffree2d(histo, dim.x);
    break;

  case Y_PLN:
    sum = Ialloc1d(dim.y);
    for (j=0; j<dim.y; j++) 
      for (k=0; k<dim.z; k++)
	for (i=0; i<dim.x; i++)
	  if (img[k][i][j] != 0) sum[j]++;

    for (j=0; j<dim.y; j++)
      if ((sum[j] == 0) && (sum[j+1] != 0)) {
	start_id = j+1;
	break;
      }
    for (j=dim.y-1; j>=0; j--)
      if ((sum[j] == 0) && (sum[j-1] != 0)) {
	end_id = j-1;
	break;
      }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, op, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);

    histo = Falloc2d(dim.y, 256);
    for (l=0; l<iter; l++) {
      for (j=focus_id; j>start_id; j--) {
	
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++) histo[j][img[k][i][j]] += 1.0;
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++) histo[j-1][img[k][i][j-1]] += 1.0;
	printf("slice: %d ", j-1);
	si = MatchingTwoHistograms(histo[j-1], histo[j], dim);
	
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++)
	    if (img[k][i][j-1] != 0) {
	      temp = (int)((img[k][i][j-1] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k][i][j-1] = (unsigned char)temp;
	    }
      }
      
      for (j=focus_id; j<end_id; j++) {
	
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++) histo[j][img[k][i][j]] += 1.0;
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++) histo[j+1][img[k][i][j+1]] += 1.0;
	printf("slice: %d ", j-1);
	si = MatchingTwoHistograms(histo[j+1], histo[j], dim);
	
	for (k=0; k<dim.z; k++)
	  for (i=0; i<dim.x; i++) 
	    if (img[k][i][j+1] != 0) {
	      temp = (int)((img[k][i][j+1] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k][i][j+1] = (unsigned char)temp;
	    }
      }
    }
    free(sum);
    Ffree2d(histo, dim.z);
    break;

  case Z_PLN:
    sum = Ialloc1d(dim.z);
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
    	for (j=0; j<dim.y; j++) 
	  if (img[k][i][j] != 0) sum[k]++;
    
    for (k=0; k<dim.z; k++) {
      if (sum[k] != 0) {
	start_id = k;
	break;
      }
      if ((sum[k] == 0) && (sum[k+1] != 0)) {
	start_id = k+1;
	break;
      }
    }
    for (k=dim.z-1; k>=0; k--) {
      if (sum[k] != 0) {
	end_id = k;
	break;
      }
      if ((sum[k] == 0) && (sum[k-1] != 0)) {
	end_id = k-1;
	break;
      }
    }
    free(sum);
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, op, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);

    for (l=0; l<iter; l++) {
      for (k=focus_id; k>start_id; k--) {
	
	SliceVol1 = SliceVol2 = 0.0;
	histo1 = Falloc1d(MaxOfUC+1);
	histo2 = Falloc1d(MaxOfUC+1);
	for (h=0; h<MaxOfUC+1; h++) histo1[h] = histo2[h] = 0.0;

	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) {
	    if (img[k][i][j] > 0) {
	      histo1[img[k][i][j]] += 1.0;
	      SliceVol1 += 1.0;
	    }
	    if (img[k-1][i][j] > 0) {
	      histo2[img[k-1][i][j]] += 1.0;
	      SliceVol2 += 1.0;
	    }
	  }
	printf("slice: %d ", k-1);fflush(stdout);
	ratio = SliceVol2/SliceVol1; printf("ratio: %f ", ratio);fflush(stdout);
	if ((ratio < 0.95) || (ratio > 1.05)) CreateWeightingSlice(img, dim, k-1, k, wsz, histo2, histo1, ratio, cost_function);
	si = MatchingTwoHistograms_special(histo2, histo1, dim, k-1);
	free(histo1);
	free(histo2);
	
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) 
	    if (img[k-1][i][j] != 0) {
	      temp = (int)((img[k-1][i][j] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k-1][i][j] = (unsigned char)temp;
	    }
      }
      
      for (k=focus_id; k<end_id; k++) {
	
	SliceVol1 = SliceVol2 = 0.0;
	histo1 = Falloc1d(MaxOfUC+1);
	histo2 = Falloc1d(MaxOfUC+1);
	for (h=0; h<MaxOfUC+1; h++) histo1[h] = histo2[h] = 0.0;

	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) {
	    if (img[k][i][j] > 0) {
	      histo1[img[k][i][j]] += 1.0;
	      SliceVol1 += 1.0;
	    }
	    if (img[k+1][i][j] > 0) {
	      histo2[img[k+1][i][j]] += 1.0;
	      SliceVol2 += 1.0;
	    }
	  }
	printf("slice: %d ", k+1);
	ratio = SliceVol2/SliceVol1; printf("ratio: %f ", ratio);fflush(stdout);
	if ((ratio < 0.95) || (ratio > 1.05)) CreateWeightingSlice(img, dim, k+1, k, wsz, histo2, histo1, ratio, cost_function);
	si = MatchingTwoHistograms_special(histo2, histo1, dim, k+1);
	free(histo1);
	free(histo2);
	
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) 
	    if (img[k+1][i][j] != 0) {
	      temp = (int)((img[k+1][i][j] - si.shift)*si.scale + 0.5);
	      if (temp < 0)   temp = 1; 
	      if (temp > MaxOfUC) temp = MaxOfUC;
	      
	      img[k+1][i][j] = (unsigned char)temp;
	    }
      }
    }
    break;

  default:
    break;
  }
      
  /* save result */
  WriteImgUC(argv[2], img, dim);
  UCfree3d(img, dim.z, dim.x);
}

void ComputingLocalHistogramSlice(unsigned char **img, Ivector3d dim, int wsz, HISTOGRAM **LH)
{
  int                  i, j, k, ii, jj;

  for (i=0; i<dim.x; i++)
    for (j=0; j<dim.y; j++)
      for (k=0; k<MaxOfUC+1; k++) LH[i][j].h[k] = 0;

  for (i=wsz/2; i<dim.x-wsz/2; i++)
    for (j=wsz/2; j<dim.y-wsz/2; j++)
      if (img[i][j] > 0) {
	for (ii=i-wsz/2; ii<=i+wsz/2; ii++)
	  for (jj=j-wsz/2; jj<=j+wsz/2; jj++) 
	    if ((ii>0) && (jj>0) && (ii<dim.x) && (jj<dim.y) && (img[ii][jj] > 0)) LH[i][j].h[img[ii][jj]] += 1.0;
      }
}

void ComputingLocalHistogramSlice_fast(unsigned char **img, Ivector3d dim, int wsz, HISTOGRAM **LH)
{
  unsigned char        **element, **status;
  Ivector3d            element_dim;
  int                  i, j, k, ii, jj;

  element_dim.x = element_dim.y = wsz+1;  element_dim.z = 0.0;
  status = UCalloc2d(dim.x, dim.y);
  for (i=0; i<dim.x; i++)
    for (j=0; j<dim.y; j++) status[i][j] = 0;

  for (i=0; i<dim.x; i++)
    for (j=0; j<dim.y; j++)
      if ((img[i][j] != 0) && (status[i][j] == 0)) {
	if ((status[i-1][j] == 0) && (status[i][j-1] == 0)) {
	  for (ii=i-wsz/2; ii<=i+wsz/2; ii++)
	    for (jj=j-wsz/2; jj<=j+wsz/2; jj++) 
	      if (img[ii][jj] > 0) LH[i][j].h[img[ii][jj]] += 1.0;
	  status[i][j] = 1;
	} else {
	  if ((status[i-1][j] != 0) && (status[i][j] == 0)) {
	    for (k=0; k<MaxOfUC+1; k++) LH[i][j].h[k] = LH[i-1][j].h[k];
	    for (jj=j-wsz/2; jj<=j+wsz/2; jj++) {
	      if (img[i-1-wsz/2][jj] > 0) LH[i][j].h[img[i-1-wsz/2][jj]] -= 1.0;
	      if (img[i+wsz/2][jj] > 0) LH[i][j].h[img[i+wsz/2][jj]] += 1.0;
	    }
	    status[i][j] = 1;
	  }
	  if ((status[i][j-1] != 0) && (status[i][j] == 0)) {
	    for (k=0; k<MaxOfUC+1; k++) LH[i][j].h[k] = LH[i][j-1].h[k];
	    for (ii=i-wsz/2; ii<=i+wsz/2; ii++) {
	      if (img[ii][jj-1-wsz/2] > 0) LH[i][j].h[img[ii][jj-1-wsz/2]] -= 1.0;
	      if (img[ii][jj+wsz/2] > 0) LH[i][j].h[img[ii][jj+wsz/2]] += 1.0;
	    }
	    status[i][j] = 1;
	  }
	}
	//	PrintHistogram(LH[i][j]);exit(1);
      }
  UCfree2d(status, dim.x);
}

unsigned char **UCalloc2d_test(int i_size,int j_size)
{
  unsigned char **array;
  int i,j;

  array=(unsigned char **) calloc(i_size,sizeof(unsigned char *));

  for(i=0;i<i_size;i++)
    array[i]=(unsigned char *) calloc(j_size,sizeof(unsigned char ));

  return(array);
}

void UCinitialization2d(unsigned char **arr, int xsz, int ysz)
{
  int        i, j;

  for (i=0; i<xsz; i++)
    for (j=0; j<ysz; j++) arr[i][j] = 0;
}

void Finitialization2d(float **arr, int xsz, int ysz)
{
  int        i, j;

  for (i=0; i<xsz; i++)
    for (j=0; j<ysz; j++) arr[i][j] = 0.0;
}

void HISTOGRAMinitialization2d(HISTOGRAM **arr, int xsz, int ysz)
{
  int        i, j, k;

  for (i=0; i<xsz; i++)
    for (j=0; j<ysz; j++) 
      for (k=0; k<MaxOfUC+1; k++) arr[i][j].h[k] = 0;
}

void CreateWeightingSlice(unsigned char ***img, Ivector3d dim, int ind_source, int ind_target, int wsz, float *histo_source, float *histo_target, float ratio, int cost_function)
{
  unsigned char        **source_slice, **target_slice, **element_source, **element_target;
  Ivector3d            element_dim, pos;
  float                **weighting, **MI, maxMI, **Rdiff, maxRdiff, threshold, **Vdiff;
  HISTOGRAM            **LH_source, **LH_target;
  int                  i, j, ii, jj, temp_source, temp_target;
  BOUNDINGBOX          bb_source, bb_target;

  source_slice = UCalloc2d_test(dim.x, dim.y); UCinitialization2d(source_slice, dim.x, dim.y);
  target_slice = UCalloc2d(dim.x, dim.y);      UCinitialization2d(target_slice, dim.x, dim.y);
  element_dim.x = element_dim.y = wsz+1;  element_dim.z = 0.0;
  element_source = UCalloc2d(element_dim.x, element_dim.y); UCinitialization2d(element_source, element_dim.x, element_dim.y);
  element_target = UCalloc2d(element_dim.x, element_dim.y); UCinitialization2d(element_target, element_dim.x, element_dim.y);
  weighting = Falloc2d(dim.x, dim.y); Finitialization2d(weighting, dim.x, dim.y);
  MI        = Falloc2d(dim.x, dim.y); Finitialization2d(MI, dim.x, dim.y);
  Rdiff     = Falloc2d(dim.x, dim.y); Finitialization2d(Rdiff, dim.x, dim.y);
  Vdiff     = Falloc2d(dim.x, dim.y); Finitialization2d(Vdiff, dim.x, dim.y);
  LH_source = HISTOGRAMalloc2d(dim.x, dim.y); HISTOGRAMinitialization2d(LH_source, dim.x, dim.y);
  LH_target = HISTOGRAMalloc2d(dim.x, dim.y); HISTOGRAMinitialization2d(LH_target, dim.x, dim.y);

  maxMI = maxRdiff = -999999.00;
  for (i=0; i<dim.x; i++)
    for (j=0; j<dim.y; j++) {
      source_slice[i][j] = img[ind_source][i][j];
      target_slice[i][j] = img[ind_target][i][j];
    }
 
  ComputingLocalHistogramSlice(source_slice, dim, wsz, LH_source); printf("source LH computed! ");fflush(stdout);
  ComputingLocalHistogramSlice(target_slice, dim, wsz, LH_target); printf("target LH computed! ");fflush(stdout);
  SliceDifference(source_slice, target_slice, dim, Vdiff);         printf("difference map computed! ");fflush(stdout);

  for (i=wsz/2; i<dim.x-wsz/2; i++)
    for (j=wsz/2; j<dim.y-wsz/2; j++)
      if (source_slice[i][j] > 0) {
	for (ii=i-wsz/2; ii<i+wsz/2; ii++)
	  for (jj=j-wsz/2; jj<j+wsz/2; jj++) {
	    element_source[ii-i+wsz/2][jj-j+wsz/2] = source_slice[ii][jj];
	    element_target[ii-i+wsz/2][jj-j+wsz/2] = target_slice[ii][jj];
	  }

	pos.x = i; pos.y = j;
	MI[i][j] = MutualInformation(element_source, element_target, pos, element_dim, LH_source, LH_target);
	if (MI[i][j] > maxMI) maxMI = MI[i][j];
	Rdiff[i][j] = ElementDifference(pos, wsz, Vdiff);
	if (Rdiff[i][j] > maxRdiff)  maxRdiff = Rdiff[i][j];
      }

  for (i=wsz/2; i<dim.x-wsz/2; i++)
    for (j=wsz/2; j<dim.y-wsz/2; j++) {

      switch (cost_function) {
      case 0: // MI for focusing only
	if (MI[i][j] > 0.001) weighting[i][j] = 1-MI[i][j]/maxMI;
	break;

      case 1: // difference for focusing only
	if (Rdiff[i][j] > 0.001) weighting[i][j] = 1-Rdiff[i][j]/maxRdiff;
	break;

      case 2: // MI*Difference for focusing only
	if ((MI[i][j] > 0.001) || (Rdiff[i][j] > 0.001)) weighting[i][j] = (1-MI[i][j]/maxMI)*(1-Rdiff[i][j]/maxRdiff);
	break;

      case 3: // MI for histogram
	if (MI[i][j] > 0.001) weighting[i][j] = 1-MI[i][j]/maxMI;
	temp_source = (int)(source_slice[i][j]*weighting[i][j] + 0.5);
	source_slice[i][j] = (unsigned char)temp_source;
	temp_target = (int)(target_slice[i][j]*weighting[i][j] + 0.5);
	target_slice[i][j] = (unsigned char)temp_target;
	break;

      case 4: // difference for histogram
	if (Rdiff[i][j] > 0.001) weighting[i][j] = 1-Rdiff[i][j]/maxRdiff;
	temp_source = (int)(source_slice[i][j]*weighting[i][j] + 0.5);
	source_slice[i][j] = (unsigned char)temp_source;
	temp_target = (int)(target_slice[i][j]*weighting[i][j] + 0.5);
	target_slice[i][j] = (unsigned char)temp_target;
	break;

      case 5: // MI*difference for histogram
	if ((MI[i][j] > 0.001) || (Rdiff[i][j] > 0.001)) weighting[i][j] = (1-MI[i][j]/maxMI)*(1-Rdiff[i][j]/maxRdiff);
	temp_source = (int)(source_slice[i][j]*weighting[i][j] + 0.5);
	source_slice[i][j] = (unsigned char)temp_source;
	temp_target = (int)(target_slice[i][j]*weighting[i][j] + 0.5);
	target_slice[i][j] = (unsigned char)temp_target;
	break;

      default:
	break;
      }
    }

  threshold = GetWeightThreshold(weighting, dim, ratio); printf("weighting threshold: %f ", threshold);
  for (i=0; i<dim.x; i++)
    for (j=0; j<dim.y; j++) 
      if (weighting[i][j]> threshold) {
	histo_source[source_slice[i][j]] += 1.0;
	histo_target[target_slice[i][j]] += 1.0;
      }

  Ffree2d(Vdiff,      dim.x);
  Ffree2d(Rdiff,      dim.x);
  HISTOGRAMfree2d(LH_source, dim.x);
  HISTOGRAMfree2d(LH_target, dim.x);
  Ffree2d(weighting, dim.x);
  Ffree2d(MI,        dim.x);
  UCfree2d(source_slice,   dim.x);
  UCfree2d(target_slice,   dim.x);
  UCfree2d(element_source,   element_dim.x);
  UCfree2d(element_target,   element_dim.x);
}

float GetWeightThreshold(float **weighting, Ivector3d dim, float ratio)
{
  int        i, j;
  float      t, total_volume, tmp_volume, scale;

  for (i=0; i<dim.x; i++)
    for (j=0; j<dim.y; j++)
      if (weighting[i][j] > 0.001) total_volume += 1.0;

  for (t=0.01; t<1.0; t+=0.01) {
    tmp_volume = 0.0;
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++)
	if ((weighting[i][j] < t) && (weighting[i][j] > 0.001)) tmp_volume += 1.0;
    ratio = tmp_volume/total_volume;
    if (((ratio > 1.05) || (ratio < 0.95)) && (scale > 0.25)) break;
    if ((((ratio < 0.9) && (ratio > 0.8)) || ((ratio > 1.1) && (ratio < 1.2))) && (scale > 0.35)) break;
    if (((ratio < 0.8) || (ratio > 1.2)) && (scale > 0.5)) break;
  }

  return t;
}

void PrintElement2D(unsigned char **ele, Ivector3d dim)
{
  int           i, j;

  for (i=0; i<dim.x; i++) {
    for (j=0; j<dim.y; j++) printf("%d ", ele[i][j]);
    printf("\n");
  }
}

float ElementDifference(Ivector3d pos, int wsz, float **diff)
{
  float         sum;
  int           i, j;

  sum = 0.0;
  for (i=pos.x-wsz/2; i<=pos.x+wsz/2; i++)
    for (j=pos.y-wsz/2; j<=pos.y+wsz/2; j++) sum += diff[i][j];

  return sqrt(sum);
}

void SliceDifference(unsigned char **s1, unsigned char **s2, Ivector3d dim, float **diff)
{
  float         sum;
  int           i, j;

  for (i=0; i<dim.x; i++)
    for (j=0; j<dim.y; j++)
      if ((s1[i][j] != 0) || (s2[i][j] != 0))
	diff[i][j] = ((float)s1[i][j]/(MaxOfUC+1) - (float)s2[i][j]/(MaxOfUC+1))*((float)s1[i][j]/(MaxOfUC+1) - (float)s2[i][j]/(MaxOfUC+1));
}

float MutualInformation(unsigned char **im1, unsigned char **im2, Ivector3d pos, Ivector3d dim, HISTOGRAM **LH_source, HISTOGRAM **LH_target)
{
  float         E1, E2, JE;

  E1 = Entropy(pos, LH_source);
  E2 = Entropy(pos, LH_target);
  JE = JointEntropy(im1, im2, dim);

  return (E1+E2-JE);
}

float JointEntropy(unsigned char **im1, unsigned char **im2, Ivector3d dim)
{
   float        **H, JE, sum;
   int          i, j, sz, k;
   Ivector3d    pos;

   sum = 0.0;
   H = Falloc2d(MaxOfUC+1, MaxOfUC+1);
   for (i=0; i<MaxOfUC+1; i++)
     for (j=0; j<MaxOfUC+1; j++) H[i][j] = 0.0;

   for (i=0; i<dim.x; i++)
     for (j=0; j<dim.y; j++)
       if ((im1[i][j] != 0) || (im2[i][j] != 0)) {
	 H[im1[i][j]][im2[i][j]] += 1.0;
	 sum += 1.0;
       }

   JE = 0.0;
   for (i=0; i<MaxOfUC+1; i++)
     for (j=0; j<MaxOfUC+1; j++) 
       if (H[i][j] > 0.001) JE += -(H[i][j]/sum)*(log2(H[i][j]/sum));

   Ffree2d(H, MaxOfUC+1);
   return JE;
}

float Entropy(Ivector3d pos, HISTOGRAM **LH)
{
   float        E, sum;
   int          i, j, sz, k;

   sum = 0.0;   E = 0.0;
   for (i=0; i<MaxOfUC+1; i++) sum += LH[pos.x][pos.y].h[i];
   for (i=0; i<MaxOfUC+1; i++) LH[pos.x][pos.y].h[i] /= sum;
   for (i=0; i<MaxOfUC+1; i++) 
     if (LH[pos.x][pos.y].h[i] > 0.001) E += -(LH[pos.x][pos.y].h[i])*(log2(LH[pos.x][pos.y].h[i]));

   return E;
}

void InhomoCorrectionImages3D(int argc, char *argv[], Ivector3d dim, int threshold, int OtherImg, char OtherImgName[1000])
{
  unsigned char        ***img, ***Oimg;
  int                  i, j, k, t, shift, temp, start_id, end_id, focus_id, *sum;
  float                scale, mindist, s, dist, distA, distB, **histo;
  FILE                 *fp, *fpp;
  float                *histoA, *histoB, *histoB_trans, *histoA_trans;
  double               TotalIntensityA, TotalIntensityB, NonZeroPnts ;
  char                 filename[1000] ;
  TRANSFORM_PARA       si;

  dim.z = GetImgZDim(argv[1], dim);
  printf("\n\ndimA: (%d, %d, %d)\n", dim.x, dim.y, dim.z);

  img = UCalloc3d(dim.x, dim.y, dim.z);
  ReadImgUC(argv[1], img, dim);
  printf("%s read ok\n", argv[1]);

  if (OtherImg == YYES) {
    Oimg = UCalloc3d(dim.x, dim.y, dim.z);
    ReadImgUC(OtherImgName, Oimg, dim);
    printf("%s read ok\n", OtherImgName);
  }

  /* To ignore background, by thresholding */
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) 
      for (j=0; j<dim.y; j++)
    	if( img[k][i][j]<threshold ) img[k][i][j]=0 ;

  /* correct in Z_PLN first */
  printf("\n\ncorrecting Z_PLN...\n\n");
  sum = Ialloc1d(dim.z);
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) 
	if (img[k][i][j] != 0) sum[k]++;
  
  for (k=0; k<dim.z; k++) {
    if (sum[k] != 0) {
      start_id = k;
      break;
    }
    if ((sum[k] == 0) && (sum[k+1] != 0)) {
      start_id = k+1;
      break;
    }
  }
  for (k=dim.z-1; k>=0; k--) {
    if (sum[k] != 0) {
      end_id = k;
      break;
    }
    if ((sum[k] == 0) && (sum[k-1] != 0)) {
      end_id = k-1;
      break;
    }
  }
  printf("image range: (%d,%d)\n", start_id, end_id);
  focus_id = PickSlice2Start(img, dim, Z_PLN, start_id, end_id);
  printf("focusing slice number: %d\n", focus_id);
  
  histo = Falloc2d(dim.z, 256);
  for (k=focus_id; k>start_id; k--) {
    
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) histo[k][img[k][i][j]] += 1.0;
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) histo[k-1][img[k-1][i][j]] += 1.0;
    printf("slice: %d ", k-1);
    si = MatchingTwoHistograms(histo[k-1], histo[k], dim);
    
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) {
	if (img[k-1][i][j] != 0) {
	  temp = (int)((img[k-1][i][j] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  
	  img[k-1][i][j] = (unsigned char)temp;
	}
	if ((OtherImg == YYES) && (Oimg[k-1][i][j] != 0)) {
	  temp = (int)((Oimg[k-1][i][j] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  Oimg[k-1][i][j] = (unsigned char)temp;
	}
      }
  }
  
  for (k=focus_id; k<end_id; k++) {
    
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) histo[k][img[k][i][j]] += 1.0;
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) histo[k+1][img[k+1][i][j]] += 1.0;
    printf("slice: %d ", k+1);
    si = MatchingTwoHistograms(histo[k+1], histo[k], dim);
    
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) {
	if (img[k+1][i][j] != 0) {
	  temp = (int)((img[k+1][i][j] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  
	  img[k+1][i][j] = (unsigned char)temp;
	}
	if ((OtherImg == YYES) && (Oimg[k+1][i][j] != 0)) {
	  temp = (int)((Oimg[k+1][i][j] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  Oimg[k+1][i][j] = (unsigned char)temp;
	}
      }
  }
  free(sum);
  Ffree2d(histo, dim.z);
  
  /* correct in X_PLN second */
  printf("\n\ncorrecting X_PLN...\n\n");
  sum = Ialloc1d(dim.x);
  for (i=0; i<dim.x; i++)
    for (k=0; k<dim.z; k++)
      for (j=0; j<dim.y; j++)
	if (img[k][i][j] != 0) sum[i]++;
  
  for (i=0; i<dim.x; i++)
    if ((sum[i] == 0) && (sum[i+1] != 0)) {
      start_id = i+1;
      break;
    }
  for (i=dim.x-1; i>=0; i--)
    if ((sum[i] == 0) && (sum[i-1] != 0)) {
      end_id = i-1;
      break;
    }
  printf("image range: (%d,%d)\n", start_id, end_id);
  focus_id = PickSlice2Start(img, dim, X_PLN, start_id, end_id);
  printf("focusing slice number: %d\n", focus_id);
  
  histo = Falloc2d(dim.x, 256);
  for (i=focus_id; i>start_id-2; i--) {
    
    for (k=0; k<dim.z; k++)
      for (j=0; j<dim.y; j++) histo[i][img[k][i][j]] += 1.0;
    for (k=0; k<dim.z; k++)
      for (j=0; j<dim.y; j++) histo[i-1][img[k][i-1][j]] += 1.0;
    printf("slice: %d ", i-1);
    si = MatchingTwoHistograms(histo[i-1], histo[i], dim);
    
    for (k=0; k<dim.z; k++)
      for (j=0; j<dim.y; j++) {
	if (img[k][i-1][j] != 0) {
	  temp = (int)((img[k][i-1][j] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  
	  img[k][i-1][j] = (unsigned char)temp;
	}
	if ((OtherImg == YYES) && (Oimg[k][i-1][j] != 0)) {
	  temp = (int)((Oimg[k][i-1][j] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  Oimg[k][i-1][j] = (unsigned char)temp;
	}
      }
  }
  
  for (i=focus_id; i<end_id-2; i++) {
    
    for (k=0; k<dim.z; k++)
      for (j=0; j<dim.y; j++) histo[i][img[k][i][j]] += 1.0;
    for (k=0; k<dim.z; k++)
      for (j=0; j<dim.y; j++) histo[i+1][img[k][i+1][j]] += 1.0;
    printf("slice: %d ", i+1);
    si = MatchingTwoHistograms(histo[i+1], histo[i], dim);
    
    for (k=0; k<dim.z; k++)
      for (j=0; j<dim.y; j++) {
	if (img[k][i+1][j] != 0) {
	  temp = (int)((img[k][i+1][j] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  
	  img[k][i+1][j] = (unsigned char)temp;
	}
	if ((OtherImg == YYES) && (Oimg[k][i+1][j] != 0)) {
	  temp = (int)((Oimg[k][i+1][j] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  Oimg[k][i+1][j] = (unsigned char)temp;
	}
      }
  }
  free(sum);
  Ffree2d(histo, dim.x);
  
  /* correct in Y_PLN third */
  printf("\n\ncorrecting Y_PLN...\n\n");
  sum = Ialloc1d(dim.y);
  for (j=0; j<dim.y; j++) 
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
	if (img[k][i][j] != 0) sum[j]++;
  
  for (j=0; j<dim.y; j++)
    if ((sum[j] == 0) && (sum[j+1] != 0)) {
      start_id = j+1;
      break;
    }
  for (j=dim.y-1; j>=0; j--)
    if ((sum[j] == 0) && (sum[j-1] != 0)) {
      end_id = j-1;
      break;
    }
  printf("image range: (%d,%d)\n", start_id, end_id);
  focus_id = PickSlice2Start(img, dim, Y_PLN, start_id, end_id);
  printf("focusing slice number: %d\n", focus_id);
  
  histo = Falloc2d(dim.y, 256);
  for (j=focus_id; j>start_id; j--) {
    
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++) histo[j][img[k][i][j]] += 1.0;
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++) histo[j-1][img[k][i][j-1]] += 1.0;
    printf("slice: %d ", j-1);
    si = MatchingTwoHistograms(histo[j-1], histo[j], dim);
    
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++) {
	if (img[k][i][j-1] != 0) {
	  temp = (int)((img[k][i][j-1] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  
	  img[k][i][j-1] = (unsigned char)temp;
	}
	if ((OtherImg == YYES) && (Oimg[k][i][j-1] != 0)) {
	  temp = (int)((Oimg[k][i][j-1] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  Oimg[k][i][j-1] = (unsigned char)temp;
	}
      }
  }
  
  for (j=focus_id; j<end_id; j++) {
    
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++) histo[j][img[k][i][j]] += 1.0;
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++) histo[j+1][img[k][i][j+1]] += 1.0;
    printf("slice: %d ", j-1);
    si = MatchingTwoHistograms(histo[j+1], histo[j], dim);
    
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++) {
	if (img[k][i][j+1] != 0) {
	  temp = (int)((img[k][i][j+1] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  
	  img[k][i][j+1] = (unsigned char)temp;
	}
	if ((OtherImg == YYES) && (Oimg[k][i][j+1] != 0)) {
	  temp = (int)((Oimg[k][i][j+1] - si.shift)*si.scale + 0.5);
	  if (temp < 0)   temp = 1; 
	  if (temp > MaxOfUC) temp = MaxOfUC;
	  Oimg[k][i][j+1] = (unsigned char)temp;
	}
      }
  }
  free(sum);
  Ffree2d(histo, dim.z);
    

  /* save result */
  WriteImgUC(argv[2], img, dim);
  UCfree3d(img, dim.z, dim.x);

  if (OtherImg == YYES) {
    sprintf(filename, "%s.correct", OtherImgName);
    printf("saving %s...\n", filename);
    WriteImgUC(filename, Oimg, dim);
    printf("%s saved!\n", filename);
    UCfree3d(Oimg, dim.z, dim.x);
  }
}

void InhomoCorrectionImages3D_improve(int argc, char *argv[], Ivector3d dim, int threshold, int OtherImg, char OtherImgName[1000], int focus_slice_no, int iter)
{
  unsigned char        ***img, ***Oimg;
  int                  i, j, k, l, t, shift, temp, start_id, end_id, focus_id, *sum, radius, start_no, end_no;
  float                scale, mindist, s, dist, distA, distB, **histo;
  FILE                 *fp, *fpp;
  float                *histoA, *histoB, *histoB_trans, *histoA_trans;
  double               TotalIntensityA, TotalIntensityB, NonZeroPnts;
  char                 filename[1000] ;
  TRANSFORM_PARA       si;

  dim.z = GetImgZDim(argv[1], dim);
  printf("\n\ndimA: (%d, %d, %d)\n", dim.x, dim.y, dim.z);

  img = UCalloc3d(dim.x, dim.y, dim.z);
  ReadImgUC(argv[1], img, dim);
  printf("%s read ok\n", argv[1]);

  if (OtherImg == YYES) {
    Oimg = UCalloc3d(dim.x, dim.y, dim.z);
    ReadImgUC(OtherImgName, Oimg, dim);
    printf("%s read ok\n", OtherImgName);
  }

  /* To ignore background, by thresholding */
  for (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++) 
      for (j=0; j<dim.y; j++)
    	if( img[k][i][j]<threshold ) img[k][i][j]=0 ;

  radius = focus_slice_no / 2;

  for (l=0; l<iter; l++) {

    /* correct in Z_PLN first */
    printf("\n\ncorrecting Z_PLN...\n\n");
    sum = Ialloc1d(dim.z);
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) 
	  if (img[k][i][j] != 0) sum[k]++;
    
    for (k=0; k<dim.z; k++) {
      if (sum[k] != 0) {
	start_id = k;
	break;
      }
      if ((sum[k] == 0) && (sum[k+1] != 0)) {
	start_id = k+1;
	break;
      }
    }
    for (k=dim.z-1; k>=0; k--) {
      if (sum[k] != 0) {
	end_id = k;
	break;
      }
      if ((sum[k] == 0) && (sum[k-1] != 0)) {
	end_id = k-1;
	break;
      }
    }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, Z_PLN, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);
    
    histo = Falloc2d(dim.z, 256);
    for (k=focus_id; k>start_id+radius; k--) {
      
      start_no = k - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = k + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[k], start_no, end_no, 0);
      
      start_no = k - 1 - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = k - 1 + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[k-1], start_no, end_no, 0);
      printf("slice: %d ", k-1);
      si = MatchingTwoHistograms(histo[k-1], histo[k], dim);
      
      ApplyTransformation(img, dim, si, start_no, end_no, 0);
    }
    
    for (k=focus_id; k<end_id-radius; k++) {
      
      start_no = k - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = k + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[k], start_no, end_no, 0);
      
      start_no = k + 1 - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = k + 1 + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[k+1], start_no, end_no, 0);
      printf("slice: %d ", k+1);
      si = MatchingTwoHistograms(histo[k+1], histo[k], dim);
      
      ApplyTransformation(img, dim, si, start_no, end_no, 0);
      
    }
    free(sum);
    Ffree2d(histo, dim.z);
    
    /* correct in X_PLN second */
    printf("\n\ncorrecting X_PLN...\n\n");
    sum = Ialloc1d(dim.x);
    for (i=0; i<dim.x; i++)
      for (k=0; k<dim.z; k++)
	for (j=0; j<dim.y; j++)
	  if (img[k][i][j] != 0) sum[i]++;
    
    for (i=0; i<dim.x; i++)
      if ((sum[i] == 0) && (sum[i+1] != 0)) {
	start_id = i+1;
	break;
      }
    for (i=dim.x-1; i>=0; i--)
      if ((sum[i] == 0) && (sum[i-1] != 0)) {
	end_id = i-1;
	break;
      }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, X_PLN, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);
    
    histo = Falloc2d(dim.x, 256);
    for (i=focus_id; i>start_id+radius; i--) {
      start_no = i - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = i + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[i], start_no, end_no, 1);
      
      start_no = i - 1 - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = i - 1 + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[i-1], start_no, end_no, 1);
      printf("slice: %d ", i-1);
      si = MatchingTwoHistograms(histo[i-1], histo[i], dim);
      
      ApplyTransformation(img, dim, si, start_no, end_no, 1);
    }
    
    for (i=focus_id; i<end_id-radius; i++) {
      
      start_no = i - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = i + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[i], start_no, end_no, 1);
      
      start_no = i + 1 - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = i + 1 + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[i+1], start_no, end_no, 1);
      printf("slice: %d ", i+1);
      si = MatchingTwoHistograms(histo[i+1], histo[i], dim);
      
      ApplyTransformation(img, dim, si, start_no, end_no, 1);
    }
    free(sum);
    Ffree2d(histo, dim.x);
    
    /* correct in Y_PLN third */
    printf("\n\ncorrecting Y_PLN...\n\n");
    sum = Ialloc1d(dim.y);
    for (j=0; j<dim.y; j++) 
      for (k=0; k<dim.z; k++)
	for (i=0; i<dim.x; i++)
	  if (img[k][i][j] != 0) sum[j]++;
    
    for (j=0; j<dim.y; j++)
      if ((sum[j] == 0) && (sum[j+1] != 0)) {
	start_id = j+1;
	break;
      }
    for (j=dim.y-1; j>=0; j--)
      if ((sum[j] == 0) && (sum[j-1] != 0)) {
	end_id = j-1;
	break;
      }
    printf("image range: (%d,%d)\n", start_id, end_id);
    focus_id = PickSlice2Start(img, dim, Y_PLN, start_id, end_id);
    printf("focusing slice number: %d\n", focus_id);
    
    histo = Falloc2d(dim.y, 256);
    for (j=focus_id; j>start_id+radius; j--) {
      start_no = j - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = j + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[j], start_no, end_no, 2);
      
      start_no = j - 1 - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = j - 1 + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[j-1], start_no, end_no, 2);
      printf("slice: %d ", j-1);
      si = MatchingTwoHistograms(histo[j-1], histo[j], dim);
      
      ApplyTransformation(img, dim, si, start_no, end_no, 2);
    }
    
    for (j=focus_id; j<end_id-radius; j++) {
      
      start_no = j - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = j + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[j], start_no, end_no, 2);
      
      start_no = j + 1 - radius;
      if (start_no < start_id) { start_no = start_id; end_no = start_no + focus_slice_no; }
      end_no = j + 1 + radius;
      if (end_no > end_id) { end_no = end_id; start_no = end_no - focus_slice_no; }
      GetHistogram(img, dim, histo[j+1], start_no, end_no, 2);
      printf("slice: %d ", j+1);
      si = MatchingTwoHistograms(histo[j+1], histo[j], dim);
      
      ApplyTransformation(img, dim, si, start_no, end_no, 2);
    }
    free(sum);
    Ffree2d(histo, dim.z);
  }    
  
  /* save result */
  WriteImgUC(argv[2], img, dim);
  UCfree3d(img, dim.z, dim.x);
  
  if (OtherImg == YYES) {
    sprintf(filename, "%s.correct", OtherImgName);
    printf("saving %s...\n", filename);
    WriteImgUC(filename, Oimg, dim);
    printf("%s saved!\n", filename);
    UCfree3d(Oimg, dim.z, dim.x);
  }
}

void GetHistogram(unsigned char ***img, Ivector3d dim, float *histo, int start_no, int end_no, int direction)
{
  int        i, j, k;

  // direction 0:z; 1:x; 2:y
  switch (direction) {

  case 0: // z direction
    for (k=start_no; k<=end_no; k++)
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) histo[img[k][i][j]]+=1.0;
    break;

  case 1: // x direction
    for (k=0; k<dim.z; k++)
      for (i=start_no; i<=end_no; i++)
	for (j=0; j<dim.y; j++) histo[img[k][i][j]]+=1.0;
    break;

  case 2: // y direction
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
	for (j=start_no; j<=end_no; j++) histo[img[k][i][j]]+=1.0;
    break;
  }
}


void ApplyTransformation(unsigned char ***img, Ivector3d dim, TRANSFORM_PARA si, int start_no, int end_no, int direction)
{
  int        i, j, k, temp;

  // direction 0: z; 1: x; 2: y
  switch (direction) {
  case 0: // z
    for (k=start_no; k<=end_no; k++)
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++)
	  if (img[k][i][j] != 0) {
	    temp = (int)((img[k][i][j] - si.shift)*si.scale + 0.5);
	    if (temp < 0)   temp = 1; 
	    if (temp > MaxOfUC) temp = MaxOfUC;
	    img[k][i][j] = (unsigned char)temp;
	  }
    break;

  case 1: // x
    for (k=0; k<dim.z; k++)
      for (i=start_no; i<end_no; i++)
	for (j=0; j<dim.y; j++)
	  if (img[k][i][j] != 0) {
	    temp = (int)((img[k][i][j] - si.shift)*si.scale + 0.5);
	    if (temp < 0)   temp = 1; 
	    if (temp > MaxOfUC) temp = MaxOfUC;
	    img[k][i][j] = (unsigned char)temp;
	  }
    break;

  case 2: // y
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
	for (j=start_no; j<=end_no; j++)
	  if (img[k][i][j] != 0) {
	    temp = (int)((img[k][i][j] - si.shift)*si.scale + 0.5);
	    if (temp < 0)   temp = 1; 
	    if (temp > MaxOfUC) temp = MaxOfUC;
	    img[k][i][j] = (unsigned char)temp;
	  }
    break;
  }
}

int PickSlice2Start(unsigned char ***img, Ivector3d dim, int op, int start_id, int end_id)
{
  int        i, j, k, max_id, mid_id, count, max_slice;
  float      max_value, m, v, c, *vec;
  
  max_value = -9999.99;
  mid_id = (start_id + end_id) / 2;
  switch (op) {
    case X_PLN:
      vec = Falloc1d(dim.y*dim.z);
      for (i=mid_id-5; i<=mid_id+5; i++) {
	count = 0;
	for (k=0; k<dim.z; k++) 
	  for (j=0; j<dim.y; j++)
	    if (img[k][i][j] != 0) {
	      vec[count] = (float)img[k][i][j];
	      count++;
	    }
	m = GetMean(vec, count);
	v = GetVariance(vec, count);
	c = (m + v) / 2;
	if (c > max_value) {
	  max_value = c;
	  max_id = i;
	}
	printf("slice no: %d\tmean: %f\tvariance:%f\tmix: %f\n", i, m, v, c);
      }
    break;
    
    case Y_PLN:
      vec = Falloc1d(dim.x*dim.z);
      for (j=mid_id-5; j<=mid_id+5; j++) {
	count = 0;
	for (k=0; k<dim.z; k++) 
	  for (i=0; i<dim.x; i++)
	    if (img[k][i][j] != 0) {
	      vec[count] = (float)img[k][i][j];
	      count++;
	    }
	m = GetMean(vec, count);
	v = GetVariance(vec, count);
	c = (m + v) / 2;
	if (c > max_value) {
	  max_value = c;
	  max_id = j;
	}
	printf("slice no: %d\tmean: %f\tvariance:%f\tmix: %f\n", j, m, v, c);
      }
    break;
    
    case Z_PLN:
      vec = Falloc1d(dim.x*dim.y);
      for (k=mid_id-5; k<=mid_id+5; k++) {
	count = 0;
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) 
	    if (img[k][i][j] != 0) {
	      vec[count] = (float)img[k][i][j];
	      count++;
	    }
	m = GetMean(vec, count);
	v = GetVariance(vec, count);
	c = (m + v) / 2;
	if (c > max_value) {
	  max_value = c;
	  max_id = k;
	}
	printf("slice no: %d\tmean: %f\tvariance:%f\tmix: %f\n", k, m, v, c);
      }
    break;
      
    default:
    break;
  }

  free(vec);
  return max_id;
}

float GetMean(float *vec, int len)
{
  int        i;
  float      total;

  total = 0;
  //  printf("vec:\n");
  for (i=0; i<len; i++) {
    total += vec[i];
    //    printf("%f ", vec[i]);
  }
  //  printf("\n");
  //  printf("mean: %f\n", (total/len));

  return (total/len);
}

float GetVariance(float *vec, int len)
{
  int        i;
  float      total, m;

  total = 0;
  m = GetMean(vec, len);
  for (i=0; i<len; i++) total += (vec[i]-m)*(vec[i]-m)/(len-1);

  return sqrt(total);
}

TRANSFORM_PARA MatchingTwoHistograms(float *histoA, float *histoB, Ivector3d dim)
{
  float                scale, mindist, s, t, dist, distA, distB, start_s, end_s, start_t, end_t;
  float                *histoB_trans, *histoA_trans, **distRecord;
  double               TotalIntensityA, TotalIntensityB, NonZeroPnts ;
  int                  i, j, shift;
  TRANSFORM_PARA       r;

  printf("in MatchingTwoHistograms");fflush(stdout);
  TotalIntensityA = NonZeroPnts = 0;
  for (i=0; i<(MaxOfUC+1); i++)
    if (histoA[i] != 0) TotalIntensityA += i*histoA[i];
  for (i=0; i<(MaxOfUC+1); i++) { 
    if(i>0) NonZeroPnts += histoA[i]; 
    histoA[i] /= (dim.x*dim.y); 
  }
  printf("TotalIntensityA=%9.1f NonZeroPnts=%8.1f ", TotalIntensityA, NonZeroPnts) ;fflush(stdout);
  if (NonZeroPnts < 10) {
    r.scale = 1.0;
    r.shift = 0.0;
    printf("(scale, shift): (%3.2f, %d)\tmindist: %4.3f\n", r.scale, r.shift, 0.0);
    return r;
  }

  TotalIntensityB = NonZeroPnts = 0;
  for (i=0; i<(MaxOfUC+1); i++)
    if (histoB[i] != 0) TotalIntensityB += i*histoB[i];
  for (i=0; i<(MaxOfUC+1); i++) { 
    if(i>0) NonZeroPnts += histoB[i]; 
    histoB[i] /= (dim.x*dim.y); 
  }
  printf("TotalIntensityB=%9.1f NonZeroPnts=%8.1f ", TotalIntensityB, NonZeroPnts) ;fflush(stdout);
  if (NonZeroPnts < 10) {
    r.scale = 1.0;
    r.shift = 0.0;
    printf("(scale, shift): (%3.2f, %d)\tmindist: %4.3f\n", r.scale, r.shift, 0.0);
    return r;
  }
    
  /* to ignore background, and normalize ...  */
  for (i=1; i<(MaxOfUC+1); i++) histoA[i] = histoA[i]/(1.0-histoA[0]) ;
  for (i=1; i<(MaxOfUC+1); i++) histoB[i] = histoB[i]/(1.0-histoB[0]) ;
  histoA[0] = histoB[0] = 0;

  /* interpolate values for those with zeros */
  LinearlyInterpolateHistogram(histoA) ;
  LinearlyInterpolateHistogram(histoB) ;
  //  printf("after LinearlyInterpolateHistogram ");fflush(stdout);

  /* smooth ... */
  smooth_histogram_byMedianFilter(histoA);
  smooth_histogram_byMedianFilter(histoB);
  //  printf("after smooth_histogram_byMedianFilter ");fflush(stdout);

  /* normalization ... */
  Vector_Normalization(histoA, (MaxOfUC+1)) ;
  Vector_Normalization(histoB, (MaxOfUC+1)) ;
  //  printf("after Vector_Normalization ");fflush(stdout);

  /* matching imgA to imgB */
  histoB_trans = Falloc1d(MaxOfUC+1);
  histoA_trans = Falloc1d(MaxOfUC+1);
  mindist = 999999.9;

  start_s = 0.9; end_s = 1.3;
  start_t = -15; end_t = 15;
  distRecord = Falloc2d(MaxOfUC+1,MaxOfUC+1);
  for (i=0; i<(MaxOfUC+1); i++)
    for (j=0; j<(MaxOfUC+1); j++) distRecord[i][j] = 1000.0;
    
  while (1) {
    for (s=start_s; s<end_s; s+=0.01)  /* adaptively adjust */
      for (t=start_t; t<end_t; t+=1.0)
        if (fabs(distRecord[(int)(s*100+0.5)][(int)(t+100+0.5)] - 1000.0) < 0.001) {
   	  /* consitence requirement */
	  /* on A space */
	  for (i=0; i<(MaxOfUC+1); i++) histoB_trans[i] = histoB[i];
	  vector_transform(histoB_trans, (MaxOfUC+1), s, t);	
	  /* fitting, smoothing, normalization */
	  LinearlyInterpolateHistogram(histoB_trans) ;
	  /*smooth_histogram_byMedianFilter(histoB_trans);*/
	  Vector_Normalization(histoB_trans, (MaxOfUC+1)) ;
	  distA = vector_distance(histoB_trans, histoA, (MaxOfUC+1));
	  
	  /* on B space */
	  for (i=0; i<(MaxOfUC+1); i++) histoA_trans[i] = histoA[i];
	  vector_InverseTransform(histoA_trans, (MaxOfUC+1), s, t);
	  /* fitting, smoothing, normalization */
	  LinearlyInterpolateHistogram(histoA_trans) ;
	  /*smooth_histogram_byMedianFilter(histoA_trans);*/
	  Vector_Normalization(histoA_trans, (MaxOfUC+1)) ;
	  distB = vector_distance(histoA_trans, histoB, (MaxOfUC+1));
	  
	  /*distA=0;*/
	  dist = distA + distB ;
	  distRecord[(int)(s*100)][(int)(t+100)] = dist;
	  
	  if (dist < mindist) {
	    mindist = dist;
	    scale = s;
	    shift = t;
	  }
	  //	  printf("(scale, shift, dist, mindist): (%f, %f, %f, %f)\n", s, t, dist, mindist);
	}
    if ((fabs(scale - start_s)>0.01) && (fabs(scale - end_s)>0.01) && (fabs(shift - start_t)>1.0) && (fabs(shift - end_t)>1.0)) break;
    if ((start_s < 0.1) || (end_s > 2.0) ) break;
    if (fabs(scale - start_s)<0.01) start_s -=0.1;
    if (fabs(scale - end_s)<0.01)   end_s   +=0.1;
    if (fabs(shift - start_t)<1.0)  start_t -= 10;
    if (fabs(shift - end_t)<1.0)    end_t   += 10;
    for (i=0; i<(MaxOfUC+1); i++)
      for (j=0; j<(MaxOfUC+1); j++) distRecord[i][j] = 1000.0;
  }
  
  printf("(scale, shift): (%3.2f, %d) mindist: %4.3f\n", scale, shift, mindist);
  r.scale = scale;
  r.shift = shift;
  
  free(histoB_trans);
  free(histoA_trans);
  Ffree2d(distRecord, 200);
  
  return r;
}

TRANSFORM_PARA MatchingTwoHistograms_special(float *histoAA, float *histoBB, Ivector3d dim, int slice_no)
{
  float                scale, mindist, s, t, dist, distA, distB, start_s, end_s, start_t, end_t;
  float                *histoA, *histoB, *histoB_trans, *histoA_trans;
  double               TotalIntensityA, TotalIntensityB, NonZeroPntsA, NonZeroPntsB;
  int                  i, j, ii, jj, shift;
  TRANSFORM_PARA       r;

  histoA = Falloc1d(MaxOfUC+1);
  histoB = Falloc1d(MaxOfUC+1);
  for (i=0; i<MaxOfUC+1; i++) {
    histoA[i] = histoAA[i];
    histoB[i] = histoBB[i];
  }
  TotalIntensityA = NonZeroPntsA =  TotalIntensityB = NonZeroPntsB = 0.0;
  for (i=0; i<(MaxOfUC+1); i++) {
    TotalIntensityA += i*histoA[i];
    TotalIntensityB += i*histoB[i];
  }
  for (i=0; i<(MaxOfUC+1); i++) { 
    if(i>0) {
      NonZeroPntsA += histoA[i];
      NonZeroPntsB += histoB[i]; 
    }
    histoA[i] /= (dim.x*dim.y); 
    histoB[i] /= (dim.x*dim.y); 
  }
  printf("TotalIntA, B=%8.1f,%8.1f NonZeroPntsA, B=%8.1f,%8.1f ", TotalIntensityA, TotalIntensityB, NonZeroPntsA, NonZeroPntsB);fflush(stdout);
  if ((NonZeroPntsA < 10) || (NonZeroPntsB < 10)) {
    r.scale = 1.0;
    r.shift = 0.0;
    printf("(scale, shift): (%3.2f, %d)\tmindist: %4.3f\n", r.scale, r.shift, 0.0);
    return r;
  }

  /* to ignore background, and normalize ...  */
  for (i=1; i<(MaxOfUC+1); i++) {
    histoA[i] = histoA[i]/(1.0-histoA[0]);
    histoB[i] = histoB[i]/(1.0-histoB[0]) ;
  }
  histoA[0] = histoB[0] = 0.0;

  /* interpolate values for those with zeros */
  LinearlyInterpolateHistogram(histoA);
  LinearlyInterpolateHistogram(histoB);

  /* smooth ... */
  smooth_histogram_byMedianFilter(histoA);
  smooth_histogram_byMedianFilter(histoB);

  /* normalization ... */
  Vector_Normalization(histoA, (MaxOfUC+1));
  Vector_Normalization(histoB, (MaxOfUC+1));

  /* matching imgA to imgB */
  histoB_trans = Falloc1d(MaxOfUC+1);
  histoA_trans = Falloc1d(MaxOfUC+1);
  mindist = 999999.9;

  start_s = 0.9; end_s = 1.3; start_t = -15; end_t = 15;
    
  while (1) {
    for (s=start_s; s<end_s; s+=0.01)  /* {0.5~1.5 by 0.01} */
      for (t=start_t; t<end_t; t+=1.0) {
	/* consitence requirement */
	/* on A space */
	for (i=0; i<(MaxOfUC+1); i++) histoB_trans[i] = histoB[i];
	vector_transform(histoB_trans, (MaxOfUC+1), s, t);	
	/* fitting, smoothing, normalization */
	LinearlyInterpolateHistogram(histoB_trans);
	/*smooth_histogram_byMedianFilter(histoB_trans);*/
	Vector_Normalization(histoB_trans, (MaxOfUC+1)) ;
	distA = vector_distance(histoB_trans, histoA, (MaxOfUC+1));
	
	/* on B space */
	for (i=0; i<(MaxOfUC+1); i++) histoA_trans[i] = histoA[i];
	vector_InverseTransform(histoA_trans, (MaxOfUC+1), s, t);	
	/* fitting, smoothing, normalization */
	LinearlyInterpolateHistogram(histoA_trans) ;
	/*smooth_histogram_byMedianFilter(histoA_trans);*/
	Vector_Normalization(histoA_trans, (MaxOfUC+1)) ;
	distB = vector_distance(histoA_trans, histoB, (MaxOfUC+1));
	
	/*distA=0;*/
	dist = distA + distB ;
	
	if (dist < mindist) {
	  mindist = dist;
	  scale = s;
	  shift = t;
	}
      }
    if ((fabs(scale - start_s)>0.001) && (fabs(scale - end_s)>0.001) && (fabs(shift - start_t)>0.1) && (fabs(shift - end_t)>0.1)) break;
    if ((start_s < 0.1) || (end_s > 2.0) ) {printf("nonsense parameter! check data\n");break;}
    if (fabs(scale - start_s)<0.001) start_s -=0.1;
    if (fabs(scale - end_s)<0.001)   end_s   +=0.1;
    if (fabs(shift - start_t)<0.1)   start_t -= 10;
    if (fabs(shift - end_t)<0.1)     end_t   += 10;
  }

  printf("(scale, shift): (%3.2f, %d) mindist: %4.3f\n", scale, shift, mindist);
  r.scale = scale;
  r.shift = shift;
  
  free(histoB_trans);
  free(histoA_trans);
  free(histoB);
  free(histoA);
  
  return r;
}


TRANSFORM_PARA MatchingTwoHistograms_old(float *histoA, float *histoB, Ivector3d dim)
{
  float                scale, mindist, s, t, dist, distA, distB;
  float                *histoB_trans, *histoA_trans;
  double               TotalIntensityA, TotalIntensityB, NonZeroPnts ;
  int                  i, j, shift;
  TRANSFORM_PARA       r;

  TotalIntensityA = NonZeroPnts = 0;
  for (i=0; i<(MaxOfUC+1); i++)
    if (histoA[i] != 0) TotalIntensityA += i*histoA[i];
  for (i=0; i<(MaxOfUC+1); i++) { if(i>0) NonZeroPnts += histoA[i]; histoA[i] /= (dim.x*dim.y); }
  printf("TotalIntensityA=%9.1f NonZeroPnts=%8.1f ", TotalIntensityA, NonZeroPnts) ;
  if (NonZeroPnts < 10) {
    r.scale = 1.0;
    r.shift = 0.0;
    printf("(scale, shift): (%3.2f, %d)\tmindist: %4.3f\n", r.scale, r.shift, 0.0);
    return r;
  }

  TotalIntensityB = NonZeroPnts = 0;
  for (i=0; i<(MaxOfUC+1); i++)
    if (histoB[i] != 0) TotalIntensityB += i*histoB[i];
  for (i=0; i<(MaxOfUC+1); i++) { if(i>0) NonZeroPnts += histoB[i]; histoB[i] /= (dim.x*dim.y); }
  printf("TotalIntensityB=%9.1f NonZeroPnts=%8.1f ", TotalIntensityB, NonZeroPnts);fflush(stdout);
  if (NonZeroPnts < 10) {
    r.scale = 1.0;
    r.shift = 0.0;
    printf("(scale, shift): (%3.2f, %d)\tmindist: %4.3f\n", r.scale, r.shift, 0.0);fflush(stdout);
    return r;
  }
    
  /* to ignore background, and normalize ...  */
  for (i=1; i<(MaxOfUC+1); i++) histoA[i] = histoA[i]/(1.0-histoA[0]) ;
  for (i=1; i<(MaxOfUC+1); i++) histoB[i] = histoB[i]/(1.0-histoB[0]) ;
  histoA[0] = histoB[0] = 0;

  /* interpolate values for those with zeros */
  LinearlyInterpolateHistogram(histoA) ;
  LinearlyInterpolateHistogram(histoB) ;
  /* smooth ... */
  smooth_histogram_byMedianFilter(histoA);
  smooth_histogram_byMedianFilter(histoB);
  /* normalization ... */
  Vector_Normalization(histoA, (MaxOfUC+1)) ;
  Vector_Normalization(histoB, (MaxOfUC+1)) ;

  //  printf("\n\nhistogram created\n");
  /* PrintVector(histoA, (MaxOfUC+1)); 
     PrintVector(histoB, (MaxOfUC+1)); Aug 25 2003 */


  /* matching imgA to imgB */
  histoB_trans = (float *)malloc((MaxOfUC+1)*sizeof(float));
  histoA_trans = (float *)malloc((MaxOfUC+1)*sizeof(float));
  mindist = 99999999.9;
  for (s=0.9; s<1.3; s+=0.01)  /* {0.5~1.5 by 0.01} */
    for (t=-15; t<15; t++) {
   	/* consitence requirement */
	/* on A space */
	for (i=0; i<(MaxOfUC+1); i++) histoB_trans[i] = histoB[i];
	vector_transform(histoB_trans, (MaxOfUC+1), s, t);	
	 /* fitting, smoothing, normalization */
 	 LinearlyInterpolateHistogram(histoB_trans) ;
	  /*smooth_histogram_byMedianFilter(histoB_trans);*/
	 Vector_Normalization(histoB_trans, (MaxOfUC+1)) ;
     	distA = vector_distance(histoB_trans, histoA, (MaxOfUC+1));

	/* on B space */
	for (i=0; i<(MaxOfUC+1); i++) histoA_trans[i] = histoA[i];
	vector_InverseTransform(histoA_trans, (MaxOfUC+1), s, t);	
	 /* fitting, smoothing, normalization */
 	 LinearlyInterpolateHistogram(histoA_trans) ;
	  /*smooth_histogram_byMedianFilter(histoA_trans);*/
	 Vector_Normalization(histoA_trans, (MaxOfUC+1)) ;
     	distB = vector_distance(histoA_trans, histoB, (MaxOfUC+1));

	/*distA=0;*/

	dist = distA + distB ;

	if (dist < mindist) {
	  mindist = dist;
	  scale = s;
	  shift = t;
	}
  }
  printf("(scale, shift): (%3.2f, %d) mindist: %4.3f\n", scale, shift, mindist);
  r.scale = scale;
  r.shift = shift;
  
  free(histoB_trans);
  free(histoA_trans);
  
  return r;
}


float vector_distance(float *vec1, float *vec2, int len)
{
  int        i;
  float      dist;

  dist = 0.0;
  for (i=0; i<len; i++) 
    {
      dist += (vec1[i] - vec2[i])*(vec1[i] - vec2[i]); /* overflow*/
      /*      printf("dist=%f  ", dist) ;*/
    } /*printf("\n\n") ;*/
  dist = sqrt(dist);
  
  return dist;
}


void LinearlyInterpolateHistogram(float *histo)
{
  int i, j, size, *tmp, Left, Right;
  float  *histoTmp, a;

  /* get new values for the positions with value 0*/
  for(i=0; i<(MaxOfUC+1); i++) {
    if( histo[i]==0 )	{
      for(Left=i; Left>0; Left--) /* background 0 */
	if( histo[Left]!=0 ) break ;

      for(Right=i; Right<(MaxOfUC+1); Right++) 
	if( histo[Right]!=0 ) break ;

      /* interpolate ... */
      if( histo[Left]!=0 && histo[Right]!=0 ) {
	a = (float)Left/(float)(Left+Right) ;
	histo[i] = (1-a)*histo[Left] + a*histo[Right] ;
      }
    }
  }
}


void smooth_histogram_byMedianFilter(float *histo)
{
  int        i, j, size ;
  float      *histoTmp, *tmp;

  size = 2;
  tmp = (float *)malloc((2*size+1)*sizeof(float));

  histoTmp = (float *)malloc((MaxOfUC+1)*sizeof(float));
  for (i=0; i<(MaxOfUC+1); i++) histoTmp[i] = histo[i];

  for (i=size; i<(MaxOfUC+1)-size; i++) {

    for (j=-size; j<size+1; j++) tmp[j+size] = histo[i+j];
    histoTmp[i] = medianvalue(tmp, 2*size+1);
  }

  for (i=0; i<(MaxOfUC+1); i++) histo[i] = histoTmp[i];

  free(tmp);
  free(histoTmp);
}

float medianvalue(float *vec, int len)
{
  int        result, i, index;

  mysort(vec, len);

  return vec[(len/2)];
}

void PrintVector(float *vec, int len)
{
  int        i;

  for (i=0; i<len; i++) printf("%f ", vec[i]);
  printf("\n");
}

void PrintHistogram(HISTOGRAM his)
{
  int       i;

  for (i=0; i<MaxOfUC; i++) printf("%f ", his.h[i]);
  printf("\n");
}

void mysort(float *vec, int len)
{
  int    i, j, count ;
  float  *vec1, max;

  for (i=0; i<len; i++) {

    vec1 = (float *)malloc((len-i)*sizeof(float));
    for (j=i; j<len; j++) vec1[j-i] = vec[j];
    max = GetMax(vec1, len-i);
    vec[i] = max;
    count = 1;
    for (j=0; j<len-i; j++)
      if (vec1[j] != max) {
	vec[i+count] = vec1[j];
	count++;
      }
     free(vec1);
  }
}

float GetMax(float *vec, int len)
{
  int    i ;
  float  max;

  max = -999999;
  for (i=0; i<len; i++)
    if (vec[i] > max) max = vec[i];

  return max;
}

void Vector_Normalization(float *vec, int len)
{
  int        i;
  float      total ;
  
  /* normalization ... */
  total = 0 ;
  for (i=0; i<len; i++)  
    total += vec[i] ; 

  for (i=0; i<len; i++)  
    vec[i] = vec[i]/total ;
}

void vector_transform(float *vec, int len, float s, int t)
{
  int        i, j, ind;
  float      *vec1 ;
  int        minV, maxV ;
  float      ProbLeft, ProbRight, total ;

  minV = 10000 ; maxV = -10000 ;
  vec1 = (float *)malloc(len*sizeof(float));
  for (i=0; i<len; i++) vec1[i] = 0;
  for (j=0; j<len; j++) 
    {
      ind = (int)((float)(j-t)*s + 0.5);
      /* used to find whether the transformation is out of domain */
      if( ind<minV )  minV = ind ;
      if( ind>maxV )  maxV = ind ;

      if (ind < 0)       vec1[j] = 0 ;/*= vec[0];*/
      if (ind > (len-1)) vec1[j] = 0 ;/*= vec[len-1];*/
      if ((ind >= 0) && (ind <= (len-1))) vec1[j] = vec[ind];
    }
  
  /* Prob was left in the left side */
  if( minV>0 )
    {
      ProbLeft = 0 ;
      for (i=0; i<minV; i++) ProbLeft += vec[i] ;
    }

  /* Prob was left in the right side */
  if( maxV<MaxOfUC )
    {
      ProbRight = 0 ;
      for (i=maxV; i<MaxOfUC; i++) ProbRight += vec[i] ;
    }

   /* add ...  */
  vec1[0] += ProbLeft ;
  vec1[MaxOfUC] += ProbRight ;


  /* transfer ... */
  for (i=0; i<len; i++) vec[i] = vec1[i];
  free(vec1);
}

void vector_InverseTransform(float *vec, int len, float s, int t)
{
  int        i, j, ind;
  float      *vec1 ;
  int        minV, maxV ;
  float      ProbLeft, ProbRight, total ;

  minV = 10000 ; maxV = -10000 ;
  vec1 = (float *)malloc(len*sizeof(float));
  for (i=0; i<len; i++) vec1[i] = 0;
  for (j=0; j<len; j++) 
    {
      ind = (int)(j/s + t + 0.5);
      /* used to find whether the transformation is out of domain */
      if( ind<minV )  minV = ind ;
      if( ind>maxV )  maxV = ind ;

      if (ind < 0)       vec1[j] = 0 ;/*= vec[0];*/
      if (ind > (len-1)) vec1[j] = 0 ;/*= vec[len-1];*/
      if ((ind >= 0) && (ind <= (len-1))) vec1[j] = vec[ind];
    }

  /* Prob was left in the left side */
  if( minV>0 )
    {
      ProbLeft = 0 ;
      for (i=0; i<minV; i++) ProbLeft += vec[i] ;
    }

  /* Prob was left in the right side */
  if( maxV<MaxOfUC )
    {
      ProbRight = 0 ;
      for (i=maxV; i<MaxOfUC; i++) ProbRight += vec[i] ;
    }

   /* add ...  */
   vec1[0] += ProbLeft ;
   vec1[MaxOfUC] += ProbRight ;


  /* transfer ... */
  for (i=0; i<len; i++) vec[i] = vec1[i];
  free(vec1);
}

int GetImgZDim(char file[1000], Ivector3d dim)
{
  FILE        *fp;

  fp = fopen(file, "r");
  fseek(fp,0,SEEK_END);
  dim.z = (ftell(fp))/(dim.x*dim.y);
  fclose(fp);

  return dim.z;
}

void ReadImgUC(char filename[80], unsigned char ***vol, Ivector3d dim)
{
   FILE   *fp;
   int    i, j, k;

   fp = fopen(filename, "r");
   for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
         fread(vol[k][i], 1, dim.y, fp);
   fclose(fp);
}

void WriteImgUC(char filename[80], unsigned char ***data, Ivector3d dim)
{
  FILE  *fp;
  int   i, k ;

  fp=fopen(filename,"w");
  for(k=0;k<dim.z;k++)
    for(i=0;i<dim.x;i++) fwrite(data[k][i],1,dim.y,fp);
  fclose(fp);
}

HISTOGRAM **HISTOGRAMalloc2d(int i_size,int j_size)
{
  HISTOGRAM **array;
  int i,j;

  array=(HISTOGRAM **)calloc(i_size,sizeof(HISTOGRAM *));

  for(i=0;i<i_size;i++)
    array[i]=(HISTOGRAM *) calloc(j_size,sizeof(HISTOGRAM));
			     
  return(array);
}

void HISTOGRAMfree2d(HISTOGRAM **array,int i_size)
{
  int i;

  for(i=0;i<i_size;i++) free(array[i]);
  free(array);
}

void WriteImgF(char filename[80], float **data, Ivector3d dim)
{
  FILE  *fp;
  int   i, k ;

  fp=fopen(filename,"w");
  for(i=0;i<dim.x;i++) fwrite(data[i],sizeof(float),dim.y,fp);
  fclose(fp);
}
