/********************************************************/
/*	File: PGstd.c					*/
/*	Purpose: Various utility functions in C.	*/
/*							*/
/********************************************************/

/* RCS Info
   $Author: pham $
   $Date: 1996/04/18 21:35:13 $
   $Log: pgstd.c,v $
 * Revision 1.5  1996/04/18  21:35:13  pham
 * Added matrix and cube routines for signed shorts
 *
 * Revision 1.5  1996/04/18  21:35:13  pham
 * Added matrix and cube routines for signed shorts
 *
 * Revision 1.4  1996/03/30  17:12:25  chenyang
 * Print message inside the allocation routines now have consistent
 * message with the new naming conventions
 *
 * Revision 1.3  1996/03/25  18:43:31  pham
 * Adopted new naming convention
 *
 * Revision 1.2  1996/01/19  18:52:53  pham
 * Minor modificiations for testing the IBM version of the library
 *
 * Revision 1.1  1996/01/10  19:29:03  pham
 * Initial revision
 *
**/

#include <stdio.h>
#include <pgstd.h>

/* Prints error message */
PGvoid pgError(char error_text[])
{
	PGvoid exit();

	fprintf(stderr,"PG Tools run-time error:\n");
	fprintf(stderr,"%s\n",error_text);
	fprintf(stderr,"Now exiting to system.\n");
	exit(1);
}



/********************************************************/
/* Vector allocation and freeing functions              */
/********************************************************/

/* Allocates a vector of unsigned chars with range [nl..nh] */
PGbyte *pgBvector(int nl,
		  int nh)
{
        PGbyte *v;

        v=(PGbyte *)malloc((unsigned) (nh-nl+1)*sizeof(PGbyte));
        if (!v) pgError("allocation failure in pgBvector()");
        return v-nl;
}

/* Frees a vector allocated by pgBvector */
PGvoid pgFreeBvector(PGbyte *v,
		     int nl,
		     int nh)
{
        free((char*) (v+nl));
}

/* Allocates a vector of unsigned shorts with range [nl..nh] */
PGushort *pgUsvector(int nl,
		     int nh)
{
        PGushort *v;

        v=(PGushort *)malloc((unsigned) (nh-nl+1)*sizeof(PGushort));
        if (!v) pgError("allocation failure in pgUsvector()");
        return v-nl;
}



/* Frees a vector allocated by pgUsvector */
PGvoid pgFreeUsvector(PGushort *v,
		       int nl,
		       int nh)
{
        free((char*) (v+nl));
}
/* Allocates a vector of unsigned shorts with range [nl..nh] */
PGshort *pgSvector(int nl,
		     int nh)
{
        PGshort *v;

        v=(PGshort *)malloc((unsigned) (nh-nl+1)*sizeof(PGshort));
        if (!v) pgError("allocation failure in pgSvector()");
        return v-nl;
}



/* Frees a vector allocated by pgUsvector */
PGvoid pgFreeSvector(PGshort *v,
		       int nl,
		       int nh)
{
        free((char*) (v+nl));
}


/* Allocates a vector of unsigned ints with range [nl..nh] */
PGuint *pgUivector(int nl,
		   int nh)
{
        PGuint *v;

        v=(PGuint *)malloc((unsigned) (nh-nl+1)*sizeof(PGuint));
        if (!v) pgError("allocation failure in pgUivector()");
        return v-nl;
}

/* Frees a vector allocated by pgUivector */
PGvoid pgFreeUivector(PGuint *v,
		       int nl,
		       int nh)
{
        free((char*) (v+nl));
}

/* Allocates a vector of ints with range [nl..nh] */
PGint *pgIvector(nl,nh)
int nl,nh;
{
	PGint *v;

	v=(PGint *)malloc((unsigned) (nh-nl+1)*sizeof(int));
	if (!v) pgError("allocation failure in pgIvector()");
	return v-nl;
}



/* Frees a vector allocated by pgIvector */
PGvoid pgFreeIvector(v,nl,nh)
int *v,nl,nh;
{
	free((char*) (v+nl));
}





/* Allocates a vector of floats with range [nl..nh] */
PGfloat *pgFvector(nl,nh)
int nl,nh;
{
	PGfloat *v;

	v=(float *)malloc((unsigned) (nh-nl+1)*sizeof(float));
	if (!v) pgError("allocation failure in pgFvector()");
	return v-nl;
}


/* Frees a vector allocated by fvector */
PGvoid pgFreeFvector(v,nl,nh)
PGfloat *v;
int nl,nh;
{
	free((char*) (v+nl));
}






/* Allocates a vector of doubles with range [nl..nh] */
PGdouble *pgDvector(nl,nh)
int nl,nh;
{
	PGdouble *v;

	v=(PGdouble *)malloc((unsigned) (nh-nl+1)*sizeof(PGdouble));
	if (!v) pgError("allocation failure in pgDvector()");
	return v-nl;
}

/* Frees a vector allocated by dvector */
PGvoid pgFreeDvector(v,nl,nh)
PGdouble *v;
int nl,nh;
{
	free((char*) (v+nl));
}

/* Allocates a vector of byte, 2-D vectors with range [nl..nh] */
PGbyte2d *pgB2dvector(nl,nh)
int nl,nh;
{
	PGbyte2d *v;

	v=(PGbyte2d *)malloc((unsigned) (nh-nl+1)*sizeof(PGbyte2d));
	if (!v) pgError("allocation failure in pgB2dvector()");
	return v-nl;
}


/* Frees a vector allocated by pgB2dvector */
PGvoid pgFreeB2dvector(v,nl,nh)
PGbyte2d *v;
int nl,nh;
{
	free((char*) (v+nl));
}

/* Allocates a vector of int, 2-D vectors with range [nl..nh] */
PGint2d *pgI2dvector(nl,nh)
int nl,nh;
{
	PGint2d *v;

	v=(PGint2d *)malloc((unsigned) (nh-nl+1)*sizeof(PGint2d));
	if (!v) pgError("allocation failure in pgI2dvector()");
	return v-nl;
}


/* Frees a vector allocated by pgI2dvector */
PGvoid pgFreeI2dvector(v,nl,nh)
PGint2d *v;
int nl,nh;
{
	free((char*) (v+nl));
}

/* Allocates a vector of float, 2-D vectors with range [nl..nh] */
PGfloat2d *pgF2dvector(nl,nh)
int nl,nh;
{
	PGfloat2d *v;

	v=(PGfloat2d *)malloc((unsigned) (nh-nl+1)*sizeof(PGfloat2d));
	if (!v) pgError("allocation failure in pgF2dvector()");
	return v-nl;
}


/* Frees a vector allocated by pgF2dvector */
PGvoid pgFreeF2dvector(v,nl,nh)
PGfloat2d *v;
int nl,nh;
{
	free((char*) (v+nl));
}

/* Allocates a vector of double, 2-D vectors with range [nl..nh] */
PGdouble2d *pgD2dvector(nl,nh)
int nl,nh;
{
	PGdouble2d *v;

	v=(PGdouble2d *)malloc((unsigned) (nh-nl+1)*sizeof(PGdouble2d));
	if (!v) pgError("allocation failure in pgD2dvector()");
	return v-nl;
}


/* Frees a vector allocated by d2dvector */
PGvoid pgFreeD2dvector(v,nl,nh)
PGdouble2d *v;
int nl,nh;
{
	free((char*) (v+nl));
}




/* Allocates a vector of byte, 3-D vectors with range [nl..nh] */
PGbyte3d *pgB3dvector(nl,nh)
int nl,nh;
{
	PGbyte3d *v;

	v=(PGbyte3d *)malloc((unsigned) (nh-nl+1)*sizeof(PGbyte3d));
	if (!v) pgError("allocation failure in pgB3dvector()");
	return v-nl;
}



/* Frees a vector allocated by pgB3dvector */
PGvoid pgFreeB3dvector(v,nl,nh)
PGbyte3d *v;
int nl,nh;
{
	free((char*) (v+nl));
}


/* Allocates a vector of int, 3-D vectors with range [nl..nh] */
PGint3d *pgI3dvector(nl,nh)
int nl,nh;
{
	PGint3d *v;

	v=(PGint3d *)malloc((unsigned) (nh-nl+1)*sizeof(PGint3d));
	if (!v) pgError("allocation failure in pgI3dvector()");
	return v-nl;
}



/* Frees a vector allocated by pgI3dvector */
PGvoid pgFreeI3dvector(v,nl,nh)
PGint3d *v;
int nl,nh;
{
	free((char*) (v+nl));
}




/* Allocates a vector of float, 3-D vectors with range [nl..nh] */
PGfloat3d *pgF3dvector(nl,nh)
int nl,nh;
{
	PGfloat3d *v;

	v=(PGfloat3d *)malloc((unsigned) (nh-nl+1)*sizeof(PGfloat3d));
	if (!v) pgError("allocation failure in pgF3dvector()");
	return v-nl;
}


/* Frees a vector allocated by pgF3dvector */
PGvoid pgFreeF3dvector(v,nl,nh)
PGfloat3d *v;
int nl,nh;
{
	free((char*) (v+nl));
}




/* Allocates a vector of double, 3-D vectors with range [nl..nh] */
PGdouble3d *pgD3dvector(nl,nh)
int nl,nh;
{
	PGdouble3d *v;

	v=(PGdouble3d *)malloc((unsigned) (nh-nl+1)*sizeof(PGdouble3d));
	if (!v) pgError("allocation failure in pgD3dvector()");
	return v-nl;
}



/* Frees a vector allocated by pgD3dvector */
PGvoid pgFreeD3dvector(v,nl,nh)
PGdouble3d *v;
int nl,nh;
{
	free((char*) (v+nl));
}




/********************************************************/
/* Matrix allocation and freeing functions              */
/********************************************************/

/* Allocates a matrix of unsigned chars with range [nrl..nrh][ncl..nch] */
PGbyte **pgBmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGbyte **m;

  bufsize = (nrh-nrl+1)*sizeof(PGbyte*)
            + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGbyte);

  m=(PGbyte **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgBmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGbyte*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGbyte *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGbyte)));
    m[j] -= ncl;
  }

  return m;
}




/* Frees a matrix allocated by bmatrix */
PGvoid pgFreeBmatrix(m,nrl,nrh,ncl,nch)
PGbyte **m;
int nrl,nrh,ncl,nch;
{
        free((char*) (m+nrl));
}

/* Allocates a matrix of unsigned shorts with range [nrl..nrh][ncl..nch] */
PGushort **pgUsmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGushort bufsize,bufptr;
  PGushort **m;

  bufsize = (nrh-nrl+1)*sizeof(PGushort*)
            + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGushort);

  m=(PGushort **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgUsmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGushort*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGushort *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGushort)));
    m[j] -= ncl;
  }

  return m;
}



/* frees a matrix allocated by usmatrix */
PGvoid pgFreeUsmatrix(m,nrl,nrh,ncl,nch)
PGushort **m;
int nrl,nrh,ncl,nch;
{
        free((char*) (m+nrl));
}

/* Allocates a matrix of signed shorts with range [nrl..nrh][ncl..nch] */
PGshort **pgSmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGshort bufsize,bufptr;
  PGshort **m;

  bufsize = (nrh-nrl+1)*sizeof(PGshort*)
            + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGshort);

  m=(PGshort **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgSmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGshort*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGshort *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGshort)));
    m[j] -= ncl;
  }

  return m;
}



/* frees a matrix allocated by Smatrix */
PGvoid pgFreeSmatrix(m,nrl,nrh,ncl,nch)
PGshort **m;
int nrl,nrh,ncl,nch;
{
        free((char*) (m+nrl));
}


/* Allocates a matrix of unsigned ints with range [nrl..nrh][ncl..nch] */
PGuint **pgUimatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGuint **m;

  bufsize = (nrh-nrl+1)*sizeof(PGuint*)
            + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGuint);

  m=(PGuint **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgUimatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGuint*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGuint *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGuint)));
    m[j] -= ncl;
  }

  return m;
}



/* Frees a matrix allocated by uimatrix */
PGvoid pgFreeUimatrix(m,nrl,nrh,ncl,nch)
PGuint **m;
int nrl,nrh,ncl,nch;
{
        free((char*) (m+nrl));
}



/* Allocates a matrix of doubles with range [nrl..nrh][ncl..nch] */
PGdouble **pgDmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGdouble **m;

  bufsize = (nrh-nrl+1)*sizeof(PGdouble*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGdouble);

  m=(PGdouble **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgDmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGdouble*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGdouble *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGdouble)));
    m[j] -= ncl;
  }

  return m;
}



/* Frees a matrix allocated by dmatrix */
PGvoid pgFreeDmatrix(m,nrl,nrh,ncl,nch)
PGdouble **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}



/* Allocates a matrix of floats with range [nrl..nrh][ncl..nch] */
PGfloat **pgFmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGfloat **m;

  bufsize = (nrh-nrl+1)*sizeof(PGfloat*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGfloat);

  m=(PGfloat **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgFmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGfloat*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGfloat *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGfloat)));
    m[j] -= ncl;
  }

  return m;
}


/* Frees a matrix allocated by fmatrix */
PGvoid pgFreeFmatrix(m,nrl,nrh,ncl,nch)
PGfloat **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}




/* Allocates a matrix of ints with range [nrl..nrh][ncl..nch] */
PGint **pgImatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGint **m;

  bufsize = (nrh-nrl+1)*sizeof(PGint*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGint);

  m=(PGint **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgImatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGint*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGint *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGint)));
    m[j] -= ncl;
  }

  return m;
}



/* Frees a matrix allocated by pgImatrix */
PGvoid pgFreeImatrix(m,nrl,nrh,ncl,nch)
PGint **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}


/* Allocates a matrix of byte, 2-D vectors with range [nrl..nrh][ncl..nch] */
PGbyte2d **pgB2dmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGbyte2d **m;

  bufsize = (nrh-nrl+1)*sizeof(PGbyte2d*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGbyte2d);

  m=(PGbyte2d **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgB2dmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGbyte2d*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGbyte2d *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGbyte2d)));
    m[j] -= ncl;
  }

  return m;
}

/* Frees a matrix allocated by pgB2dmatrix */
PGvoid pgFreeB2dmatrix(m,nrl,nrh,ncl,nch)
PGbyte2d **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}


/* Allocates a matrix of int, 2-D vectors with range [nrl..nrh][ncl..nch] */
PGint2d **pgI2dmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGint2d **m;

  bufsize = (nrh-nrl+1)*sizeof(PGint2d*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGint2d);

  m=(PGint2d **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgI2dmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGint2d*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGint2d *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGint2d)));
    m[j] -= ncl;
  }

  return m;
}

/* Frees a matrix allocated by pgI2dmatrix */
PGvoid pgFreeI2dmatrix(m,nrl,nrh,ncl,nch)
PGint2d **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}


/* Allocates a matrix of float, 2-D vectors with range [nrl..nrh][ncl..nch] */
PGfloat2d **pgF2dmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGfloat2d **m;

  bufsize = (nrh-nrl+1)*sizeof(PGfloat2d*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGfloat2d);

  m=(PGfloat2d **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgF2dmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGfloat2d*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGfloat2d *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGfloat2d)));
    m[j] -= ncl;
  }

  return m;
}

/* Frees a matrix allocated by pgF2dmatrix */
PGvoid pgFreeF2dmatrix(m,nrl,nrh,ncl,nch)
PGfloat2d **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}




/* Allocates a matrix of double, 2-D vectors with range [nrl..nrh][ncl..nch] */
PGdouble2d **pgD2dmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGdouble2d **m;

  bufsize = (nrh-nrl+1)*sizeof(PGdouble2d*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGdouble2d);

  m=(PGdouble2d **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgD2dmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGdouble2d*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGdouble2d *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGdouble2d)));
    m[j] -= ncl;
  }

  return m;
}

/* Frees a matrix allocated by d2dmatrix */
PGvoid pgFreeD2dmatrix(m,nrl,nrh,ncl,nch)
PGdouble2d **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}

/* Allocates a matrix of byte, 3-D vectors with range [nrl..nrh][ncl..nch] */
PGbyte3d **pgB3dmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGbyte3d **m;

  bufsize = (nrh-nrl+1)*sizeof(PGbyte3d*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGbyte3d);

  m=(PGbyte3d **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgB3dmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGbyte3d*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGbyte3d *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGbyte3d)));
    m[j] -= ncl;
  }

  return m;
}


/* Frees a matrix allocated by pgB3dmatrix */
PGvoid pgFreeB3dmatrix(m,nrl,nrh,ncl,nch)
PGbyte3d **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}


/* Allocates a matrix of int, 3-D vectors with range [nrl..nrh][ncl..nch] */
PGint3d **pgI3dmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGint3d **m;

  bufsize = (nrh-nrl+1)*sizeof(PGint3d*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGint3d);

  m=(PGint3d **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgI3dmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGint3d*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGint3d *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGint3d)));
    m[j] -= ncl;
  }

  return m;
}


/* Frees a matrix allocated by pgI3dmatrix */
PGvoid pgFreeI3dmatrix(m,nrl,nrh,ncl,nch)
PGint3d **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}


/* Allocates a matrix of float, 3-D vectors with range [nrl..nrh][ncl..nch] */
PGfloat3d **pgF3dmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGfloat3d **m;

  bufsize = (nrh-nrl+1)*sizeof(PGfloat3d*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGfloat3d);

  m=(PGfloat3d **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgF3dmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGfloat3d*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGfloat3d *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGfloat3d)));
    m[j] -= ncl;
  }

  return m;
}


/* Frees a matrix allocated by pgF3dmatrix */
PGvoid pgFreeF3dmatrix(m,nrl,nrh,ncl,nch)
PGfloat3d **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}



/* Allocates a matrix of double, 3-D vectors with range [nrl..nrh][ncl..nch] */
PGdouble3d **pgD3dmatrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
  int j;
  PGuint bufsize,bufptr;
  PGdouble3d **m;

  bufsize = (nrh-nrl+1)*sizeof(PGdouble3d*)
	    + (nrh-nrl+1)*(nch-ncl+1)*sizeof(PGdouble3d);

  m=(PGdouble3d **) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgD3dmatrix()");
  m -= nrl;

  bufptr = ((unsigned) (m+nrl)) + (nrh-nrl+1)*sizeof(PGdouble3d*);
  for(j=nrl;j<=nrh;j++) {
    m[j] = ((PGdouble3d *) (bufptr+(j-nrl)*(nch-ncl+1)*sizeof(PGdouble3d)));
    m[j] -= ncl;
  }

  return m;
}


/* Frees a matrix allocated by d3dmatrix */
PGvoid pgFreeD3dmatrix(m,nrl,nrh,ncl,nch)
PGdouble3d **m;
int nrl,nrh,ncl,nch;
{
	free((char*) (m+nrl));
}



/********************************************************/
/* Cube allocation and freeing functions                */
/********************************************************/

/* Allocates a cube of unsigned shorts with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGushort ***pgUscube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGushort ***m;

  bufsize = (nph-npl+1)*sizeof(PGushort**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGushort*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGushort);

  m=(PGushort ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgUscube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGushort**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGushort **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGushort*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGushort*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGushort *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGushort)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGushort)));
      m[i][j] -= ncl;
    }

  return m;
}

/* Frees a cube allocated by uscube */
PGvoid pgFreeUscube(m,npl,nph,nrl,nrh,ncl,nch)
PGushort ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}

/* Allocates a cube of signed shorts with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGshort ***pgScube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGshort ***m;

  bufsize = (nph-npl+1)*sizeof(PGshort**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGshort*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGshort);

  m=(PGshort ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgScube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGshort**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGshort **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGshort*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGshort*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGshort *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGshort)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGshort)));
      m[i][j] -= ncl;
    }

  return m;
}

/* Frees a cube allocated by scube */
PGvoid pgFreeScube(m,npl,nph,nrl,nrh,ncl,nch)
PGshort ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}

/* Allocates a cube of doubles with range [npl..nph][nrl..nrh][ncl..nch] */
PGdouble ***pgDcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGdouble ***m;

  bufsize = (nph-npl+1)*sizeof(PGdouble**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGdouble*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGdouble);

  m=(PGdouble ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgDcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGdouble**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGdouble **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGdouble*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGdouble*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGdouble *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGdouble)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGdouble)));
      m[i][j] -= ncl;
    }

  return m;
}

/* Frees a cube allocated by dcube */
PGvoid pgFreeDcube(m,npl,nph,nrl,nrh,ncl,nch)
PGdouble ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}




/* Allocates a cube of integers with range [npl..nph][nrl..nrh][ncl..nch] */
PGint ***pgIcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGint ***m;

  bufsize = (nph-npl+1)*sizeof(PGint**)
            + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGint*)
            + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGint);

  m=(PGint ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgIcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGint**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGint **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGint*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGint*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGint *) (bufptr
                + (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGint)
                + (j-nrl)*(nch-ncl+1)*sizeof(PGint)));
      m[i][j] -= ncl;
    }

  return m;
}


/* Frees a cube allocated by icube */
PGvoid pgFreeIcube(m,npl,nph,nrl,nrh,ncl,nch)
PGint ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}



/* Allocates a cube of PGbyte2d, 2D vectors with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGbyte2d ***pgB2dcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGbyte2d ***m;

  bufsize = (nph-npl+1)*sizeof(PGbyte2d**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGbyte2d*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGbyte2d);

  m=(PGbyte2d ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgB2dcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGbyte2d**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGbyte2d **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGbyte2d*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGbyte2d*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGbyte2d *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGbyte2d)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGbyte2d)));
      m[i][j] -= ncl;
    }

  return m;
}

/* Frees a cube allocated by pgB2dcube */
PGvoid pgFreeB2dcube(m,npl,nph,nrl,nrh,ncl,nch)
PGbyte2d ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}

/* Allocates a cube of PGint2d, 2D vectors with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGint2d ***pgI2dcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGint2d ***m;

  bufsize = (nph-npl+1)*sizeof(PGint2d**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGint2d*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGint2d);

  m=(PGint2d ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgI2dcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGint2d**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGint2d **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGint2d*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGint2d*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGint2d *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGint2d)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGint2d)));
      m[i][j] -= ncl;
    }

  return m;
}

/* Frees a cube allocated by pgI2dcube */
PGvoid pgFreeI2dcube(m,npl,nph,nrl,nrh,ncl,nch)
PGint2d ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}


/* Allocates a cube of PGfloat2d, 2D vectors with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGfloat2d ***pgF2dcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGfloat2d ***m;

  bufsize = (nph-npl+1)*sizeof(PGfloat2d**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGfloat2d*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGfloat2d);

  m=(PGfloat2d ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgF2dcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGfloat2d**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGfloat2d **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGfloat2d*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGfloat2d*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGfloat2d *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGfloat2d)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGfloat2d)));
      m[i][j] -= ncl;
    }

  return m;
}

/* Frees a cube allocated by pgF2dcube */
PGvoid pgFreeF2dcube(m,npl,nph,nrl,nrh,ncl,nch)
PGfloat2d ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}


/* Allocates a cube of double, 2D vectors with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGdouble2d ***pgD2dcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGdouble2d ***m;

  bufsize = (nph-npl+1)*sizeof(PGdouble2d**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGdouble2d*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGdouble2d);

  m=(PGdouble2d ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgD2dcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGdouble2d**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGdouble2d **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGdouble2d*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGdouble2d*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGdouble2d *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGdouble2d)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGdouble2d)));
      m[i][j] -= ncl;
    }

  return m;
}

/* Frees a cube allocated by pgD2dcube */
PGvoid pgFreeD2dcube(m,npl,nph,nrl,nrh,ncl,nch)
PGdouble2d ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}


/* Allocates a cube of byte, 3D vectors with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGbyte3d ***pgB3dcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGbyte3d ***m;

  bufsize = (nph-npl+1)*sizeof(PGbyte3d**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGbyte3d*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGbyte3d);

  m=(PGbyte3d ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgB3dcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGbyte3d**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGbyte3d **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGbyte3d*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGbyte3d*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGbyte3d *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGbyte3d)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGbyte3d)));
      m[i][j] -= ncl;
    }

  return m;
}

/* Frees a cube allocated by pgB3dcube */
PGvoid pgFreeB3dcube(m,npl,nph,nrl,nrh,ncl,nch)
PGbyte3d ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}

/* Allocates a cube of int, 3D vectors with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGint3d ***pgI3dcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGint3d ***m;

  bufsize = (nph-npl+1)*sizeof(PGint3d**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGint3d*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGint3d);

  m=(PGint3d ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgI3dcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGint3d**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGint3d **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGint3d*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGint3d*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGint3d *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGint3d)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGint3d)));
      m[i][j] -= ncl;
    }

  return m;
}

/* Frees a cube allocated by pgI3dcube */
PGvoid pgFreeI3dcube(m,npl,nph,nrl,nrh,ncl,nch)
PGint3d ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}



/* Allocates a cube of float, 3-D vectors with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGfloat3d ***pgF3dcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGfloat3d ***m;

  bufsize = (nph-npl+1)*sizeof(PGfloat3d**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGfloat3d*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGfloat3d);

  m=(PGfloat3d ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgF3dcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGfloat3d**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGfloat3d **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGfloat3d*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGfloat3d*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGfloat3d *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGfloat3d)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGfloat3d)));
      m[i][j] -= ncl;
    }

  return m;
}




/* Frees a cube allocated by f3dcube */
PGvoid pgFreeF3dcube(m,npl,nph,nrl,nrh,ncl,nch)
PGfloat3d ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}



/* Allocates a cube of double, 3D vectors with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGdouble3d ***pgD3dcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGdouble3d ***m;

  bufsize = (nph-npl+1)*sizeof(PGdouble3d**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGdouble3d*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGdouble3d);

  m=(PGdouble3d ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgD3dcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGdouble3d**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGdouble3d **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGdouble3d*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGdouble3d*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGdouble3d *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGdouble3d)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGdouble3d)));
      m[i][j] -= ncl;
    }

  return m;
}

/* Frees a cube allocated by d3dcube */
PGvoid pgFreeD3dcube(m,npl,nph,nrl,nrh,ncl,nch)
PGdouble3d ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}



/* Allocates a cube of floats with range [npl..nph][nrl..nrh][ncl..nch] */
PGfloat ***pgFcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGfloat ***m;

  bufsize = (nph-npl+1)*sizeof(PGfloat**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGfloat*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGfloat);

  m=(PGfloat ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgFcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGfloat**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGfloat **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGfloat*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGfloat*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGfloat *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGfloat)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGfloat)));
      m[i][j] -= ncl;
    }

  return m;
}




/* Frees a cube allocated by fcube */
PGvoid pgFreeFcube(m,npl,nph,nrl,nrh,ncl,nch)
PGfloat ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}




/* Allocates a cube of unsigned chars with 
   range [npl..nph][nrl..nrh][ncl..nch] */
PGbyte ***pgBcube(npl,nph,nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch,npl,nph;
{
  int i,j;
  PGuint bufsize,bufptr;
  PGbyte ***m;

  bufsize = (nph-npl+1)*sizeof(PGbyte**)
	    + (nph-npl+1)*(nrh-nrl+1)*sizeof(PGbyte*)
	    + (nph-npl+1)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGbyte);

  m=(PGbyte ***) malloc(bufsize);
  if (!m) pgError("allocation failure 1 in pgBcube()");
  m -= npl;

  bufptr = ((unsigned) (m+npl)) + (nph-npl+1)*sizeof(PGbyte**);
  for(i=npl;i<=nph;i++) {
    m[i] = ((PGbyte **) (bufptr+(i-npl)*(nrh-nrl+1)*sizeof(PGbyte*)));
    m[i] -= nrl;
  }

  bufptr += (unsigned) (nph-npl+1)*(nrh-nrl+1)*sizeof(PGbyte*);
  for(i=npl;i<=nph;i++)
    for(j=nrl;j<=nrh;j++) {
      m[i][j] = ((PGbyte *) (bufptr
		+ (i-npl)*(nrh-nrl+1)*(nch-ncl+1)*sizeof(PGbyte)
		+ (j-nrl)*(nch-ncl+1)*sizeof(PGbyte)));
      m[i][j] -= ncl;
    }

  return m;
}




/* Frees a cube allocated by bcube */
PGvoid pgFreeBcube(m,npl,nph,nrl,nrh,ncl,nch)
PGbyte ***m;
int nrl,nrh,ncl,nch,npl,nph;
{
	free((char*) (m+npl));
}



/*************************************************************/
/*        Following image i/o operations added by            */
/*        Zhiqiang Lao                                       */
/*        3/28/2007                                          */
/*  Don't know why they are missing from original version    */
/*************************************************************/
int pgFileSize(file_name)
char *file_name;
{
  FILE        *fp;
  int         sz;
  
  fp = fopen(file_name, "r");
  if (fp == NULL) {
    printf("can not open %s\n", file_name);
    sz = PG_ERROR;
  } else {
    fseek(fp, 0, SEEK_END);
    sz = ftell(fp);
  }
  fclose(fp);

  return sz;
}


int pgReadByte(Bbuffer, length, start, file_name)
PGbyte *Bbuffer;
int    length;
int    start;
char   *file_name;
{
  int         sz;
  FILE        *fp;
  
  sz = pgFileSize(file_name);
  if ((sz==PG_ERROR) || (sz<length)) return PG_ERROR;
  fp = fopen(file_name, "r");
  fseek(fp, start*sizeof(PGbyte), SEEK_SET);
  fread(Bbuffer, sizeof(PGbyte), length, fp);
  fclose(fp);
  
  return length;
}


int pgReadShort(Sbuffer, length, start, file_name)
PGshort *Sbuffer;
int     length;
int     start;
char    *file_name;
{
  int         sz;
  FILE        *fp;
  
  sz = pgFileSize(file_name);
  if ((sz==PG_ERROR) || (sz<length)) return PG_ERROR;
  fp = fopen(file_name, "r");
  fseek(fp, start*sizeof(PGshort), SEEK_SET);
  fread(Sbuffer, sizeof(PGshort), length, fp);
  fclose(fp);
  
  return length;
}


int pgReadUshort(Usbuffer, length, start, file_name)
PGushort *Usbuffer;
int      length;
int      start;
char     *file_name;
{
  int         sz;
  FILE        *fp;
  
  sz = pgFileSize(file_name);
  if ((sz==PG_ERROR) || (sz<length)) return PG_ERROR;
  fp = fopen(file_name, "r");
  fseek(fp, start*sizeof(PGushort), SEEK_SET);
  fread(Usbuffer, sizeof(PGushort), length, fp);
  fclose(fp);
  
  return length;
}


int pgWriteByte(Bbuffer, length, file_name)
PGbyte *Bbuffer;
int    length;
char   *file_name;
{
  int         sz;
  FILE        *fp;
  
  fp = fopen(file_name, "w");
  if (fp==NULL) {
    printf("can not write %s, check disk space\n", file_name);
    return PG_ERROR;
  }
  fwrite(Bbuffer, sizeof(PGbyte), length, fp);
  fclose(fp);
  
  return length;
}


int pgWriteShort(Sbuffer, length, file_name)
PGshort *Sbuffer;
int    length;
char   *file_name;
{
  int         sz;
  FILE        *fp;
  
  fp = fopen(file_name, "w");
  if (fp==NULL) {
    printf("can not write %s, check disk space\n", file_name);
    return PG_ERROR;
  }
  fwrite(Sbuffer, sizeof(PGshort), length, fp);
  fclose(fp);
  
  return length;
}


int pgWriteUshort(Usbuffer, length, file_name)
PGushort *Usbuffer;
int      length;
char     *file_name;
{
  int         sz;
  FILE        *fp;
  
  fp = fopen(file_name, "w");
  if (fp==NULL) {
    printf("can not write %s, check disk space\n", file_name);
    return PG_ERROR;
  }
  fwrite(Usbuffer, sizeof(PGushort), length, fp);
  fclose(fp);
  
  return length;
}





