#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <math.h>
#include <matrixSHEN.h>
#include <mvcd.h>

#define BASIC_R 0

typedef struct {
  char        t1[200], t2[200], pd[200], flair[200], mask[200], nonlesion_mask[200], ven_mask[200];
} FILEGROUP;

typedef struct {
  unsigned char        ***t1, ***t2, ***pd, ***flair, ***mask, ***nonlesion_mask, ***ven_mask;
} VOLUMEGROUP;

typedef struct {
  float        t1, t2, pd, flair;
} WEIGHTGROUP;

int         GetLineNumber(char filename[100]);
void        show_usage(int argc, char **argv);
void        NormalizeVector(float *vec, int vec_len);
void        ReadImg(char filename[80], unsigned short ***vol, Ivector3d dim);
void        ReadImgUC(char filename[80], unsigned char ***vol, Ivector3d dim);
void        ReadImg_FLOAT(char filename[80], float ***vol, Ivector3d dim);
void        ReadFeatureVector(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim);
void        ReadFeatureVector_FLOAT(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim);
void        PrintVector(float *vec, int vec_len);
int         IsValidVoxel(unsigned char ***img, Ivector3d dim, Ivector3d pos);
//int         IsValidVoxel_Neighbor(unsigned char ***img, Ivector3d dim, Ivector3d pos, int radius);
int         IsValidVoxel_Neighbor(unsigned char ***img, Ivector3d dim, Fvector3d res, Ivector3d pos, Ivector3d *bubble, int pixelNumInBubble);
void        WriteImgUC(char filename[80], unsigned char ***data, Ivector3d dim);
void        calculate_hood_special(Fvector3d *bubble,int bubble_size, int *pixelNumInBubble, Fvector3d res);
void        calculate_hood_special_improve(Ivector3d *bubble,int bubble_size, int *pixelNumInBubble, Fvector3d res);
int         IsPtInLst(Fvector3d pt, Ivector3d *lst, int lst_len);

void show_usage(int argc, char **argv)
{
   printf("USAGE: %s t1.img t2.img pd.img flair.img mask.img feature_vector\n\
\t -d <int>,<int>             : image dimension in XY (default: 256,256)\n\
\t -R <float>,<float>,<float> : image resolution (default: 0.9375,0.9375,3.0)\n\
\t -t <int>                   : type of feature vectors\n\
\t                              0: lesion (default: 0)\n\
\t                              1: non-lesion \n\
\t -m <int>                   : methods for getting feature vectors (default: 0)\n\
\t                              0: voxel-wise intensity\n\
\t                              1: voxel-wise with neighborhood defined by -r option over smoothed volume\n\
\t -r <int>                   : neighbor size (default: 1)\n\
\t -b <float>                 : blur 3D image by Guassian filter (0.5)\n\
\t -n <int>                   : number of voxels to select (default: 1000)\n\
\t -O <string>                : output nonlesion mask image \n\
\t -M <string>                : mask image that only focus on \n\
\t -P <string>                : output position file for selected voxels \n\
\t -w <float>,<float>,<float>,<float>   : weighting parameters for different modalities, t1, t2, pd, flair respectively (default: 1.0,1.0,1.0,1.0)\n\
\t -S                         : special case, get lesion feature but labeled as nonlesion feature\n\
", argv[0]);
   exit(1);
}

int main(int argc, char **argv)
{
  FILE          *fp, *fpw, *fp_pos;
  int           i, j, k, ii, jj, kk, iii, jjj, kkk, c, s, vec_len, subno, type;
  int           subid, index, selected_voxel_number, x, y, z, ind, method, special_case;
  int           save_nonlesion_mask, only_preventricle, radius, save_position;
  Ivector3d     *pos, dim, tmp_pos;
  Fvector3d     res;
  extern char   *optarg;
  VOLUMEGROUP   ImgGrp;
  FILEGROUP     FileGrp, FileGrpSmooth;
  float         *vec, tmpx, tmpy, ratio, smooth_factor;
  char          cmd[1000], position_file[1000];
  unsigned char ***ven_mask;
  WEIGHTGROUP   WeightGrp;
  int           bubble_size, pixelNumInBubble, t;
  Ivector3d     *bubble;
  
  /* read in arguments */
  if (argc < 6) show_usage(argc, argv);
  
  dim.x = dim.y = 256; dim.z = 124;
  res.x = res.y = 0.9375; res.z = 3.0;
  type = method = 0;
  smooth_factor = 0.5;
  save_nonlesion_mask = 0;
  only_preventricle = 0;
  radius = 1;
  special_case = 0;
  res.x = res.y = 0.9375; res.z = 3.0;
  save_position = 0;
  WeightGrp.t1 = WeightGrp.t2 = WeightGrp.pd = WeightGrp.flair = 1.0;
  while((c=getopt(argc-6,argv+6,"d:t:n:m:b:w:O:M:r:R:SP:")) != -1) {
    switch(c) {
    case 'd':
      sscanf(optarg, "%d,%d", &dim.x, &dim.y);
      break;

    case 't':
      type = atoi(optarg);
      break;

    case 'n':
      selected_voxel_number = atoi(optarg);
      break;

    case 'm':
      method = atoi(optarg);
      break;

    case 'b':
      smooth_factor = atof(optarg);
      break;

    case 'w':
      sscanf(optarg, "%f,%f,%f,%f", &WeightGrp.t1, &WeightGrp.t2, &WeightGrp.pd, &WeightGrp.flair);
      break;

    case 'O':
      save_nonlesion_mask = 1;
      sscanf(optarg, "%s", FileGrp.nonlesion_mask);
      break;

    case 'M':
      only_preventricle = 1;
      sscanf(optarg, "%s", FileGrp.ven_mask);
      break;

    case 'r':
      radius = atoi(optarg);
      break;

    case 'S':
      special_case = 1;
      break;

    case 'R':
      sscanf(optarg, "%f,%f,%f", &res.x, &res.y, &res.z);
      break;

    case 'P':
      save_position = 1;
      sscanf(optarg, "%s", position_file);

    default:
      break;
    }
  }
  
  strcpy(FileGrp.t1,    argv[1]);
  strcpy(FileGrp.t2,    argv[2]);
  strcpy(FileGrp.pd,    argv[3]);
  strcpy(FileGrp.flair, argv[4]);
  strcpy(FileGrp.mask,  argv[5]);

  fp = fopen(FileGrp.t1,"r");
  fseek(fp,0,SEEK_END);
  dim.z = (ftell(fp))/(dim.x*dim.y);
  fclose(fp);
  printf("image dimension: (%d, %d, %d)\n", dim.x, dim.y, dim.z);
  printf("image resolution: (%f, %f, %f)\n", res.x, res.y, res.z);
  printf("selected voxel number: %d\n", selected_voxel_number);
  printf("type: %d\n", type);
  printf("method: %d\n", method);
  printf("smooth factor: %f\n", smooth_factor);
  printf("weighting: (%f, %f, %f, %f)\n", WeightGrp.t1, WeightGrp.t2, WeightGrp.pd, WeightGrp.flair);
  printf("neighbor size: %d\n", radius);
  printf("special case: %d\n", special_case);
  if (save_nonlesion_mask == 1) printf("nonlesion mask image: %s\n", FileGrp.nonlesion_mask);
  if (only_preventricle == 1) printf("pre ventricle mask image: %s\n", FileGrp.ven_mask);
  if (save_position == 1) printf("output position file: %s\n", position_file);

  ImgGrp.t1    = UCalloc3d(dim.x, dim.y, dim.z);
  ImgGrp.t2    = UCalloc3d(dim.x, dim.y, dim.z);
  ImgGrp.pd    = UCalloc3d(dim.x, dim.y, dim.z);
  ImgGrp.flair = UCalloc3d(dim.x, dim.y, dim.z);
  ImgGrp.mask  = UCalloc3d(dim.x, dim.y, dim.z);
  if (save_nonlesion_mask) ImgGrp.nonlesion_mask = UCalloc3d(dim.x, dim.y, dim.z);
  if (only_preventricle) ImgGrp.ven_mask = UCalloc3d(dim.x, dim.y, dim.z);

  //  bubble = Fvector3dalloc1d(10000);
  bubble = Ivector3dalloc1d(10000);
  bubble_size = radius;
  //  calculate_hood_special(bubble, bubble_size, &pixelNumInBubble, res);
  calculate_hood_special_improve(bubble, bubble_size, &pixelNumInBubble, res);
  printf("bubble_size=%d, pixelNumInBubble=%d\n", bubble_size, pixelNumInBubble);

  // read input images according to different options
  switch (method) {
  case 0: // read original input images directly
    ReadImgUC(FileGrp.t1,    ImgGrp.t1,    dim);
    ReadImgUC(FileGrp.t2,    ImgGrp.t2,    dim);
    ReadImgUC(FileGrp.pd,    ImgGrp.pd,    dim);
    ReadImgUC(FileGrp.flair, ImgGrp.flair, dim);
    ReadImgUC(FileGrp.mask,  ImgGrp.mask,  dim);
    if (only_preventricle) ReadImgUC(FileGrp.ven_mask,  ImgGrp.ven_mask,  dim);
    break;

  case 1: // smooth original input images first, then read them
    ReadImgUC(FileGrp.t1,    ImgGrp.t1,    dim);
    ReadImgUC(FileGrp.t2,    ImgGrp.t2,    dim);
    ReadImgUC(FileGrp.pd,    ImgGrp.pd,    dim);
    ReadImgUC(FileGrp.flair, ImgGrp.flair, dim);
    ReadImgUC(FileGrp.mask,  ImgGrp.mask,  dim);
    if (only_preventricle) ReadImgUC(FileGrp.ven_mask,  ImgGrp.ven_mask,  dim);
    break;

  default:
    break;
  }

  fp = fopen(argv[6], "w");
  if (save_position == 1) fp_pos = fopen(position_file, "w");
  if (type == 0) { // lesion feature vector
    switch (method) {
    case 0: // no smooth, only voxel intensity at mask locations
      index = 0;
      for (k=0; k<dim.z; k++) 
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++)
	    if (only_preventricle == 0) {
	      if (ImgGrp.mask[k][i][j] == 255) {
		if (special_case == 0) fprintf(fp, "%f %f %f %f        -1\n", 
					       WeightGrp.t1*ImgGrp.t1[k][i][j], WeightGrp.t2*ImgGrp.t2[k][i][j], 
					       WeightGrp.pd*ImgGrp.pd[k][i][j], WeightGrp.flair*ImgGrp.flair[k][i][j]);
		if (special_case == 1) fprintf(fp, "%f %f %f %f        1\n", 
					       WeightGrp.t1*ImgGrp.t1[k][i][j], WeightGrp.t2*ImgGrp.t2[k][i][j], 
					       WeightGrp.pd*ImgGrp.pd[k][i][j], WeightGrp.flair*ImgGrp.flair[k][i][j]);
		index++;
	      }
	    } else {
	      if ((ImgGrp.mask[k][i][j] == 255) && (ImgGrp.ven_mask[k][i][j] != 0)) {
		if (special_case == 0) fprintf(fp, "%f %f %f %f        -1\n", 
					       WeightGrp.t1*ImgGrp.t1[k][i][j], 
					       WeightGrp.t2*ImgGrp.t2[k][i][j], 
					       WeightGrp.pd*ImgGrp.pd[k][i][j], 
					       WeightGrp.flair*ImgGrp.flair[k][i][j]);
		if (special_case == 1) fprintf(fp, "%f %f %f %f        1\n", 
					       WeightGrp.t1*ImgGrp.t1[k][i][j], 
					       WeightGrp.t2*ImgGrp.t2[k][i][j], 
					       WeightGrp.pd*ImgGrp.pd[k][i][j], 
					       WeightGrp.flair*ImgGrp.flair[k][i][j]);
		index++;
	      }
	    }
      printf("total lesion voxel number: %d\n", index);
      break;
      
    case 1: // already smoothed, get orthoginal direct neighborhood in addition to masked voxels
      index = 0;
      for (k=radius; k<dim.z-radius; k++)
	for (i=radius; i<dim.x-radius; i++)
	  for (j=radius; j<dim.y-radius; j++)

	    if (only_preventricle == 0) {
	      if (ImgGrp.mask[k][i][j] == 255) {
		
		for (ii=0; ii<pixelNumInBubble; ii++) {

		  tmp_pos.x = i + bubble[ii].x; tmp_pos.y = j + bubble[ii].y; tmp_pos.z = k + bubble[ii].z;
		  if ((tmp_pos.x>=0) && (tmp_pos.x<dim.x) && (tmp_pos.y>=0) && 
		      (tmp_pos.y<dim.y) && (tmp_pos.z>=0) && (tmp_pos.z<dim.z))
		    fprintf(fp, "%f %f %f %f ", WeightGrp.t1*ImgGrp.t1[tmp_pos.z][tmp_pos.x][tmp_pos.y], 
			    WeightGrp.t2*ImgGrp.t2[tmp_pos.z][tmp_pos.x][tmp_pos.y],
			    WeightGrp.pd*ImgGrp.pd[tmp_pos.z][tmp_pos.x][tmp_pos.y],
			    WeightGrp.flair*ImgGrp.flair[tmp_pos.z][tmp_pos.x][tmp_pos.y]);

		}

		if (special_case == 0) 
		  if (save_position == 0) fprintf(fp, "        -1\n");
		  else {fprintf(fp, "\n");fprintf(fp_pos, "%d %d %d\n", i, j, k);}
		if (special_case == 1) fprintf(fp, "        1\n");
		index++;
	      }
	    } else {
	      if ((ImgGrp.mask[k][i][j] == 255) && (ImgGrp.ven_mask[k][i][j] != 0)) {
		
		for (ii=0; ii<pixelNumInBubble; ii++) {

		  tmp_pos.x = i + bubble[ii].x;
		  tmp_pos.y = j + bubble[ii].y;
		  tmp_pos.z = k + bubble[ii].z;
		  if ((tmp_pos.x>=0) && (tmp_pos.x<dim.x) && (tmp_pos.y>=0) && 
		      (tmp_pos.y<dim.y) && (tmp_pos.z>=0) && (tmp_pos.z<dim.z))
		    fprintf(fp, "%f %f %f %f ", WeightGrp.t1*ImgGrp.t1[tmp_pos.z][tmp_pos.x][tmp_pos.y], 
			    WeightGrp.t2*ImgGrp.t2[tmp_pos.z][tmp_pos.x][tmp_pos.y],
			    WeightGrp.pd*ImgGrp.pd[tmp_pos.z][tmp_pos.x][tmp_pos.y],
			    WeightGrp.flair*ImgGrp.flair[tmp_pos.z][tmp_pos.x][tmp_pos.y]);

		}

		if (special_case == 0) fprintf(fp, "        -1\n");
		if (special_case == 1) fprintf(fp, "        1\n");
		if (save_position == 1) {fprintf(fp_pos, "%d %d %d\n", i, j, k);fprintf(fp, "\n");}
		index++;
	      }
	    }
      break;
      
    default:
      break;
    }
    printf("total lesion voxel number: %d\n", index);
  }

  if (type == 1) { // nonlesion feature vector

    pos = Ivector3dalloc1d(dim.x*dim.y*dim.z);
    index = 0;
    switch (method) {
    case 0:
      for (k=0; k<dim.z; k++) 
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) 
	    if (only_preventricle == 0) {
	      if ((ImgGrp.t1[k][i][j] != 0) && (ImgGrp.mask[k][i][j] != 255)) {
		pos[index].x = i; pos[index].y = j; pos[index].z = k;
		index++;
	      }
	    } else {
	      if ((ImgGrp.t1[k][i][j] != 0) && (ImgGrp.mask[k][i][j] != 255) && (ImgGrp.ven_mask[k][i][j] != 0)) {
		pos[index].x = i; pos[index].y = j; pos[index].z = k;
		index++;
	      }
	    }
      printf("final index: %d\n", index);
      
    case 1:
      for (k=0; k<dim.z; k++) 
	for (i=0; i<dim.x; i++)
	  for (j=0; j<dim.y; j++) {
	    tmp_pos.x = i;	    tmp_pos.y = j;	    tmp_pos.z = k;

	    if (only_preventricle == 0) {
	      if ((ImgGrp.mask[k][i][j] != 255) && 
		  (IsValidVoxel_Neighbor(ImgGrp.t1, dim, res, tmp_pos, bubble, pixelNumInBubble)) && 
		  (IsValidVoxel_Neighbor(ImgGrp.t2, dim, res, tmp_pos, bubble, pixelNumInBubble)) && 
		  (IsValidVoxel_Neighbor(ImgGrp.pd, dim, res, tmp_pos, bubble, pixelNumInBubble)) && 
		  (IsValidVoxel_Neighbor(ImgGrp.flair, dim, res, tmp_pos, bubble, pixelNumInBubble))) {
		pos[index].x = i; pos[index].y = j; pos[index].z = k;
		index++;
	      }
	    } else {
	      if ((ImgGrp.mask[k][i][j] != 255) && 
		  (IsValidVoxel_Neighbor(ImgGrp.t1, dim, res, tmp_pos, bubble, pixelNumInBubble)) && 
		  (IsValidVoxel_Neighbor(ImgGrp.t2, dim, res, tmp_pos, bubble, pixelNumInBubble)) && 
		  (IsValidVoxel_Neighbor(ImgGrp.pd, dim, res, tmp_pos, bubble, pixelNumInBubble)) && 
		  (IsValidVoxel_Neighbor(ImgGrp.flair, dim, res, tmp_pos, bubble, pixelNumInBubble)) 
		  && (ImgGrp.ven_mask[k][i][j] != 0)) {
		pos[index].x = i; pos[index].y = j; pos[index].z = k;
		index++;
	      }
	    }
	  }
      printf("final index: %d\n", index);
      break;

    default:
      break;
    }

    switch (method) {
    case 0:
      printf("ratio: ");
      for (i=0; i<selected_voxel_number; i++) {
	ratio = rand() / (double)RAND_MAX;
	printf("%f ", ratio);
	ind = (int)(ratio*index + 0.5);
	x = pos[ind].x;	y = pos[ind].y;	z = pos[ind].z;
	fprintf(fp, "%f %f %f %f        1\n", WeightGrp.t1*ImgGrp.t1[z][x][y], WeightGrp.t2*ImgGrp.t2[z][x][y], WeightGrp.pd*ImgGrp.pd[z][x][y], WeightGrp.flair*ImgGrp.flair[z][x][y]);
	if (save_nonlesion_mask == 1) ImgGrp.nonlesion_mask[z][x][y] = 255;
      }
      break;

    case 1:
      //      printf("ratio: ");
      for (s=0; s<selected_voxel_number; s++) {
	ratio = rand() / (double)RAND_MAX;
	//	printf("%f ", ratio);
	ind = (int)(ratio*index + 0.5);
	x = pos[ind].x;	y = pos[ind].y;	z = pos[ind].z;	
	for (i=0; i<pixelNumInBubble; i++) {
	  tmp_pos.x = x + bubble[i].x; tmp_pos.y = y + bubble[i].y; tmp_pos.z = z + bubble[i].z;
	  fprintf(fp, "%f %f %f %f ", WeightGrp.t1*ImgGrp.t1[tmp_pos.z][tmp_pos.x][tmp_pos.y], 
		  WeightGrp.t2*ImgGrp.t2[tmp_pos.z][tmp_pos.x][tmp_pos.y],
		  WeightGrp.pd*ImgGrp.pd[tmp_pos.z][tmp_pos.x][tmp_pos.y], 
		  WeightGrp.flair*ImgGrp.flair[tmp_pos.z][tmp_pos.x][tmp_pos.y]);
	}
	
	fprintf(fp, "        1\n");
	if (save_nonlesion_mask == 1) ImgGrp.nonlesion_mask[z][x][y] = 255;
      }
      break;
      
    default:
      break;
    }
    free(pos);
  }
  fclose(fp);
  if (save_position == 1) fclose(fp_pos);
  
  if (save_nonlesion_mask) WriteImgUC(FileGrp.nonlesion_mask, ImgGrp.nonlesion_mask, dim);

  UCfree3d(ImgGrp.t1,    dim.z, dim.x);
  UCfree3d(ImgGrp.t2,    dim.z, dim.x);
  UCfree3d(ImgGrp.pd,    dim.z, dim.x);
  UCfree3d(ImgGrp.flair, dim.z, dim.x);
  UCfree3d(ImgGrp.mask,  dim.z, dim.x);
  free(bubble);
  if (save_nonlesion_mask) UCfree3d(ImgGrp.nonlesion_mask,  dim.z, dim.x);

  return 0;
}

int IsValidVoxel(unsigned char ***img, Ivector3d dim, Ivector3d pos)
{
  int        i, j, k, count;

  count = 0;
  for (k=pos.z-1; k<=pos.z+1; k++)
    for (i=pos.x-1; i<=pos.x+1; i++)
      for (j=pos.y-1; j<=pos.y+1; j++)
	if ((i>=0) && (i<dim.x) && (j>=0) && (j<dim.y) && (k>=0) && (k<dim.z) && (img[k][i][j] != 0)) count++;

  if (count < 27) return 0;

  return 1;
}



int IsValidVoxel_Neighbor(unsigned char ***img, Ivector3d dim, Fvector3d res, Ivector3d pos, Ivector3d *bubble, int pixelNumInBubble)
{
  int        i, j, k, count, x, y, z, total;
  Ivector3d  tmp_pos;

  for (i=0; i<pixelNumInBubble; i++) {
    tmp_pos.x = pos.x + bubble[i].x;    tmp_pos.y = pos.y + bubble[i].y;    tmp_pos.z = pos.z + bubble[i].z;

    if ((tmp_pos.x<0) || (tmp_pos.y<0) || (tmp_pos.z<0) || 
	(tmp_pos.x>=dim.x) || (tmp_pos.y>=dim.y) || (tmp_pos.z>=dim.z) ||
	(img[tmp_pos.z][tmp_pos.x][tmp_pos.y] == 0)) return 0;
  }

  return 1;
}



//int IsValidVoxel_Neighbor(unsigned char ***img, Ivector3d dim, Ivector3d pos, int radius)
//{
//  int        i, j, k, count, thre;

//  count = 0;
//  for (k=pos.z-radius; k<=pos.z+radius; k++)
//    for (i=pos.x-radius; i<=pos.x+radius; i++)
//      for (j=pos.y-radius; j<=pos.y+radius; j++)
//	if ((i>=0) && (i<dim.x) && (j>=0) && (j<dim.y) && (k>=0) && (k<dim.z) && (img[k][i][j] != 0)) count++;

//  thre = (2*radius+1)*(2*radius+1)*(2*radius+1);
//  if (count < thre) return 0;

//  return 1;
//}


void ReadFeatureVector(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim)
{
  int                   i, j, k;
  unsigned short        ***img;

  img = USalloc3d(dim.x, dim.y, dim.z);
  ReadImg(filename, img, dim);
  for (i=0; i<vec_len; i++) vec[i] = (float)img[pos[i].z][pos[i].x][pos[i].y];

  USfree3d(img, dim.z, dim.x);
}

void ReadFeatureVector_FLOAT(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim)
{
  int          i, j, k;
  float        ***img;

  img = Falloc3d(dim.x, dim.y, dim.z);
  ReadImg_FLOAT(filename, img, dim);
  for (i=0; i<vec_len; i++) {
    vec[i] = img[pos[i].z][pos[i].x][pos[i].y];
  }

  Ffree3d(img, dim.z, dim.x);
}

void ReadImg(char filename[80], unsigned short ***vol, Ivector3d dim)
{
   FILE   *fp;
   int    i, j, k;

   fp = fopen(filename, "r");
   for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
         fread(vol[k][i], 2, dim.y, fp);
   fclose(fp);
}

void ReadImgUC(char filename[80], unsigned char ***vol, Ivector3d dim)
{
   FILE   *fp;
   int    i, j, k;

   fp = fopen(filename, "r");
   for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
         fread(vol[k][i], 1, dim.y, fp);
   fclose(fp);
}


void ReadImg_FLOAT(char filename[80], float ***vol, Ivector3d dim)
{
   FILE   *fp;
   int    i, j, k;

   fp = fopen(filename, "r");
   for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
         fread(vol[k][i], sizeof(float), dim.y, fp);
   fclose(fp);
}

void NormalizeVector(float *vec, int vec_len)
{
  int        i;
  float      sum;

  sum = 0.0;
  for (i=0; i<vec_len; i++) sum += vec[i]*vec[i];
  sum = sqrt(sum);
  for (i=0; i<vec_len; i++) vec[i] /= sum;
}

int GetLineNumber(char filename[100])
{
  FILE        *fp;
  char        ch;
  int         count;

  count = 0;
  fp = fopen(filename, "r");
  while (feof(fp) == 0) {
    ch = fgetc(fp);
    /*    printf("%c", ch); */
    if (ch == '\n') count++;
  }
  fclose(fp);

  return count;
}

void PrintVector(float *vec, int vec_len)
{
  int        i;
  
  for (i=0; i<vec_len; i++) printf("%f ", vec[i]);
  printf("\n");
}

void WriteImgUC(char filename[80], unsigned char ***data, Ivector3d dim)
{
  FILE  *fp;
  int   i, k ;

  fp=fopen(filename,"w");
  for(k=0;k<dim.z;k++)
    for(i=0;i<dim.x;i++) fwrite(data[k][i],1,dim.y,fp);
  fclose(fp);
}

void calculate_hood_special(Fvector3d *bubble,int bubble_size, int *pixelNumInBubble, Fvector3d res)
{
  int i,j,k,t,rad_sqrd,radius;

  t=0;
  radius=bubble_size+BASIC_R;
  rad_sqrd=radius*radius;
  for(i=-radius;i<=radius;i++)
    for(j=-radius;j<=radius;j++)
      for(k=-radius;k<=radius;k++)
	if((i*i+j*j+k*k)<=rad_sqrd)
	  {
	    bubble[t].x=((float) i)/res.x;
	    bubble[t].y=((float) j)/res.y;
	    bubble[t].z=((float) k)/res.z;
	    t++;
	  }
  (*pixelNumInBubble) = t;
}

void calculate_hood_special_improve(Ivector3d *bubble,int bubble_size, int *pixelNumInBubble, Fvector3d res)
{
  int         i,j,k,t,rad_sqrd,radius, count;
  Fvector3d   *bubble_float;

  bubble_float = Fvector3dalloc1d(10000);
  t=0;
  radius=bubble_size+BASIC_R;
  rad_sqrd=radius*radius;
  for(i=-radius;i<=radius;i++)
    for(j=-radius;j<=radius;j++)
      for(k=-radius;k<=radius;k++)
	if((i*i+j*j+k*k)<=rad_sqrd)
	  {
	    bubble_float[t].x=((float) i)/res.x;
	    bubble_float[t].y=((float) j)/res.y;
	    bubble_float[t].z=((float) k)/res.z;
	    t++;
	  }

  printf("radius: %d\n", radius);
  printf("pixelNumInBubble: %d\n", t);
  for (i=0; i<t; i++) printf("%d: (%f, %f, %f)\n", i, bubble_float[i].x, bubble_float[i].y, bubble_float[i].z);

  count = 0;
  bubble[count].x = (int)bubble_float[0].x;
  bubble[count].y = (int)bubble_float[0].y;
  bubble[count].z = (int)bubble_float[0].z;

  for (i=0; i<t; i++) {
    if (!IsPtInLst(bubble_float[i], bubble, count)) {
      bubble[count].x = (int)bubble_float[i].x;
      bubble[count].y = (int)bubble_float[i].y;
      bubble[count].z = (int)bubble_float[i].z;
      count++;
    }
  }

  printf("pixelNumInBubble: %d\n", count);
  for (i=0; i<count; i++) printf("%d: (%d, %d, %d)\n", i, bubble[i].x, bubble[i].y, bubble[i].z);

  free(bubble_float);
  (*pixelNumInBubble) = count;
}

int IsPtInLst(Fvector3d pt, Ivector3d *lst, int lst_len)
{
  int        i;
  Ivector3d  tmp_pt;

  tmp_pt.x = (int)pt.x;
  tmp_pt.y = (int)pt.y;
  tmp_pt.z = (int)pt.z;

  for (i=0; i<lst_len; i++)
    if ((tmp_pt.x == lst[i].x) && (tmp_pt.y == lst[i].y) && (tmp_pt.z == lst[i].z)) return 1;


  return 0;
}
