#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <math.h>
#include <matrixSHEN.h>
#include <mvcd.h>

#define BASIC_R 0

typedef struct {
  Ivector3d        pos;
  float            value;
} VOXEL;

typedef struct {
  char        t1[200], t2[200], pd[200], flair[200], mask[200];
} FILEGROUP;

typedef struct {
  unsigned char        ***t1, ***t2, ***pd, ***flair, ***mask;
} VOLUMEGROUP;

typedef struct {
  float        t1, t2, pd, flair;
} WEIGHTGROUP;

int         GetLineNumber(char filename[100]);
void        show_usage(int argc, char **argv);
void        NormalizeVector(float *vec, int vec_len);
void        ReadImg(char filename[80], unsigned short ***vol, Ivector3d dim);
void        ReadImgUC(char filename[80], unsigned char ***vol, Ivector3d dim);
void        WriteImgUC(char filename[80], unsigned char ***data, Ivector3d dim);
void        WriteImgF(char filename[80], float ***data, Ivector3d dim);
void        ReadImg_FLOAT(char filename[80], float ***vol, Ivector3d dim);
void        ReadFeatureVector(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim);
void        ReadFeatureVector_FLOAT(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim);
void        PrintVector(float *vec, int vec_len);
int         IsValidVoxel(unsigned char ***img, Ivector3d dim, Ivector3d pos);
int         IsValidVoxel_flair(unsigned char ***img, Ivector3d dim, Ivector3d pos, int threshold);
int         GetImgThreshold(unsigned char ***img, Ivector3d dim, float intensity_ratio_threshold);
int         IsValidVoxel_Neighbor(unsigned char ***img, Ivector3d dim, Ivector3d pos, Ivector3d *bubble, int pixelNumInBubble);
void        calculate_hood_special(Fvector3d *bubble,int bubble_size, int *pixelNumInBubble, Fvector3d res);
void        calculate_hood_special_improve(Ivector3d *bubble,int bubble_size, int *pixelNumInBubble, Fvector3d res);
int         IsPtInLst(Fvector3d pt, Ivector3d *lst, int lst_len);

void show_usage(int argc, char **argv)
{
   printf("USAGE: %s t1.img t2.img pd.img flair.img svm_model lesion_segmented.img\n\
\t -d <int>,<int>        : image dimension in XY (default: 256,256)\n\
\t -v <float>,<float>,<float>: image resolution (default: 0.9375,0.9375,3.0)\n\
\t -m <int>              : methods for getting feature vectors (default: 0)\n\
\t                         0: voxel-wise intensity\n\
\t                         1: voxel-wise direct neighborhood over smoothed volume\n\
\t -M <string>           : file name of mask for testing\n\
\t -r <int>              : neighbor size (default: 1)\n\
\t -w <float>,<float>,<float>,<float>   : weighting parameters for different modalities, t1, t2, pd, flair respectively (default: 1.0,1.0,1.0,1.0)\n\
", argv[0]);
   exit(1);
}

int main(int argc, char **argv)
{
  FILE          *fp;
  int           i, j, k, ii, jj, kk, c, vec_len, index, x, y, z, ind, pixelNumInBubble;
  int           total_voxel_number, method, threshold_flair, threshold_volume, radius;
  int           have_mask;
  Ivector3d     dim, tmp_pos, *bubble;
  Fvector3d     res;
  extern char   *optarg;
  FILEGROUP     InputFile, InputFileSmooth;
  VOLUMEGROUP   InputImg;
  unsigned char ***img_mask;
  char          *sub, cmd[1000];
  float         *vec, ratio, tmp, smooth_factor, function_threshold, intensity_ratio_threshold, ***img_result;
  VOXEL         *voxel_list;
  WEIGHTGROUP   WeightGrp;
 
  /* read in arguments */
  if (argc < 6) show_usage(argc, argv);
  
  dim.x = dim.y = 256; dim.z = 124;
  res.x = res.y = 0.9375; res.z = 3.0;

  method = 0;
  smooth_factor = 0.5;
  function_threshold = 0.0;
  intensity_ratio_threshold = 0.99;
  have_mask = 0;
  radius = 1;
  while((c=getopt(argc-6,argv+6,"d:m:w:M:r:v:")) != -1) {
    switch(c) {
    case 'd':
      sscanf(optarg, "%d,%d", &dim.x, &dim.y);
      break;

    case 'v':
      sscanf(optarg, "%f,%f,%f", &res.x, &res.y, &res.z);
      break;

    case 'm':
      method = atoi(optarg);
      break;

    case 'w':
      sscanf(optarg, "%f,%f,%f,%f", &WeightGrp.t1, &WeightGrp.t2, &WeightGrp.pd, &WeightGrp.flair);
      break;

    case 'M':
      have_mask = 1;
      sscanf(optarg, "%s", InputFile.mask);
      break;

    case 'r':
      radius = atoi(optarg);
      break;
      
   default:
      break;
    }
  }

  strcpy(InputFile.t1,    argv[1]);
  strcpy(InputFile.t2,    argv[2]);
  strcpy(InputFile.pd,    argv[3]);
  strcpy(InputFile.flair, argv[4]);

  fp = fopen(InputFile.t1,"r");
  fseek(fp,0,SEEK_END);
  dim.z = (ftell(fp))/(dim.x*dim.y);
  fclose(fp);
  printf("image dimension: (%d, %d, %d)\n", dim.x, dim.y, dim.z);
  printf("image resolution: (%f, %f, %f)\n", res.x, res.y, res.z);
  printf("method: %d\n", method);
  printf("weighting: (%f, %f, %f, %f)\n", WeightGrp.t1, WeightGrp.t2, WeightGrp.pd, WeightGrp.flair);
  printf("neighbor size: %d\n", radius);
  if (have_mask == 1) printf("mask file: %s\n", InputFile.mask);

  bubble = Ivector3dalloc1d(10000);
  calculate_hood_special_improve(bubble, radius, &pixelNumInBubble, res);
  printf("after hood calculation, pixelNumInBubble: %d\n", pixelNumInBubble);
  for (i=0; i<pixelNumInBubble; i++) printf("%d: (%d, %d, %d)\n", i, bubble[i].x, bubble[i].y, bubble[i].z);

  InputImg.t1    = UCalloc3d(dim.x, dim.y, dim.z);
  InputImg.t2    = UCalloc3d(dim.x, dim.y, dim.z);
  InputImg.pd    = UCalloc3d(dim.x, dim.y, dim.z);
  InputImg.flair = UCalloc3d(dim.x, dim.y, dim.z);
  if (have_mask == 1) InputImg.mask = UCalloc3d(dim.x, dim.y, dim.z);
  printf("allocate ok\n");

  switch (method) {
  case 0: //voxel-wise intensity
    ReadImgUC(InputFile.t1,    InputImg.t1,    dim);
    ReadImgUC(InputFile.t2,    InputImg.t2,    dim);
    ReadImgUC(InputFile.pd,    InputImg.pd,    dim);
    ReadImgUC(InputFile.flair, InputImg.flair, dim);
    if (have_mask == 1) ReadImgUC(InputFile.mask, InputImg.mask, dim);
  break;

  case 1: // voxel-wise neighbor
    printf("enter 1\n");
    sprintf(InputFileSmooth.t1,    "%s", InputFile.t1);
    sprintf(InputFileSmooth.t2,    "%s", InputFile.t2);
    sprintf(InputFileSmooth.pd,    "%s", InputFile.pd);
    sprintf(InputFileSmooth.flair, "%s", InputFile.flair);
    if (have_mask == 1) sprintf(InputFileSmooth.mask, "%s", InputFile.mask);
    
    ReadImgUC(InputFileSmooth.t1,    InputImg.t1,    dim);
    ReadImgUC(InputFileSmooth.t2,    InputImg.t2,    dim);
    ReadImgUC(InputFileSmooth.pd,    InputImg.pd,    dim);
    ReadImgUC(InputFileSmooth.flair, InputImg.flair, dim);
    if (have_mask == 1) ReadImgUC(InputFileSmooth.mask, InputImg.mask, dim);
    printf("%s\n %s\n %s\n %s\n read ok\n", InputFileSmooth.t1, InputFileSmooth.t2, InputFileSmooth.pd, InputFileSmooth.flair);
    if (have_mask == 1) printf("%s read ok\n", InputFileSmooth.mask);
    printf("image read ok\n");
    break;

  default:
    break;
  }

  total_voxel_number = 0;
  switch (method) {
  case 0: // voxel-wise intensity
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) 
	  if (have_mask == 1) {
	    if ((InputImg.t1[k][i][j] != 0) && (InputImg.mask[k][i][j] != 0)) total_voxel_number++;
	  } else {
	    if (InputImg.t1[k][i][j] != 0) total_voxel_number++;
	  }
    
    voxel_list = (VOXEL *)malloc(total_voxel_number*sizeof(VOXEL));
    index = 0;
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) 
	  if (have_mask == 1) {
	    if ((InputImg.t1[k][i][j] != 0) && (InputImg.mask[k][i][j] != 0)) {
	      voxel_list[index].pos.x = i;
	      voxel_list[index].pos.y = j;
	      voxel_list[index].pos.z = k;
	      voxel_list[index].value = 0;
	      index++;
	    }
	  } else {
	    if (InputImg.t1[k][i][j] != 0) {
	      voxel_list[index].pos.x = i;
	      voxel_list[index].pos.y = j;
	      voxel_list[index].pos.z = k;
	      voxel_list[index].value = 0;
	      index++;
	    }
	  }
    break;

  case 1: // voxel-wise neighbor
    printf("enter 1 again\n");
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) {
	  tmp_pos.x = i;      tmp_pos.y = j;      tmp_pos.z = k;
	  if (have_mask == 1) {

	    if (IsValidVoxel_Neighbor(InputImg.t1, dim, tmp_pos, bubble, pixelNumInBubble) && 
		IsValidVoxel_Neighbor(InputImg.t2, dim, tmp_pos, bubble, pixelNumInBubble) && 
		IsValidVoxel_Neighbor(InputImg.pd, dim, tmp_pos, bubble, pixelNumInBubble) && 
		IsValidVoxel_Neighbor(InputImg.flair, dim, tmp_pos, bubble, pixelNumInBubble) && 
		(InputImg.mask[k][i][j] != 0)) total_voxel_number++;
	  } else {
	    if (IsValidVoxel_Neighbor(InputImg.t1, dim, tmp_pos, bubble, pixelNumInBubble) && 
		IsValidVoxel_Neighbor(InputImg.t2, dim, tmp_pos, bubble, pixelNumInBubble) && 
		IsValidVoxel_Neighbor(InputImg.pd, dim, tmp_pos, bubble, pixelNumInBubble) && 
		IsValidVoxel_Neighbor(InputImg.flair, dim, tmp_pos, bubble, pixelNumInBubble)) 
	      total_voxel_number++;
	  }
	}
    printf("total_voxel_number: %d\n", total_voxel_number);

    voxel_list = (VOXEL *)malloc(total_voxel_number*sizeof(VOXEL));
    index = 0;
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++) {
	  tmp_pos.x = i;      tmp_pos.y = j;      tmp_pos.z = k;

	  if (have_mask == 1) {
	    if (IsValidVoxel_Neighbor(InputImg.t1, dim, tmp_pos, bubble, pixelNumInBubble) &&
		IsValidVoxel_Neighbor(InputImg.t2, dim, tmp_pos, bubble, pixelNumInBubble) &&
		IsValidVoxel_Neighbor(InputImg.pd, dim, tmp_pos, bubble, pixelNumInBubble) &&
		IsValidVoxel_Neighbor(InputImg.flair, dim, tmp_pos, bubble, pixelNumInBubble) &&
		(InputImg.mask[k][i][j] != 0)) {
	      voxel_list[index].pos.x = tmp_pos.x;   
	      voxel_list[index].pos.y = tmp_pos.y;    
	      voxel_list[index].pos.z = tmp_pos.z;
	      voxel_list[index].value = -1.0;
	      index++;
	    }
	  } else {
	    if (IsValidVoxel_Neighbor(InputImg.t1, dim, tmp_pos, bubble, pixelNumInBubble) && 
		IsValidVoxel_Neighbor(InputImg.t2, dim, tmp_pos, bubble, pixelNumInBubble) && 
		IsValidVoxel_Neighbor(InputImg.pd, dim, tmp_pos, bubble, pixelNumInBubble) && 
		IsValidVoxel_Neighbor(InputImg.flair, dim, tmp_pos, bubble, pixelNumInBubble)) {
	      voxel_list[index].pos.x = tmp_pos.x;   
	      voxel_list[index].pos.y = tmp_pos.y;    
	      voxel_list[index].pos.z = tmp_pos.z;
	      voxel_list[index].value = -1.0;
	      index++;
	    }
	  }
	}
    printf("index value: %d\n", index);
    if (index != total_voxel_number) {
      printf("valid voxel number doesn't match\n");
      exit(2);
    }
    break;

  default:
    break;
  }
  
  fp = fopen("test.example", "w");
  switch (method) {
  case 0: // voxel-wise intensity
    fprintf(fp, "%d 4\n", total_voxel_number);
    for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
	for (j=0; j<dim.y; j++)
	  if (InputImg.t1[k][i][j] != 0)
	    fprintf(fp, "%f %f %f %f\n", INITIAL_T1_WEIGHT*WeightGrp.t1*InputImg.t1[k][i][j], INITIAL_T2_WEIGHT*WeightGrp.t2*InputImg.t2[k][i][j], 
		    INITIAL_PD_WEIGHT*WeightGrp.pd*InputImg.pd[k][i][j], INITIAL_FL_WEIGHT*WeightGrp.flair*InputImg.flair[k][i][j]);
    break;

  case 1: // voxel-wise neighbor
    fprintf(fp, "%d %d\n", total_voxel_number, pixelNumInBubble*4);
    printf("%d %d\n", total_voxel_number, pixelNumInBubble*4);

    for (i=0; i<total_voxel_number; i++) {
      for (j=0; j<pixelNumInBubble; j++) {
	
	tmp_pos.x = voxel_list[i].pos.x + bubble[j].x;
	tmp_pos.y = voxel_list[i].pos.y + bubble[j].y;
	tmp_pos.z = voxel_list[i].pos.z + bubble[j].z;
	if ((tmp_pos.x>=0) & (tmp_pos.y>=0) & (tmp_pos.z>=0) & (tmp_pos.x<dim.x) & (tmp_pos.y<dim.y) & (tmp_pos.z<dim.z))
	  fprintf(fp, "%f %f %f %f ", WeightGrp.t1*InputImg.t1[tmp_pos.z][tmp_pos.x][tmp_pos.y],
		  WeightGrp.t2*InputImg.t2[tmp_pos.z][tmp_pos.x][tmp_pos.y],
		  WeightGrp.pd*InputImg.pd[tmp_pos.z][tmp_pos.x][tmp_pos.y],
		  WeightGrp.flair*InputImg.flair[tmp_pos.z][tmp_pos.x][tmp_pos.y]);
      }
      fprintf(fp, "\n");
    }
    break;

  default:
    break;
  }
  fclose(fp);
  printf("Before SVMTest\n");

  img_result = Falloc3d(dim.x, dim.y, dim.z);
  //printf("svm_test -ae -no -oa test.result %s test.example", argv[5]);
  //sprintf(cmd, "svm_test  -ae -no -oa test.result %s test.example", argv[5]);
  sprintf(cmd, "SVMTest -no -oa test.result %s test.example", argv[5]);
  system(cmd);
  fp = fopen("test.result", "r");
  for (i=0; i<total_voxel_number; i++) {
    fscanf(fp, "%f", &tmp);
    img_result[voxel_list[i].pos.z][voxel_list[i].pos.x][voxel_list[i].pos.y] = tmp;
  }
  fclose(fp);
  printf("voxel_list created\n");
  printf("finished creating mask\n");

  WriteImgF(argv[6], img_result, dim);

  UCfree3d(InputImg.t1,    dim.z, dim.x);
  UCfree3d(InputImg.t2,    dim.z, dim.x);
  UCfree3d(InputImg.pd,    dim.z, dim.x);
  UCfree3d(InputImg.flair, dim.z, dim.x);
  Ffree3d(img_result,     dim.z, dim.x);
  free(voxel_list);
  free(bubble);

  return 0;
}


int IsValidVoxel_Neighbor(unsigned char ***img, Ivector3d dim, Ivector3d pos, Ivector3d *bubble, int pixelNumInBubble)
{
  int        i, j, k, count, x, y, z, total;
  Ivector3d  tmp_pos;

  //  for (i=0; i<pixelNumInBubble; i++) printf("%d: (%d, %d, %d)\n", i, bubble[i].x, bubble[i].y, bubble[i].z);
  //  exit(1);

  for (i=0; i<pixelNumInBubble; i++) {
    tmp_pos.x = pos.x + bubble[i].x;
    tmp_pos.y = pos.y + bubble[i].y;
    tmp_pos.z = pos.z + bubble[i].z;
    if ((tmp_pos.x<0) || (tmp_pos.y<0) || (tmp_pos.z<0) || 
	(tmp_pos.x>=dim.x) || (tmp_pos.y>=dim.y) || (tmp_pos.z>=dim.z) || 
	(img[tmp_pos.z][tmp_pos.x][tmp_pos.y] == 0)) return 0;

    //    if ((tmp_pos.x>=0) && (tmp_pos.y>=0) && (tmp_pos.z>=0) && 
    //	(tmp_pos.x<dim.x) && (tmp_pos.y<dim.y) && (tmp_pos.z<dim.z)) {
    //      if (img[tmp_pos.z][tmp_pos.x][tmp_pos.y] != 0) count++;
    //      total++;
    //    }
  }

  //  if (count < total) return 0;

  return 1;
}

int IsValidVoxel(unsigned char ***img, Ivector3d dim, Ivector3d pos)
{
  int        i, j, k, count;

  count = 0;
  for (k=pos.z-1; k<=pos.z+1; k++)
    for (i=pos.x-1; i<=pos.x+1; i++)
      for (j=pos.y-1; j<=pos.y+1; j++)
	if ((i>=0) && (i<dim.x) && (j>=0) && (j<dim.y) && (k>=0) && (k<dim.z) && (img[k][i][j] != 0)) count++;

  if (count < 27) return 0;

  return 1;
}

int IsValidVoxel_flair(unsigned char ***img, Ivector3d dim, Ivector3d pos, int threshold)
{
  int        i, j, k, count;

  if (img[pos.z][pos.x][pos.y] < threshold) return 0;

  count = 0;
  for (k=pos.z-1; k<=pos.z+1; k++)
    for (i=pos.x-1; i<=pos.x+1; i++)
      for (j=pos.y-1; j<=pos.y+1; j++)
	if ((i>=0) && (i<dim.x) && (j>=0) && (j<dim.y) && (k>=0) && (k<dim.z) && (img[k][i][j] != 0)) count++;

  if (count < 27) return 0;

  return 1;
}


void ReadFeatureVector(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim)
{
  int                   i, j, k;
  unsigned short        ***img;

  img = USalloc3d(dim.x, dim.y, dim.z);
  ReadImg(filename, img, dim);
  for (i=0; i<vec_len; i++) vec[i] = (float)img[pos[i].z][pos[i].x][pos[i].y];
  /*  NormalizeVector(vec, vec_len);*/
  /*PrintVector(vec, vec_len);*/

  USfree3d(img, dim.z, dim.x);
}

void ReadFeatureVector_FLOAT(char filename[200], float *vec, int vec_len, Ivector3d *pos, Ivector3d dim)
{
  int          i, j, k;
  float        ***img;

  /*  printf("enter ReadFeatureVector_FLOAT\n"); */
  img = Falloc3d(dim.x, dim.y, dim.z);
  ReadImg_FLOAT(filename, img, dim);
  for (i=0; i<vec_len; i++) {
    vec[i] = img[pos[i].z][pos[i].x][pos[i].y];
    /*    printf("%f ", vec[i]); */
  }
  /*  printf("\n"); */
  /*  NormalizeVector(vec, vec_len);*/
  /*PrintVector(vec, vec_len);*/

  Ffree3d(img, dim.z, dim.x);
}

void ReadImg(char filename[80], unsigned short ***vol, Ivector3d dim)
{
   FILE   *fp;
   int    i, j, k;

   fp = fopen(filename, "r");
   for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
         fread(vol[k][i], 2, dim.y, fp);
   fclose(fp);
}

void ReadImgUC(char filename[80], unsigned char ***vol, Ivector3d dim)
{
   FILE   *fp;
   int    i, j, k;

   fp = fopen(filename, "r");
   for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
         fread(vol[k][i], 1, dim.y, fp);
   fclose(fp);
}

void WriteImgUC(char filename[80], unsigned char ***data, Ivector3d dim)
{
  FILE  *fp;
  int   i, k ;

  fp=fopen(filename,"w");
  for(k=0;k<dim.z;k++)
    for(i=0;i<dim.x;i++) fwrite(data[k][i],1,dim.y,fp);
  fclose(fp);
}

void WriteImgF(char filename[80], float ***data, Ivector3d dim)
{
  FILE  *fp;
  int   i, k ;

  fp=fopen(filename,"w");
  for(k=0;k<dim.z;k++)
    for(i=0;i<dim.x;i++) fwrite(data[k][i],sizeof(float),dim.y,fp);
  fclose(fp);
}

void ReadImg_FLOAT(char filename[80], float ***vol, Ivector3d dim)
{
   FILE   *fp;
   int    i, j, k;

   fp = fopen(filename, "r");
   for (k=0; k<dim.z; k++)
      for (i=0; i<dim.x; i++)
         fread(vol[k][i], sizeof(float), dim.y, fp);
   fclose(fp);
}

void NormalizeVector(float *vec, int vec_len)
{
  int        i;
  float      sum;

  /*PrintVector(vec, vec_len);*/
  sum = 0.0;
  for (i=0; i<vec_len; i++) sum += vec[i]*vec[i];
  sum = sqrt(sum);
  for (i=0; i<vec_len; i++) vec[i] /= sum;
}

int GetLineNumber(char filename[100])
{
  FILE        *fp;
  char        ch;
  int         count;

  count = 0;
  fp = fopen(filename, "r");
  while (feof(fp) == 0) {
    ch = fgetc(fp);
    /*    printf("%c", ch); */
    if (ch == '\n') count++;
  }
  fclose(fp);

  return count;
}

void PrintVector(float *vec, int vec_len)
{
  int        i;
  
  for (i=0; i<vec_len; i++) printf("%f ", vec[i]);
  printf("\n");
}


int GetImgThreshold(unsigned char ***img, Ivector3d dim, float intensity_ratio_threshold)
{
  int        i, j, k, histo[256], sum, total_volume, threshold;
  float      ratio, ratio_threshold;

  for (i=0; i<256; i++) histo[i] = 0;
  for  (k=0; k<dim.z; k++)
    for (i=0; i<dim.x; i++)
      for (j=0; j<dim.y; j++) histo[img[k][i][j]]++;
  histo[0] = 0;

  for (i=0; i<256; i++) printf("%d ", histo[i]);
  printf("\n");

  total_volume = 0;
  for (i=0; i<256; i++) total_volume += histo[i];
  printf("total_volume: %d\n", total_volume);

  ratio_threshold = 0.99;
  sum = 0; threshold = 0;
  for (i=0; i<256; i++) {
    sum += histo[i];
    ratio = (float)sum / total_volume;
    printf("ratio: %f\n", ratio);
    //    if (ratio > ratio_threshold) {
    if (ratio > intensity_ratio_threshold) {
      threshold = i;
      break;
    }
  }

  return threshold;
}

void calculate_hood_special(Fvector3d *bubble,int bubble_size, int *pixelNumInBubble, Fvector3d res)
{
  int         i,j,k,t,rad_sqrd,radius;

  t=0;
  printf("res: (%f, %f, %f)\n", res.x, res.y, res.z);
  radius=bubble_size+BASIC_R;
  rad_sqrd=radius*radius;
  for(i=-radius;i<=radius;i++)
    for(j=-radius;j<=radius;j++)
      for(k=-radius;k<=radius;k++)
	if((i*i+j*j+k*k)<=rad_sqrd)
	  {
	    bubble[t].x=((float) i)/res.x;
	    bubble[t].y=((float) j)/res.y;
	    bubble[t].z=((float) k)/res.z;
	    t++;
	  }

  printf("radius: %d\n", radius);
  printf("pixelNumInBubble: %d\n", t);
  for (i=0; i<t; i++) printf("%d: (%f, %f, %f)\n", i, bubble[i].x, bubble[i].y, bubble[i].z);

  (*pixelNumInBubble) = t;
}

void calculate_hood_special_improve(Ivector3d *bubble,int bubble_size, int *pixelNumInBubble, Fvector3d res)
{
  int         i,j,k,t,rad_sqrd,radius, count;
  Fvector3d   *bubble_float;

  bubble_float = Fvector3dalloc1d(10000);
  t=0;
  radius=bubble_size+BASIC_R;
  rad_sqrd=radius*radius;
  for(i=-radius;i<=radius;i++)
    for(j=-radius;j<=radius;j++)
      for(k=-radius;k<=radius;k++)
	if((i*i+j*j+k*k)<=rad_sqrd)
	  {
	    bubble_float[t].x=((float) i)/res.x;
	    bubble_float[t].y=((float) j)/res.y;
	    bubble_float[t].z=((float) k)/res.z;
	    t++;
	  }

  printf("radius: %d\n", radius);
  printf("pixelNumInBubble: %d\n", t);
  for (i=0; i<t; i++) printf("%d: (%f, %f, %f)\n", i, bubble_float[i].x, bubble_float[i].y, bubble_float[i].z);

  count = 0;
  bubble[count].x = (int)bubble_float[0].x;
  bubble[count].y = (int)bubble_float[0].y;
  bubble[count].z = (int)bubble_float[0].z;

  for (i=0; i<t; i++) {
    if (!IsPtInLst(bubble_float[i], bubble, count)) {
      bubble[count].x = (int)bubble_float[i].x;
      bubble[count].y = (int)bubble_float[i].y;
      bubble[count].z = (int)bubble_float[i].z;
      count++;
    }
  }

  printf("pixelNumInBubble: %d\n", count);
  for (i=0; i<count; i++) printf("%d: (%d, %d, %d)\n", i, bubble[i].x, bubble[i].y, bubble[i].z);

  free(bubble_float);
  (*pixelNumInBubble) = count;
}

int IsPtInLst(Fvector3d pt, Ivector3d *lst, int lst_len)
{
  int        i;
  Ivector3d  tmp_pt;

  tmp_pt.x = (int)pt.x;
  tmp_pt.y = (int)pt.y;
  tmp_pt.z = (int)pt.z;

  for (i=0; i<lst_len; i++)
    if ((tmp_pt.x == lst[i].x) && (tmp_pt.y == lst[i].y) && (tmp_pt.z == lst[i].z)) return 1;


  return 0;
}
