function [template,rank] = wfu_pick_template(dirName)

% Selects appropriate template based on the criterion 
% listed below
% FORMAT [template,rank] = wfu_pick_template(dirName)
%
%   template    - image file (analyze format) selected as 
%                 template for spm2 normalization
%   rank        - structure of images ranked for best norm
%                 template
%   dirName     - directory containing possible templates
%___________________________________________________________

if nargin<1, return; end

% search for .info files
options.displayFiles = 0;
options.displayDirectories = 0; 
infoFiles = wfu_find_files('.info$',dirName,0,options);
vecLength = length(infoFiles);

if vecLength == 0, template = []; return 
else
    for i = 1:vecLength
        infoStruct(i)   = wfu_read_textfile(infoFiles{i});
        %vectors: 
        vecTR(i)        = infoStruct(i).RepetitionTime;
        vecTE(i)        = infoStruct(i).EchoTime; 
        vecSlice(i)     = infoStruct(i).NumberOfSlices;
        vecAngle(i)     = infoStruct(i).FlipAngle;
        vecName{i}      = infoStruct(i).PulseSeqName;
        vecPlane{i}     = infoStruct(i).PlaneType; 
        vecSeriesDesc{i} = infoStruct(i).SeriesDescription; 
        %structures: 
        [path,name]             = fileparts(infoFiles{i});
        rank(i).FileName        = [name,'.img']; 
        rank(i).RepetitionTime  = vecTR(i);
        rank(i).EchoTime        = vecTE(i);
        rank(i).NumberOfSlices  = vecSlice(i);
        rank(i).FlipAngle       = vecAngle(i);
        rank(i).PulseSeqName    = vecName{i};
        rank(i).PlaneType       = vecPlane{i}; 
        rank(i).SeriesDesc      = vecSeriesDesc{i};
    end
    
    % disqualify 3-plane localizers
    for i = 1:vecLength
        disqualified(i) = ( (sum(findstr(infoStruct(i).PulseSeqName,'3Plane')) >=1)      | ...
                            (sum(findstr(infoStruct(i).SeriesDescription,'ASSET')) >= 1) | ...
                            (sum(findstr(infoStruct(i).SeriesDescription,'3pl')) >= 1)     ...
                           );
    end
    
    % select 1/0 arrays for desireable images
    goodTR = local_binary(find(vecTR<=500),vecLength);
    goodTE = local_binary(find(vecTE<=70),vecLength);
    goodSlice = local_binary(find(vecSlice==max(vecSlice)),vecLength);
    goodAngle = local_binary(find(vecAngle==min(vecAngle)),vecLength); 
    
    
    % First, try to find a suitable T1 
    T1_Index = find((goodTR+goodTE).*~disqualified==2); 
    if ~isempty(T1_Index)
        if length(T1_Index)>1
            T1_Slices=zeros(1,vecLength);T1_Slices(T1_Index)=vecSlice(T1_Index);
            bestT1 = find(T1_Slices==max(T1_Slices));   %T1 w/ most slices
            template = infoFiles{bestT1};
        else
            template = infoFiles{T1_Index};             %Take the only T1; 
        end
    end
    % try to find suitable T2's 
    %---- need to do this later

    [tp,tn]  = fileparts(template); 
    template = fullfile(tp,[tn,'.img']);
    V = spm_vol(template);
    if length(V) > 1, template = [template ',1']; end

end


%_______________________________________________________
%_______________________________________________________
function outV = local_binary(inV,vecLength)
outV = zeros(1,vecLength);
outV(inV) = 1;
