function reftext = wfu_read_refs(filenames)

% Read wfu reference functions and return variables in struct
% FORMAT reftext = wfu_read_refs(filenames)
%
% reftext       - structure containing variables and values
%                 read from reference functions
% filenames     - cell or character array of wfu reference
%                 functions
%___________________________________________________________

if ischar(filenames)
    filenames = cellstr(filenames);
end

for f = 1:length(filenames)
    if isempty(filenames{f})
        continue
    end

    setSep1 = ':';
    setSep2 = ',';
    timSep1 = ',';
    timSep2 = ',';

    oldLine = '';
    field   = [];
    value   = [];
    text_var    = [];
    half    = 1;
    fid     = fopen(filenames{f},'rt');
    while 1
        line = fgetl(fid);
        if ~ischar(line)
            break
        end

        line = [oldLine,line];
        line = line(~isspace(line));
        if ~isempty(line) & ~strcmp(';',line(1))
            if line(end)=='$'
                oldLine = line(1:end-1);
                continue
            else
                oldLine = '';
            end

            if strncmp(line,'begin',5)
                half = 2;
                continue
            elseif strncmp(line,'end',3)
                break
            end

            if half == 1
                %----- general parameters -----%
                fieldSep    = min(find(line==setSep1));
                field       = line(1:fieldSep-1);
                lineEnd     = line(fieldSep+1:end);
                if isempty(lineEnd)
                    value  = 0; 
                elseif strcmp('[',lineEnd(1))
                    value  = local_arrange_cells(lineEnd,setSep2);
                else
                    tryNumber = str2num(lineEnd);
                    if ~isempty(tryNumber)
                        value = tryNumber;
                    else
                        valueSep = find(lineEnd==setSep2);
                        if isempty(valueSep)
                            valueString = lineEnd(:);
                            if ischar(valueString)
                                value = valueString';
                            else
                                value = 'NonCharNonNumber...uncertainType';
                            end
                        else
                            offset  = 1;
                            value   = [];
                            for i = 1:length(valueSep)
                                value{i}    = lineEnd(offset:valueSep(i)-1);
                                offset      = valueSep(i)+1;
                            end
                            value{end+1}    = lineEnd(valueSep(end)+1:end);
                        end
                    end
                end
                if any(strcmp(field,{'names','contrast_names','contrast_types'})) & ~iscell(value)
                    value = {value};
                end

            elseif half == 2
                %----- timing -----%
                value       = [];
                fieldSep    = min(find(line==timSep1));
                field       = line(1:fieldSep-1);
                lineEnd     = line(fieldSep+1:end);
                valueString = local_arrange_cells(lineEnd,timSep2);
                event       = str2num(field);

                if strcmp(field,'blocks')
                    text_var.blocks = [];
                    field       = 'skipfield';
                    for i = 1:length(valueString);
                        text_var.blocks(i) = str2num(valueString{i});
                    end
                elseif strcmp(field,'time')
                    text_var.time   = [];
                    field       = 'skipfield';
                    if length(valueString) == 1
                        try
                            text_var.time = ones(1,length(text_var.blocks))*str2num(valueString{1});
                        catch
                            text_var.time = ones(1,100)*str2num(valueString{1});
                        end
                    else
                        for i = 1:length(valueString)
                            text_var.time(i) = str2num(valueString{i});
                        end
                    end
                elseif ~isempty(event)
                    field = 'skipfield';
                    if ~isfield(text_var,'events')
                        text_var.events.num = [];
                        text_var.events.dur = [];
                        text_var.events.ons = [];
                    end
                    exist_entry=valueString{1};
                    if exist_entry
                        text_var.events.num = [text_var.events.num,ones(1,length(valueString))*event];
                        text_var.events.dur = [text_var.events.dur,zeros(1,length(valueString))];
                        for i = 1:length(valueString)
                            text_var.events.ons(end+1) = str2num(valueString{i});
                        end
                    end
                elseif strcmp(field,'durations')
                    field = 'skipfield';
                    for i = 1:length(valueString)
                        text_var.events.dur(end-length(valueString) + i) = str2num(valueString{i});
                    end
                else
                    value{end+1} = valueString;
                end
            end

            %----- set structure field-----%
            if ~strcmp(field,'skipfield')
                text_var = wfu_setfield(text_var,field,value);
            end
        end
    end

    %----- adjust onsets for SPM -----%
    text_var = local_SPM_onsets(text_var);

    %----- fixed effects -----%
    if isfield(text_var,'combined_paradigms')
        text_var.fixedref   = 1;
        dontneed        = {'tr','conditions','images','ER','fixed', ...
            'mixed','names','events','SPMons','SPMdur'};
        for i = 1:length(dontneed)
            if isfield(text_var,dontneed{i})
                text_var = rmfield(text_var,dontneed{i});
            end
        end
    else
        text_var.fixedref   = 0;
    end

    %----- add filename to structure -----%
    text_var.refname = filenames{f};

    fclose(fid);
    text_var.native_contrast_values = text_var.contrast_values;
    reftext{f} = text_var;
end % for f = 1:length(filenames)
%-------Reset Lasterr for all the catches so that grid engine recovery doesn't detect the errors------
lasterr('');
reftext = wfu_permute_contrasts(reftext);
return



%_______________________________________________________________________
%_______________________________________________________________________
function output = local_arrange_cells(input,delim)

output  = {};
openBr  = find(input=='[');
closeBr = find(input==']');

if isempty(openBr)
    sep = find(input==delim);
    if sep~=0
        offset = 1;
        for i = 1:length(sep)
            output{end+1}   = input(offset:sep(i)-1);
            offset          = sep(i)+1;
        end
        output{end+1} = input(offset:end);
    else
        output{end+1} = input;
    end
else
    skip = 0;
    for i = 1:length(openBr)
        if i < skip
            continue
        else
            current = length(output)+1;
            if i < length(openBr) & closeBr(1) > openBr(i+1)
                subInput        = input(min(openBr)+1:max(closeBr)-1);
                output{current} = local_arrange_cells(subInput,delim);
                skip            = i+length(output{end})+1;
            else
                seg = input(openBr(i):closeBr(i));
                oBr = find(seg=='[');
                cBr = find(seg==']');
                sep = find(seg==delim);
                offset  = min(oBr)+1;
                if ~isempty(sep)
                    offset  = min(oBr)+1;
                    for x = 1:length(sep)
                        output{current}{x} = seg(offset:sep(x)-1);
                        offset = sep(x)+1;
                    end
                    output{current}{end+1} = seg(sep(end)+1:max(cBr)-1);
                else
                    output{current}{1} = seg(min(oBr)+1:min(cBr-1));
                end
            end
        end
    end
end
return

%___________________________________________________________
function text_var = local_SPM_onsets(text_var)

%----- complete onsets for blocks -----%
if isfield(text_var,'blocks') & isfield(text_var,'time')
    clock = 0;
    for i = 1:length(text_var.blocks)
        if text_var.blocks(i) == 0
            clock = clock + text_var.time(i);
        else
            event = text_var.blocks(i);
            try
                idx = length(text_var.SPMons{event}) + 1;
            catch
                text_var.SPMons{event}  = [];
                text_var.SPMdur{event}  = [];
                idx                 = 1;
            end
            text_var.SPMons{event}(idx) = clock;
            text_var.SPMdur{event}(idx) = text_var.time(i);
            clock                   = clock + text_var.time(i);
        end
    end
end
%----- complete onsets for events -----%
if isfield(text_var,'events')
    for i = 1:length(text_var.events.num)
        event = text_var.events.num(i);
        try
            idx = length(text_var.SPMons{event}) + 1;
        catch
            text_var.SPMons{event}  = [];
            text_var.SPMdur{event}  = [];
            idx                 = 1;
        end
        text_var.SPMons{event}(idx) = text_var.events.ons(i);
        text_var.SPMdur{event}(idx) = text_var.events.dur(i);
    end
end
return
