function [inDirStep, outDirStep] = wfu_DICOM(inDir,outDir,clearFlag,prefix,suffix)

%   Converts 2D DICOM files into volumes
%   FORMAT wfu_DICOM(inDir,outDir,prefix,suffix)
%   
%   inDir       - directory containing DICOM format images
%   outDir      - directory to write volumes
%   clearFlag   - when clearFlag==1, if output directory is 
%                 not empty at start files will be deleted 
%                 *** by default, set to ON (clearFlag=1)
%   prefix      - prefix of DICOM images to read
%   suffix      - suffix of DICOM images to read
%
%   these optional output arguments, when requested, prevent
%   execution of the wfu_extra_processing_steps function:
%    inDirStep -  the input directory in use
%    outDirStep-  the output directory in use
%
%   ##  v649.1, Aaron Baer, Wake Forest University        ##
%   ##  based upon code from spm_dicom_convert.m(v2.17)   ##
%   ##  and spm_dicom_headers.m(v2.14) by John Ashburner  ##
%____________________________________________________________

global defaults; spm_defaults;          % defaults.analyze.flip = 1; 
dircon = wfu_directory_configuration;   % dircon.anatdir setting    
startdir = pwd;
cpuSpeed = .0025; 											% relative speed of proccessing.  
																				% this times number of images is 
																				% the "estimate to to completion".
																				% .0025 is for a dual core 3.0GHz box

if nargin < 1 | isempty(inDir)
    if exist('spm_get')
        inDir = spm_get(-1,'*','Select dicom directory',pwd);
    else
        inDir = spm_select(1, 'dir', 'Select dicom directory',[], pwd);
    end
    inDir = spm_str_manip(inDir,'r');
else
    inDir = wfu_full_path(inDir); 
end

if nargin < 2 | isempty(outDir)
    [path, name, ext] = fileparts(inDir);
    outDir = fullfile(path, dircon.anatdir);
    [ok,msg,msgID] = mkdir(path, dircon.anatdir);
     if ~ok | ~isempty(msg)
        disp(sprintf('Output directory already exists...')); 
    end
    if exist('spm_get')
        existingFiles = spm_get('files',outDir,'*');
    else
        existingFiles = spm_select('List', outDir, '.*');
    end
end
if ~isdir(outDir)
	[ok,msg,msgID] = mkdir(outDir);
  if ~ok
      error('could not create output dir: %s', outDir)
	end
end
if exist('spm_get')
  existingFiles = spm_get('files',outDir,'*');
else
  [existingFiles existingDirectories] = spm_select('List',outDir,'.*');
end
if ~isempty(existingFiles) | ~isempty(existingDirectories)
    if nargin < 3 | clearFlag == 1
        disp(sprintf('Deleting contents of directory: %s',outDir)); 
        existingDir = fullfile(outDir,'*');
        delStr = sprintf('!rm -rf %s',existingDir); 
        eval(delStr);
    end
end     

if nargin < 4 | isempty(prefix), prefix = '';
else, prefix = deblank(prefix); 
end

if nargin < 5 | isempty(suffix), suffix = '';
    defaultWarn = '(no suffix specified)';
else, suffix = deblank(suffix); defaultWarn = ''; 
end


%----- dicom conversion -----%
warning off MATLAB:singularMatrix;
fprintf('Dicom-->%s conversion began at: %s\n\n',dircon.anatdir, datestr(now,14));
fprintf('Dicom directory: %s\nOutput directory: %s\nPrefix: ''%s''\nSuffix: ''%s'' %s\n\n',inDir,outDir,prefix,suffix,defaultWarn);
if exist('clearFlag', 'var')
    disp(sprintf('Clear output directory = %d', clearFlag));
else
    disp(sprintf('Clear output directory = %d', 1));
end
options.displayFiles        = 0; 
options.displayDirectories  = 0;
[dcmFiles,dcmDirs]          = wfu_find_files([prefix,'.*',suffix,'$'],inDir,true,options);
ndir                        = length(dcmDirs);
cd(outDir);
for d = 1:ndir
    [seriespath, seriesname] = fileparts(dcmDirs{d}); 
    seriesdir = sprintf('convert_%s', seriesname);
    disp(sprintf('Series Dir: %s', seriesdir));
    outDirTmp = fullfile(outDir, seriesdir);
    if isdir(outDirTmp)
        [success, msg, msgid] = rmdir(outDirTmp, 's');
        if ~success
            error('cannot remove directory: %s', outDirTmp);
        end
    end
    [success, msg, msgid] = mkdir(outDirTmp);
    if ~success
        error('cannot create directory: %s', outDirTmp);
    end
    cd(outDirTmp);
    disp(sprintf('Processing directory %d of %d ...', d, ndir));
    [dcmFiles, dcmDirs2]    = wfu_find_files([prefix,'.*',suffix,'$'],dcmDirs{d},true,options);
    P                       = char(dcmFiles);
    nImg                    = size(P,1);
    fprintf('Reading dicom headers (%i images) ...\n',nImg);
    fprintf('Estimated time to completion: %i minutes\n',ceil(nImg*cpuSpeed)); 
    try 
        hdr = spm_dicom_headers(P);
        fprintf('Converting dicom to %s format ...\n', dircon.anatdir); 
        wfu_spm_dicom_convert(hdr);
    catch
        errorInfo = lasterror;
        fprintf('Error received: %s\n',errorInfo.message);
        fprintf('Warning: skipping directory on SPM DICOM conversion error!\n');
    end

    %----- merge series with various PSD's -----%
    %local_merge_multivols(outDirTmp,{'epi2','epiflex','bkraft', 'fgre'});
    local_merge_multivols(outDirTmp,{'epi2','epiflex','bkraft', 'fgre', 'epi', 'despot'});

    %----- save dicom image for each series -----%
    local_copy_dicom(outDirTmp);

    %----- rename according to wfu preferences -----%
    local_rename_output(outDirTmp);

    %----- cleanup instance number in name (ask bkraft) -----%
    %local_cleanup_instance(outDirTmp); 

    % move the converted files to the outDir above
%    existingFiles = spm_get('files' ,outDirTmp, '*');
    if exist('spm_get')
      existingFiles = spm_get('files',outDirTmp,'*');
    else
      [existingFiles existingDirectories] = spm_select('List',outDirTmp,'.*');
    end
    if ~isempty(existingFiles)
        [success, msg, msgid] = movefile('*', outDir, 'f');
        if ~success
            error('could not move converted files to output directory');
        end
    end
    cd(outDir);
    [success, msg, msgid] = rmdir(outDirTmp, 's');
    if ~success
        error('cannot remove directory: %s', outDirTmp);
    end
end
cd(startdir);

% ... DO ALL OF THIS INSIDE THE PER SERIES LOOP ABOVE
%----- merge series with various PSD's -----%
%local_merge_multivols(outDir,{'epi2','epiflex','bkraft', 'fgre'});

%----- save dicom image for each series -----%
%local_copy_dicom(outDir);

%----- rename according to wfu preferences -----%
%local_rename_output(outDir);

%----- cleanup instance number in name (ask bkraft) -----%
%local_cleanup_instance(outDir); 
% ...

fprintf('Conversion complete.\nTime of completion: %s\n',datestr(now));

%----- continue WFU processing steps, if any, before return -----%
% don't do this if return arguments are requested for a separate call to
% wfu_extract_processing_steps
%
inDirStep = inDir;
outDirStep = outDir;
if nargout == 0
    wfu_extra_processing_steps(inDir, outDir);
else
    disp('... wfu_DICOM was called with output arguments');
    disp('... skipping wfu_extra_processing_steps within wfu_DICOM');
end
return 




%_______________________________________________________________
%_______________________________________________________________
function local_merge_multivols(outDir,PSD)

for p = 1:length(PSD)
    
    try
        usedSeries  = [];
        dti         = [];
        nSlices     = [];
        infoText    = local_read_info_files(outDir);
        
        %--- sort images into 4D volumes ---% 
        for i = 1:length(infoText)
            text = infoText{i};
            if strfind(lower(text.PulseSeqName),PSD{p}) & text.NumberOfVolumes == 1
                imgName     = wfu_str_fix(text.InfoFilename,'se','img');
                matchIndex  = find(ismember(usedSeries,text.SeriesNumber)==1);
                if isempty(matchIndex)
                    %-- first of series --%
                    dti{end+1}{1}           = imgName; 
                    usedSeries(end+1)       = text.SeriesNumber;
                    nSlices{end+1}          = text.NumberOfSlices;
                else
                    %-- add to series --%
                    dti{matchIndex}{end+1}  = imgName;
                    nSlices{matchIndex}     = nSlices{matchIndex}+text.NumberOfSlices;
                end
            end
        end
        
        %--- DTI ---%
        for i = 1:length(dti)
            imgSet  = dti{i};
            nVols   = length(imgSet);
            pref    = wfu_str_fix(imgSet{1},'kf');
            pref4d  = [pref,'_temp4d'];
            name4d  = wfu_str_fix(imgSet{1},'sf',pref4d);
            
            %--- write 4D image ---%
            mergeOK = wfu_merge_3d(imgSet,name4d);
            
            if mergeOK
                
                %--- copy 4D .info file ---%
                oldInfo     = wfu_str_fix(imgSet{1},'se','info');
                newInfo     = wfu_str_fix(name4d,'se','info');
                copyInfoStr = sprintf('!cp %s %s',oldInfo,newInfo); 
                eval(copyInfoStr);       
                
                %--- modify 4D .info file  ---%
                infoFid = fopen(newInfo,'r+');
                if infoFid == -1
                    warning(sprintf('4D info file could not be created.'));
                    infoOK = 0;
                else
                    infoTxt = textread(newInfo,'%s','delimiter','\n','whitespace','');
                    for x = 1:length(infoTxt)
                        if strncmp(infoTxt{x},'NumberOfSlices',14)
                            infoTxt{x} = sprintf('NumberOfSlices\t%i',nSlices{i});
                        elseif strncmp(infoTxt{x},'NumberOfVolumes',15)
                            infoTxt{x} = sprintf('NumberOfVolumes\t%i',nVols);
                        end
                        fprintf(infoFid,'%s\n',infoTxt{x});
                    end
                    fclose(infoFid);
                    infoOK = 1; 
                end
                
                if infoOK
                    
                    %--- modify mat fields ---% 
                        wfu_mat_fields(name4d);
                    
                    %--- erase old 3D files ---%
                    fprintf('Erasing old 3D files ...\n');
                    %for i = 1:length(imgSet)
                        %currFile    = imgSet{i};
                    for j = 1:length(imgSet)
                         currFile    = imgSet{j};
                         [cP,cN]     = fileparts(currFile);
                        delString   = [fullfile(cP,cN),'.*'];
                        delete(delString);
                    end
                    
                end
            end
        end
    catch %-if merging fails somehow, don't crash 
        errorInfo = lasterror;
        fprintf('Error received: %s\n',errorInfo.message);
        disp('trapping DICOM conversion merge failure in local function: local_merge_multivols of wfu_DICOM');
        break
    end
end


%_______________________________________________________________
function local_copy_dicom(outDir)

infoText = local_read_info_files(outDir); 
fprintf('Saving first dicom image of each series ...\n');
for i = 1:length(infoText)
    text = infoText{i}; 
    if isfield(text,'Filename')
        firstDicomOfVolume  = text.Filename;
        newDicomName        = wfu_str_fix(text.InfoFilename,'se','dicom');
%       copyStr             = sprintf('!(yes | cp -r %s %s)',firstDicomOfVolume,newDicomName); 
%       eval(copyStr);
        if exist(newDicomName, 'file')
            delete(newDicomName);
        end
        copyStr             = sprintf('!(cp -r %s %s)',firstDicomOfVolume,newDicomName); 
        eval(copyStr);
    else
        warning(sprintf('Cannot copy dicom files..\nInfo file is incomplete.')); 
    end
end
return


%_____________________________________________________________
function local_rename_output(volDir)
fprintf('Renaming output files to wfu convention ...\n');

planeCount  = 1;
infoText    = local_read_info_files(volDir);
for i = 1:length(infoText); 
    text = infoText{i};
    %--- crop off PatientID and SeriesNumber ---% 
    oldName = wfu_str_fix(text.InfoFilename,'kf');
    dashes  = find(oldName=='-');
    segment = oldName(1:dashes(2));
    %--- remove leading 's' from spm output names ---%
    if strcmp(segment(1),'s')
        segment = wfu_str_fix(segment,'rlc',1);
    end
    %--- add identifying fields to name ---%
    prefix = sprintf('%s%03d-%04d-%04d-', ...
        segment,text.NumberOfVolumes,text.InstanceNumber,text.NumberOfSlices);
    %--- add PulseSeqName if exists ---% 
    if ~isempty(text.PulseSeqName)
        %--- adjust name for 3Plane localizer ---% 
        if strcmp(text.PulseSeqName,'3Plane')
            psd         = text.PulseSeqName;
            newPref     = sprintf('%s%s%i',prefix,psd,planeCount);
            planeCount  = planeCount+1;
        else
            psd         = text.PulseSeqName; 
            newPref     = sprintf('%s%s',prefix,psd);
        end
    else
        psd     = 'UnknownPulseSeq';
        newPref = sprintf('%s%s',prefix,psd);
    end
    
    %--- rename files ---%
    if exist('spm_select')
        P    = cellstr(spm_select('List', volDir, oldName));
    else  
        P    = cellstr(spm_list_files_expand(volDir,sprintf('%s*',oldName),'noexpand'));
    end 
    used = [];   
    for x = 1:length(P)
        oldFilename = fullfile(volDir,P{x});
        newFilename = wfu_str_fix(oldFilename,'sf',newPref); 
        if ~any(strcmp(oldFilename,used)) & ~any(strcmp(newFilename,used))
            moveString  = sprintf('!mv %s %s',oldFilename,newFilename);
            eval(moveString);
            used{end+1} = oldFilename;
            used{end+1} = newFilename; 
        end
    end
end
return


%_______________________________________________________________
function infoText = local_read_info_files(outDir)

if exist('spm_get')
  infoFiles = cellstr(spm_get('files',outDir,'*info'));
else
	infoFiles = cellstr(spm_select('List', outDir, '.*\.info'));
	if ~isempty(infoFiles{1})
	  for i = 1:length(infoFiles)
	    infoFiles{i} = [outDir, '/', infoFiles{i}];
	  end
	end
end

if length(infoFiles) == 1 && isempty(infoFiles{1})
    infoText = [];
    return;
end

for i = 1:length(infoFiles) 
    infoText{i}              = wfu_read_info(infoFiles{i});
    infoText{i}.InfoFilename = infoFiles{i};
end
return

%_______________________________________________________________
function local_cleanup_instance(outDir) 

usedSeries  = [];
usedInst    = [];
instCount   = []; 
if exist('spm_get')
  imgFiles = cellstr(spm_get('files',outDir,'*img'));
else
  imgFiles = cellstr(spm_select('List', outDir, '.*\.img'));
  for i = 1:length(imgFiles)
    imfFiles{i} = [outDir, '/', imgFiles{i}];
  end
end
for i = 1:length(imgFiles)
    [path,name] = fileparts(imgFiles{i}); 
    dashes      = find(name=='-');
    series      = str2num(name(dashes(1)+1:dashes(2)-1));
    instance    = str2num(name(dashes(3)+1:dashes(4)-1)); 
    matchidx    = find(ismember(series,usedSeries)==1);
    if isempty(matchidx) & instance == 1 
        usedSeries(end+1)   = series;
        instCount(end+1)    = 1;
    else %--- rename the following ---% 
        if isempty(matchidx)
            usedSeries          = series; 
            instCount(end+1)    = 1; 
            instance            = sprintf('%04i',1);
        else
            instCount(matchidx) = instCount(matchidx) + 1;
            instance            = sprintf('%04i',instCount(matchidx)); 
        end
        oldname = fullfile(path,name); 
        newname = fullfile(path,[name(1:dashes(3)),instance,name(dashes(4):end)]);
        exts    = {'.img','.hdr','.info','.dicom','.mat'};
        for e = 1:length(exts)
            rename = sprintf('!mv %s %s',[oldname,exts{e}],[newname,exts{e}]);
            disp(rename); 
            eval(rename)
        end
    end
end
return
