function wfu_convert_to_dicom(input_filename, dicom_dir, clear_dir, verbose, modifyhdr, dicom_template_filename)
%__________________________________________________________________________
%
% Syntax:
%     wfu_convert_to_dicom(input_filename, dicom_dir, clear_dir, ...
%                          verbose, modifyhdr, dicom_template_filename)
%     wfu_convert_to_dicom(input_filename, dicom_dir, clear_dir, ...
%                          verbose, modifyhdr)
%
% Required Arguments:
%     input_filename        file name of 3d image volume or single image
%     dicom_dir              directory to hold the DICOM images created
%
% Optional Arguments:
%     clear_dir              toggle to indicate if dicom_dir is to be
%                            cleared of contents before images created
%                              (default is 1)
%     verbose                toggle to indicate if text feedback is wanted
%                              (default is 1)
%     modifyhdr              structure to hold user modifications to
%                            DICOM header; current options checked are:
%                              modifyhdr.newhdr.SeriesNumber
%                              modifyhdr.newhdr.StudyDescription
%                              modifyhdr.newhdr.SeriesDescription
%                              modifyhdr.newhdr.AdditionalPatientHistory
%                              modifyhdr.existinghdr.SeriesNumberAdd
%                              modifyhdr.existinghdr.SeriesNumberOverwrite
%                              modifyhdr.existinghdr.SeriesDescription
%                              (default is [])
%   dicom_template_filename  file name of first DICOM file in series
%                              (default is NONE SPECIFIED)
%
% Description:
% If the input_filename has an extention recognized by MATLAB as an
% image format, the input will be treated as an image object.  Otherwise,
% the input will be treated as a 3d image volume.
%
% The single image object or each slice from the 3d image file is read and
% converted to DICOM.  Each DICOM file created represents the image object
% or a single slice from the 3d image file.
%
% The DICOM files are written to the directory, dicom_dir.  If the
% clear_dir toggle is on, all f*.dcm files in dicom_dir will be deleted
% before any new images are created.  If the verbose toggle is on, text
% feedback will be provided.  Any newhdr fields in the modifyhdr structure
% will be applied to a new header created from scratch.  The existinghdr
% structure includes fields for SeriesNumberAdd and SeriesNumberOverwrite.
% With the optional dicom_template, if SeriesNumberOverwrite is set, that
% value will be used for the series.  If SeriesNumberAdd is set (and 
% SeriesNumberOverwrite is not set), then the SeriesNumber in the output
% header will be set equal to the original series number plus the 
% SeriesNumberAdd value.  If these modifyhdr.existinghdr fields are not
% set, then 90 will be added to the original series number.  The series
% description may also be modified for an existing header.  For a new
% header created from scratch, when no modifyhdr.newhdr values are set
% these will be used:
%        SeriesNumber = 777;
%        StudyDescription = 'FMRI Study';
%        SeriesDescription = 'FMRI Series';
%        AdditionalPatientHistory = 'FUSION History';
%
% Limitation:
% The format of each generated DICOM file is "f%04d.dcm".  This limits the
% output DICOM directory to a maximum of 9999 files.
%
% Possible future modifications:
%     - add hook for feedback of slice number within a loop
%     - allow any legitimate DICOM header fields in modify.newhdr
%
% KLP
% 09-15-06
%__________________________________________________________________________
%
    if nargin < 2
        error('at least 2 arguments required');
    end
    if ~exist(input_filename, 'file')
        error('Cannot locate input file: %s', input_filename);
    end
    if ~exist('clear_dir', 'var')
        clear_dir = 1;
    end
    if ~exist('verbose', 'var')
        verbose = 0;
    end
%
% determine whether the input file represents a single image object or
% a 3d image volume
%
    fmt = what_image_format(input_filename);
    if verbose
        if isempty(fmt)
            disp('Input treated as a 3d image volume');
        else
            disp('Input treated as an image object');
        end
    end
    if isempty(fmt)
        image_object = [];
        V  = spm_vol(input_filename);
    else
        image_object = imread(input_filename);
        V.dim = size(image_object);
        V.dim(3) = 1;
        V.mat = [1 0 0 0; 0 1 0 0; 0 0 1 0; 0 0 0 1];
    end
%
% assume that the first file # will be 1 (DICOM file name = f00001.dcm);
% if f*.dcm files exist in the output directory, then the first file
% # to use will be one more than the number of files already there
%
    first_fileno = 1;
%
% if clear_dir, erase f*.dcm files in an existing dicom_dir;
% if the directory doesn't exist, then make it
%
    if isdir(dicom_dir)
        dirstr = sprintf('%s/f*.dcm', dicom_dir);
        dirlist = dir(dirstr);
        n = 0;
        if length(dirlist) > 0
            dicom_fnames = cellstr(char(dirlist.name));
            n = length(dicom_fnames);
        end
        if clear_dir
            if n > 0
                if verbose
                    disp(sprintf('removing files in existing temp directory: %s', dicom_dir));
                end
                for i = 1:n
                    delete(fullfile(dicom_dir, dicom_fnames{i}));
                end
            end
        else
            first_fileno = n + 1;
        end
    else
        if exist(dicom_dir, 'file')
            delete(dicom_dir);
        end
        if verbose
            disp(sprintf('creating new temp diretory: %s', dicom_dir));
        end
        [status, message, messageid] = mkdir(dicom_dir);
        if ~status
            error('cannot create new temp dir: %s', dicom_dir);
        end
    end
%
    dicom_output_format = 'f%04d.dcm';
    if verbose
        disp(sprintf('dicom_output_format: %s', dicom_output_format));
        disp(sprintf('writing to convert temp directory: %s', dicom_dir));
        disp(sprintf('input filename: %s', input_filename));
    end
%
% if a DICOM template is specified, use its header and modify only the
% series number (and/or series description) in the header if requested;
% add 90 to original series number if no setting in:
%          modifyhdr.existinghdr.SeriesNumberAdd
%     (or) modifyhdr.exsitinghdr.SeriesNumberOverwrite
%
    if exist('dicom_template_filename', 'var') && exist(dicom_template_filename, 'file')
        if verbose
            disp(sprintf('reading DICOM template file: %s', dicom_template_filename));
        end
        dh = dicominfo(dicom_template_filename);
        starting_series = dh.SeriesNumber;
        starting_description = dh.SeriesDescription;
        dh.SeriesNumber = starting_series + 90;
        if exist('modifyhdr', 'var') && isfield(modifyhdr, 'existinghdr')
            if isfield(modifyhdr.existinghdr, 'SeriesNumberAdd')
                dh.SeriesNumber = starting_series + modifyhdr.existinghdr.SeriesNumberAdd;
            end
            if isfield(modifyhdr.existinghdr, 'SeriesNumberOverwrite')
                dh.SeriesNumber = modifyhdr.existinghdr.SeriesNumberOverwrite;
            end
            if isfield(modifyhdr.existinghdr, 'SeriesDescription')
				%
				% Check the length of text input
				%
                series_description = modifyhdr.existinghdr.SeriesDescription;
                maxlen = 60;
                strformat = sprintf('%c%dc', '%', maxlen);
                if length(series_description) > maxlen
                    series_description = strread(series_description, strformat, 1);
                end
                dh.SeriesDescription = series_description;
            end
        end
        if verbose
            disp(sprintf('Series Number was:           %d', starting_series));
            disp(sprintf('Series Number reset to:      %d', dh.SeriesNumber));
            disp(sprintf('Series Description was:      %s', starting_description));
            disp(sprintf('Series Description reset to: %s', dh.SeriesDescription));
        end
        %%dh.PatientOrientation = 'L\P';
%
% if no DICOM template is given, create a header from scratch;
% if any modifyhdr.newhdr variables are specified, use those for
% certain fields
%
    else
        if verbose
            disp('no DICOM template file; creating one');
        end
        series_number = 777;
        study_description = 'FMRI Study';
        series_description = 'FMRI Series';
        history = 'FUSION History';
        if exist('modifyhdr', 'var') && isfield(modifyhdr, 'newhdr')
            if isfield(modifyhdr.newhdr, 'SeriesNumber')
                series_number = modifyhdr.newhdr.SeriesNumber;
            end
            if isfield(modifyhdr.newhdr, 'StudyDescription')
                study_description = modifyhdr.newhdr.StudyDescription;
            end
            if isfield(modifyhdr.newhdr, 'SeriesDescription')
                series_description = modifyhdr.newhdr.SeriesDescription;
            end
            if isfield(modifyhdr.newhdr, 'AdditionalPatientHistory')
                history = modifyhdr.newhdr.AdditionalPatientHistory;
            end
        end
%
% Check the lengths of text inputs
%
        maxlen = 60;
        strformat = sprintf('%c%dc', '%', maxlen);
        if length(history) > maxlen
            history = strread(history, strformat, 1);
        end
        if length(study_description) > maxlen
            study_description = strread(study_description, strformat, 1);
        end
        if length(series_description) > maxlen
            series_description = strread(series_description, strformat, 1);
        end
        dh = wfu_make_dicom_header;
%
        %accession_number = '100000';
        today = strrep(datestr(now, 'yyyy/mm/dd'), '/', '');
        %%
        dh.Filename                      =  'No locfile';
        dh.StudyDate                     = 	today;
        dh.SeriesDate                    = 	today;
        dh.ContentDate                   = 	today;
        %dh.AccessionNumber               = 	accession_number;
        dh.InstitutionName               =	'MATLAB';
        dh.StudyDescription              = 	study_description;
        dh.SeriesDescription             =  series_description;
        dh.AdditionalPatientHistory      =  history;
        %dh.PatientName                   = 	'Research Patient';
        %dh.OtherPatientName              = 	['Research Patient'];
        %dh.PatientID                     = 	'100000';
        dh.SeriesNumber                  =  series_number;
        if verbose
            disp(sprintf('in new header, setting series number to: %d', dh.SeriesNumber));
        end
        DIM = V.dim;                   
        VOX = abs(diag(V.mat));
        dh.PixelSpacing = [VOX(1) VOX(2)];
        thick = VOX(3);
        gap = 0;
        spacing = thick + gap;
        dh.Rows = DIM(1);
        dh.Columns = DIM(2);
        dh.SliceThickness = thick;
        dh.SpacingBetweenSlices = spacing;
        %
        % get Julian seconds since Jan 1, 1970 GMT; Eastern US time zone is 5 hours from GMT
        %
        timestamp = etime(clock, [1970 1 1 0 0 0]) + (5 * 60 * 60); 
        %
        dicomroot = '1.3.6.1.4.1';
        % MatLab's registration number
        spmid = '9590';
        landmark = '100.00';
        accession_number = '100000';
        dh.FrameOfReferenceUID = sprintf('%s.%s.%0.f.%s.%s', dicomroot, spmid, timestamp, accession_number, landmark);
    end
    wfu_dicom_write_all_slices(dicom_dir, dicom_output_format, dh, V, first_fileno, image_object, verbose);
% % % % end
%
function wfu_dicom_write_all_slices(dicom_dir, dicom_output_format, dh, V, first_fileno, image_object, verbose)
%__________________________________________________________________________
%
% Syntax:
%   wfu_dicom_write_all_slices(dicom_dir, dicom_output_format, ...
%                                       dh, V, first_fileno, verbose)
%
% Required Arguments:
%   dicom_dir               output directory for DICOM images created
%   dicom_output_format     format for DICOM file names (e.g. 'f%04d.dcm')
%   dh                      DICOM header structure
%   V                       spm volume header (or dimensions if img object)
%   first_fileno            first file number for created DICOM file names
%   image_object            image read from image object input (empty if V)
%   verbose                 toggle to indicate if text feedback is wanted
%
% Description:
% All slices in the input (with the dh header struct) will be
% written as DICOM files to the directory dicom_dir using the
% dicom_output_format as the output file name for each slice. 
%
% KLP
% 01-26-05
%__________________________________________________________________________
  
    DIM = V.dim;                   
    VOX = abs(diag(V.mat));
    h.PixelSpacing = [VOX(1) VOX(2)];
    thick = VOX(3);
    gap = 0;
    spacing = thick + gap;
    dh.Rows = DIM(1);
    dh.Columns = DIM(2);
    zsize = DIM(3);

    % request unique instance UID
    %
    dh.SeriesInstanceUID = dicomuid;
    dh.ImagesInAcquisition = zsize;
    dh.ImagesInSeries = zsize;
    %
    dh.PixelRepresentation = 1;
    dh.BitsAllocated = 16;
    dh.BitsStored = 16;
    dh.HighBit = 15;

    % write out the image for each slice with new computed position
    %
    iop = dh.ImageOrientationPatient;
    ipp = dh.ImagePositionPatient;
    slice0 = dh.SliceLocation;
    slice = [];
    for n = 1:zsize
        if isempty(image_object)
            slice = spm_slice_vol(V, spm_matrix([0 0 n]), V.dim(1:2), 0);
            slicemax = max(slice(:));
            slicemin = min(slice(:));
        else
            slicemax = max(image_object(:));
            slicemin = min(image_object(:));
        end
	width = double(slicemax) - double(slicemin);
        center = double(slicemin) + fix(width/2);
        dh.WindowWidth = width;
        dh.WindowCenter = center;
        dh.InstanceNumber = n;
        dh.ImagePositionPatient(1) = ipp(1) + ((n - 1) * (spacing * (iop(2) * iop(6) - iop(3) * iop(5))));
        dh.ImagePositionPatient(2) = ipp(2) + ((n - 1) * (spacing * (iop(3) * iop(4) - iop(1) * iop(6))));
        dh.ImagePositionPatient(3) = ipp(3) + ((n - 1) * (spacing * (iop(1) * iop(5) - iop(2) * iop(4))));
        dh.SliceLocation = slice0 + (spacing * (n - 1));
        dcmname = sprintf(dicom_output_format, n + first_fileno - 1);
        dicom_output_file = sprintf('%s/%s', dicom_dir, dcmname);
        if verbose
            report = sprintf('Writing range [%g,%g] window[%g,%g] slice %d of %d to DICOM file:\n\t%s', ...
		     double(slicemin), double(slicemax), dh.WindowCenter, dh.WindowWidth, n, zsize, dicom_output_file);
            disp(report);
        end
        if isempty(image_object)
            dicomwrite(int16(rot90(slice)), dicom_output_file, dh, 'ObjectType', 'MR Image Storage', 'VR', 'implicit', ...
                'SmallestImagePixelValue', slicemin, 'LargestImagePixelValue', slicemax);
        else
            dicomwrite(image_object, dicom_output_file, dh,  'ObjectType', 'Secondary Capture Image Storage', 'VR', 'implicit', ...
		       'SmallestImagePixelValue', slicemin, 'LargestImagePixelValue', slicemax);
        end
    end
% % % % end
function dh = wfu_make_dicom_header(dx)
%__________________________________________________________________________
%
% Syntax:
%     function dh = wfu_make_dicom_header(dx)
%
% Optional argument:
%     dx    input DICOM header
%
% Returned value:
%    dh    output DICOM header
%
% Description
% Fill in the essential fields in an empty DICOM header.
% Optional input argument: an existing header for modification
%
% KLP
% 01-21-05
%__________________________________________________________________________

    if (nargin == 1)
        dh = dx;
    end
    %
    % This function was translated from the IDL routine, make_dicom_header.pro.
    %
    % The following header fields previously referenced from IDL exist
    % under different names in the default MatLab DICOM data dictionary.
    %
    % IDL Name                      MatLab Name                 Group   Element
    % ----------------------------  --------------------------  -----   -------
    % ImageDate                     ContentDate                 0008    0023
    % ImageTime                     ContentTime                 0008    0033
    % AttendingPhysiciansName       PerformingPhysicianName     0008    1050
    % NameofPhysiciansReadingStudy  PhysicianReadingStudy       0008    1060
    % OperatorsName                 OperatorName                0008    1070
    % OtherPatientNames             OtherPatientName            0010    1001
    % PercentSamplingFieldofView    PercentPhaseFieldofView     0018    0094
    % CardiacNumberofFrames         CardiacNumberofImages       0018    1090 
    % ReceivingCoil                 ReceiveCoilName             0018    1250
    % TransmittingCoil              TransmitCoilName            0018    1251
    % VariableFlipAngle             VariableFlipAngleFlag       0018    1315
    % ImageNumber                   InstanceNumber              0020    0013
    %--------------------------------------------------------------------------
    %
    % POSSIBLY UNDEFINED
    % dh.PatientOrientation    =   'L\P';                                           % group 0020 element 0020
    %
    dh.SpecificCharacter_Set   =   'ISO_IR 100';                                    % group 0008 element 0005
    % MatLab sets OTHER ImageType/uncompressed by default
    %dh.ImageType               =   'DERIVED\PRIMARY';                              % group 0008 element 0008
    dh.SOPClassUID             =   '1.2.840.10008.5.1.4.1.1.4';                     % group 0008 element 0016
    dh.SOPInstanceUID =   '1.2.840.200000.2.1.1.2702429029.814.936099513.654';      % group 0008 element 0018
    dh.StudyDate               =   '19990831';                                      % group 0008 element 0020
    dh.SeriesDate              =   '19990831';                                      % group 0008 element 0021
    dh.ContentDate             =   '19990831';                                      % group 0008 element 0023
    dh.StudyTime               =   '142522';                                        % group 0008 element 0030
    dh.SeriesTime              =   '144813';                                        % group 0008 element 0031
    dh.ContentTime             =   '144813';                                        % group 0008 element 0033
    %dh.AccessionNumber        =   '00000000';                                      % group 0008 element 0050
    dh.AccessionNumber         =   '100000';                                        % group 0008 element 0050
    dh.Modality                =   'MR';                                            % group 0008 element 0060
    dh.Manufacturer            =   'GE MEDICAL SYSTEMS';                            % group 0008 element 0070
    dh.InstitutionName         =   'Wake Forest University';                        % group 0008 element 0080
    %dh.InstitutionName        =   'IDLSPM';                                        % group 0008 element 0080
    dh.ReferringPhysiciansName =   'RESEARCH';           % MATLAB6 %                % group 0008 element 0090                    
    dh.ReferringPhysicianName  =   'RESEARCH';           % MATLAB7 %                % group 0008 element 0090                    
    dh.StationName             =   'MR06MR06';                                      % group 0008 element 1010
    dh.StudyDescription	       =   'RESEARCH';                                      % group 0008 element 1030
    dh.SeriesDescription       =   'RESEARCH';                                      % group 0008 element 103e
    dh.PerformingPhysicianName =   'RESEARCH';                                      % group 0008 element 1050
    dh.PhysicianReadingStudy   =   'NEURO';                                         % group 0008 element 1060
    dh.OperatorName            =   'AF';                                            % group 0008 element 1070
    dh.ManufacturerModelName   =   'SIGNA_EXCITE';                                  % group 0008 element 1090
    % MatLab sets OTHER ImageType/uncompressed by default
    %dh.DerivationDescription  =   'Lossy Compressed';                              % group 0008 element 2111
    dh.PatientsName            =   'RESEARCH PATIENT';   % MATLAB6 %                % group 0010 element 0010
    dh.PatientName             =   'RESEARCH PATIENT';   % MATLAB7 %                % group 0010 element 0010
    %dh.PatientID              =   '00000000';                                      % group 0010 element 0020
    dh.PatientID               =   '100000';                                        % group 0010 element 0020
    dh.PatientBirthDate        =   '19570303';                                      % group 0010 element 0030
    dh.PatientSex              =   'M';                                             % group 0010 element 0040
    dh.OtherPatientName        =   'RESEARCH PATIENT';                              % group 0010 element 1001
    dh.PatientAge              =   '042Y';                                          % group 0010 element 1010
    dh.PatientWeight           =   86.183;                                          % group 0010 element 1030
    dh.AdditionalPatientHistory=   'RESEARCH FUSION';                               % group 0010 element 21b0
    dh.ContrastBolusAgent      =   'NONE';                                          % group 0018 element 0010
    dh.ScanningSequence        =   'SE';                                            % group 0018 element 0020
    %%dh.ScanningSequence      =   'IR';                                            % group 0018 element 0020
    dh.SequenceVariant         =   'NONE';                                          % group 0018 element 0021
    dh.ScanOptions             =   'FC\SP';                                         % group 0018 element 0022
    dh.MRAcquisitionType       =   '2D';                                            % group 0018 element 0023
    dh.SequenceName            =   'fcmemp';                                        % group 0018 element 0024
    dh.AngioFlag               =   'Y';                                             % group 0018 element 0025
    dh.SliceThickness          =   5.0;                                             % group 0018 element 0050
    dh.RepetitionTime          =   566.664001;                                      % group 0018 element 0080
    dh.EchoTime                =   20.0;                                            % group 0018 element 0081
    dh.NumberOfAverages        =   1.0;                                             % group 0018 element 0083
    dh.ImagingFrequency        =   63.874856;                                       % group 0018 element 0084
    dh.ImagedNucleus           =   'H1';                                            % group 0018 element 0085
    dh.EchoNumber              =   1;                                               % group 0018 element 0086
    dh.MagneticFieldStrength   =   15000;                                           % group 0018 element 0087
    dh.SpacingBetweenSlices    =   5.0;                                             % group 0018 element 0088
    dh.EchoTrainLength         =   0;                                               % group 0018 element 0091
    dh.PercentSampling         =   100.0;                                           % group 0018 element 0093
    dh.PercentPhaseFieldOfView =   75.0;                                            % group 0018 element 0094
    dh.PixelBandwidth          =   125.0;                                           % group 0018 element 0095
    dh.SoftwareVersion         =   '11';                                            % group 0018 element 1020
    dh.ContrastBolusRoute      =   'IV';                                            % group 0018 element 1040
    dh.SpatialResolution       =   1.145833;                                        % group 0018 element 1050
    dh.NominalInterval         =   566;                                             % group 0018 element 1062
    dh.HeartRate               =   0;                                               % group 0018 element 1088
    dh.CardiacNumberOfImages   =   0;                                               % group 0018 element 1090
    dh.TriggerWindow           =   0;                                               % group 0018 element 1094
    dh.ReconstructionDiameter  =   220.0;                                           % group 0018 element 1100
    dh.ReceiveCoilName         =   'HEAD';                                          % group 0018 element 1250
    dh.TransmitCoilName        =   'HEAD';                                          % group 0018 element 1251
    dh.AcquisitionMatrix       =   [0; 0; 0; 0];                                    % group 0018 element 1310
    dh.FlipAngle               =   90;                                              % group 0018 element 1314
    dh.VariableFlipAngleFlag   =   'N';                                             % group 0018 element 1315
    dh.SAR                     =   0.038486;                                        % group 0018 element 1316
    dh.PatientPosition         =   'HFS';                                           % group 0018 element 5100
    dh.StudyInstanceUID  =   '1.2.840.200000.2.1.1.2702429029.814.936099513.377';   % group 0020 element 000d
    dh.SeriesInstanceUID =   '1.2.840.200000.2.1.1.2702429029.814.936099513.634';   % group 0020 element 000e
    dh.StudyID                 =   '2956707';                                       % group 0020 element 0010
    dh.SeriesNumber            =   6;                                               % group 0020 element 0011
    dh.AcquisitionNumber       =   1;                                               % group 0020 element 0012
    dh.InstanceNumber          =   10;                                              % group 0020 element 0013
    dh.ImagePositionPatient    =   [-115.0; -110.0; -40.0];                         % group 0020 element 0032
    dh.ImageOrientationPatient =   [1.0; 0.0; 0.0; 0.0; 1.0; 0.0];                  % group 0020 element 0037
    %dh.ImagePositionPatient   =   [32.1954; -131.239; 129.482];                    % group 0020 element 0032
    %dh.ImageOrientationPatient=   [0.0; 1.0; 0.0; 0.0; 0.0; -1.0];                 % group 0020 element 0037
    dh.FrameofReferenceUID = '1.2.840.113619.2.1.1.2702429029.809.936099446.28';    % group 0020 element 0052
    dh.ImagesInAcquisition     =   29;                                              % group 0020 element 1002
    dh.PositionReferenceIndicator = 'NA';                                           % group 0020 element 1040
    dh.SliceLocation           =   -40.0;                                           % group 0020 element 1041
    %dh.SliceLocation          =   -32.1954;                                        % group 0020 element 1041
    dh.SamplesPerPixel         =   1;                                               % group 0028 element 0002
    dh.PhotometricInterpretation = 'MONOCHROME2';                                   % group 0028 element 0004
    dh.Rows                    =   256;                                             % group 0028 element 0010
    dh.Columns                 =   256;                                             % group 0028 element 0011
    dh.PixelSpacing            =   [0.8593750000; 0.8593750000];                    % group 0028 element 0030
    dh.PixelRepresentation     =   1;                                               % group 0028 element 0103
    %%
    %% also required by wfu_infofields_dicom
    %% -- BUT, if (0018,0020) scanning sequence does not have value IR
    %% -- then InversionTime should not be present in the image and won't be
    %% -- written by MatLab
    dh.AcquisitionDate         =   dh.StudyDate;                                    % group 0008 element 0022
    dh.InversionTime           =   0.0;                                             % group 0018 element 0082
% % % % end
%
function [fmt] = what_image_format(filename)
%
% if the extension of the filename matches an image format recognized by
% MATLAB, return that format; otherwise set the returned fmt to empty
%   
fmt = [];
formats = imformats;
nformats = length(formats);
[pathstr, name, ext] = fileparts(filename);
volume_ext = strrep(ext, '.', '');
for i = 1:nformats
    nformatsext = length(formats(i).ext);
    for j = 1:nformatsext
        if strcmp(char(formats(i).ext(j)), volume_ext)
            fmt = char(formats(i).ext(j));
            break;
        end
    end
end
