function wfu_site_specific_dicom(fname,hdr)

%   Write .info file for each analyze image produced and 
%   and modify .mat to include wfu-specific fields. 
%   This function is inserted into spm_dicom_convert and does
%   not return a variable
%   FORMAT wfu_site_specific_dicom(fname,hdr)
%
%   fname   - name of current analyze file (function is 
%             executed within a loop of spm_dicom_convert.m) 
%   hdr     - header structure for the first 2D dicom file of
%             the image series
%
%   ##  v2.1, Aaron Baer, Wake Forest University  ##
%____________________________________________________________


%----- extract meaningful dicom fields -----%
infofileText = local_dicom_fields(hdr);

%----- create .info file -----%
infofileName = wfu_str_fix(fname,'se','info');
ok           = wfu_write_infofile(infofileName,infofileText);
if ~ok
		warning('Problems writing .info file...'); 
    return
%----- add extra fields to .mat -----% 
else
	wfu_mat_fields(fname);
end
return



%________________________________________________________

function infoStruct = local_dicom_fields(hdr)

if size(hdr) > 1
    error(sprintf('%s can only accept one header at a time',mfilename)); 
elseif iscell(hdr)
    hdr = hdr{1}; 
end

infoStruct = local_get_dicom_fields(hdr);
infoStruct = local_private_fields(infoStruct,hdr); 
infoStruct = local_clean_fields(infoStruct);

%________________________________________________________

function infoStruct = local_get_dicom_fields(hdr)
%----- pair .info file fields with dicom hdr fields -----% 
infoCells = { ...
				{'StationName',          'StationName'}, ...
        {'ScanDate',             'AcquisitionDate'}, ...
        {'RepetitionTime',       'RepetitionTime'}, ...
        {'EchoTime',             'EchoTime'}, ...
        {'InversionTime',        'InversionTime'}, ...
        {'FlipAngle',            'FlipAngle'}, ...
        {'PulseSeqName',         'PulseSeqName'}, ...
        {'PlaneType',            'PlaneType'}, ...
        {'PatientPosition',      'PatientPosition'}, ...
        {'PatientEntry',         'PatientEntry'}, ...
        {'PatientID',            'PatientID'}, ...
        {'Protocol',             'NotApplicable'}, ...
        {'Xdim',                 'PixelSpacing'}, ...
        {'Ydim',                 'PixelSpacing'}, ...
        {'NumberOfSlices',       'LocationsInAcquisition'}, ...
        {'InstanceNumber',       'InstanceNumber'}, ...
        {'AcquisitionNumber',    'AcquisitionNumber'}, ...
        {'NumberOfVolumes',      '--------------'}, ...
        {'SliceThickness',       'SliceThickness'}, ...
        {'SpacingBetweenSlices', 'SpacingBetweenSlices'}, ...
        {'FrameOfReferenceUID',  'FrameofReferenceUID'}, ...
        {'HorizFrameOfRef',      'HorizFrameOfRef'}, ...
        {'StudyID',              'StudyID'}, ...
        {'SeriesNumber',         'SeriesNumber'}, ...
        {'SeriesDescription',    'SeriesDescription'}, ...
        {'ImagingOptions',       'NotApplicable'}, ...
        {'EpiflexVersion',       'NotApplicable'}, ...
        {'AcquisitionCode',      'NotApplicable'}, ...
        {'AcquisitionName',      'NotApplicable'}, ...
        {'fMRIParadigmNumber',   'NotApplicable'}, ...
        {'Filename',             'Filename'}, ...
        {'PixelSizeX',           'PixelSpacing'}, ...
        {'PixelSizeY',           'PixelSpacing'}, ...
        {'PixelSizeZ',           'SliceThickness'}, ...
    };

%----- create output structure -----%
infoStruct = []; 
for i = 1:length(infoCells)
    outField = infoCells{i}{1};
    hdrField = infoCells{i}{2}; 
    if isfield(hdr,hdrField)
        infoStruct = setfield(infoStruct,outField,getfield(hdr,hdrField));
    else
        if strcmp(hdrField,'NotApplicable')
            infoStruct = setfield(infoStruct,outField,'NotApplicable');
        else
            infoStruct = setfield(infoStruct,outField,'');
        end
    end
end
%____________________________________________________________

function infoStruct = local_private_fields(infoStruct,hdr)
A = fieldnames(infoStruct); 
B = struct2cell(infoStruct); 
for i = 1:length(A)
    if isempty(B{i})
        notFound = 'field_not_available_in_dcm_header';
        switch A{i}
            case 'PulseSeqName'
                if isfield(hdr,'Private_0019_109c')
                    infoStruct.PulseSeqName = char(getfield(hdr,'Private_0019_109c'));
                else infoStruct.PulseSeqName = notFound;
                end
                
            case 'PlaneType'
                if isfield(hdr,'Private_0027_1035')
                    infoStruct.PlaneType = getfield(hdr,'Private_0027_1035');
                    infoStruct.PlaneType = infoStruct.PlaneType(1);
                else infoStruct.PlaneType = notFound;
                end
                
            case 'NumberOfSlices'
                if isfield(hdr,'Private_0021_104f')
                     infoStruct.NumberOfSlices = getfield(hdr,'Private_0021_104f');
                     infoStruct.NumberOfSlices = infoStruct.NumberOfSlices(1);
                else infoStruct.NumberOfSlices = notFound;
                end
                
            case 'NumberOfVolumes'
                %===== perhaps modify this after checking for DTI/4Ds? ==%
                infoStruct.NumberOfVolumes = 1; 
                
            case 'HorizFrameOfRef'
                if isfield(hdr,'Private_0019_100f')
                     infoStruct.HorizFrameOfRef = getfield(hdr,'Private_0019_100f');
										 if length(infoStruct.HorizFrameOfRef) > 1
										   try
										     infoStruct.HorizFrameOfRef = eval(char(infoStruct.HorizFrameOfRef));
										   catch
										     infoStruct.HorizFrameOfRef = notFound;
										   end
										end
                else infoStruct.HorizFrameOfRef = notFound;
                end
        end
    end
end


%____________________________________________________________

function infoStruct = local_clean_fields(infoStruct)

F = fieldnames(infoStruct); 
V = struct2cell(infoStruct);
for i = 1:length(F)
    field = F{i};
    value = V{i};
    if isa(value,'char')
        switch field
            case {'Filename','FrameOfReferenceUID'}
                value = wfu_remove_bad_chars(value,'.','/','_','-');
            case 'PatientPosition'
                value = deblank(value);
                if     strcmp(value,'HFS'),  value = 'head_first_supine'; 
                elseif strcmp(value,'HFP'),  value = 'head_first_prone';
                elseif strcmp(value,'HFDR'), value = 'head_first_decubitus_right';
                elseif strcmp(value,'HFDL'), value = 'head_first_decubitus_left';
                elseif strcmp(value,'FFS'),  value = 'feet_first_supine'; 
                elseif strcmp(value,'FFP'),  value = 'feet_first_prone';
                elseif strcmp(value,'FFDR'), value = 'feet_first_decubitus_right';
                elseif strcmp(value,'FFDL'), value = 'feet_first_decubitus_left';
                else value = 'patient_position_unknown';
                end
            case 'PatientEntry'
                if isempty(value)
                    value = 'patient_entry_unavailable'; 
                end
            otherwise
                value = wfu_remove_bad_chars(value);
        end
        value       = wfu_fill_blanks_with_underscores(value);
        infoStruct  = setfield(infoStruct,field,value);
    elseif isa(value,'double')
        switch field
            case 'ScanDate'
                value = datestr(value,23);
            case {'Xdim','PixelSizeX'}
                value = value(1);
            case {'Ydim','PixelSizeY'}
                value = value(2);
            case 'PlaneType'
                if isequal(value,2),       value = 'ax'; 
                elseif isequal(value,4),   value = 'sag'; 
                elseif isequal(value,8),   value = 'cor';
                elseif isequal(value,16),  value = 'obl'; 
                else,                      value = 'unknown_plane_type';
                end
        end
        infoStruct  = setfield(infoStruct,field,value); 
    else
        warning(sprintf('Class unknown for variable:\n%s\n'));
    end
end
return
