%
% PURPOSE: - To copy an SPM Analyze file from one directory to another.
%            Matlab system command is not used to maintain compatability 
%            across platforms. 
%
% CATEGORY: Analyze Utility
%
% INPUTS:
% 
%    fileName - Cell list of SPM analyze files to move.
%
%    outDirectory - Directory where to move the files.
% 
%
% OUTPUTS: 
%
%    moveOK  - Variable that returns true of false depending on whether merge
%               was successful.
%
% EXAMPLE:
%
%

function varargout = wfu_extract_spm(inFileNames, copyDirNames, volumeIndex, displayFlag, extractEachVolumeFlag )

if nargin < 5 || isempty(extractEachVolumeFlag)
   extractEachVolumeFlag = true; 
end

if nargin < 4 || isempty(displayFlag)
   displayFlag = true; 
end


if nargin < 3 || isempty(volumeIndex)
   volumeIndex = ':'; 
end


wfu_exist_spm;

%
% Convert to Cell if necessary
%

if ~iscell(inFileNames)
    inFileNames = cellstr(inFileNames); 
end

if ~iscell(copyDirNames)
    copyDirNames = cellstr(copyDirNames); 
end

nFileNames    = length(inFileNames);
nCopyDirNames = length(copyDirNames); 

copyOK       = zeros(nCopyDirNames, nFileNames); 
outFileNames = cell(nCopyDirNames, nFileNames); 
    

%
% Create new 4D file
%

nTotalCopyCommands = nCopyDirNames*nFileNames;

for jj=1:nCopyDirNames
    for ii=1:nFileNames
                    
        currentFileName = inFileNames{ii};
        spmHdr = wfu_read_spm_hdr(currentFileName);
        
        if exist(copyDirNames{jj}, 'dir') ==7 % If this is a directory           
            outFileNames{ii,jj} = strcat(copyDirNames{jj},filesep,wfu_fileparts(char(currentFileName),true,true));  
        else
            outFileNames{ii,jj} = copyDirNames{jj};                        
        end
        
        if strcmp(outFileNames{ii,jj}, currentFileName) == true
            warning('Input filename and output filename are the same. Skipping SPM copy');
        end
        
        if displayFlag == true
            disp(sprintf('\n\t Copying %s to \n\t\t\t        %s (%d,%d;%d/%d)\n', ...
                 currentFileName, outFileNames{ii,jj}, jj, ii, (jj-1)*nFileNames + ii, nTotalCopyCommands));           
        end
        

    %
    % Copy image data
    %
    
    [ volumeIndex nVolumeIndex ]     = wfu_create_index_array(volumeIndex, spmHdr.n );

    dimensions    = wfu_get_spm_dimensions(spmHdr);      % Reset dimensions from 4D to 3D
    dimensions(4) = nVolumeIndex;
    
    if ~(exist( outFileNames{ii,jj}, 'file'  ) == 2);

        wfu_init_spm_hdr(spmHdr, outFileNames{ii,jj}, dimensions, true );  %Create Flag = true

        if extractEachVolumeFlag == true
            for kk = 1:nVolumeIndex
                spmVol = wfu_read_spm_img( currentFileName,':', volumeIndex(kk) );
                wfu_write_spm_img(outFileNames{ii,jj}, spmVol, ':', kk);
            end
        else
            spmVol = wfu_read_spm_img( currentFileName,':', volumeIndex );
            wfu_write_spm_img(outFileNames{ii,jj}, spmVol, ':', 1:nVolumeIndex);
        end

    else

        warning('Skipping extract: %s already exists.', outFileNames{ii,jj});
        
    end
    
    %
    % Copy Mat file if it exists
    %

    currentFileName = wfu_get_spm_filename(currentFileName,'mat');
    outFileName =  wfu_get_spm_filename(outFileNames{ii,jj},'mat');
    
    if exist( currentFileName, 'file'  );

        spmMatVariables = load(char(wfu_get_spm_filename( currentFileName,'mat')));
        spmFieldNames   = fieldnames(spmMatVariables);
        
        for kk=1:length(spmFieldNames)
            currentFieldName = char(spmFieldNames{kk});
    
            evalCommand = sprintf('%s = spmMatVariables.%s;',currentFieldName, currentFieldName);
            eval(evalCommand);            
            
            evalCommand = sprintf('wfu_save_mat(''%s'',''%s'');', outFileName, currentFieldName);
            eval(evalCommand);
        end    
    end

    
    %
    % Copy DICOM Info file if it exists
    %

    currentFileName = wfu_get_spm_filename(currentFileName,'dicom');
    outFileName     = wfu_get_spm_filename(outFileNames{ii,jj},'dicom');
    
    if exist( currentFileName, 'file' );

        wfu_copy_dicom(currentFileName, outFileName);

    end

         
    copyOK(ii,jj) = true;
    
    end
    
end

%
% Return Output if requested
%

if nargout > 0
    varargout{1}  = outFileNames;    
end

if nargout > 1
    varargout{2}  = copyOK;    
end
