function  varargout = wfu_init_spm_hdr(spmStructure, inFileName,dimensions, createFlag, dataType, volumeIndex )
%
% PURPOSE: Generates an SPM structure and writes the SPM img file if desired.
%
% CATEGORY: SPM IO 
%
% INPUTS: 
%
%   spmStruture - SPM header structure.  If empty than a SPM header
%                 structure is created from the remaining inputs. If the structure is not
%                 empty than its contents are changed based upon the remaining inputs.
%
%   inFileName - Filename for the SPM header. The IMG extension is added to the name if the extension is no
%                'hdr' or 'img'
% 
%
% KEYWORD PARAMETERS:
%
%   dimensions - Dimensions of the SPM header (Xdim, Ydim, Zdim, tDim). If a more than three dimension are
%                specified SPMs third dimensions is set to prod(dimensions(3:end))
%
%   createFlag -
%
%   dataType   - Flag
%
%   volumeIndex - The volume index of the SPM hdr structure.  Default is
%                 ':'.
%
%
%   OTHER:
%
%   defaultSpmStructure is the starting SPM structure used with spmHdr is not passed
%   into the function.
%
%   defaultSpmStructure.mat        = eye(4);
%   defaultSpmStructure.dataType   = 'int16';
%   defaultSpmStructure.dim        = dimensions;
%   defaultSpmStructure.descrip    = '80 character note field';
%   defaultSpmStructure.pinfo      = [1; 0; 0];
%   defaultSpmStructure.n          = 1;
%
% OUTPUTS: 
%
%   SPM Volume Structure - an SPM header struture.
%
% EXAMPLE:
%
%    spmVolumeStructure  = wfu_init_spm_hdr('test', [64 64] )
%

%
% Check if SPM exists
%

[spmExist spmVer ] = wfu_exist_spm;

%
% Initialize input variables
%

if nargin < 6 
    volumeIndex = ':';
end

if nargin < 5
    dataType = [];
end

if nargin < 4 || isempty(createFlag)   
    createFlag = 'noopen';
end

if nargin < 3 
    dimensions = [];
end


if ~isempty(dimensions)
   dimensions = [dimensions  ones(1,4-length(dimensions))];
end

if nargin < 2
   inFileName = [];  
end

if nargin < 1
   spmStructure = [];  
end


%
% Select which SPM is running
%

if strcmpi(spmVer,'spm2')

    spmHdr = local_init_spm2_hdr(spmStructure, inFileName,dimensions, createFlag, dataType, volumeIndex );
    
else
    
    spmHdr = local_init_spm5_hdr(spmStructure, inFileName,dimensions, createFlag, dataType, volumeIndex );
   
end

%
% Return Output Header 
%

if nargout > 0
    varargout{1} = spmHdr;
end

return






function  spmHdr = local_init_spm2_hdr(spmStructure, inFileName,dimensions, createFlag, dataType, volumeIndex )

%
% Initialize default SPM structure
%

defaultSpmStructure.mat        = eye(4);

maxDim                    = 3;  % SPM file types only support three dimensions (trick to 4D by extending 3D)

defaultDimensions = [64 64 1 1];
defaultDataType   = 'int16';

    
    defaultSpmStructure.dim   = [defaultDimensions(1) defaultDimensions(2) defaultDimensions(3) ...
                                   spm_type(defaultDataType)]; % Set unspecified dimensions to 1[64 64 1 1];
    defaultSpmStructure.n          = 1;

                              
defaultSpmStructure.descrip    = '80 character note field';
defaultSpmStructure.pinfo      = [1; 0; 0];

%
% Initialize spmStructure and force fname to Analyze Img File Name
%

if ~isempty(inFileName)

    if iscell(inFileName)
        inFileName = inFileName{1};
    end

    spmStructure.fname = wfu_get_spm_filename(inFileName,'img');

else
    spmStructure.fname = '';    
end

%
% Reset Dimensions
%
if ~isempty(dimensions)

    spmStructure.dim(1) = dimensions(1);
    spmStructure.dim(2) = dimensions(2);
    spmStructure.dim(3) = dimensions(3);


        spmStructure.n   = dimensions(4);

        %
        % Reset dataType
        %

        if ~isempty(dataType)
            
            if strfind(dataType,'float')
                dataType = 'float';
            end
            
            spmStructure.dim(4) = spm_type(dataType);
        end

        if length(spmStructure.dim) < 4
           spmStructure.dim(4) = defaultSpmStructure.dim(4);    
        end

end



spmStructure       = wfu_set_function_options(spmStructure, defaultSpmStructure);

    spmStructure.private.hdr.dime.dim(1:5) = [spmStructure.dim(4) spmStructure.dim(1:3) spmStructure.n];
%
% Only set fileName 
%
[fileDir shortFileName fileExt] = fileparts(spmStructure.fname);
shortFileName                   = [shortFileName fileExt];

if ~isempty(inFileName)  
    spmStructure.fname = char(inFileName);
end

%
% Create SPM header file
%
%
%  The for loop is necessary so that each entry in the mat file is the
%  same for 4D files.

[ volumeIndex nVolumeIndex ]     = wfu_create_index_array(volumeIndex, spmStructure.n(1)  );

for ii=1:nVolumeIndex
    spmStructure.n     = volumeIndex(ii);
    spmHdr = spm_create_vol(spmStructure, createFlag );
end

%
% Create the image file
%

if ~strcmpi(createFlag,'noopen')
    wfu_write_spm_img( spmStructure.fname );
end

%
% Close SPM image volume
%

    spm_close_vol(spmHdr);
 
return




function  spmHdr = local_init_spm5_hdr(spmStructure, inFileName,dimensions, createFlag, dataType, volumeIndex )

%
% Kathy's code from /ansir1/kpearson/spm5io/step2.m
%
% %% create a new nifti file with inputs from calls to MATLAB DICOM functions
% % -- first, set up the SPM5 header
% nz = size(names,1);
% fname = 'nifti/tmp.img';
% dim = [256 256 nz];
% dt = [spm_type('int16') spm_platform('bigend')];
% pinfo = [1; 0; 0];
% descrip = '80 char note field';
% V2 = struct('fname', fname, 'dim', dim, 'dt', dt, 'mat', dicommat, 'pinfo', pinfo, 'descrip', descrip);
% V2 = spm_create_vol(V2, 'noopen');

% Initialize default SPM structure
%

defaultSpmStructure.mat   = eye(4);

maxDim                    = 3;  % SPM file types only support three dimensions (trick to 4D by extending 3D)

defaultDimensions = [64 64 1];

defaultDataType                = 'int16';
defaultSpmStructure.dim        = [defaultDimensions(1) defaultDimensions(2) defaultDimensions(3) ]; % Set unspecified dimensions to 1[64 64 1 1];    defaultSpmStructure.n     = [1 1 ];
defaultSpmStructure.dt         = [spm_type(defaultDataType) spm_platform('bigend')];
                              
defaultSpmStructure.descrip    = '80 character note field';
defaultSpmStructure.pinfo      = [1; 0; 0];

%
% Initialize spmStructure and force fname to Analyze Img File Name
%

if ~isempty(inFileName)

    if iscell(inFileName)
        inFileName = inFileName{1};
    end

    spmStructure.fname = wfu_get_analyze_filename(inFileName,'img');

else
    spmStructure.fname = '';    
end

%
% Reset Dimensions
%
if ~isempty(dimensions)

    spmStructure.dim(1) = dimensions(1);
    spmStructure.dim(2) = dimensions(2);
    spmStructure.dim(3) = dimensions(3);

    spmStructure.n      = [dimensions(4) 1];
    
    %
    % Reset dataType
    %

    if ~isempty(dataType)

        if strfind(dataType,'float')
            dataType = 'float32';
        end

        spmStructure.dt = [ spm_type(dataType) spm_platform('bigend')];
    end
end

spmStructure       = wfu_set_function_options(spmStructure, defaultSpmStructure);

%
% Only set fileName 
%

[fileDir shortFileName fileExt] = fileparts(spmStructure.fname);
shortFileName                   = [shortFileName fileExt];

if ~isempty(inFileName)  
    spmStructure.fname = char(inFileName);
end

spmStructure.fname = wfu_get_spm_filename(spmStructure.fname,'img');


%
% Create SPM header file
%
%
%  The for loop is necessary so that each entry in the mat file is the
%  same for 4D files.

%spmHdr = spm_create_vol(spmStructure, 'noopen');
spmHdr = spm_create_vol(spmStructure, createFlag);

%spm5 doesn't create the .img files automatically  this fixes that:
if ~strcmpi(createFlag,'noopen')
    fid = fopen(spmStructure.fname,'w');
    fclose(fid);
end

%
% Create the image file
%

if ~strcmp(lower(createFlag),'noopen')
    wfu_write_spm_img( spmStructure.fname );
end
