function [hdr, spmHdr, otherEndian] = wfu_read_spm_hdr(fileName, volumeIndex)
%
% PURPOSE: Wrapper function to write a SPM header structure to a file.
%
% CATEGORY: SPM IO 
%
% INPUTS: 
%
%   fileName    - SPM  file name to read. (May be the img or hdr file).
%   volumeIndex - Volume index to read.  Default is 'end'
%
% OUTPUTS: 
%
%   hdr         - structure containing Analyze header
%   spmHdr      - Unadultered SPM Header
%   otherendian - byte swapping necessary flag
%
% EXAMPLE:
%
%    [ hdr otherEndian ] = wfu_read_spm_hdr( 'test' )
%

%
% Check if SPM exists
%

[spmFlag spmVer] = wfu_exist_spm;


% 
% Error only supports SPM02 or greater
%

if strcmpi('spm99',spmVer)
    error('This function does not support %s',spmVer);
end

%
%
%

if ~iscell(fileName)
    fileName = {fileName};
end

nFileName = length(fileName);

spmHdr      = cell(1,nFileName);
hdr         = cell(1,nFileName);
otherEndian = cell(1,nFileName);

%
% 
%

if nargin < 2 || isempty(volumeIndex)
    volumeIndex = 'end';
end

%
% Loop through all files
%

for ii=1:nFileName

    hdrFileName = wfu_get_spm_filename(fileName{ii}, 'hdr');
    imgFileName = wfu_get_spm_filename(fileName{ii}, 'img');

    [tmpSpmHdr, tmpOtherEndian] = spm_read_hdr(hdrFileName);

    tmpHdr = spm_vol(imgFileName);
    %
    % Volume index
    %

    nVolumes = tmpSpmHdr.dime.dim(5);
    
    [ tmpVolumeIndex nVolumeIndex ]  = wfu_create_index_array(volumeIndex, nVolumes );
    
    switch lower(spmVer)
        
        case 'spm5'
            
            hdr{ii}     = tmpHdr(tmpVolumeIndex);  % Only return the headers requested.
            
        otherwise
            
            tmpHdr.n(1) = tmpVolumeIndex(1);                    
            hdr{ii}     = tmpHdr;
            
    end

    spmHdr{ii}      = tmpSpmHdr;
    otherEndian{ii} = tmpOtherEndian;
    
end

%
% Remove Cell if only one file is read
%

if nFileName==1    
    spmHdr = spmHdr{1};   
    hdr    = hdr{1};
    otherEndian = otherEndian{1};
end
