function [spmVol, spmHdr] = wfu_read_spm_img(fileNameCell, sliceIndex, volumeIndex, options) 

%
% PURPOSE: - To read EPI raw data to complex k-space array.  Keyword
%            parameter timeReverseFlag determines if every other line of 
%            k-space is reversed
%
% CATEGORY: EPIFLEX Reconstruction
%
% INPUTS:
%
%    fileName  - Name of the EPIFLEX p file to be read 
%
%    sliceIndex  - Index of slices to be read
%
%    volumeIndex - Index of volumes to be read
%
% OPTIONS:
%
%    options.channelsPerVolume = Represents the number of channels acquired per volume. For example,
%                                Magnitude, Phase, Real, and Imaginginary images (or a combination)
%                                can be stored for each volume. SPM images may be 4 dimensional. The
%                                channels are interleaved in the order that they were acquired. For example, 
%                                if the Magnitude, Real, and Imaginary channels were stored
%                                then there would be 3 channels per volume. By setting
%                                options.channlesPerVolume to 3 the array is reshaped to store the
%                                channels in the 5th dimension.
%
%
% OUTPUTS: 
%
%    imageStack - A 2D, 3D, or 4D array containing the complex EPI k-space.  
%
%    V          - SPM volume structure
%
% EXAMPLE:
%
%
%    >> volumeStack = wfu_read_spm_vol('analyze.hdr',':',1)
%    >> imagesc(volumeStack(:,:,14))                      % Display slice 14 of volume stack
%
%  - Read slice 14 across all volumes and return a slice stack
%
%    sliceStack = wfu_read_spm_vol('analyze.img',14,':')
%    movie( im2frame(sliceStack))
%
%  - Read in a 4D data set (64 64 11 13] with Magnitude, Real, and Imaginary channels
%  stored for each volume
%
%   >> options.channlesPerVolume = 1
%   >> data = wfu_read_spm_img('analyze.img',[],[],options);
%   >> size(data)
%
%       ans = [64 64 11 39];
%
%
%   >> options.channelsPerVolume = 3;
%   >> data = wfu_read_spm_img('analyze.img',[],[],options);
%   >> size(data)
%
%       ans = [64 64 11 13 3];
%
%==========================================================================
% C H A N G E   L O G
% 
%--------------------------------------------------------------------------

%
% Initialize Inputs
%

defaultOptions.channelsPerVolume = 1;

if nargin < 4	
    options = [];
end

options = wfu_set_function_options(options,defaultOptions);


if (nargin < 3) || isempty(volumeIndex)
    volumeIndex = ':';
end

if (nargin < 2) || isempty(sliceIndex)
    sliceIndex = ':';
end

if ~iscell(fileNameCell)
    fileNameCell = {fileNameCell};
end

nFileNameCell = length(fileNameCell);
spmVol  = cell(1,nFileNameCell);
spmHdr  = cell(1,nFileNameCell);


for ii=1:nFileNameCell

    fileName = wfu_get_spm_filename(char(fileNameCell{ii}),'img');

    [spmVol{ii}, spmHdr{ii}] = local_read_spm_img(fileName, sliceIndex, volumeIndex, options.channelsPerVolume);

end

%
% Remove Cell structure if only one file name is given
%

if nFileNameCell == 1
   spmVol = spmVol{1};
   spmHdr = spmHdr{1};
end




function [spmVol, spmHdr] = local_read_spm_img(fileName, sliceIndex, volumeIndex,  channelsPerVolume)

%
% Read Analyze Header
%

spmHdr = wfu_read_spm_hdr(fileName);  % Get SPM header structure for first volume

%
% Interpret header structure and put into easy to remember variables.
%

[nXres nYres nSliceMax nVolumeMax  ] = wfu_get_spm_dimensions(spmHdr);

%
%
%

nVolumeMax = nVolumeMax / channelsPerVolume;

%
% Create slice and volume index arrays
%

[ sliceIndex  nSliceIndex  ] = wfu_create_index_array(sliceIndex,  nSliceMax );
[ volumeIndex nVolumeIndex ] = wfu_create_index_array(volumeIndex, nVolumeMax );

%
% Increase Volume Index by Volume Multiplier
%
offsetVolumeIndex = (channelsPerVolume-1)*(0:nVolumeMax);
offsetVolumeIndex = offsetVolumeIndex(volumeIndex);

volumeIndex  = volumeIndex  + offsetVolumeIndex;

%
% Initialize spmVol to zero
%

spmVol = zeros(nXres, nYres, nSliceIndex, nVolumeIndex, channelsPerVolume);


for ii=1:channelsPerVolume

    iiVolumeIndex = volumeIndex + (ii-1);
    
    switch lower(spm('ver'))
              
        case 'spm5'

            V = wfu_read_spm_hdr(fileName, iiVolumeIndex);   % Get SPM header structure for a specific volume
            Y = spm_read_vols(V);                          % Read all slices for specified volume

            spmVol(:,:,:,:,ii) = Y(:,:,sliceIndex,:);  % Grab only the slices that are requested by slice Index

        otherwise

            for jj=1:nVolumeIndex

                spmFileName = sprintf('%s,%d', fileName, iiVolumeIndex(jj));  % Appending on a

                V = spm_vol(spmFileName);   % Get SPM header structure for a specific volume
                Y = spm_read_vols(V);       % Read all slices for specified volume

                spm_close_vol(V);        % Close SPM img file

                spmVol(:,:,:,jj,ii) = Y(:,:,sliceIndex);  % Grab only the slices that are requested by slice Index
            end
    end
end
%
% Bounds checking
%

spmVol(isnan(spmVol)) = 0;
spmVol(isinf(spmVol)) = 0;