function [mat, landmark, M, vx, origin] = wfu_get_magnet_transform(fileNameOrHeader, matrixSize, fieldOfView, coordSystem )

%--------------------------------------------------------
% Function: wfu_magnet_transform
%
% Purpose:  Calculates the magnet transform for a volume in a P-file.
%           Performs a y flip at the end (for rotate 7) images will need to be rotated separately
%
% Input:    pfileName   = name of Pfile to use, or '' if using image file.
%
%           matrixSize  = Matrix size of final reconstructed image along X and Y. If not specified field of view is read from
%                         p file header.
%
%           fieldOfView = field of view along X and Y.  If not specified field of view is read from
%                         p file header.
%
%           coordSystem = 'spm' or 'other'.  The two coordinate systems are either
%                         SPM or something else.  The something else was to calculate the
%                         scanner coordinate system correctly. Default is 'spm'
%
% Returns:  mt       = 4x4 magnet transform matrix
%           landmark = landmark value read directly from series header
%           M        =
%           vx       =
%           origin   = Origin of image.  If coordinate system is SPM then image origin is interms of 
%                      integer pixels. If coordinate system is something else origin is give with respect
%                      to the scanner isocenter.
%
% Note:     on error, will return mt=[], landmark=[]
%
% Examples of usage:
%
%           [mt,landmark] = wfu_magnet_transform('pilot01.09_48_19.06_06_04.P02560'); % using a pfile
%
%==========================================================================
% C H A N G E   L O G
% JAM  10/01/04	    mat is the magnet transform, M takes into account flipping in SPM2
% RAK  05/20/05     Added flag to change scanner coordinate system of origin.  Default is 'spm'
%------------------------------------------------------------------------
mt=[]; landmark=[]; % in case get out early

%------------------------------------------------------------
% If user supplied a pfile name, then wants to read a pfile:
%------------------------------------------------------------
if ~isempty(fileNameOrHeader)

    if iscell(fileNameOrHeader)
        fileNameOrHeader = fileNameOrHeader{1};
    end
    
    if ischar(fileNameOrHeader)
        % Read in the pfile headers:
        hdr = wfu_get_p_header(fileNameOrHeader);
    else
        hdr = fileNameOrHeader;
    end
        
    nSlices = hdr.NumberOfSlices;
else
    %----------------------
    % Else bad calling args:
    %----------------------
    error('Invalid calling arg(s). Syntax: wfu_get_magnet_transform(''pfileName'');');
end


if nargin < 2 || isempty(matrixSize)
    matrixSize = [hdr.ImageYdim hdr.ImageXdim ];
end

if nargin < 3 || isempty(fieldOfView)
    fieldOfView      = [ hdr.FieldOfView  nSlices*hdr.SliceThickness];
    fieldOfView(1:2) = fliplr(fieldOfView(1:2));
else

    if length(fieldOfView) == 2
        fieldOfView = [fieldOfView  nSlices*hdr.SliceThickness ];
    end
end

if length(fieldOfView) ~= 3
    error( 'wfu_get_magnet_transform: FOV must specified in 3 dimensions not %d', length(fieldOfView));   
end


if nargin < 4 || isempty(coordSystem)
    coordSystem = 'scanner';
end

if strcmpi(coordSystem, 'oldspm')
    coordOffset =   1;  % Originally 1
else
    coordOffset =  -1;  % Scanner coordinates.  The pre SPM5 the origin was shifted by 2 pixels.
                        % This has been changed in SPM5.  Kathy Pearson updated our version of
                        % SPM 2 to be compatible with SPM5.
end


%------------------------------------------
% Get landmark, tlhc, trhc and brhc values 
%------------------------------------------
landmark = hdr.Landmark;

tlhcScanner = hdr.TopLeftHandCorner;      % Top left hand corner
trhcScanner = hdr.TopRightHandCorner;     % Top right hand corner
brhcScanner = hdr.BottomRightHandCorner;  % Bottom right hand corner

norm     = hdr.RasNorm;                   % Norm
	
%
% Determine Phase and Frequency Direction and force Phase to always
% be along the first dimension
% 

diffLR = tlhcScanner - trhcScanner;
diffTB = trhcScanner - brhcScanner;

scannerFieldOfView = [ sqrt(diffLR'*diffLR) sqrt(diffTB' * diffTB) ];

%scannerFieldOfView = fieldOfView;

% Force Phase to always along the first dimension

nPixels = [ matrixSize(1:2) nSlices]; % Slice direction is 1 because each image is 2D

pixelSize = fieldOfView ./ nPixels;

%
% The corner points provided in the GE header file are the corner points after
% the image is reconstructed according to their reconstruction code. In particular,
% zero filling and zero padding are included in the corner points. 
%
% These corner points are accurate for square field of views but not for asymmetric
% field of views or changes in the matrix resolution.
%

differenceFieldOfView = (scannerFieldOfView(1:2) - fieldOfView(1:2)).';

tlhc      = tlhcScanner;
tlhc(1:2) = tlhc(1:2) - differenceFieldOfView(1:2)/2;

trhc      = trhcScanner;
trhc(1:2) = trhc(1:2) - differenceFieldOfView(1:2)/2;

brhc      = brhcScanner;
brhc(1:2) = brhc(1:2) - differenceFieldOfView(1:2)/2;


%--------------------------------------------------
% SPM mat computation based on DICOM image orientation patient and image position patient
%---------------------------------------------------

% IPP image position patient position of top left corner in radiological format
% IOP image orientation patient   direction cosines of the 
% norms with respect to each of the x,y, and z directions  norm always points towards their feet.


[ipp, iop] = wfu_ge2dcm(tlhc, trhc, brhc, norm, pixelSize(1), pixelSize(2));
[mat]      = wfu_spm_mt(nPixels(1), nPixels(2), iop,ipp, pixelSize(3), pixelSize(1:2), coordOffset);

% Orientation etc...
%---------------------------------------------------------------------------------
%The matrix is already right-left flipped
%mat is the native matrix
%M is what SPM99 will interpret
%
%So if spm_flip_analyze_images is 0 then flip it back
%---------------------------------------------------------------------------------
M = mat;
M = diag([-1 1 1 1])*M; %M is + for spm99

flipper=0;

if exist('spm','file') == 2
    if strcmp(spm('Ver'),'SPM2')
        if ~spm_flip_analyze_images
            flipper=1;
        end
        if flipper
            mat = M;
        end
    end
end

vx = sqrt(sum(M(1:3,1:3).^2));
if det(M(1:3,1:3))<0, vx(1) = -vx(1); end;
origin = M\[0 0 0 1]';

origin = origin(1:3);

if strcmp(lower(coordSystem), 'spm')
    origin = round(origin);
else
    origin = origin';
    origin = (origin-(nPixels + [1 1 1])/2) .* pixelSize;;    
    origin = origin';    
end


return;

