function varargout = wfu_get_mri_info(fileNames)
%
% PURPOSE: - Reads the header of a GE DICOM file or P file.  If the header is created by the EPIFLEX psd then the header information is 
%            interpreted when possible.
%
% INPUTS:
%
%       fileNames - Cell list of filenames
%
%
% OUTPUTS: 
%
%       mriInfo   - Structure containing selected information read in from the input file.
%       rawHdr    - Structure containing the raw header. This information is only returned when requested.
%
% EXAMPLE:
%
%      >> wfu_lsa
%
%      >> mriInfo = wfu_get_mri_info(a(1));
%
%==========================================================================
% C H A N G E   L O G
%
%--------------------------------------------------------------------------

if nargin < 1
   options.displayFiles = false;
   fileNames = wfu_find_pfiles([],[],options); 
end

if ~iscell(fileNames)
    fileNames = {fileNames};
end

%
% Loop over all file names
%

nFileNames = length(fileNames);

mriInfoCell  = cell(1,nFileNames);
rawInfo      = cell(1,nFileNames);

for ii=1:nFileNames

    if wfu_is_epiflex_pfile(fileNames{ii})

        mriInfoCell{ii} = wfu_get_epiflex_header(fileNames{ii});

    elseif wfu_is_p_file(fileNames{ii})

        mriInfoCell{ii} = wfu_get_p_header(fileNames{ii});

    else

        [ fileDir fileName fileExt ] = wfu_fileparts(fileNames{ii});
        
        if( strcmpi(fileExt(2:end), 'dcm'))
            
            % This image is a raw DICOM image from the scanner. The filename does not need to be manipulated
            
            dicomFileName = fileNames{ii};
            dicomExtension = 'dcm';
            
        else
            
            % Look for corresponding dicom image associated with filename.
            dicomFileName = wfu_get_spm_filename(fileNames{ii},'dicom');
            dicomExtension = 'dicom';
        end
        
        if exist(dicomFileName,'file')==2
            
            try
                [mriInfoCell{ii} rawInfo{ii}] = local_get_dicom_mri_info(dicomFileName, dicomExtension );
            catch
                disp(sprintf('%d) %s', ii, dicomFileName));
            end
            
        else

            %dicomFileName = fileNames{ii};

            %if exist(dicomFileName,'file')==2

            %    [mriInfoCell{ii} rawInfo{ii}] = local_get_dicom_mri_info(dicomFileName, []);

            %end

        end
    end
end


%
% Write Output (not implemented yet)
%

if nargout > 0
    if nFileNames == 1
        varargout{1} = mriInfoCell{1};
    else
        varargout{1} = mriInfoCell;
    end          

    if nargout > 1
        if nFileNames == 1
            varargout{2} = rawInfo{1};    
        else
            varargout{2} = rawInfo;        
        end          
    end
else
    % If no output is requested show structure
    wfu_show_structure(mriInfoCell);
end







function [mriInfo, dicomInfo] = local_get_dicom_mri_info(fileName, dicomExtension)

if nargin < 2
    dicomExtension = 'dicom';
end

v             = ver('Matlab');
versionMatlab = str2double(v.Version(1));



if strcmpi(dicomExtension, 'dicom')
    dicomFileName = wfu_get_spm_filename(fileName, 'dicom');
else
    dicomFileName = fileName;
end

dicomInfo     = [];

%
% Initialize MRI Info Structure
%
mriInfo.MRN                  = 'DICOM Unknown';
mriInfo.PatientName          = 'DICOM Unknown';
mriInfo.MRN                  = 'DICOM Unknown';
mriInfo.PatientID            = 'DICOM Unknown';  
mriInfo.PatientWeight        = 'DICOM Unknown';
mriInfo.PatientAge           = 'DICOM Unknown';
mriInfo.Technologist         = 'DICOM Unknown';  
mriInfo.AcquisitionDate      = 'DICOM Unknown';
mriInfo.AcquisitionTime      = 'DICOM Unknown';
mriInfo.AcquisitionDuration  = 'DICOM Unknown';

mriInfo.PatientPosition      = 'DICOM Unknown';  
mriInfo.PatientEntry         = 'DICOM Unknown';  
mriInfo.Protocol             = 'DICOM Unknown';

mriInfo.Scanner                = 'DICOM Unknown';
mriInfo.MagneticFieldStrength  = 'DICOM Unknown';

mriInfo.StudyID              = 'DICOM Unknown';  
mriInfo.SeriesNumber         = 'DICOM Unknown';     
mriInfo.SeriesDescription    = 'DICOM Unknown'; 
mriInfo.ReceiveCoilName      = 'DICOM Unknown'; 

mriInfo.TG                   = 'DICOM Unknown';
    
mriInfo.RepetitionTime       = 'DICOM Unknown';
mriInfo.EchoTime             = 'DICOM Unknown';  
mriInfo.InversionTime        = 'DICOM Unknown';  
mriInfo.PulseSeqName         = 'DICOM Unknown';  
mriInfo.InternalPulseSeqName = 'DICOM Unknown';  
mriInfo.PlaneType            = 'DICOM Unknown';    

mriInfo.SliceThickness       = 'DICOM Unknown';  
mriInfo.SpacingBetweenSlices = 'DICOM Unknown';  
mriInfo.FrameOfReferenceUID  = 'DICOM Unknown';  
mriInfo.HorizFrameOfRef      = 'DICOM Unknown';    
mriInfo.FlipAngle            = 'DICOM Unknown';
mriInfo.PixelBandwidth       = 'DICOM Unknown';
mriInfo.FileName             = 'DICOM Unknown';         
mriInfo.FileType             = 'DICOM Unknown';         
mriInfo.FileFormat           = 'DICOM Unknown';         
mriInfo.Xdim                 = 'DICOM Unknown';   
mriInfo.Ydim                 = 'DICOM Unknown';             
mriInfo.NumberOfSlices       = 'DICOM Unknown';   
mriInfo.NumberOfVolumes      = 'DICOM Unknown';    
mriInfo.nPassReps            = 'DICOM Unknown';  
mriInfo.PixelSizeX           = 'DICOM Unknown';     
mriInfo.PixelSizeY           = 'DICOM Unknown';   
mriInfo.PixelSizeZ           = 'DICOM Unknown'; 
        
mriInfo.opUserCvs            = 'DICOM Unknown';


mriInfo.EpiflexVersion       = '';  
mriInfo.AcquisitionCode      = '';    
mriInfo.AcquisitionName      = '';  
mriInfo.fMRIParadigmNumber   = '';    
        
%
% Set MRI Info from DICOM Header
%

if exist(dicomFileName, 'file') == 2   
    
    dicomInfo = dicominfo(fileName);
    
    [tmpFileDir tmpFileName fileExt ] = wfu_fileparts(fileName);
    
    
    mriInfo.FileDir                = tmpFileDir;
    mriInfo.FileName               = local_get_dicom_field(dicomInfo,'Filename');
    mriInfo.FileType               = fileExt(2:end);
    mriInfo.FileFormat             = fileExt(2:end);

    %
    % This subject directory needs a little more help
    %
    filesepPosition                = findstr(mriInfo.FileName,filesep);
    
    if ~isempty(filesepPosition) && ((filesepPosition(end)-1)>0)
        mriInfo.SubjectDirectory       = mriInfo.FileName(1:(filesepPosition(end)-1));
    else
        mriInfo.SubjectDirectory       = [];
    end

    mriInfo.AcquisitionDate        = local_get_dicom_field(dicomInfo,'AcquisitionDate');
    mriInfo.AcquisitionTime        = local_get_dicom_field(dicomInfo,'AcquisitionTime');   
    mriInfo.ScanDuration           = local_get_dicom_field(dicomInfo,'Private_0019_105a')/100000;  % Convert to Seconds
    mriInfo.Format                 = local_get_dicom_field(dicomInfo,';Format');
    mriInfo.Scanner                = local_get_dicom_field(dicomInfo,'StationName');
    mriInfo.MagneticFieldStrength  = dicomInfo.MagneticFieldStrength;
    mriInfo.SoftwareVersion        = local_get_dicom_field(dicomInfo,'SoftwareVersion');
    mriInfo.ReceiveCoilName        = local_get_dicom_field(dicomInfo,'ReceiveCoilName');
    mriInfo.StudyID                = local_get_dicom_field(dicomInfo,'StudyID');
    mriInfo.PatientID              = local_get_dicom_field(dicomInfo,'PatientID');
    mriInfo.MRN                    = local_get_dicom_field(dicomInfo,'PatientID');     % MRN and Patient ID are the same thing. It is included here to match
                                                                                       % WFU naming convention  
    mriInfo.PatientWeight          = local_get_dicom_field(dicomInfo,'PatientWeight');


    %
    % A bug exists in the dicom dictionary for Matlab6 where the dictionary has
    % PatientsName for Matlab6 and PatientName for Matlab7.  Instead of
    % changing the dictionary and having unintended consequences fixing the bug
    % here.
    %
   
    if (versionMatlab < 7)
        mriInfo.PatientName            = local_get_dicom_field(local_get_dicom_field(dicomInfo, 'PatientsName'),'FamilyName');
    else
        mriInfo.PatientName            = local_get_dicom_field(local_get_dicom_field(dicomInfo, 'PatientName'),'FamilyName');
    end

    mriInfo.PatientAge             = local_get_dicom_field(dicomInfo,'PatientAge');
    mriInfo.PatientSex             = local_get_dicom_field(dicomInfo,'PatientSex');
    mriInfo.PatientWeight          = local_get_dicom_field(dicomInfo,'PatientWeight');
    
    mriInfo.ProtocolName           = local_get_dicom_field(dicomInfo,'ProtocolName');
    mriInfo.Protocol               = local_get_dicom_field(dicomInfo,'ProtocolName');

    mriInfo.Technologist           = local_get_dicom_field( local_get_dicom_field(dicomInfo, 'OperatorName'), 'FamilyName');
 
    mriInfo.AccessionNumber        = local_get_dicom_field(dicomInfo,'AccessionNumber');
    
%         entry = { ...
%      'head_first', ...  HS
%      'feet_first', ...  FS
%      'axial_entry', ...
%      'side_entry', ...
%      'vertical_entry', ...
%       'reserved', ...
%      'head_first_plus25degrees', ...
%      'head_first_minus25degrees', ...
%      'feet_first_plus25degrees', ...
%      'feet_first_minus25degrees' ...
%     };

    mriInfo.PatientPosition        = 1;
    
%        position = { ...
%          'supine', ...
%          'prone', ...
%          'left', ...
%          'right' ...
%      };

    mriInfo.PatientEntry           = 1;

    mriInfo.PulseSeqName           = local_get_dicom_field(dicomInfo, 'Private_0019_109c');    
    mriInfo.InternalPulseSeqName   = local_get_dicom_field(dicomInfo, 'Private_0019_109e');    
    mriInfo.PlaneType              = local_get_dicom_field(dicomInfo, 'Private_0027_1035');

     switch mriInfo.PlaneType
         case  2
             mriInfo.PlaneTypeString            = 'Axial';
         case  4
             mriInfo.PlaneTypeString            = 'Coronal';
         case  8
             mriInfo.PlaneTypeString            = 'Sagittal';
         case 16
             mriInfo.PlaneTypeString            = 'Oblique';
         otherwise
             mriInfo.PlaneTypeString            = 'Unknown';
     end

    mriInfo.SeriesNumber           = local_get_dicom_field(dicomInfo,'SeriesNumber');
    mriInfo.SeriesDescription      = local_get_dicom_field(dicomInfo,'SeriesDescription');

    mriInfo.RepetitionTime         = local_get_dicom_field(dicomInfo,'RepetitionTime');
    mriInfo.EchoTime               = local_get_dicom_field(dicomInfo,'EchoTime');
   
    mriInfo.InversionTime          = local_get_dicom_field(dicomInfo,'InversionTime');
    
    mriInfo.FlipAngle              = local_get_dicom_field(dicomInfo,'FlipAngle');
  
    mriInfo.PixelBandwidth         = local_get_dicom_field(dicomInfo,'PixelBandwidth');
    
    
    mriInfo.ReconstructionDiameter = local_get_dicom_field(dicomInfo,'ReconstructionDiameter');   % RAK - I don't remember what this is or 
                                                                                                  %       why it is being read. 

    tmp                            = local_get_dicom_field(dicomInfo, 'Private_0019_101e');    

    if isnan(mriInfo.ReconstructionDiameter)
        mriInfo.FieldOfView            = double(  tmp * [ 1 (dicomInfo.PercentPhaseFieldOfView/100.0)] );
    else
        mriInfo.FieldOfView            = double( [ dicomInfo.ReconstructionDiameter (tmp * dicomInfo.PercentPhaseFieldOfView/100.0) ]);
    end
    
    %
    % mriInfo.PointSize            = dicomInfo.raw.point_size;dicomInfo.
    %
    % mriInfo.PrescribedXdim       = dicomInfo.raw.frame_size;
    % mriInfo.PrescripedYdim       = dicomInfo.raw.nframes;
    %
    % mriInfo.AcquiredXdim         = dicomInfo.raw.da_xres;
    % mriInfo.AcquiredYdim         = dicomInfo.raw.da_yres;
    %
    % mriInfo.ReconXdim            = dicomInfo.raw.rc_xres;
    % mriInfo.ReconYdim            = dicomInfo.raw.rc_yres;
    %
    % mriInfo.ImageXdim            = dicomInfo.image.dim_X;
    % mriInfo.ImageYdim            = dicomInfo.image.dim_Y;
    %
    % mriInfo.vrgfFlag             = dicomInfo.raw.vrgf;
    % mriInfo.vrgfXres             = dicomInfo.raw.vrgfxres;
    %
    % mriInfo.NumberOfCoils        = dicomInfo.raw.dab(2,1) - dicomInfo.raw.dab(1,1) + 1;
    %
    % mriInfo.NumberOfBaselines    = max([ 1 dicomInfo.raw.baseline_views ]);
    % mriInfo.PointSize            = dicomInfo.raw.point_size;
    %
    % mriInfo.Landmark              = dicomInfo.series.lmhor;
    % mriInfo.TopLeftHandCorner     = dicomInfo.image.tlhc;    % Top left hand corner
    % mriInfo.TopRightHandCorner    = dicomInfo.image.trhc;    % Top right hand corner
    % mriInfo.BottomRightHandCorner = dicomInfo.image.brhc;    % Bottom right hand corner
    % mriInfo.RasNorm               = dicomInfo.image.norm;
    %
    
    mriInfo.NumberOfSlices             = local_get_dicom_field(dicomInfo,'Private_0021_104f'); % Locations in Acquisition
    
    mriInfo.NumberOfVolumesPrescribed  = max([1 local_get_dicom_field(dicomInfo,'Private_0019_10f2')]); % Fast Phases

    mriInfo.NumberOfVolumes            = max([1 local_get_dicom_field(dicomInfo,'Private_0019_10f2')]); % Number of Acquisitions

    mriInfo.nPassReps                  = mriInfo.NumberOfVolumes;
    
    mriInfo.NumberOfAcquisition        = local_get_dicom_field(dicomInfo, 'Private_0025_1011'); % Number of Acquisitions

    mriInfo.LastImageNumberUsed        = local_get_dicom_field(dicomInfo, 'Private_0025_1019');  % Number of Acquisitions    
    mriInfo.SliceThickness             = local_get_dicom_field(dicomInfo,'SliceThickness');
    mriInfo.SpacingBetweenSlices       = dicomInfo.SliceThickness - local_get_dicom_field(dicomInfo,'SpacingBetweenSlices');

    mriInfo.FrameOfReferenceUID        = 0;

    mriInfo.HorizFrameOfRef            =  local_get_dicom_field(dicomInfo, 'Private_0019_100f'); 
   
    mriInfo.AcquisitionMatrix          = double(local_get_dicom_field(dicomInfo,'AcquisitionMatrix'));

    mriInfo.Xdim                       = max([1 max(mriInfo.AcquisitionMatrix)]);
    mriInfo.Ydim                       = max([1 min(mriInfo.AcquisitionMatrix(mriInfo.AcquisitionMatrix>0))]);

    %
    %
     mriInfo.PixelSizeX           = mriInfo.FieldOfView(1)/mriInfo.Xdim;
     mriInfo.PixelSizeY           = mriInfo.FieldOfView(2)/mriInfo.Ydim;
     mriInfo.PixelSizeZ           = mriInfo.SliceThickness;
    %
    % % Prescan Values
    % %

   
    
    mriInfo.TG              = local_get_dicom_field(dicomInfo, 'Private_0019_10f9'); % TG /*actual transmit gain (.1 db)*/      
    mriInfo.R1              = local_get_dicom_field(dicomInfo, 'Private_0019_108a'); % R1 /*actual transmit gain (.1 db)*/    
    mriInfo.R2              = local_get_dicom_field(dicomInfo, 'Private_0019_108b'); % R2 /*actual transmit gain (.1 db)*/  

    mriInfo.Prescan.autoTG = local_get_dicom_field(dicomInfo,  'Private_0019_1094'); % TG /*actual transmit gain (.1 db)*/       
    mriInfo.Prescan.autoR1 = local_get_dicom_field(dicomInfo,  'Private_0019_1095'); %:	typint    ,	 /*actual receive gain analog (.1 db)*/  
    mriInfo.Prescan.autoR2 = local_get_dicom_field(dicomInfo,  'Private_0019_1096'); %:	typint    ,	 /*actual receive gain digital (.1 db)*/ 
    mriInfo.Prescan.autoAX = local_get_dicom_field(dicomInfo,  'Private_0019_1093');   %:	typint    ,	 /*actual transmit frequency (Hz) */
    
    %
    % User Cvs
    %
    
    mriInfo.opUserCvs = zeros([1 49]);

    mriInfo.opUserCvs(  1 ) =  local_get_dicom_field(dicomInfo, 'Private_0019_10a7');
    mriInfo.opUserCvs(  2 ) =  local_get_dicom_field(dicomInfo, 'Private_0019_10a8');
    mriInfo.opUserCvs(  3 ) =  local_get_dicom_field(dicomInfo, 'Private_0019_10a9');
    mriInfo.opUserCvs(  4 ) =  local_get_dicom_field(dicomInfo, 'Private_0019_10aa');
    mriInfo.opUserCvs(  5 ) =  local_get_dicom_field(dicomInfo, 'Private_0019_10ab');

    mriInfo.opUserCvs(  6 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10ac');
    mriInfo.opUserCvs(  7 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10ad');
    mriInfo.opUserCvs(  8 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10ae');
    mriInfo.opUserCvs(  9 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10af');
    mriInfo.opUserCvs( 10 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10b0'); 

    mriInfo.opUserCvs( 11 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10b1');
    mriInfo.opUserCvs( 12 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10b2');
    mriInfo.opUserCvs( 13 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10b3');
    mriInfo.opUserCvs( 14 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10b4');
    mriInfo.opUserCvs( 15 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10b5'); 

    mriInfo.opUserCvs( 16 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10b6');
    mriInfo.opUserCvs( 17 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10b7');
    mriInfo.opUserCvs( 18 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10b8');
    mriInfo.opUserCvs( 19 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10b9');
    mriInfo.opUserCvs( 20 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10ba');

    mriInfo.opUserCvs( 21 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10bb');
    mriInfo.opUserCvs( 22 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10bc');
    mriInfo.opUserCvs( 23 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10bd');
    mriInfo.opUserCvs( 24 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10df');
    mriInfo.opUserCvs( 25 )  = local_get_dicom_field(dicomInfo, 'Private_0019_10e0');  % mriInfo.opUserCvs(24) on scanner
    mriInfo.opUserCvs(26:49) = local_get_dicom_field(dicomInfo, 'Private_0043_1038');
    
    %
    %
    % %************************************
    % % Get magnet transform info from pFile
    % %************************************
    % if wfu_exist_spm
    %
    % [mat,landmark,M,vx,origin]   = wfu_get_magnet_transform(mriInfo);
    %
    % mriInfo.LandMark  = landmark;
    % mriInfo.mat       = mat;
    % mriInfo.M         = M;
    % mriInfo.vx        = vx;
    % mriInfo.origin    = origin;
    %
    % end
    %
    % %************************************
    % % rhdacqctrl
    % %************************************
    % % /* ******************************
    % %    bit patterns for rhdacqctrl
    % %    ----------------------------
    % %    bit 0	0= NO_REC data, 1=RAW data
    % %    bit 1	Set if even echo phase flip desired.
    % %    bit 2	Set if odd echo phase flip desired.
    % %    bit 3	Set if even echo frequency flip desired.
    % %    bit 4	Set if odd echo frequency flip desired.
    % %    bit 5	Set if RAW collection without DAB SSP packets is desired.
    % %    bit 6	Set if RAW collection should wrap around to the top of the
    % % 	  	    buffer if overflow occurs.dicomInfo.
    % %    bit 10 Set if fcine echotrain data acq. mode is used.
    % %                MRIge56094                RH_FCINE_ET = 1024
    % %    bit 11 Set if passthrough data acq. mode is used.;
    % %                                           RH_PASS_THROUGH_CERD_FLAG = 2048
    % %    ********************************************* */
    %
    % mriInfo.rh.dacqctrl = dicomInfo.raw.dacq_ctrl;
    % mriInfo.rh.rawdata  = rawDataFlag;
    %
    % %
    % % EPI Specific Information
    % %eval( sprintf('dicomInfo.%s', fieldName) )
    % switch dicomInfo.raw.kydir   % Phase Order 0=TopDown, 1=CenterOut, 2=BottomUp
    %     case 0
    %         mriInfo.epi.kydir  = 'TopDown';
    %     case 1
    %         mriInfo.epi.kydir  = 'CenterOut';
    %     case 2
    %         mriInfo.epi.kydir  = 'BottomUp';
    % end
    %
    % mriInfo.epi.ileaves = dicomInfo.raw.ileaves;
    %
    % %*****************************************
    % % Imaging Options
    % %*****************************************
    %
    % bitMaskImageOption = bitget(dicomInfo.image.iopt,1:30);
    % mriInfo.imopt.ImagingOptions  = dicomInfo.image.iopt;
    % mriInfo.imopt.SquarePixel     = bitMaskImageOption(13);
    % mriInfo.imopt.FlowCompFreq    = bitMaskImageOption(24);
    % mriInfo.imopt.FlowCompSlice   = bitMaskImageOption(25);
    % mriInfo.imopt.EPI             = bitMaskImageOption(27);
    % mriInfo.imopt.EDR             = 0*bitMaskImageOption(27); % Extended Dynamic Range

    %mriInfo.imopt.InversionPrep   = dicomInfo.image.iopt;

    %*****************************************
    % EPI and HomoDyne Reconstruction
    %*****************************************

    %mriInfo.homodyne.offsetKy = ;


else

end



function dicomField = local_get_dicom_field(dicomInfo, fieldName)

    if isfield(dicomInfo, fieldName)
        dicomField  = eval( sprintf('dicomInfo.%s', fieldName) );  %#ok<AGROW,AGROW> 
    else
        dicomField  = nan;
    end
