function varargout = wfu_get_p_header(fileNames);
%
% PURPOSE: - Read the P file hdr{ii} and return a structure containing information about the P file.
%
% INPUTS:
%
%
% OUTPUTS: 
%
%
% EXAMPLE:
%
%
%==========================================================================
% C H A N G E   L O G
%
%--------------------------------------------------------------------------

if nargin < 1
   options.displayFiles = false;
   fileNames = wfu_find_pfiles([],[],options); 
end

if ~iscell(fileNames)
    fileNames = {fileNames};
end

%
% Loop over all file names
%

nFileNames = length(fileNames);

pHdrCell  = cell(1,nFileNames);

for ii=1:nFileNames
    
[ iiFileDir iiFileName ] = wfu_fileparts(fileNames{ii},true);
  
hdr{ii} = wfu_read_p_header(fileNames{ii});

pHdrCell{ii}.FileDir              = iiFileDir;
pHdrCell{ii}.FileName             = iiFileName;
pHdrCell{ii}.rdbmRevision         = hdr{ii}.rdbmRevision;
pHdrCell{ii}.FileType             = 'P';
pHdrCell{ii}.FileFormat           = sprintf('GE P File %d', hdr{ii}.rdbmRevision);
pHdrCell{ii}.StudyID              = hdr{ii}.exam.study_uid;
pHdrCell{ii}.Protocol             = hdr{ii}.series.prtcl;
pHdrCell{ii}.PatientID            = hdr{ii}.exam.patid;
pHdrCell{ii}.PatientName          = hdr{ii}.exam.patname;
pHdrCell{ii}.PatientPosition      = hdr{ii}.series.position;
pHdrCell{ii}.PatientEntry         = hdr{ii}.series.entry;
pHdrCell{ii}.PulseSeqType         = hdr{ii}.raw.scan_type;
pHdrCell{ii}.PulseSeqMode         = hdr{ii}.image.pseqmode;
pHdrCell{ii}.PulseSeqName         = hdr{ii}.image.psdname;
pHdrCell{ii}.InternalPulseSeqName = hdr{ii}.image.psdname;

pHdrCell{ii}.PlaneType            = hdr{ii}.image.plane;

if isempty(pHdrCell{ii}.PlaneType)
    pHdrCell{ii}.PlaneType = NaN;
end

switch pHdrCell{ii}.PlaneType

    case 2
            pHdrCell{ii}.PlaneName            = 'Axial';
    case 4
            pHdrCell{ii}.PlaneName            = 'Sagittal';        
    case 8
            pHdrCell{ii}.PlaneName            = 'Coronal';
    case 16
            pHdrCell{ii}.PlaneName            = 'Oblique';
    otherwise
            pHdrCell{ii}.PlaneName            = 'Unknown';
end



pHdrCell{ii}.SeriesNumber         = hdr{ii}.series.se_no;
pHdrCell{ii}.SeriesDescription    = hdr{ii}.series.se_desc;

pHdrCell{ii}.RepetitionTime       = hdr{ii}.image.tr/1000; % Convert us to ms
pHdrCell{ii}.EchoTime             = hdr{ii}.image.te/1000; % Convert us to ms
pHdrCell{ii}.InversionTime        = hdr{ii}.image.ti/1000; % Convert us to ms
pHdrCell{ii}.FlipAngle            = hdr{ii}.image.mr_flip;

pHdrCell{ii}.FieldOfView          = hdr{ii}.image.dfov * [ 1 1 ];

if hdr{ii}.image.dfov_rect ~= 0
    pHdrCell{ii}.FieldOfView(2)       = hdr{ii}.image.dfov_rect;   
end

pHdrCell{ii}.SwapPhaseFreq        = hdr{ii}.image.swappf;

pHdrCell{ii}.PointSize            = hdr{ii}.raw.point_size;

pHdrCell{ii}.PrescribedXdim       = hdr{ii}.image.dim_X;
pHdrCell{ii}.PrescribedYdim       = hdr{ii}.image.dim_Y;

pHdrCell{ii}.AcquiredXdim         = hdr{ii}.raw.da_xres;
pHdrCell{ii}.AcquiredYdim         = hdr{ii}.raw.da_yres;

pHdrCell{ii}.ReconXdim            = hdr{ii}.raw.rc_xres;
pHdrCell{ii}.ReconYdim            = hdr{ii}.raw.rc_yres;

pHdrCell{ii}.ImageXdim            = hdr{ii}.image.dim_X;
pHdrCell{ii}.ImageYdim            = hdr{ii}.image.dim_Y;

pHdrCell{ii}.vrgfFlag             = (hdr{ii}.raw.vrgf) || (hdr{ii}.raw.da_xres >hdr{ii}.image.dim_X);
pHdrCell{ii}.vrgfXres             = hdr{ii}.raw.da_xres; % vrgfxres;

pHdrCell{ii}.Xdim                 = hdr{ii}.raw.frame_size;
pHdrCell{ii}.Ydim                 = hdr{ii}.raw.nframes;
pHdrCell{ii}.NumberOfCoils        = hdr{ii}.raw.dab(2,1) - hdr{ii}.raw.dab(1,1) + 1;

pHdrCell{ii}.NumberOfBaselines    = max([ 1 hdr{ii}.raw.baseline_views ]);
pHdrCell{ii}.PointSize            = hdr{ii}.raw.point_size;

pHdrCell{ii}.Landmark              = hdr{ii}.series.lmhor;
pHdrCell{ii}.TopLeftHandCorner     = hdr{ii}.image.tlhc;    % Top left hand corner
pHdrCell{ii}.TopRightHandCorner    = hdr{ii}.image.trhc;    % Top right hand corner
pHdrCell{ii}.BottomRightHandCorner = hdr{ii}.image.brhc;    % Bottom right hand corner

pHdrCell{ii}.RasNorm               = hdr{ii}.image.norm;
 
% There are two versions of p that exist. The old version
% uses the excitation loop to store the data similar to epibold.
% The newer version uses the pass_rep loop.  
% For, the older version the number of slices is equal
% to the hdr{ii}.raw.navs for the newer version the number of volumes is 
% equal to hdr{ii}.raw.reps.  The CVs are complementary, which means that if
% one CV is not equal to 1 then the other is equal to 1.  For this reason,
% the maximum value of these two values determines the number of
% volumes.
rawDataFlag = mod(hdr{ii}.raw.dacq_ctrl,2);

pHdrCell{ii}.nPassReps = hdr{ii}.raw.npasses;


if rawDataFlag == true
    pHdrCell{ii}.NumberOfSlices             = hdr{ii}.raw.nslices;

    pHdrCell{ii}.NumberOfVolumesPrescribed  = max( [hdr{ii}.raw.navs, hdr{ii}.raw.reps] ) ;
    pHdrCell{ii}.NumberOfVolumes            = max( [hdr{ii}.raw.navs, hdr{ii}.raw.reps] ) ;
else

    nVolumesPrescribed                      = max( [hdr{ii}.raw.navs, hdr{ii}.raw.reps] ) ;
    pHdrCell{ii}.NumberOfSlices             = hdr{ii}.raw.nslices / nVolumesPrescribed;

    pHdrCell{ii}.NumberOfVolumesPrescribed  = nVolumesPrescribed;
    pHdrCell{ii}.NumberOfVolumes            = nVolumesPrescribed;

end

pHdrCell{ii}.SliceThickness           = hdr{ii}.image.slthick;
pHdrCell{ii}.SpacingBetweenSlices     = hdr{ii}.image.slthick;
pHdrCell{ii}.FrameOfReferenceUID      = 0;
pHdrCell{ii}.HorizFrameOfRef          = hdr{ii}.series.lmhor;


pHdrCell{ii}.PixelSizeX           = hdr{ii}.image.pixsize_X;
pHdrCell{ii}.PixelSizeY           = hdr{ii}.image.pixsize_Y;
pHdrCell{ii}.PixelSizeZ           = hdr{ii}.image.slthick;

% Prescan Values
% 
pHdrCell{ii}.Prescan.manualTG = hdr{ii}.raw.ps_mps_tg; %:	typint    ,	 /*actual transmit gain (.1 db)*/        %      196
pHdrCell{ii}.Prescan.manualR1 = hdr{ii}.raw.ps_mps_r1; %:	typint    ,	 /*actual receive gain analog (.1 db)*/  %      198
pHdrCell{ii}.Prescan.manualR2 = hdr{ii}.raw.ps_mps_r2; %:	typint    ,	 /*actual receive gain digital (.1 db)*/ %      200
pHdrCell{ii}.Prescan.manualAX = hdr{ii}.raw.ps_mps_freq;   %:	typint    ,	 /*actual transmit frequency (Hz) */ %      200

pHdrCell{ii}.Prescan.autoTG = hdr{ii}.raw.ps_aps_tg; %:	typint    ,	 /*actual transmit gain (.1 db)*/        %      196
pHdrCell{ii}.Prescan.autoR1 = hdr{ii}.raw.ps_aps_r1; %:	typint    ,	 /*actual receive gain analog (.1 db)*/  %      198
pHdrCell{ii}.Prescan.autoR2 = hdr{ii}.raw.ps_aps_r2; %:	typint    ,	 /*actual receive gain digital (.1 db)*/ %      200
pHdrCell{ii}.Prescan.autoAX = hdr{ii}.raw.ps_aps_freq;   %:	typint    ,	 /*actual transmit frequency (Hz) */ %      200


%************************************
% Get magnet transform info from pFile
%************************************
%if wfu_exist_spm
    
[mat,landmark,M,vx,origin]   = wfu_get_magnet_transform(pHdrCell{ii});    

pHdrCell{ii}.LandMark  = landmark;
pHdrCell{ii}.mat       = mat;
pHdrCell{ii}.M         = M;
pHdrCell{ii}.vx        = vx;
pHdrCell{ii}.origin    = origin;

%end

%************************************
% rhdacqctrl
%************************************
% /* ******************************
%    bit patterns for rhdacqctrl
%    ----------------------------
%    bit 0	0= NO_REC data, 1=RAW data
%    bit 1	Set if even echo phase flip desired.
%    bit 2	Set if odd echo phase flip desired.
%    bit 3	Set if even echo frequency flip desired.
%    bit 4	Set if odd echo frequency flip desired.
%    bit 5	Set if RAW collection without DAB SSP packets is desired.
%    bit 6	Set if RAW collection should wrap around to the top of the
% 	  	    buffer if overflow occurs.
%    bit 10 Set if fcine echotrain data acq. mode is used.
%                MRIge56094                RH_FCINE_ET = 1024
%    bit 11 Set if passthrough data acq. mode is used.; 
%                                           RH_PASS_THROUGH_CERD_FLAG = 2048
%    ********************************************* */
   
pHdrCell{ii}.rh.dacqctrl = hdr{ii}.raw.dacq_ctrl;
pHdrCell{ii}.rh.rawdata  = rawDataFlag; 

%
% EPI Specific Information
%
switch hdr{ii}.raw.kydir   % Phase Order 0=TopDown, 1=CenterOut, 2=BottomUp
    case 0      
        pHdrCell{ii}.epi.kydir  = 'TopDown';  
    case 1 
        pHdrCell{ii}.epi.kydir  = 'CenterOut'; 
    case 2 
        pHdrCell{ii}.epi.kydir  = 'BottomUp';  
end
        
pHdrCell{ii}.epi.ileaves     = hdr{ii}.raw.ileaves;
pHdrCell{ii}.epi.EchoSpacing = hdr{ii}.image.effechospace;

%*****************************************
% Imaging Options
%*****************************************

bitMaskImageOption = bitget(hdr{ii}.image.iopt,1:30);
pHdrCell{ii}.imopt.ImagingOptions  = hdr{ii}.image.iopt;
pHdrCell{ii}.imopt.SquarePixel     = bitMaskImageOption(13);
pHdrCell{ii}.imopt.FlowCompFreq    = bitMaskImageOption(24);
pHdrCell{ii}.imopt.FlowCompSlice   = bitMaskImageOption(25);
pHdrCell{ii}.imopt.EPI             = bitMaskImageOption(27);
pHdrCell{ii}.imopt.EDR             = 0*bitMaskImageOption(27); % Extended Dynamic Range

%pHdrCell{ii}.imopt.InversionPrep   = hdr{ii}.image.iopt;

%*****************************************
% EPI and HomoDyne Reconstruction
%*****************************************

%pHdrCell{ii}.homodyne.offsetKy = ;

end

%
% Write Output (not implemented yet)
%

if nargout > 0
    if nFileNames == 1
        varargout{1} = pHdrCell{1};
    else
        varargout{1} = pHdrCell;
    end          

    if nargout > 1
        if nFileNames == 1
            varargout{2} = hdr{1};    
        else
            varargout{2} = hdr;        
        end          
    end
else
    % If no output is requested show structure
    wfu_show_structure(pHdrCell);
end
