%-----------------------------------------------------
% Function: wfu_get_p_header_size(file)
% Purpose:  Determines the size of the header in the given P-File.
%
% Arguments:file =  File Id of P-File to read (must be set via prior call to Open())
%               or  Filename of file to read (will open, then close).
%
% Returns:  header_size = size in bytes of the P-File header
%           isa_new_pfile_flag = 1 if this is a new style (longer) P-File header, 0 otherwise
%-----------------------------------------------------

%     07          7.0      7.x                       39984    Big
%     08          7.0      8.x                       39984    Big
%     09          7.0      9.0 except for 9.0 CNV4   39984    Big
%     09          8.0      9.0 CNV4, 9.1, VH3        60464    Big
%     10          8.0      10.0 last SGI               "      Big
%     11          9.0      11.0 (1st PC/Linux)       61464   Little
%     12                   12.0                      66072   Little
%
%     The 12.0 header increased in two places: the RDB_DATA_ACQ_TAB structure increased 
%     to 45056 bytes from 40960 bytes (+4096 bytes), and the SERIESDATATYPE structure 
%     increased to 2048 bytes from 1536 bytes (+512 bytes).
%
%
% Where genesis_ver = MRIMAGEDATATYPE.im_verscre
%      rdb_hdr_rev = RDB_HEADER_REC.rdb_hdr_rdbm_rev
% 
%-JBob

function [header_size, rdbmRevision, endian ] = wfu_get_p_header_size(fileName)

%------------------------------------------------------------------------------------------------------------
% 
%------------------------------------------------------------------------------------------------------------
    
minRdbmRevision =  9;
maxRdbmRevision = 14.3;

if iscell(fileName)
    fileName = fileName{1};
end

% Open the P-file and assume that it is big endian for the moment

endian      = 'ieee-le';
fid = fopen(fileName, 'r', endian);
if fid==-1
      error('Error opening file: %s.',fileName);
      return;
end
  
rdbmRevision = fread(fid, 1, 'float32');

rdbmRevision = round(10*rdbmRevision)/10;

if ( (rdbmRevision < minRdbmRevision) || (rdbmRevision>maxRdbmRevision) )
    fclose(fid);
    error('GE_Pfile:invalidPfile','Invalid revision number for P file header');    
end

%
% Check Little Endian Assumption
%

if ( (rdbmRevision < minRdbmRevision) || (rdbmRevision>maxRdbmRevision) )
    endian      = 'ieee-be';

    fid = fopen(fileName, 'r', endian); 

    if fid==-1  % Unnecessary check 
        error(sprintf('Error opening file: %s.',fileName));
        return;
    end
    
    rdbmRevision = fread(fid, 1, 'float32');
end

%
% Close fid
%

fclose(fid);

%
% Check Range of rdbmRevision again
%
if( (rdbmRevision <minRdbmRevision) || (rdbmRevision>maxRdbmRevision) )
    error('GE_Pfile:invalidPfile','Invalid revision number for P file header');
end


%
% Set Header size
%

hdr = wfu_get_pfile_offsets(rdbmRevision);

header_size = hdr.header_size;


return;
