function varargout = wfu_read_p_header(fileNames, type, nSlices)
%-----------------------------------------------------
% Function: wfu_read_p_header(file, type)
%
% Purpose:  Reads header values from a P-File for all or a specific type of header.
%
% Inputs:
%
%     fileName =  File name of P-file to read.
%
%     type - Specifies which header to read ('all')
%                   'all' for all headers
%                   'raw' for raw file header
%                   'daq' for data acquisition table
%                   'exam' for exam header
%                   'series' for series header
%                   'image' for image header
%
%     nSlices  - Optional arguement for reading nSlices from the data
%                acquistion order table. If empty then all entries 
%               (512 or 1024) are returned.
%
% Outputs:  
%
%     header = M-array structure "header" containing header field values.
%
% External References: 
%           wfu_read_ge_header()   %in GE/private folder
%==========================================================================
% Change Log
%------------------------------------------------------------------------
if nargin < 3 || isempty(nSlices)
    nSlices = [];
end

if nargin < 2 || isempty(type) || strcmp(type,'all')
    type = { 'raw', 'daq', 'exam', 'series', 'image', 'series'};
end

if ~iscell(type)
    type = {type};
end

if ~iscell(fileNames)
    fileNames = {fileNames};
end

nFileNames = length(fileNames);

pHdrCell   = cell(1,nFileNames);

for ii=1:nFileNames
    [ pHdrCell{ii} pHdrCell{ii}.rdbmRevision ] = local_read_p_header(fileNames{ii},type,nSlices);
end

%
% Return Output or Show Structure
%

if nargout > 0

        if nFileNames == 1
            varargout{1} = pHdrCell{1};
        else
            varargout{1} = pHdrCell;
        end
    
else

    % Pick first filename for title of figure.
    [tmpFileDirectory tmpFileName tmpFileExtension ] = fileparts(fileNames{1});
    tmpFullFileName = strcat(tmpFileName, tmpFileExtension);

    wfu_show_p_header(pHdrCell, type, tmpFullFileName);

end


function  [ header, rdbmRevision ] = local_read_p_header(fileName, type, nSlices)

%
% Open P File
%

[ rdbmRevision, endian ]  = wfu_get_p_header_revision(fileName);

fid =  fopen(fileName, 'r', endian);

if fid==-1
    error('Error opening file: %s.',filename_or_fileid);
    return;
end


%
% Loop over all types
% 

nType = length(type);

for ii=1:nType
    
    switch lower( char(type{ii}))
        case 'raw'
            header.raw    = private_read_ge_raw_header(fid,rdbmRevision);
        case 'daq'
            header.daq    = private_read_ge_daq_header(fid,'pfile',rdbmRevision, nSlices);
        case 'exam';
            header.exam   = private_read_ge_exam_header(fid,rdbmRevision);
        case 'series'
            header.series = private_read_ge_series_header(fid,rdbmRevision);
        case 'image'
            header.image  = private_read_ge_image_header(fid,rdbmRevision);
        otherwise
            
            fclose(fid);
            error('Second arg must be ''raw'',''daq'',''exam'',''series'', ''image'', or ''all'' ');
    end   
end

%
% Close file
%

fclose(fid);
