function wfu_postprocess_epi(fileOrDirNames, reconEpiType, options)

%
% PURPOSE: - To reconstruct epi images after reconstruction.
%
% CATEGORY: epi Reconstruction
%
% INPUTS:
%
%    fileOrDirNames  - Names of the epi P files or directories to
%                      be reconstructed. This function is called
%                      recursively until all the P files and
%                      directories with P files are reconstructed.
%
%                      This function reconstructs all directories first
%                      regardless of thier order in the list, before
%                      reconstructing individual files in the list.
%
%    reconEpiType - Only reconstruct this epi data type.
%
%
% OUTPUTS:
%
%    None
%
%
% NOTES:
%  
%  infoState.Post processing          = 0;
%  infoState.ReconstructionCompleted = 1;
%  infoState.ReconstructionFailed    = 2;
%  infoState.ReconstructionSkipped   = 3;
%
% EXAMPLE:
%
%
%==========================================================================
% C H A N G E   L O G
%
%--------------------------------------------------------------------------

%
% Internal Variable
%
defaultOptions.pacsServer          = [];
defaultOptions.pasl.minPaslVolumes = 32;
defaultOptions.pasl.model          = [];

if nargin < 3
    options = [];
end

options = wfu_set_function_options(options, defaultOptions);


if nargin < 2 || isempty(reconEpiType)
    reconEpiType = 'all';
end

if nargin < 1 || isempty(fileOrDirNames)
    try
        % Assume current directory
        fileOrDirNames2 = wfu_find_epi_ifiles(reconEpiType, [], false, false );

        fileOrDirNames  = [fileOrDirNames1, fileOrDirNames2];
        
    catch
        disp(sprintf('\n\t!!!!! Unable to find valid P-files to reconstruct !!!!!\n'));
        return
    end
end

if ~iscell(fileOrDirNames)
    fileOrDirNames = {fileOrDirNames};
end

if( ~ischar(reconEpiType) )
    error('epi Reconstruction Type must be ''all'', ''fmri'', ''dccsi'', ''dcpsf'', ''pasl'', or ''t1map''');
end

if(     (~strcmpi(reconEpiType,'all')  && ...
        ~strcmpi(reconEpiType,'fmri')  && ...
        ~strcmpi(reconEpiType,'dccsi') && ...
        ~strcmpi(reconEpiType,'dcpsf') && ...
        ~strcmpi(reconEpiType,'pasl')  && ...
        ~strcmpi(reconEpiType,'t1map')) )
    error('epi Reconstruction Type must be ''all'', ''fmri'', ''dccsi'', ''dcpsf'', ''pasl'', or ''t1map''');
end

%
% Loop over fileOrDirNames and sort fileNames and Directories
%
nFileOrDirNames = length(fileOrDirNames);

fileNames = {};
iiFileCounter = 1;

for ii=1:nFileOrDirNames
    try

        if isdir(fileOrDirNames{ii})

            %
            % Recursively call directories to be reconstructed
            %

            dirFileNames = wfu_find_epi_pfiles(reconEpiType, fileOrDirNames{ii}, false, false );
            wfu_postprocess_epi(dirFileNames, reconEpiType, options)

        else
            fileNames{iiFileCounter} = fileOrDirNames{ii};
            iiFileCounter = iiFileCounter + 1;
        end

    catch
        %
        % With reconstruction State Flag set to 2 if an error occurs
        %

        disp(sprintf('\n\t!!!!! Error finding Pfiles to reconstruct !!!!!\n'));
    end
end

%
% Loop over fileNames and reconstruct each epi P-file
%

nFileNames = length(fileNames);

if nFileNames > 0
    
    disp(' ');
    disp(sprintf(   '************************************************************************************************'));
    disp('epi Post Processing:  Post processing the following files ...         ');
    
    wfu_list_files( fileOrDirNames );
    
    for ii=1:nFileNames

        absoluteFileName = fileOrDirNames{ii};

       
        %
        % Post Process P files
        %
        
        disp(sprintf('\n-------------------------------------------------------------------------------------------------\n'));        
        
        if wfu_is_epi_ifile(absoluteFileName) 

            local_postprocess_epi_ifiles(absoluteFileName, reconEpiType, options );

        else

            disp(sprintf('\tSkipping post processing for non-epi P file %s.', absoluteFileName));

        end
        
    end
end

%**************************************************************************
%
%


function local_postprocess_epi_ifiles(absoluteFileName, reconEpiType, options)


[fileDirectory fileNameBase fileNameExtension ] = fileparts(absoluteFileName);           
fileName = strcat(fileNameBase, fileNameExtension);
        

stateFlag = 1;

infoState.ReconstructionInProgress = 0;
infoState.ReconstructionCompleted  = 1;
infoState.ReconstructionFailed     = 2;
infoState.ReconstructionSkipped    = 3;

            
epiInfo     = wfu_get_epi_info( absoluteFileName );

epiType     = epiInfo.AcquisitionCode;
epiParadigm = epiInfo.fMRIParadigmNumber;


if (strcmp(epiInfo.AcquisitionName,reconEpiType) || strcmp('all',reconEpiType)) && ...
        ( stateFlag == infoState.ReconstructionCompleted)

    %
    % Determine Parent Directory
    %

    [analyzeDir baseFileName ] = wfu_fileparts(wfu_get_analyze_filename(absoluteFileName,'base'));
    fileSepPosition            = strfind(analyzeDir,filesep);
    examParentDir              = analyzeDir(1:fileSepPosition(end)-1);

    try

        switch epiType

            case 1 %fmri

                minFmriVolumes = 17;  % options.pasl.minPaslVolumes;
                
                if (epiInfo.NumberOfVolumes > minFmriVolumes) && ...
                   (epiParadigm > 0 )
              
                    disp(sprintf('Post processing for fMRI Analyze file %s.\n', fileName));
                                        
                    %
                    % Read epi info from dicom file
                    %
                    
                    epiInfo       = wfu_get_epi_info(absoluteFileName);
                                        
                    %
                    % Extract fMRI volumes without DDAs
                    %

                    mvAbsoluteFileName = sprintf('%s%s%s_4D', examParentDir, filesep, baseFileName);

                    nDDA = epiInfo.ddaN;
                    
                    wfu_extract_spm(absoluteFileName, mvAbsoluteFileName, sprintf('%d:end',nDDA+1), false );

                    %
                    % Write Info File
                    %
                    
                     epiInfo.FileName = sprintf('%s%s%s_4D', examParentDir, filesep, baseFileName);
                     wfu_write_epiflex_pinfo_from_dicom(epiInfo, infoState.ReconstructionCompleted);  % Use EPIFLEX P info file for writing
                    
                else

                    disp(sprintf(['Skipping post processing of epi fMRI Analyze file %s.\n' ...
                                  'Not enough volumes or unknown paradigm number'], fileName));
                   
                end


            otherwise

              disp(sprintf('\tSkipping unknown EPI Analyze file %s.', fileName));
              
        end

    catch

        %
        % With reconstruction State Flag set to 2 if an error occurs
        %

        disp(sprintf('\n\t!!!!! Error occurred during post processing !!!!!\n'));

        errorMessage = regexprep(lasterr,sprintf('\n'), sprintf('\n\t'));
        disp(sprintf('\t%s',errorMessage));
    end
else

    %
    % Create state message
    %

    switch stateFlag

        case infoState.ReconstructionInProgress
            stateMessage = 'Reconstruction in progress.';

        case infoState.ReconstructionFailed
            stateMessage = 'Reconstruction failed.';

        case infoState.ReconstructionInProgress
            stateMessage = 'Reconstruction skipped.';

        otherwise
            stateMessage = 'Reconstruction state unknown.';
    end

    %
    % Display skipped message
    %

    switch epiType

        case 1 %fmri
            disp(sprintf('\tSkipping fMRI post processing P file %s. \n\t%s', fileName,  stateMessage));

        otherwise

            disp(sprintf('\tSkipping post processing for unknown epi P file %s. \n\t %s', fileName,  stateMessage));

    end
end