function wfu_postprocess_epiflex(fileOrDirNames, reconEpiflexType, options)

%
% PURPOSE: - To reconstruct EPIFLEX images after reconstruction.
%
% CATEGORY: EPIFLEX Reconstruction
%
% INPUTS:
%
%    fileOrDirNames  - Names of the EPIFLEX P files or directories to
%                      be reconstructed. This function is called
%                      recursively until all the P files and
%                      directories with P files are reconstructed.
%
%                      This function reconstructs all directories first
%                      regardless of thier order in the list, before
%                      reconstructing individual files in the list.
%
%    reconEpiflexType - Only reconstruct this EPIFLEX data type.
%
%
% OUTPUTS:
%
%    None
%
%
% NOTES:
%  
%  infoState.Post processing          = 0;
%  infoState.ReconstructionCompleted = 1;
%  infoState.ReconstructionFailed    = 2;
%  infoState.ReconstructionSkipped   = 3;
%
% EXAMPLE:
%
%
%==========================================================================
% C H A N G E   L O G
%
%--------------------------------------------------------------------------
%	2008-11-26 BCW	Added Dircon

%
% Internal Variable
%

defaultOptions.pacsServer               = [];
defaultOptions.checkAccessionNumber     = true;
defaultOptions.pasl.minPaslVolumes      = 32;
defaultOptions.pasl.model               = [];


if nargin < 3
    options = [];
end

options = wfu_set_function_options(options, defaultOptions);


if nargin < 2 || isempty(reconEpiflexType)
    reconEpiflexType = 'all';
end

if nargin < 1 || isempty(fileOrDirNames)
    try
        % Assume current directory
        fileOrDirNames1 = wfu_find_epiflex_pfiles(reconEpiflexType, [], false, false );
        fileOrDirNames2 = wfu_find_epiflex_ifiles(reconEpiflexType, [], false, false );

        fileOrDirNames  = [fileOrDirNames1, fileOrDirNames2];
        
    catch
        errorInfo = lasterror;
        fprintf('Error received: %s\n',errorInfo.message);
        disp(sprintf('\n\t!!!!! Unable to find valid P-files to reconstruct !!!!!\n'));
        return
    end
end

if ~iscell(fileOrDirNames)
    fileOrDirNames = {fileOrDirNames};
end

if( ~ischar(reconEpiflexType) )
    error('Epiflex Reconstruction Type must be ''all'', ''fmri'', ''dccsi'', ''dcpsf'', ''pasl'', or ''t1map''');
end

if(     (~strcmpi(reconEpiflexType,'all')  && ...
        ~strcmpi(reconEpiflexType,'fmri')  && ...
        ~strcmpi(reconEpiflexType,'dccsi') && ...
        ~strcmpi(reconEpiflexType,'dcpsf') && ...
        ~strcmpi(reconEpiflexType,'pasl')  && ...
        ~strcmpi(reconEpiflexType,'t1map')) )
    error('Epiflex Reconstruction Type must be ''all'', ''fmri'', ''dccsi'', ''dcpsf'', ''pasl'', or ''t1map''');
end

dircon = wfu_directory_configuration;   % dircon.anatdir setting    

%
% Loop over fileOrDirNames and sort fileNames and Directories
%
nFileOrDirNames = length(fileOrDirNames);

fileNames = {};
iiFileCounter = 1;

for ii=1:nFileOrDirNames
    try

        if isdir(fileOrDirNames{ii})

            %
            % Recursively call directories to be reconstructed
            %

            dirFileNames = wfu_find_epiflex_pfiles(reconEpiflexType, fileOrDirNames{ii}, false, false );
            wfu_postprocess_epiflex(dirFileNames, reconEpiflexType, options)

        else
            fileNames{iiFileCounter} = fileOrDirNames{ii};
            iiFileCounter = iiFileCounter + 1;
        end

    catch
        %
        % With reconstruction State Flag set to 2 if an error occurs
        %

        errorInfo = lasterror;
        fprintf('Error received: %s\n',errorInfo.message);
        disp(sprintf('\n\t!!!!! Error finding Pfiles to reconstruct !!!!!\n'));
    end
end

%
% Loop over fileNames and reconstruct each EPIFLEX P-file
%

nFileNames = length(fileNames);

if nFileNames > 0
    
    disp(' ');
    disp(sprintf(   '************************************************************************************************'));
    disp('EPIFLEX Post Processing:  Post processing the following files ...         ');
    
    wfu_list_files( fileOrDirNames );
    
    for ii=1:nFileNames

        absoluteFileName = fileOrDirNames{ii};

       
        %
        % Post Process P files
        %
        
        disp(sprintf('\n-------------------------------------------------------------------------------------------------\n'));
        
        
        if wfu_is_epiflex_pfile(absoluteFileName) 

        %
        % Set options here to account for file dependent option, reorderKy
        % 

            defaultOptions = wfu_recon_epiflex_default_options(absoluteFileName);
  
            completeOptions = wfu_set_function_options(options,defaultOptions);

            local_postprocess_epiflex_pfiles(absoluteFileName, reconEpiflexType, completeOptions);
            
        elseif wfu_is_epiflex_ifile(absoluteFileName) 

            local_postprocess_epiflex_ifiles(absoluteFileName, reconEpiflexType, options );

        else

            disp(sprintf('\tSkipping post processing for non-EPIFLEX P file %s.', absoluteFileName));

        end
        
    end
end

%**************************************************************************
%
%


function local_postprocess_epiflex_ifiles(absoluteFileName, reconEpiflexType, options)


[fileDirectory fileNameBase fileNameExtension ] = fileparts(absoluteFileName);           
fileName = strcat(fileNameBase, fileNameExtension);
        

stateFlag = 1;

infoState.ReconstructionInProgress = 0;
infoState.ReconstructionCompleted  = 1;
infoState.ReconstructionFailed     = 2;
infoState.ReconstructionSkipped    = 3;

dircon = wfu_directory_configuration(false);   % dircon.anatdir setting    
            
epiflexInfo     = wfu_get_epiflex_info( absoluteFileName );

epiflexType     = epiflexInfo.AcquisitionCode;
epiflexParadigm = epiflexInfo.fMRIParadigmNumber;


if (strcmp(epiflexInfo.AcquisitionName,reconEpiflexType) || strcmp('all',reconEpiflexType)) && ...
        ( stateFlag == infoState.ReconstructionCompleted)

    %
    % Determine Parent Directory
    %

    [analyzeDir baseFileName ] = wfu_fileparts(wfu_get_analyze_filename(absoluteFileName,'base'));
    fileSepPosition            = strfind(analyzeDir,filesep);
    examParentDir              = analyzeDir(1:fileSepPosition(end)-1);

    try
        
        %
        % Determine SPM Version
        %

        [spmExist spmVer spmNumericVersion] = wfu_exist_spm;

        switchSpm5Flag                      = (~isempty( strfind(analyzeDir, '/ansir4/clinical/')) | ...
                                              ~isempty( strfind(analyzeDir, '/ansir9/clinical/'))) & ...
                                              (spmNumericVersion ~= 5);

 %        switchSpm5Flag                      = (~isempty( strfind(analyzeDir, '/DHSMind/bkraft/pasl_rr'))) & ...
 %                                              (spmNumericVersion ~= 5);

        switchSpm5Flag       = false;  % Override SPM5 until segmentation problem is fixed.          
                                          
        switchToSpmVersion   = 5;

        %
        % Postprocess data according to EPIFLEX type
        %

        % tmp change for MA
        
        epiflexParadigm = 1600;
        
        switch epiflexType

            case 1 %fmri

                minFmriVolumes = 17;  % options.pasl.minPaslVolumes;
                
                if ((epiflexInfo.NumberOfVolumes > minFmriVolumes) && ...
                    (epiflexParadigm > 0 ))
              
                    disp(sprintf('Post processing for fMRI %s file %s.\n', dircon.anatdir, fileName));
                                        
                    %
                    % Read epiflex info from dicom file
                    %
                    
                    epiflexInfo       = wfu_get_epiflex_info(absoluteFileName);
                    
                    %
                    % Link image file to parent directory
                    %
%                     unixCommand = sprintf('ln -sf %s %s', wfu_get_spm_filename(absoluteFileName,'hdr'), '..');
%                     system(unixCommand);
%                     
%                     unixCommand = sprintf('ln -sf %s %s', wfu_get_spm_filename(absoluteFileName,'img'), '..');
%                     system(unixCommand);
%                     
%                     unixCommand = sprintf('ln -sf %s %s', wfu_get_spm_filename(absoluteFileName,'info'), '..');
%                     system(unixCommand)

                    
                    %
                    % Extract fMRI volumes without DDAs
                    %

                    % seriesNumber      = epiflexInfo.SeriesNumber;


                    mvAbsoluteFileName = sprintf('%s%s%s_4D', examParentDir, filesep, baseFileName);

                    nDDA        = epiflexInfo.ddaN;
                    indexVolume = sprintf('%d:end',nDDA+1);
                    wfu_extract_spm(absoluteFileName, mvAbsoluteFileName, indexVolume, false );

                    %
                    % Write Info File
                    %
                    
                     epiflexInfo.FileName = sprintf('%s%s%s_4D', examParentDir, filesep, baseFileName);
                     wfu_write_epiflex_pinfo_from_dicom(epiflexInfo, infoState.ReconstructionCompleted);
                    
                else

                    disp(sprintf(['Skipping post processing of EPIFLEX fMRI %s file %s.\n' ...
                                  'Not enough volumes or unknown paradigm number'], dircon.anatdir, fileName));
                   
                end


            case 2 % DC

                disp(sprintf('No post processing for DC Field Map %s file %s', dircon.anatdir, fileName ));

            case 3 % PSF

                disp(sprintf('No post processing for PSF Map %s file %s.', dircon.anatdir, fileName));

            case 4 % Perfusion

                minPaslVolumes = 17;  % options.pasl.minPaslVolumes;
                

                [analyzeDir baseFileName ] = wfu_fileparts(wfu_get_analyze_filename(absoluteFileName,'base'));
                paslDir                    = wfu_get_epiflex_subdir(absoluteFileName);
 

                if switchSpm5Flag == true
                    gospm(switchToSpmVersion);
                end
                
                if epiflexInfo.NumberOfVolumes > minPaslVolumes
 
                    disp(sprintf('Post processing PASL %s file %s.', dircon.anatdir, fileName));

                    %[aDir baseFileName ] = wfu_fileparts(wfu_get_analyze_filename(absoluteFileName,'base'));
                    %fileSepPosition      = strfind(aDir,filesep);
                    %pDir                 = aDir(1:fileSepPosition(end)-1);

                    paramList = {'mPERF', 'M0', 'mBOLD', 'sPERF' };
                    reconOptions.showParams  = paramList;
 
                    %
                    % Hard code DDA limits until a better method is
                    % determined
                    %

                                      
                    if( epiflexInfo.EpiflexVersion < 140000 ) 
                        
                        reconOptions.dda            = [10 3 6 0];
                   
                    elseif ( (epiflexInfo.EpiflexVersion >= 140700) && ...
                             (strcmpi(epiflexInfo.Scanner,'MR01') || ...
                               strcmpi(epiflexInfo.Scanner,'MR02') || ...
                               strcmpi(epiflexInfo.Scanner,'MR03') || ...
                               strcmpi(epiflexInfo.Scanner,'MR04') || ...
                               strcmpi(epiflexInfo.Scanner,'MR05') || ...
                               strcmpi(epiflexInfo.Scanner,'GEMR') || ...
                               strcmpi(epiflexInfo.Scanner,'SIMC') || ...
                               strcmpi(epiflexInfo.Scanner,'SIMCGEMR') || ...
                               strcmpi(epiflexInfo.Scanner,'MRRES') || ...
                               strcmpi(epiflexInfo.Scanner,'mr06mr06')))

                        reconOptions.dda            = [10 2 7 0 ];
                        
                    else
                        reconOptions.dda            = [6 0 6 0  ];
                    end


                   [ tmp paslDir] = wfu_recon_epiflex_pasl_image(absoluteFileName, reconOptions);
 
                    % Convert the mPERF and M0 and send them to PACS.
                    % This is done now in case there is a problem with
                    % generating the CBF images.


                    dicomFileName  = wfu_get_spm_filename(absoluteFileName,'dicom');
                    dicomHdr       = dicominfo(dicomFileName);

                    convertOptions.pacsServer = options.pacsServer;

                    if ~isempty(convertOptions.pacsServer)


                       % if ( ~(isempty(dicomHdr.AccessionNumber)) || (options.checkAccessionNumber == false) )
                       %     wfu_convert_epiflex_pasl_cbf(paslDir, paramList, [], convertOptions);
                       % end
                    end

                    %
                    % Calculate CBF maps.
                    %
                        
                    model = wfu_pasl_gkm_default( dicomHdr.MagneticFieldStrength  ); 
                    wfu_recon_epiflex_pasl_cbf_image(  paslDir, baseFileName, model, options );

                    
                    % Create Post Script Figures
                    paramList = {'mPERF', 'mBOLD', 'CBF', 'qpCBF', 'M0', 'T1', 'QP', 'efgre3d'};
                    wfu_show_epiflex_pasl_cbf(paslDir, paramList, [ false true], false);

                    % Convert the CBF images and send them to PACS.
                    if ( ~isempty(convertOptions.pacsServer) && ( ~(isempty(dicomHdr.AccessionNumber)) || (options.checkAccessionNumber == false) ))                        
                        paramList = {'M0', 'CBF'};
                        wfu_convert_epiflex_pasl_cbf(paslDir, paramList, [], convertOptions);
                    end
                    
                else

                    disp(sprintf(['Skipping post processing PASL %s file %s.' ...
                                  '\n Only %d volumes acquired, which is less than %d required.'], dircon.anatdir, fileName, epiflexInfo.NumberOfVolumes, minPaslVolumes));
                    
                end

                    %
                    % Return to previous SPM version
                    %

                if switchSpm5Flag
                    gospm(spmNumericVersion);
                end

     
            case 5 % T1 map
                
                disp(sprintf('Post processing for T1 Map %s file %s.', dircon.anatdir, fileName));

                if switchSpm5Flag == true
                    gospm(switchToSpmVersion);
                end

                % wfu_recon_epiflex_t1map_image(absoluteFileName);
                t1Options.t1Dir     = examParentDir;
                [t1FilePath]        = wfu_recon_t1map(absoluteFileName, [], [], t1Options);
               
                if switchSpm5Flag
                    gospm(spmNumericVersion);
                end
                
                % after t1maps are reconstructed, go back to the pasl
                % directory and create the qpCBF maps
                [ analyzeDir baseFileName ]     = wfu_fileparts(wfu_get_analyze_filename(absoluteFileName,'base'));
                [ fileDir fileName paslDir ]    = fileparts(analyzeDir);
                paslDirName                     = wfu_find_dir('pasl_.*', fileDir);

                paslDir                         = strcat(fileDir, filesep, paslDirName);

                disp('*** Calculating qpCBF based on T1 map calculation ***');
                qpOptions.pasl.t1mapFileName      = t1FilePath.t1MapFileName;
                qpOptions.pasl.t1mapM0FileName    = t1FilePath.m0AcqFileName;

                wfu_recon_epiflex_pasl_cbf_image(paslDir{1}, [], [], qpOptions);
                wfu_show_epiflex_pasl_cbf(paslDir, {'qpCBF', 't1map'}, [false true], false);

            otherwise

              disp(sprintf('\tSkipping unknown EPIFLEX %s file %s.', dircon.anatdir, fileName));
              
        end

    catch

        %
        % With reconstruction State Flag set to 2 if an error occurs
        %

        errorMessage = regexprep(lasterr,sprintf('\n'), sprintf('\n\t'));
        disp(sprintf('\t%s',errorMessage));

        disp(sprintf('\n\t!!!!! Error occurred during post processing !!!!!\n'));
    end
else

    %
    % Create state message
    %

    switch stateFlag

        case infoState.ReconstructionInProgress
            stateMessage = 'Reconstruction in progress.';

        case infoState.ReconstructionFailed
            stateMessage = 'Reconstruction failed.';

        case infoState.ReconstructionInProgress
            stateMessage = 'Reconstruction skipped.';

        otherwise
            stateMessage = 'Reconstruction state unknown.';
    end

    %
    % Display skipped message
    %

    switch epiflexType

        case 1 %fmri
            disp(sprintf('\tSkipping fMRI post processing P file %s. \n\t%s', fileName,  stateMessage));

        case 2 % DC
            disp(sprintf('\tSkipping DC Field Map post processing P file %s. \n\t%s', fileName,  stateMessage));

        case 3 % PSF
            disp(sprintf('\tSkipping PSF Map P file %s. \n\t%s', fileName,  stateMessage));

        case 4 % Perfusion
            disp(sprintf('\tSkipping PASL P file %s. \n\t%s', fileName,  stateMessage));

        case 5 % T1 map

            disp(sprintf('\tSkipping post processing for T1 map P file %s. \n\t %s', fileName,  stateMessage));

        otherwise

            disp(sprintf('\tSkipping post processing for unknown EPIFLEX P file %s. \n\t %s', fileName,  stateMessage));

    end
end


%**************************************************************************
%
%


function local_postprocess_epiflex_pfiles(absoluteFileName, reconEpiflexType, options)

[fileDirectory fileNameBase fileNameExtension ] = fileparts(absoluteFileName);           
fileName = strcat(fileNameBase, fileNameExtension);
        


infoState.ReconstructionInProgress = 0;
infoState.ReconstructionCompleted  = 1;
infoState.ReconstructionFailed     = 2;
infoState.ReconstructionSkipped    = 3;

stateFlag = wfu_get_epiflex_info_state(absoluteFileName);

epiflexInfo = wfu_get_epiflex_info( absoluteFileName );

epiflexType     = epiflexInfo.AcquisitionCode;
% epiflexParadigm = epiflexInfo.fMRIParadigmNumber;


if (strcmp(epiflexInfo.AcquisitionName,reconEpiflexType) || strcmp('all',reconEpiflexType)) && ...
        ( stateFlag == infoState.ReconstructionCompleted)


    try

        switch epiflexType

            case 1 %fmri

                disp(sprintf('No post processing for fMRI P file %s \n', fileName));


            case 2 % DC

                disp(sprintf('No post processing for DC Field Map P file %s', fileName));

            case 3 % PSF

                disp(sprintf('No post processing for PSF Map P file %s', fileName));

            case 4 % Perfusion

                disp(sprintf('Post processing PASL P file %s', fileName));

                wfu_recon_epiflex_pasl_cbf(absoluteFileName, options.pasl.model, options);

                pDir            = wfu_get_fileparts(absoluteFileName);
                fileSepPosition = strfind(pDir,filesep);
                shortPDir       = pDir(fileSepPosition(end)+1:end);
                csvFullFileName = strcat( pDir, filesep, shortPDir, '_REPORT.csv');

                wfu_report_epiflex_pasl_cbf(absoluteFileName, [],[],csvFullFileName);
                wfu_show_epiflex_pasl_cbf(absoluteFileName, [],true, false);

            case 5 % T1 map

                disp(sprintf('No post processing for T1 Map P file %s', fileName));


            otherwise

                disp(sprintf('Skipping Unknown EPIFLEX P file %s', fileName));
        end

    catch

        %
        % With reconstruction State Flag set to 2 if an error occurs
        %

        disp(sprintf('\n\t!!!!! Error occurred during post processing !!!!!\n'));

        errorMessage = regexprep(lasterr,sprintf('\n'), sprintf('\n\t'));
        disp(sprintf('\t%s',errorMessage));
    end
else

    %
    % Create state message
    %

    switch stateFlag

        case infoState.ReconstructionInProgress
            stateMessage = 'Reconstruction in progress.';

        case infoState.ReconstructionFailed
            stateMessage = 'Reconstruction failed.';

        case infoState.ReconstructionInProgress
            stateMessage = 'Reconstruction skipped.';

        otherwise
            stateMessage = 'Reconstruction state unknown.';
    end

    %
    % Display skipped message
    %

    switch epiflexType

        case 1 %fmri
            disp(sprintf('\tSkipping fMRI post processing P file %s. \n\t%s', fileName, stateMessage));

        case 2 % DC
            disp(sprintf('\tSkipping DC Field Map post processing P file %s. \n\t%s', fileName, stateMessage));

        case 3 % PSF
            disp(sprintf('\tSkipping PSF Map P file %s. \n\t%s', fileName, stateMessage));

        case 4 % Perfusion
            disp(sprintf('\tSkipping PASL P file %s. \n\t%s', fileName, stateMessage));

        case 5 % T1 map

            disp(sprintf('\tSkipping post processing for T1 map P file %s. \n\t %s', fileName, stateMessage));

        otherwise

            disp(sprintf('\tSkipping post processing for unknown EPIFLEX P file %s. \n\t %s', fileName, stateMessage));

    end
end
