function varargout = wfu_write_epiflex_pinfo_from_dicom(fileNamesOrHeaders, reconState)
%
% PURPOSE: - Writes the Pfile info table
%
% CATEGORY: EPIFLEX Write
%
% INPUTS:
%
%
% OUTPUTS: 
%
%    None
%
% EXAMPLE:
%
%
%==========================================================================
% C H A N G E   L O G
%
%--------------------------------------------------------------------------

if ~iscell(fileNamesOrHeaders)
    fileNamesOrHeaders = {fileNamesOrHeaders};
end

nHeaders = length(fileNamesOrHeaders);

if nargin < 2 || isempty(reconState)
    reconState = zeros(1,nHeaders);  % 0 means still processing
end

if length(reconState) <= nHeaders
     reconState = [ reconState zeros(1, nHeaders - length(reconState)) ];
end

%
% Loop over all file names
%


epiflexHdrCell      = cell(1,nHeaders);
epiflexPinfoHdrCell = cell(1,nHeaders);

for ii=1:nHeaders
    
    %
    % Get EPIFLEX Header
    %
    
    if isstruct( fileNamesOrHeaders{ii}) 
        epiflexHdrCell{ii} = fileNamesOrHeaders{ii};
        interpretFlag      = false;
    else
        epiflexHdrCell{ii} = wfu_get_epiflex_info(fileNamesOrHeaders{ii});        
        interpretFlag      = true;
    end
    
    %
    % Select struture elements
    %
    
    pInfoList = { ...
            'Scanner';                         ...
            'RepetitionTime';       'EchoTime'; ...
            'InversionTime';        'PulseSeqName'; ...
            'PlaneType';            'PatientPosition'; ...
            'PatientEntry';         'PatientID'; ...
            'Protocol';             'SliceThickness'; ...
            'SpacingBetweenSlices'; 'FrameOfReferenceUID'; ...
            'HorizFrameOfRef';      'StudyID'; ...
            'SeriesNumber';         'SeriesDescription'; ...
                                    'EpiflexVersion'; ...
            'AcquisitionCode';      'AcquisitionName'; ...
            'fMRIParadigmNumber';   'FlipAngle'; ...
            'FileName';             'Xdim'; ... 
            'Ydim';                 'NumberOfSlices'; ... 
            'NumberOfVolumes';      'nPassReps'; ...
            'PixelSizeX';           'PixelSizeY';  ...
            'PixelSizeZ'; ...
            };
    
    unwantedHdr             = rmfield(epiflexHdrCell{ii}, pInfoList);
    epiflexPinfoHdrCell{ii} = rmfield(epiflexHdrCell{ii}, fieldnames(unwantedHdr));
    
    
    %
    % Translate appropriate fields
    %
    epiflexPinfoHdrCell{ii}.PlaneType       = wfu_translate(epiflexPinfoHdrCell{ii}.PlaneType,'plane');
    epiflexPinfoHdrCell{ii}.PatientEntry    = wfu_translate(epiflexPinfoHdrCell{ii}.PatientEntry,'entry');
    epiflexPinfoHdrCell{ii}.PatientPosition = wfu_translate(epiflexPinfoHdrCell{ii}.PatientPosition,'position');

    %
    % Calculate the number of Volumes in fMRI
    %
    
    if ( (epiflexPinfoHdrCell{ii}.fMRIParadigmNumber > 0) && (interpretFlag == true))
        
        %
        % Calculate DDA
        %

        TR      = epiflexPinfoHdrCell{ii}.RepetitionTime;
        ddaTime = 20000;
        nDDA    = floor(ddaTime/TR);

        nBOLD   = epiflexPinfoHdrCell{ii}.NumberOfVolumes - nDDA;

        %
        % If PASL fMRI cut number of volumes in half
        %

        if strcmp(epiflexPinfoHdrCell{ii}.AcquisitionName,'pasl')

            nBOLD = floor(nBOLD/2);       % Divide by two to account for C+L=BOLD
            nBOLD = nBOLD - mod(nBOLD,1); % Force to be even

            epiflexPinfoHdrCell{ii}.RepetitionTime  =  2*epiflexPinfoHdrCell{ii}.RepetitionTime;
        end

        epiflexPinfoHdrCell{ii}.NumberOfVolumes = nBOLD;

    end

    %
    %
    %
    
    epiflexPinfoHdrCell{ii}.State = reconState(ii);
    
    %
    % Write Info File
    %
    
    fileName     = wfu_get_spm_filename(epiflexHdrCell{ii}.FileName,'base');
    infoFileName = strcat(fileName,'.info');
    
    wfu_infowrite_pfile( infoFileName,  fileName, epiflexPinfoHdrCell{ii});
    
end


%
% Write Output
%

if nargout > 0
    if nFileNames == 1
        varargout{1} = epiflexPinfoHdrCell{1};
    else
        varargout{1} = epiflexPinfoHdrCell;
    end
end

if nargout > 1
    if nFileNames == 1
        varargout{2} = epiflexHdrCell{1};
    else
        varargout{2} = epiflexHdrCell;
    end
end