function wfu_postprocess(fileOrDirNames, controlFlag, options)

%
% PURPOSE: - Post process GE raw data P files and Analyze files that have 
%            already been reconstructed.  This function is a wrapper 
%            function, which calls wfu_postprocess_pfiles and wfu_postprocess_analyze.
%
% CATEGORY: Post Processing 
%
% INPUTS:
%
%    fileOrDirNames  - Names of the P files, Analyze files, or directories to
%                      be reconstructed. This function is called
%                      recursively until all files and directories listed are 
%                      post processed.  
%
%                      This function reconstructs all directories first
%                      regardless of thier order in the list, before 
%                      post processing individual files in the list.
%
%    controlFlag     - controlFlag is a 1x4 boolean array to determine
%                      which portions of post processing will be run.
%
%               controlFlag(1) = Post process analyze images
%               controlFlag(2) = Post process p files
%               controlFlag(3) = Send data to PACS
%               controlFlag(4) = Convert all Mat files to Matlab 6.5 format
%
%               controlFlag(3) and controlFlag(4) require Matlab 7 or higher. Therfore
%               these control flags are forced to false when Matlab 6.n or 
%               lower is being run.
%
%    options         - A set of options to pass into the function. These
%                      options are passed to lower level functions to
%                      control their behaviour.
%
%          options.pacsServer - Contains the name of the pacs server to
%                               send the data to.
%
% OUTPUTS: 
%
%    None
%
% EXAMPLE:
%
%
%   To send images to the PACS during post processing must set the
%   controlFlag(3) to true.
%   
%   >> postprocess([], [1 1 1 1])  
%
%==========================================================================
% C H A N G E   L O G
%
%--------------------------------------------------------------------------
%	2008-11-26 BCW	Added Dircon


%
% Set input variables
%

v = ver('Matlab');
versionMatlab = str2num(v.Version(1));


%
% Initializing input parameters
%


if nargin < 2  || isempty(controlFlag)
   controlFlag = [ true true false true ];  % [ Analyze pFiles send convert ]
end

analyzePostProcessFlag    =  controlFlag(1);
pFilesPostProcessFlag     =  controlFlag(2); 
sendToPacsFlag            =  controlFlag(3) && (versionMatlab >= 6);
convertMatPostProcessFlag =  controlFlag(4) && (versionMatlab >= 7);


% Set function options


if sendToPacsFlag == true
     defaultOptions.pacsServer   = 'wfm3';
else
     defaultOptions.pacsServer = '';
end

if nargin < 3 
   options = [];  
end

options = wfu_set_function_options(options, defaultOptions);


% Use current directory if input directory is empty

if nargin < 1 || isempty(fileOrDirNames)
    fileOrDirNames = pwd;  % Grab current directory
end


dircon = wfu_directory_configuration;   % dircon.anatdir setting    

%
% Post Process Analyze files
%

if ~iscell(fileOrDirNames)
    fileOrDirNames = {fileOrDirNames};
end



nFileNames = length(fileOrDirNames);


if analyzePostProcessFlag == true

    disp(sprintf('\n*** Post processing %s files ... \n',dircon.anatdir)); % analyze or nifti

    for ii=1:nFileNames

        if isdir(fileOrDirNames{ii})

            findOptions.displayFiles = false;

            %
            % Search for Analyze images in specified directory
            %   

            aFiles1 = wfu_find_files( '.*.img', fileOrDirNames{ii}, 0, findOptions);

            if ~isempty(aFiles1)
                wfu_postprocess_analyze(aFiles1, options);
            end
            
            %
            % Search for Analyze/Nifti images in analyze/Nifti subdirectory for post
            % processing.
            %   
            
            aFiles2 = wfu_find_files( '.*.img', strcat(fileOrDirNames{ii}, ...
                filesep, dircon.anatdir), 0, findOptions);

            if ~isempty(aFiles2)
                wfu_postprocess_analyze(aFiles2, options);
            end
            
        else

            wfu_postprocess_analyze(fileOrDirNames{ii}, options);
        end
    end
end


%
% Post process P files
%

if pFilesPostProcessFlag == true
    

    disp(sprintf('\n*** Post processing P files ... \n'));
    
    for ii=1:nFileNames

        if isdir(fileOrDirNames{ii})
            pFiles = wfu_find_pfiles(fileOrDirNames{ii});
        else
            pFiles  = fileOrDirNames{ii};
        end

        wfu_postprocess_pfiles(pFiles, options);
    end

end

%
% Convert all mat files to Matlab 6.5
%


if (convertMatPostProcessFlag == true)

    disp(sprintf('\n*** Convert all mat files to 6.5 ... \n'));

    for ii=1:nFileNames
        wfu_convert_mat(fileOrDirNames{ii}, true, true);
    end
    
else

    disp(sprintf('\n*** Skipping conversion of all mat files to 6.5 ... \n'));

end
