function wfu_postprocess_analyze(fileOrDirNames, options)

%
% PURPOSE: - To post process GE raw data P files that have already been
%            reconstructed.  This function is a wrapper function. It
%            determines the type of P file then calls the lower level 
%            post processing function.
%
% CATEGORY: Post Processing 
%
% INPUTS:
%
%    fileOrDirNames  - Names of the P files or directories to
%                      be reconstructed. This function is called
%                      recursively until all the P files and
%                      directories with P files are reconstructed.  
%
%                      This function reconstructs all directories first
%                      regardless of thier order in the list, before 
%                      reconstructing individual files in the list.
%
% OUTPUTS: 
%
%    None
%
% EXAMPLE:
%
%
%==========================================================================
% C H A N G E   L O G
%
%--------------------------------------------------------------------------

%
% Internal Variable
%


if nargin < 1 || isempty(fileOrDirNames)
    % Assume current directory 
    findOptions.displayFiles = false;     
    fileOrDirNames = wfu_find_files('.*.img',[], false, findOptions);
end

if ~iscell(fileOrDirNames)
    fileOrDirNames = {fileOrDirNames};
end


defaultOptions.timeLimit = 3; % hours

if nargin < 2 
   options = [];
end
   
options = wfu_set_function_options(options,defaultOptions);
   

%
% Loop over fileOrDirNames and sort fileNames and Directories
% 
nFileOrDirNames = length(fileOrDirNames);

fileNames = {};
iiFileCounter = 1;

for ii=1:nFileOrDirNames
    
    if isdir(fileOrDirNames{ii})
        
        % Find and segment T1 weighted image
        wfu_segment_t1weighted_image(fileOrDirNames{ii});
        
        % Search for all analyze files in the directory to be processed
        fileNames = wfu_lsa;        

        % Post process each analyze file found
        wfu_postprocess_analyze(fileNames, options );
        
    else        
        fileNames{iiFileCounter} = fileOrDirNames{ii};
        iiFileCounter = iiFileCounter + 1;
    end
    
end

%
% Loop over fileNames and reconstruct each P-file if known
%

nFileNames = length(fileNames);

tic
t0 = toc;

while (t0 < 60*60*options.timeLimit) && nFileNames > 0  % Post process data for three hours before aborting or all
                                                        % file names have been processed

    jj=1;
    postProcessedFileNames = {};
    
    for ii=1:length(fileNames)

        absoluteFileName = fileNames{ii};

        %
        % Post processing P files
        %

        [fileDirectory fileNameBase fileNameExtension ] = fileparts(absoluteFileName);
        fileName = strcat(fileNameBase, fileNameExtension);

        hdr       = wfu_get_mri_info(absoluteFileName);
%disp([hdr.FileName ' ( ' absoluteFileName ' ) paradigm is ' hdr.fMRIParadigmNumber]);
        infoState = 1; % wfu_get_info_state(absoluteFileName);

        if (infoState == 1) && ~isempty(hdr)  % Reconstruction completed successfully

            [directory, psdNameInPFile, extension] = fileparts( hdr.PulseSeqName );

            if wfu_is_epiflex(absoluteFileName)

                %disp(sprintf('\tPost processing EPIFLEX Image File %s',absoluteFileName));
                wfu_postprocess_epiflex(absoluteFileName, [], options);

            elseif wfu_is_epi(absoluteFileName)

                wfu_postprocess_epi(absoluteFileName, [], options);

            elseif any(strfind(  lower(psdNameInPFile), lower('spiral') ))

                %disp(sprintf('Post processing spiral data'));

            elseif any(strfind(  lower(psdNameInPFile), lower('fgre') ))

                %disp(sprintf('Post processing fgre data'));

            else

                % disp(sprintf('\tSkipping post processing of unknown P file %s (%d/%d)', fileName,ii,nFileOrDirNames));
            end

            postProcessedFileNames{jj} = absoluteFileName;
            jj= jj+1;
    
        else
            
            disp(sprintf('\t !!! No DICOM file found. Skipping post processing of %s', absoluteFileName));

            postProcessedFileNames{jj} = absoluteFileName;
            jj= jj+1;
            
        end
    end % of for loop


    %
    % Figure out which P files have been processed
    %
    
    kk = 1;
    newFileNames = [];
    
    for ii=1:length(fileNames)
        tryAgainFlag = true;        
        
        jj=1;
        
        while (tryAgainFlag == true) && ( jj<=length(postProcessedFileNames))
             tryAgainFlag = ~((tryAgainFlag==true) && strcmp(fileNames{ii}, postProcessedFileNames{jj}));
             jj = jj + 1;
        end
    
        if tryAgainFlag == true
            newFileNames{kk} = fileNames{ii};
            kk = kk + 1;
        end
    end

    fileNames = newFileNames;
    t0 = toc;
    nFileNames = length(fileNames);
        
end   % of while loop

%
% Return error message if time limit has expired.
%

if (t0 > 60*60*options.timeLimit)
    disp(sprintf('\n\n*** Post processing of P files aborted. Time limit of %f hours has expired\n\n', options.timeLimit));    
end


    
end
