function wfu_postprocess_pfiles(fileOrDirNames, options)

%
% PURPOSE: - To post process GE raw data P files that have already been
%            reconstructed.  This function is a wrapper function. It
%            determines the type of P file then calls the lower level 
%            post processing function.
%
% CATEGORY: Post Processing 
%
% INPUTS:
%
%    fileOrDirNames  - Names of the P files or directories to
%                      be reconstructed. This function is called
%                      recursively until all the P files and
%                      directories with P files are reconstructed.  
%
%                      This function reconstructs all directories first
%                      regardless of thier order in the list, before 
%                      reconstructing individual files in the list.
%
% OUTPUTS: 
%
%    None
%
% EXAMPLE:
%
%
%==========================================================================
% C H A N G E   L O G
%
%--------------------------------------------------------------------------

%
% Internal Variable
%


findOptions.displayFiles = false; 
    

if nargin < 1 || isempty(fileOrDirNames)
    % Assume current directory 
    fileOrDirNames = wfu_find_pfiles([],[], findOptions);
end

if ~iscell(fileOrDirNames)
    fileOrDirNames = {fileOrDirNames};
end

defaultOptions.timeLimit = 3; % hours

if nargin < 2 
   options = [];
end
   
options = wfu_set_function_options(options,defaultOptions);


%
% Loop over fileOrDirNames and sort fileNames and Directories
% 

nFileOrDirNames = length(fileOrDirNames);

fileNames = {};
iiFileCounter = 1;

for ii=1:nFileOrDirNames
    
    if isdir(fileOrDirNames{ii})
        
        %
        % Recursively call directories to be reconstructed
        %
        
        dirFileNames = wfu_find_pfiles(fileOrDirNames{ii}, false, findOptions );
        wfu_postprocess_pfiles(dirFileNames, options)        
        
    else   
        
        if wfu_is_p_file(fileOrDirNames{ii})
            fileNames{iiFileCounter} = fileOrDirNames{ii};
            iiFileCounter = iiFileCounter + 1;
        end

    end
end


%
% Loop over fileNames and reconstruct each P-file if known
%

nFileNames = length(fileNames);

tic
t0 = toc;

while (t0 < 60*60*options.timeLimit) && nFileNames > 0  % Post process data for three hours before aborting or all
                                        % file names have been processed

    jj=1;
    postProcessedFileNames = {};
    
    for ii=1:length(fileNames)

        absoluteFileName = fileNames{ii};

        %
        % Post processing P files
        %

        [fileDirectory fileNameBase fileNameExtension ] = fileparts(absoluteFileName);
        fileName = strcat(fileNameBase, fileNameExtension);

        hdr       = wfu_get_p_header(absoluteFileName);
        infoState = wfu_get_info_state(absoluteFileName);

        if infoState == 1  % Reconstruction completed successfully

            [directory, psdNameInPFile, extension] = fileparts( hdr.PulseSeqName );

            if wfu_is_epiflex_pfile(absoluteFileName)

                % disp(sprintf('Post processing EPIFLEX  data'));
                wfu_postprocess_epiflex(absoluteFileName, [], options);

            elseif any(strfind(  lower(psdNameInPFile), lower('spiral') ))

                %disp(sprintf('Post processing spiral data'));

            elseif any(strfind(  lower(psdNameInPFile), lower('fgre') ))

                %disp(sprintf('Post processing fgre data'));

            else

                % disp(sprintf('\tSkipping post processing of unknown P file %s (%d/%d)', fileName,ii,nFileOrDirNames));
            end

            postProcessedFileNames{jj} = absoluteFileName;
            jj= jj+1;
    
        elseif infoState == 0

            disp(sprintf('\t!!! Reconstruction in progress.  Postponing post processing of P file %s for 60 seconds', absoluteFileName));
            pause(60);  % Wait 60 seconds before trying again.  This will cut 
                        % down on the number of times this error message is
                        % displayed.

        else
            
            disp(sprintf('\t !!! Reconstruction error.  Skipping post processing of P file %s', absoluteFileName));

            postProcessedFileNames{jj} = absoluteFileName;
            jj= jj+1;
            
        end
    end % of for loop


    %
    % Figure out which P files have been processed
    %
    
    kk = 1;
    newFileNames = [];
    
    for ii=1:length(fileNames)
        tryAgainFlag = true;        
        
        jj=1;
        
        while (tryAgainFlag == true) && ( jj<=length(postProcessedFileNames))
             tryAgainFlag = ~((tryAgainFlag==true) && strcmp(fileNames{ii}, postProcessedFileNames{jj}));
             jj = jj + 1;
        end
    
        if tryAgainFlag == true
            newFileNames{kk} = fileNames{ii};
            kk = kk + 1;
        end
    end

    fileNames = newFileNames;
    t0 = toc;
    nFileNames = length(fileNames);
        
end   % of while loop

%
% Return error message if time limit has expired.
%

if (t0 > 60*60*options.timeLimit)
    disp(sprintf('\n\n*** Post processing of P files aborted. Time limit of %f hours has expired\n\n', options.timeLimit));    
end


